/*
 * dwepdump v0.2 [dwepdump.c]
 * by h1kari - (c) Dachb0den Labs 2002
 */

/*
 * Copyright (c) 2002 Dachb0den Labs.
 *      David Hulton <h1kari@dachb0den.com>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by David Hulton.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY David Hulton AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL David Hulton OR THE VOICES IN HIS HEAD
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <signal.h>
#include <sys/types.h>
#include <errno.h>

#include <pcap.h>
#include <bat/common.h>

/*
 * definitions
 */
#define PCAP_SNAPLEN  2304
#define PCAP_PROMISC  0
#define PCAP_TOMS     256
#define IV_OFF        60
#define PROBABLE      60
#define LOW_KEY_SIZE  5
#define HIGH_KEY_SIZE 13
#define WEP_KEY_SIZE  (strong ? HIGH_KEY_SIZE : LOW_KEY_SIZE)

#define BANNER \
 "* dwepdump v0.2 by h1kari <h1kari@dachb0den.com> *\n" \
 "* Copyright (c) Dachb0den Labs 2002 [http://dachb0den.com] *\n"

/*
 * macros
 */
#ifdef __FreeBSD__
#define le16toh(x) (x)
#else
#ifdef __OpenBSD__
#define le16toh(x) letoh16(x)
#endif
#endif

#ifndef MAX
#define MAX(x, y) ((x) > (y) ? (x) : (y))
#endif
#ifndef AMOD
#define AMOD(x, y) ((x) % (y) < 0 ? ((x) % (y)) + (y) : (x) % (y))
#endif

/*
 * declarations
 */
static int weakkey, strong, all, maxsize, nodevice, infiles, cnt, filecnt;
static pcap_t *p;
static pcap_dumper_t *dump;

#define SAMPLES_KEY_LEN 3
static u_int samples_len[SAMPLES_KEY_LEN];

extern char *optarg;
extern int optind;

/*
 * update the line of text with progress info
 */
void
updateprog(void)
{
  int i;
  putchar('\r');

  if(weakkey)
  {
    for(i = 0; i < SAMPLES_KEY_LEN; i++)
      printf("%01x%c%02x ", i,
       samples_len[i] >= PROBABLE ? ':' : '.', samples_len[i]);
    printf(":: %d", cnt);
    fflush(stdout);
    return;
  }

  printf("packets received: %d", cnt);
  fflush(stdout);
}

/*
 * if we receive packets from pcap_loop, make sure they're data packets before
 * logging them to the pcap logfile.
 */
#define N 256
void
handler(u_char *dump, const struct pcap_pkthdr *h, const u_char *pkt)
{
  static time_t last = -1;

  int x, y, z, a, b, B;
  u_char *iv;
  struct wi_rx_frame *wi_h;
  time_t this;

  if(last == -1 && !infiles)
  {
    last = time(NULL);
    updateprog();
  }

  wi_h = (struct wi_rx_frame *)pkt;

  if(maxsize && le16toh(wi_h->wi_dat_len) > maxsize)
    return;

  if(!(le16toh(wi_h->wi_status) & WI_STAT_BADCRC) &&
   (le16toh(wi_h->wi_frame_ctl) & WI_FCTL_FTYPE) == WI_FTYPE_DATA)
  {
    cnt++; filecnt++;

    this = time(NULL);
    if(this > last && !infiles)
    {
      updateprog();
      last = this;
    }

    if(weakkey)
    {
      /*
       * we're gonna be logging every packet, but make sure we update stats
       * for the first key, because that is the most important one (and
       * predictable one) for gathering statistics. Use the standard equation
       * as illustrated in "Weaknesses in the Key Scheduling Algorithm of RC4"
       * and the one illustrated in
       * "Attacking additional RC4 PRGA bytes in a WEP environment".
       */
      iv = (u_char *)pkt + IV_OFF;

      x = iv[0];
      y = iv[1];
      z = iv[2];

      a = (x + y) % N;
      b = AMOD((x + y) - z, N);

      for(B = 0; B < WEP_KEY_SIZE; B++)
      {
        if((((0 <= a && a < B) ||
         (a == B && b == (B + 1) * 2)) &&
         (B % 2 ? a != (B + 1) / 2 : 1)) ||
         (a == B + 1 && (B == 0 ? b == (B + 1) * 2 : 1)) ||
         (x == B + 3 && y == N - 1) ||
         (B != 0 && !(B % 2) ? (x == 1 && y == (B / 2) + 1) ||
         (x == (B / 2) + 2 && y == (N - 1) - x) : 0))
        {
          if(!all)
          {
            pcap_dump((u_char *)dump, h, pkt);
            fflush((FILE *)dump);
          }

          if(B < SAMPLES_KEY_LEN)
            samples_len[B]++;
        }
      }
    }

    if(!weakkey || all)
    {
      pcap_dump((u_char *)dump, h, pkt);
      fflush((FILE *)dump);
    }
  }
}

/*
 * make sure we exit cleanly on kill or ctrl+c
 */
void
exit_handler(int val)
{
  printf("\n\nstopping pcap capture loop...\n\n");

  fflush(stdout);
  fflush((FILE *)dump);

  pcap_close(p);
  pcap_dump_close(dump);

  exit(0);
}

/*
 * print program usage
 */
void
usage(char *progname)
{
  printf("usage: %s [-w [-s | -a]] [-m <max>] [-d | <device>] "
   "[<in> [...]] <logfile>\n"
   " -w: log weak keys to logfile for use with dwepcrack -w (40-bit default)\n"
   " -s: log packets that use 104-bit encryption\n"
   " -a: log all packets intead of just ones with weak ivs\n"
   " -m: maximum size packet to allow through\n"
   " -d: specify that the device isn't specifed, just join the in files\n",
   progname);
  exit(2);
}

/*
 * open device and log all data packets to file
 */
int
main(int argc, char *argv[])
{
  int c, i;
  char *device, *logfile, errbuf[PCAP_ERRBUF_SIZE], *progname = argv[0];
  pcap_t *tp;

  weakkey = 0;
  strong = 0;
  all = 0;
  maxsize = 0;
  nodevice = 0;
  infiles = 0;

  while((c = getopt(argc, argv, "wsm:d")) != -1)
  {
    switch(c)
    {
      case 'w':
        weakkey = 1;
        break;
      case 's':
        strong = 1;
        break;
      case 'a':
        all = 1;
      case 'm':
        maxsize = MAX(atoi(optarg), 1);
        break;
      case 'd':
        nodevice = 1;
        break;
      default:
        break;
    }
  }

  argc -= optind;
  argv += optind;

  if(argc < 2)
    usage(progname);

  logfile = argv[1];

  printf("\n%s\n", BANNER);
  printf("starting pcap capture loop...\n");

  if(!nodevice)
  {
    device = argv[0]; argv++; argc--;
    printf(" device: %s\n", device);
    if((p = pcap_open_live(device, PCAP_SNAPLEN, PCAP_PROMISC, PCAP_TOMS,
     errbuf)) == NULL)
    {
      fprintf(stderr, "error: unable to open device for packet capture: %s\n",
       errbuf);
      exit(2);
    }

    logfile = argv[argc - 1];
    printf(" logfile: %s\n", logfile);
    if((dump = pcap_dump_open(p, logfile)) == NULL)
    {
      fprintf(stderr, "error: unable to open pcap logfile: %s\n",
       pcap_geterr(p));
      exit(2);
    }
  }

  if(weakkey)
    memset(samples_len, 0, SAMPLES_KEY_LEN * sizeof(u_int));

  cnt = 0;
  signal(SIGINT, exit_handler);

  for(i = 0; i < argc - 1; i++)
  {
    infiles++;

    if((tp = pcap_open_offline(argv[i], errbuf)) == NULL)
    {
      fprintf(stderr, "error: unable to open pcap file for reading: %s\n",
       errbuf);
      exit(2);
    }

    if(i == 0 && nodevice)
    {
      logfile = argv[argc - 1];
      printf(" logfile: %s\n\n", logfile);
      if((dump = pcap_dump_open(tp, logfile)) == NULL)
      {
        fprintf(stderr, "error: unable to open pcap logfile: %s\n",
         pcap_geterr(tp));
        exit(2);
      }
    }

    filecnt = 0;
    pcap_dispatch(tp, 0, handler, (u_char *)dump);
    printf(" infile #%d: %s (%d)\n", i + 1, argv[i], filecnt);

    pcap_close(tp);
  }
  infiles = 0;

  putchar('\n');

  if(!nodevice)
    pcap_loop(p, -1, handler, (u_char *)dump);
  else
  {
    updateprog();
    printf("\n\n");
  }

  exit(0);
}
