/* 
 *	Copyright (C) Chia-chen Kuo - April 2001
 *
 *  This file is part of DVD2AVI, a free MPEG-2 decoder
 *	
 *  DVD2AVI is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  DVD2AVI is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include "global.h"

#define MAX_AVI_SIZE	2073600000

__forceinline static void Store_RGB24(unsigned char *src[], DWORD frame);
__forceinline static void Store_YUY2(unsigned char *src[], DWORD frame);

static void Luminance_Filter(unsigned char *src, unsigned char *dst);
static void conv420to422(unsigned char *src, unsigned char *dst);
static void conv422to444(unsigned char *src, unsigned char *dst);
static void conv444toRGB24odd(unsigned char *py, unsigned char *pu, unsigned char *pv, unsigned char *dst);
static void conv444toRGB24even(unsigned char *py, unsigned char *pu, unsigned char *pv, unsigned char *dst);
static void Flush_RGB24(void);
static void conv422toyuy2odd(unsigned char *py, unsigned char *pu, unsigned char *pv, unsigned char *dst);
static void conv422toyuy2even(unsigned char *py, unsigned char *pu, unsigned char *pv, unsigned char *dst);
static void Flush_YUY2(void);
static void AVIKill(void);

static int gcd(int x, int y);

PAVIFILE pfile;
PAVISTREAM ps, psCompressed;
AVICOMPRESSOPTIONS opts;
LPAVICOMPRESSOPTIONS lpopts = &opts;
AVISTREAMINFO strinfo;
ICCOMPRESSFRAMES iccf;
COMPVARS compvars;
PCOMPVARS pcompvars = &compvars;

BITMAPINFOHEADER birgb;
LPBITMAPINFOHEADER lpbirgb = &birgb;

BITMAPINFOHEADER biyuv;
LPBITMAPINFOHEADER lpbiyuv = &biyuv;

static char VideoOut[_MAX_PATH];
static unsigned char *y444;
static int AVI_Init, avi_size, avi_count, frame_count;
static int TFF, RFF, TFB, BFB, playback, Old_Playback, frame_size, frame_type;

static int Resize_Flag;
static int Clip_Width, Clip_Height, Resize_Width, Resize_Height;
static int DOUBLE_WIDTH, HALF_WIDTH, LUM_AREA, PROGRESSIVE_HEIGHT, INTERLACED_HEIGHT;
static int HALF_WIDTH_D8, RGB_DOWN1, RGB_DOWN2;
static int NINE_CLIP_WIDTH, QUAD_CLIP_WIDTH, DOUBLE_CLIP_WIDTH, HALF_CLIP_WIDTH;
static int CLIP_AREA, HALF_CLIP_AREA, CLIP_STEP, CLIP_HALF_STEP;

__int64 LumOffsetMask, LumGainMask;

static char *FrameType[3] = {
	"", "Interlaced", "Progressive"
};

static const __int64 mmmask_0001 = 0x0001000100010001;
static const __int64 mmmask_0002 = 0x0002000200020002;
static const __int64 mmmask_0003 = 0x0003000300030003;
static const __int64 mmmask_0004 = 0x0004000400040004;
static const __int64 mmmask_0005 = 0x0005000500050005;
static const __int64 mmmask_0007 = 0x0007000700070007;
static const __int64 mmmask_0064 = 0x0040004000400040;
static const __int64 mmmask_0128 = 0x0080008000800080;

void Write_Frame(unsigned char *src[], D2VData d2v, DWORD frame)
{
	int repeat;

	if (Fault_Flag)
	{
		if (Fault_Flag < CRITICAL_ERROR_LEVEL)
		{
			SetDlgItemText(hDlg, IDC_INFO, "V.E.!");
			Fault_Flag = 0;		// fault tolerance
		}
		else
		{
			if (AVI_Flag)
				AVIKill();
			ThreadKill();
		}
	}

	frame_type = d2v.pf;
	TFF = d2v.trf>>1;
	RFF = d2v.trf & 0x01;

	if (!frame)
	{
		char *ext, szTemp[_MAX_PATH];
		int i;

		AVI_Init = 1; TFB = BFB = 0; Old_Playback = 0;
		frame_size = 0; avi_size = 0; avi_count = 1; frame_count = 0; playback = 0;

		Resize_Flag = 0;
		Clip_Width = Resize_Width = Coded_Picture_Width;
		Clip_Height = Resize_Height = Coded_Picture_Height;
		CLIP_AREA = HALF_CLIP_AREA = CLIP_STEP = CLIP_HALF_STEP = 0;

		if (ClipResize_Flag)
		{
			if (Clip_Top || Clip_Bottom || Clip_Left || Clip_Right)
			{
				Clip_Width -= Clip_Left+Clip_Right;
				Clip_Height -= Clip_Top+Clip_Bottom;
				Resize_Width = Clip_Width;
				Resize_Height = Clip_Height;

				CLIP_AREA = Coded_Picture_Width * Clip_Top;
				HALF_CLIP_AREA = (Coded_Picture_Width>>1) * Clip_Top;

				CLIP_STEP = Coded_Picture_Width * Clip_Top + Clip_Left;
				CLIP_HALF_STEP = (Coded_Picture_Width>>1) * Clip_Top + (Clip_Left>>1);
			}

			if (Squeeze_Width || Squeeze_Height)
			{
				Resize_Flag = 1;
				Resize_Width -= Squeeze_Width;
				Resize_Height -= Squeeze_Height;
			}
		}

		NINE_CLIP_WIDTH = Clip_Width * 9;
		QUAD_CLIP_WIDTH = Clip_Width<<2;
		DOUBLE_CLIP_WIDTH = Clip_Width<<1;
		HALF_CLIP_WIDTH = Clip_Width>>1;

		LUM_AREA = Coded_Picture_Width * Clip_Height;
		DOUBLE_WIDTH = Coded_Picture_Width<<1;
		HALF_WIDTH = Coded_Picture_Width>>1;
		HALF_WIDTH_D8 = (Coded_Picture_Width>>1) - 8;
		PROGRESSIVE_HEIGHT = (Coded_Picture_Height>>1) - 2;
		INTERLACED_HEIGHT = (Coded_Picture_Height>>2) - 2;
		RGB_DOWN1 = Clip_Width * (Clip_Height - 1) * 3;
		RGB_DOWN2 = Clip_Width * (Clip_Height - 2) * 3;

		LumGainMask = ((__int64)LumGain<<48) + ((__int64)LumGain<<32) + ((__int64)LumGain<<16) + (__int64)LumGain;
		LumOffsetMask = ((__int64)LumOffset<<48) + ((__int64)LumOffset<<32) + ((__int64)LumOffset<<16) + (__int64)LumOffset;

		sprintf(szBuffer, "DVD2AVI - [%d / %d] ", File_Flag+1, File_Limit);
		ext = strrchr(Infilename[File_Flag], '\\');
		strncat(szBuffer, ext+1, strlen(Infilename[0])-(int)(ext-Infilename[0]));

		sprintf(szTemp, " %d x %d", Resize_Width, Resize_Height);
		strcat(szBuffer, szTemp);

		if (Resize_Width%32 == 0)
			strcat(szBuffer, " [32X]");
		else if (Resize_Width%16 == 0)
			strcat(szBuffer, " [16X]");
		else if (Resize_Width%8 == 0)
			strcat(szBuffer, " [8X]");
		else
			strcat(szBuffer, " [ ]");

		if (Resize_Height%32 == 0)
			strcat(szBuffer, "[32X]");
		else if (Resize_Height%16 == 0)
			strcat(szBuffer, "[16X]");
		else if (Resize_Height%8 == 0)
			strcat(szBuffer, "[8X]");
		else
			strcat(szBuffer, "[ ]");

		SetWindowText(hWnd, szBuffer);

		ResizeWindow(Clip_Width, Clip_Height);

		ZeroMemory(&birgb, sizeof(BITMAPINFOHEADER));
		birgb.biSize = sizeof(BITMAPINFOHEADER);
		birgb.biWidth = Clip_Width;
		birgb.biHeight = Clip_Height;
		birgb.biPlanes = 1;
		birgb.biBitCount = 24;
		birgb.biCompression = BI_RGB;
		birgb.biSizeImage = Clip_Width * Clip_Height * 3;

		ZeroMemory(&biyuv, sizeof(BITMAPINFOHEADER));
		biyuv = birgb;
		biyuv.biBitCount = 16;
		biyuv.biCompression = mmioFOURCC('Y','U','Y','2');
		biyuv.biSizeImage = Clip_Width * Clip_Height * 2;

		if (process.locate==LOCATE_RIP && FO_Flag==FO_SWAP)
		{
			if (TFF)
				BFB = 1;
			else
				TFB = 1;

			// mask
			ZeroMemory(rgb24, Coded_Picture_Width * Clip_Height * 3);
			for (i=0; i<Coded_Picture_Width * Coded_Picture_Height * 2; i+=2)
			{
				yuy2[i] = 0;
				yuy2[i+1] = 128;
			}
		}

		if (FO_Flag!=FO_FILM)
		{
			if ((TFF && FO_Flag!=FO_SWAP) || (!TFF && FO_Flag==FO_SWAP))
				SetDlgItemText(hDlg, IDC_INFO, "T");
			else
				SetDlgItemText(hDlg, IDC_INFO, "B");
		}
	}

	repeat = DetectVideoType(frame, d2v.trf);

	if (FO_Flag!=FO_FILM || repeat)
	{
		if (Store_Flag==STORE_YUY2)
			Store_YUY2(src, frame);
		else
			Store_RGB24(src, frame);
	}

	if (FO_Flag==FO_FILM && repeat==2)
	{
		if (Store_Flag==STORE_YUY2)
			Store_YUY2(src, frame);
		else
			Store_RGB24(src, frame);
	}

	if (Statistics_Flag && process.locate==LOCATE_RIP)
	{
		sprintf(szBuffer, "%s", FrameType[frame_type+1]);
		SetDlgItemText(hDlg, IDC_FRAME_TYPE, szBuffer);

		sprintf(szBuffer, "%d", frame+1);
		SetDlgItemText(hDlg, IDC_CODED_NUMBER, szBuffer);

		sprintf(szBuffer, "%d", playback);
		SetDlgItemText(hDlg, IDC_PLAYBACK_NUMBER, szBuffer);

		if (AVI_Flag)
		{
			sprintf(szBuffer, "%d", avi_count-1);
			SetDlgItemText(hDlg, IDC_FILE, szBuffer);
	
			sprintf(szBuffer, "%d MB", avi_size>>20);
			SetDlgItemText(hDlg, IDC_FILE_SIZE, szBuffer);
		}

		if ((frame & 63) == 63)
		{
			process.ed = timeGetTime();

			sprintf(szBuffer, "%.1f", 1000.0*(playback-Old_Playback)/(process.ed-process.mi+1));
			SetDlgItemText(hDlg, IDC_FPS, szBuffer);
			SendDlgItemMessage(hDlg, IDC_PROGRESS, PBM_SETPOS, (int)(Bitrate_Meter*Frame_Rate/(playback-Old_Playback))>>7, 0);

			Bitrate_Meter = 0;
			process.mi = process.ed;
			Old_Playback = playback;
		}
	}

	if (AVI_Flag && avi_size>=MAX_AVI_SIZE)
		AVIKill();
}

static void Store_RGB24(unsigned char *src[], DWORD frame)
{
	// store AVI
	if (AVI_Flag && AVI_Init)
	{
		if (!frame)
		{
			compvars.cbSize = sizeof(compvars);

			if (!ICCompressorChoose(hWnd, ICMF_CHOOSE_DATARATE | ICMF_CHOOSE_KEYFRAME,
				lpbirgb, NULL, pcompvars, NULL))
				ThreadKill();

			// set AVI header
			ZeroMemory(&strinfo, sizeof(AVISTREAMINFO));
			strinfo.fccType					= streamtypeVIDEO;
			strinfo.fccHandler				= compvars.fccHandler;
			strinfo.dwQuality				= -1;
			strinfo.dwSuggestedBufferSize	= birgb.biSizeImage;
			strinfo.dwScale	= 1000;
			strinfo.dwRate = (unsigned int)((FO_Flag==FO_FILM) ? frame_rate*800 : frame_rate*1000);

			// set AVI save options
			opts.cbFormat = sizeof(birgb);
			opts.fccType = streamtypeVIDEO;
			opts.fccHandler = compvars.fccHandler;
			opts.dwKeyFrameEvery = compvars.lKey;
			opts.dwQuality = compvars.lQ;
			opts.dwBytesPerSecond = compvars.lDataRate<<10;
			opts.dwFlags = AVICOMPRESSF_VALID | AVICOMPRESSF_KEYFRAMES | AVICOMPRESSF_DATARATE;
			opts.dwInterleaveEvery = 0;
			opts.lpFormat = lpbirgb;
			opts.cbParms = compvars.cbState;
			opts.lpParms = compvars.lpState;
		}

		AVI_Init = 0; avi_size = 0;
		AVIFileInit();

		sprintf(VideoOut, "%s.%02d.avi", szOutput, avi_count++);

		if (AVIFileOpen(&pfile, VideoOut, OF_WRITE | OF_CREATE, NULL) != AVIERR_OK)
		{
			AVIKill();
			ThreadKill();
		}

		if (AVIFileCreateStream(pfile, &ps, &strinfo) != AVIERR_OK)
		{
			AVIKill();
			ThreadKill();
		}

		if (AVIMakeCompressedStream(&psCompressed, ps, lpopts, NULL) != AVIERR_OK)
		{
			AVIKill();
			ThreadKill();
		}

		if (AVIStreamSetFormat(psCompressed, 0, lpbirgb, birgb.biSize) != AVIERR_OK)
		{
			AVIKill();
			ThreadKill();
		}
	}

	if (chroma_format==CHROMA420)
	{
		conv420to422(src[1], u422);
		conv420to422(src[2], v422);

		conv422to444(u422, u444);
		conv422to444(v422, v444);
	}
	else
	{
		conv422to444(src[1], u444);
		conv422to444(src[2], v444);
	}

	if (Luminance_Flag)
	{
		Luminance_Filter(src[0], lum);
		y444 = lum;
	}
	else
		y444 = src[0];

	if (BFB)
	{
		conv444toRGB24odd(y444, u444, v444, rgb24);
		Flush_RGB24();
		conv444toRGB24even(y444, u444, v444, rgb24);
		Flush_RGB24();
	}
	else
	{
		conv444toRGB24even(y444, u444, v444, rgb24);
		Flush_RGB24();
		conv444toRGB24odd(y444, u444, v444, rgb24);
		Flush_RGB24();
	}

	if (FO_Flag!=FO_FILM && RFF)
		if (TFF)
		{
			TFB = 1;
			Flush_RGB24();
		}
		else
		{
			BFB = 1;
			Flush_RGB24();
		}
}

static void Flush_RGB24()
{
	if (TFB & BFB)
	{
		if (AVI_Flag)
		{
			if (AVIStreamWrite(psCompressed, frame_count++, 1, rgb24,
				birgb.biSizeImage, 0, NULL, &frame_size) != AVIERR_OK)
			{
				AVIKill();
				ThreadKill();
			}

			avi_size += frame_size;
		}

		if (Display_Flag)
			RenderRGB24();

		playback++;
		TFB = BFB = 0;
	}
}

static void Store_YUY2(unsigned char *src[], DWORD frame)
{
	// Save AVI
	if (AVI_Flag && AVI_Init)
	{
		if (!frame)
		{
			compvars.cbSize = sizeof(compvars);

			if (!ICCompressorChoose(hWnd, ICMF_CHOOSE_DATARATE | ICMF_CHOOSE_KEYFRAME,
				lpbiyuv, NULL, pcompvars, NULL))
				ThreadKill();

			// set AVI header
			ZeroMemory(&strinfo, sizeof(AVISTREAMINFO));
			strinfo.fccType					= streamtypeVIDEO;
			strinfo.fccHandler				= compvars.fccHandler;
			strinfo.dwQuality				= -1;
			strinfo.dwSuggestedBufferSize	= biyuv.biSizeImage;
			strinfo.dwScale	= 1000;
			strinfo.dwRate = (unsigned int)((FO_Flag==FO_FILM) ? frame_rate*800 : frame_rate*1000);

			// set AVI save options
			opts.cbFormat = sizeof(biyuv);
			opts.fccType = streamtypeVIDEO;
			opts.fccHandler = compvars.fccHandler;
			opts.dwKeyFrameEvery = compvars.lKey;
			opts.dwQuality = compvars.lQ;
			opts.dwBytesPerSecond = compvars.lDataRate<<10;
			opts.dwFlags = AVICOMPRESSF_VALID | AVICOMPRESSF_KEYFRAMES | AVICOMPRESSF_DATARATE;
			opts.dwInterleaveEvery = 0;
			opts.lpFormat = lpbiyuv;
			opts.cbParms = compvars.cbState;
			opts.lpParms = compvars.lpState;

			iccf.dwRate = strinfo.dwRate;
			iccf.dwScale = strinfo.dwScale;
			iccf.lQuality = compvars.lQ;
			iccf.lDataRate = compvars.lDataRate<<10;
			iccf.lKeyRate = compvars.lKey;

			ICSendMessage(compvars.hic, ICM_COMPRESS_FRAMES_INFO, (WPARAM)&iccf, (DWORD)sizeof(ICCOMPRESSFRAMES));
		}

		AVI_Init = 0; avi_size = 0;
		AVIFileInit();

		sprintf(VideoOut, "%s.%02d.avi", szOutput, avi_count++);

		if (AVIFileOpen(&pfile, VideoOut, OF_WRITE | OF_CREATE, NULL) != AVIERR_OK)
		{
			AVIKill();
			ThreadKill();
		}

		if (AVIFileCreateStream(pfile, &ps, &strinfo) != AVIERR_OK)
		{
			AVIKill();
			ThreadKill();
		}

		if (AVIMakeCompressedStream(&psCompressed, ps, lpopts, NULL) != AVIERR_OK)
		{
			AVIKill();
			ThreadKill();
		}

		if (AVIStreamSetFormat(psCompressed, 0, lpbirgb, birgb.biSize) != AVIERR_OK)
		{
			AVIKill();
			ThreadKill();
		}

		if (!frame && !ICSeqCompressFrameStart(pcompvars, (LPBITMAPINFO)lpbiyuv))
		{
			AVIKill();
			ThreadKill();
		}
	}

	if (chroma_format==CHROMA420)
	{
		conv420to422(src[1], u422);
		conv420to422(src[2], v422);
	}
	else
	{
		u422 = src[1];
		v422 = src[2];
	}

	if (Luminance_Flag)
	{
		Luminance_Filter(src[0], lum);
		y444 = lum;
	}
	else
		y444 = src[0];

	if (BFB)
	{
		conv422toyuy2odd(y444, u422, v422, yuy2);
		Flush_YUY2();

		conv422toyuy2even(y444, u422, v422, yuy2);
		Flush_YUY2();
	}
	else
	{
		conv422toyuy2even(y444, u422, v422, yuy2);
		Flush_YUY2();

		conv422toyuy2odd(y444, u422, v422, yuy2);
		Flush_YUY2();
	}

	if (FO_Flag!=FO_FILM && RFF)
		if (TFF)
		{
			TFB = 1;
			Flush_YUY2();
		}
		else
		{
			BFB = 1;
			Flush_YUY2();
		}
}

static void Flush_YUY2()
{
	void *yuy2c;
	int key_flag, yuy2c_size;

	if (TFB & BFB)
	{
		if (AVI_Flag)
		{
			if ((yuy2c = ICSeqCompressFrame(pcompvars, 0, yuy2, &key_flag, &yuy2c_size)) == NULL)
			{
				AVIKill();
				ThreadKill();
			}

			if (AVIStreamWrite(ps, frame_count++, 1, yuy2c, yuy2c_size,
				key_flag ? AVIIF_KEYFRAME : 0, NULL, &frame_size) != AVIERR_OK)
			{
				AVIKill();
				ThreadKill();
			}

			avi_size += frame_size;
		}

		if (DDOverlay_Flag && Display_Flag)
			RenderYUY2();

		playback++;
		TFB = BFB = 0;
	}
}

static void Luminance_Filter(unsigned char *src, unsigned char *dst)
{
	src += CLIP_AREA;
	dst += CLIP_AREA;

	__asm
	{
		mov			eax, [src]
		mov			ebx, [dst]
		mov			esi, 0x00
		mov			edi, [LUM_AREA]
		pxor		mm0, mm0
		movq		mm5, [LumOffsetMask]
		movq		mm6, [LumGainMask]
		movq		mm7, [mmmask_0064]

lumconv:
		movq		mm1, [eax+esi]
		movq		mm2, mm1

		punpcklbw	mm1, mm0
		punpckhbw	mm2, mm0

		pmullw		mm1, mm6
		pmullw		mm2, mm6

		paddw		mm1, mm7
		paddw		mm2, mm7

		psrlw		mm1, 7
		psrlw		mm2, 7

		paddw		mm1, mm5
		paddw		mm2, mm5

		packuswb	mm1, mm0
		packuswb	mm2, mm0

		add			esi, 0x08
		cmp			esi, edi
		movq		[ebx+esi-8], mm1
		movq		[ebx+esi-4], mm2
		jl			lumconv
	}
}

static void conv422to444(unsigned char *src, unsigned char *dst)
{
	src += HALF_CLIP_AREA;
	dst += CLIP_AREA;

	__asm
	{
		mov			eax, [src]
		mov			ebx, [dst]
		mov			edi, [Clip_Height]

		movq		mm1, [mmmask_0001]
		pxor		mm0, mm0

convyuv444init:
		movq		mm7, [eax]
		mov			esi, 0x00

convyuv444:
		movq		mm2, mm7
		movq		mm7, [eax+esi+8]
		movq		mm3, mm2
		movq		mm4, mm7

		psrlq		mm3, 8
		psllq		mm4, 56
		por			mm3, mm4

		movq		mm4, mm2
		movq		mm5, mm3

		punpcklbw	mm4, mm0
		punpcklbw	mm5, mm0

		movq		mm6, mm4
		paddusw		mm4, mm1
		paddusw		mm4, mm5
		psrlw		mm4, 1
		psllq		mm4, 8
		por			mm4, mm6

		punpckhbw	mm2, mm0
		punpckhbw	mm3, mm0

		movq		mm6, mm2
		paddusw		mm2, mm1
		paddusw		mm2, mm3

		movq		[ebx+esi*2], mm4

		psrlw		mm2, 1
		psllq		mm2, 8
		por			mm2, mm6

		add			esi, 0x08
		cmp			esi, [HALF_WIDTH_D8]
		movq		[ebx+esi*2-8], mm2
		jl			convyuv444

		movq		mm2, mm7
		punpcklbw	mm2, mm0
		movq		mm3, mm2

		psllq		mm2, 8
		por			mm2, mm3

		movq		[ebx+esi*2], mm2

		punpckhbw	mm7, mm0
		movq		mm6, mm7

		psllq		mm6, 8
		por			mm6, mm7

		movq		[ebx+esi*2+8], mm6

		add			eax, [HALF_WIDTH]		
		add			ebx, [Coded_Picture_Width]
		dec			edi
		cmp			edi, 0x00
		jg			convyuv444init
	}
}

static void conv420to422(unsigned char *src, unsigned char *dst)
{
	if (frame_type)
	{
		__asm
		{
			mov			eax, [src]
			mov			ebx, [dst]
			mov			ecx, ebx
			add			ecx, [HALF_WIDTH]
			mov			esi, 0x00
			movq		mm3, [mmmask_0003]
			pxor		mm0, mm0
			movq		mm4, [mmmask_0002]

			mov			edx, eax
			add			edx, [HALF_WIDTH]
convyuv422topp:
			movd		mm1, [eax+esi]
			movd		mm2, [edx+esi]
			movd		[ebx+esi], mm1
			punpcklbw	mm1, mm0
			pmullw		mm1, mm3
			paddusw		mm1, mm4
			punpcklbw	mm2, mm0
			paddusw		mm2, mm1
			psrlw		mm2, 0x02
			packuswb	mm2, mm0

			add			esi, 0x04
			cmp			esi, [HALF_WIDTH]
			movd		[ecx+esi-4], mm2
			jl			convyuv422topp

			add			eax, [HALF_WIDTH]
			add			ebx, [Coded_Picture_Width]
			add			ecx, [Coded_Picture_Width]
			mov			esi, 0x00

			mov			edi, [PROGRESSIVE_HEIGHT]
convyuv422p:
			movd		mm1, [eax+esi]

			punpcklbw	mm1, mm0
			mov			edx, eax

			pmullw		mm1, mm3
			sub			edx, [HALF_WIDTH]

			movd		mm5, [edx+esi]
			movd		mm2, [edx+esi]

			punpcklbw	mm5, mm0
			punpcklbw	mm2, mm0
			paddusw		mm5, mm1
			paddusw		mm2, mm1
			paddusw		mm5, mm4
			paddusw		mm2, mm4
			psrlw		mm5, 0x02
			psrlw		mm2, 0x02
			packuswb	mm5, mm0
			packuswb	mm2, mm0

			mov			edx, eax
			add			edx, [HALF_WIDTH]
			add			esi, 0x04
			cmp			esi, [HALF_WIDTH]
			movd		[ebx+esi-4], mm5
			movd		[ecx+esi-4], mm2

			jl			convyuv422p

			add			eax, [HALF_WIDTH]
			add			ebx, [Coded_Picture_Width]
			add			ecx, [Coded_Picture_Width]
			mov			esi, 0x00
			dec			edi
			cmp			edi, 0x00
			jg			convyuv422p

			mov			edx, eax
			sub			edx, [HALF_WIDTH]
convyuv422bottomp:
			movd		mm1, [eax+esi]
			movd		mm5, [edx+esi]
			punpcklbw	mm5, mm0
			movd		[ecx+esi], mm1

			punpcklbw	mm1, mm0
			pmullw		mm1, mm3
			paddusw		mm5, mm1
			paddusw		mm5, mm4
			psrlw		mm5, 0x02
			packuswb	mm5, mm0

			add			esi, 0x04
			cmp			esi, [HALF_WIDTH]
			movd		[ebx+esi-4], mm5
			jl			convyuv422bottomp
		}
	}
	else
	{
		__asm
		{
			mov			eax, [src]
			mov			ecx, [dst]
			mov			esi, 0x00
			pxor		mm0, mm0
			movq		mm3, [mmmask_0003]
			movq		mm4, [mmmask_0004]
			movq		mm5, [mmmask_0005]

convyuv422topi:
			movd		mm1, [eax+esi]
			mov			ebx, eax
			add			ebx, [HALF_WIDTH]
			movd		mm2, [ebx+esi]
			movd		[ecx+esi], mm1
			punpcklbw	mm1, mm0
			movq		mm6, mm1
			pmullw		mm1, mm3

			punpcklbw	mm2, mm0
			movq		mm7, mm2
			pmullw		mm2, mm5
			paddusw		mm2, mm1
			paddusw		mm2, mm4
			psrlw		mm2, 0x03
			packuswb	mm2, mm0

			mov			edx, ecx
			add			edx, [HALF_WIDTH]
			pmullw		mm6, mm5
			movd		[edx+esi], mm2

			add			ebx, [HALF_WIDTH]
			movd		mm2, [ebx+esi]
			punpcklbw	mm2, mm0
			pmullw		mm2, mm3
			paddusw		mm2, mm6
			paddusw		mm2, mm4
			psrlw		mm2, 0x03
			packuswb	mm2, mm0

			add			edx, [HALF_WIDTH]
			add			ebx, [HALF_WIDTH]
			pmullw		mm7, [mmmask_0007]
			movd		[edx+esi], mm2

			movd		mm2, [ebx+esi]
			punpcklbw	mm2, mm0
			paddusw		mm2, mm7
			paddusw		mm2, mm4
			psrlw		mm2, 0x03
			packuswb	mm2, mm0

			add			edx, [HALF_WIDTH]
			add			esi, 0x04
			cmp			esi, [HALF_WIDTH]
			movd		[edx+esi-4], mm2

			jl			convyuv422topi

			add			eax, [Coded_Picture_Width]
			add			ecx, [DOUBLE_WIDTH]
			mov			esi, 0x00

			mov			edi, [INTERLACED_HEIGHT]
convyuv422i:
			movd		mm1, [eax+esi]
			punpcklbw	mm1, mm0
			movq		mm6, mm1
			mov			ebx, eax
			sub			ebx, [Coded_Picture_Width]
			movd		mm3, [ebx+esi]
			pmullw		mm1, [mmmask_0007]
			punpcklbw	mm3, mm0
			paddusw		mm3, mm1
			paddusw		mm3, mm4
			psrlw		mm3, 0x03
			packuswb	mm3, mm0

			add			ebx, [HALF_WIDTH]
			movq		mm1, [ebx+esi]
			add			ebx, [Coded_Picture_Width]
			movd		[ecx+esi], mm3

			movq		mm3, [mmmask_0003]
			movd		mm2, [ebx+esi]

			punpcklbw	mm1, mm0
			pmullw		mm1, mm3
			punpcklbw	mm2, mm0
			movq		mm7, mm2
			pmullw		mm2, mm5
			paddusw		mm2, mm1
			paddusw		mm2, mm4
			psrlw		mm2, 0x03
			packuswb	mm2, mm0

			pmullw		mm6, mm5
			mov			edx, ecx
			add			edx, [HALF_WIDTH]
			movd		[edx+esi], mm2

			add			ebx, [HALF_WIDTH]
			movd		mm2, [ebx+esi]
			punpcklbw	mm2, mm0
			pmullw		mm2, mm3
			paddusw		mm2, mm6
			paddusw		mm2, mm4
			psrlw		mm2, 0x03
			packuswb	mm2, mm0

			pmullw		mm7, [mmmask_0007]
			add			edx, [HALF_WIDTH]
			add			ebx, [HALF_WIDTH]
 			movd		[edx+esi], mm2

			movd		mm2, [ebx+esi]
			punpcklbw	mm2, mm0
			paddusw		mm2, mm7
			paddusw		mm2, mm4
			psrlw		mm2, 0x03
			packuswb	mm2, mm0

			add			edx, [HALF_WIDTH]
			add			esi, 0x04
			cmp			esi, [HALF_WIDTH]
			movd		[edx+esi-4], mm2

			jl			convyuv422i
			add			eax, [Coded_Picture_Width]
			add			ecx, [DOUBLE_WIDTH]
			mov			esi, 0x00
			dec			edi
			cmp			edi, 0x00
			jg			convyuv422i

convyuv422bottomi:
			movd		mm1, [eax+esi]
			movq		mm6, mm1
			punpcklbw	mm1, mm0
			mov			ebx, eax
			sub			ebx, [Coded_Picture_Width]
			movd		mm3, [ebx+esi]
			punpcklbw	mm3, mm0
			pmullw		mm1, [mmmask_0007]
			paddusw		mm3, mm1
			paddusw		mm3, mm4
			psrlw		mm3, 0x03
			packuswb	mm3, mm0

			add			ebx, [HALF_WIDTH]
			movq		mm1, [ebx+esi]
			punpcklbw	mm1, mm0
			movd		[ecx+esi], mm3

			pmullw		mm1, [mmmask_0003]
			add			ebx, [Coded_Picture_Width]
			movd		mm2, [ebx+esi]
			punpcklbw	mm2, mm0
			movq		mm7, mm2
			pmullw		mm2, mm5
			paddusw		mm2, mm1
			paddusw		mm2, mm4
			psrlw		mm2, 0x03
			packuswb	mm2, mm0

			mov			edx, ecx
			add			edx, [HALF_WIDTH]
			pmullw		mm7, [mmmask_0007]
			movd		[edx+esi], mm2

			add			edx, [HALF_WIDTH]
			movd		[edx+esi], mm6

			punpcklbw	mm6, mm0
			paddusw		mm6, mm7
			paddusw		mm6, mm4
			psrlw		mm6, 0x03
			packuswb	mm6, mm0

			add			edx, [HALF_WIDTH]
			add			esi, 0x04
			cmp			esi, [HALF_WIDTH]
			movd		[edx+esi-4], mm6

			jl			convyuv422bottomi
		}
	}
}

static void conv422toyuy2odd(unsigned char *py, unsigned char *pu, unsigned char *pv, unsigned char *dst)
{
	py += CLIP_STEP;
	pu += CLIP_HALF_STEP;
	pv += CLIP_HALF_STEP;

	TFB = 1;

	__asm
	{
		mov			eax, [py]
		mov			ebx, [pu]
		mov			ecx, [pv]
		mov			edx, [dst]
		mov			esi, 0x00
		mov			edi, [Clip_Height]

yuy2conv:
		movd		mm2, [ebx+esi]
		movd		mm3, [ecx+esi]
		punpcklbw	mm2, mm3
		movq		mm1, [eax+esi*2]
		movq		mm4, mm1
		punpcklbw	mm1, mm2
		punpckhbw	mm4, mm2

		add			esi, 0x04
		cmp			esi, [HALF_CLIP_WIDTH]
		movq		[edx+esi*4-16], mm1
		movq		[edx+esi*4-8], mm4
		jl			yuy2conv

		add			eax, [DOUBLE_WIDTH]
		add			ebx, [Coded_Picture_Width]
		add			ecx, [Coded_Picture_Width]
		add			edx, [QUAD_CLIP_WIDTH]
		sub			edi, 0x02
		mov			esi, 0x00
		cmp			edi, 0x00
		jg			yuy2conv

		emms
	}
}

static void conv422toyuy2even(unsigned char *py, unsigned char *pu, unsigned char *pv, unsigned char *dst)
{
	py += Coded_Picture_Width + CLIP_STEP;
	pu += HALF_WIDTH + CLIP_HALF_STEP;
	pv += HALF_WIDTH + CLIP_HALF_STEP;
	dst += DOUBLE_CLIP_WIDTH;

	BFB = 1;

	__asm
	{
		mov			eax, [py]
		mov			ebx, [pu]
		mov			ecx, [pv]
		mov			edx, [dst]
		mov			esi, 0x00
		mov			edi, [Clip_Height]

yuy2conv:
		movd		mm2, [ebx+esi]
		movd		mm3, [ecx+esi]
		punpcklbw	mm2, mm3
		movq		mm1, [eax+esi*2]
		movq		mm4, mm1
		punpcklbw	mm1, mm2
		punpckhbw	mm4, mm2

		add			esi, 0x04
		cmp			esi, [HALF_CLIP_WIDTH]
		movq		[edx+esi*4-16], mm1
		movq		[edx+esi*4-8], mm4
		jl			yuy2conv

		add			eax, [DOUBLE_WIDTH]
		add			ebx, [Coded_Picture_Width]
		add			ecx, [Coded_Picture_Width]
		add			edx, [QUAD_CLIP_WIDTH]
		sub			edi, 0x02
		mov			esi, 0x00
		cmp			edi, 0x00
		jg			yuy2conv

		emms
	}
}

static void conv444toRGB24odd(unsigned char *py, unsigned char *pu, unsigned char *pv, unsigned char *dst)
{
	py += CLIP_STEP;
	pu += CLIP_STEP;
	pv += CLIP_STEP;
	dst += RGB_DOWN1;

	TFB = 1;

	__asm
	{
		mov			eax, [py]
		mov			ebx, [pu]
		mov			ecx, [pv]
		mov			edx, [dst]
		mov			edi, [Clip_Height]
		mov			esi, 0x00
		pxor		mm0, mm0

convRGB24:
		movd		mm1, [eax+esi]
		movd		mm3, [ebx+esi]
		punpcklbw	mm1, mm0
		punpcklbw	mm3, mm0
		movd		mm5, [ecx+esi]
		punpcklbw	mm5, mm0
		movq		mm7, [mmmask_0128]
		psubw		mm3, mm7
		psubw		mm5, mm7

		psubw		mm1, [RGB_Offset]
		movq		mm2, mm1
		movq		mm7, [mmmask_0001]
		punpcklwd	mm1, mm7
		punpckhwd	mm2, mm7
		movq		mm7, [RGB_Scale]
		pmaddwd		mm1, mm7
		pmaddwd		mm2, mm7

		movq		mm4, mm3
		punpcklwd	mm3, mm0
		punpckhwd	mm4, mm0
		movq		mm7, [RGB_CBU]
		pmaddwd		mm3, mm7
		pmaddwd		mm4, mm7
		paddd		mm3, mm1
		paddd		mm4, mm2
		psrld		mm3, 13
		psrld		mm4, 13
		packuswb	mm3, mm0
		packuswb	mm4, mm0

		movq		mm6, mm5
		punpcklwd	mm5, mm0
		punpckhwd	mm6, mm0
		movq		mm7, [RGB_CRV]
		pmaddwd		mm5, mm7
		pmaddwd		mm6, mm7
		paddd		mm5, mm1
		paddd		mm6, mm2

		psrld		mm5, 13
		psrld		mm6, 13
		packuswb	mm5, mm0
		packuswb	mm6, mm0

		punpcklbw	mm3, mm5
		punpcklbw	mm4, mm6
		movq		mm5, mm3
		movq		mm6, mm4
		psrlq		mm5, 16
		psrlq		mm6, 16
		por			mm3, mm5
		por			mm4, mm6

		movd		mm5, [ebx+esi]
		movd		mm6, [ecx+esi]
		punpcklbw	mm5, mm0
		punpcklbw	mm6, mm0
		movq		mm7, [mmmask_0128]
		psubw		mm5, mm7
		psubw		mm6, mm7

		movq		mm7, mm6
		punpcklwd	mm6, mm5
		punpckhwd	mm7, mm5		
		movq		mm5, [RGB_CGX]
		pmaddwd		mm6, mm5
		pmaddwd		mm7, mm5
		paddd		mm6, mm1
		paddd		mm7, mm2

		psrld		mm6, 13
		psrld		mm7, 13
		packuswb	mm6, mm0
		packuswb	mm7, mm0

		punpcklbw	mm3, mm6
		punpcklbw	mm4, mm7

		movq		mm1, mm3
		movq		mm5, mm4
		movq		mm6, mm4

		psrlq		mm1, 32
		psllq		mm1, 24
		por			mm1, mm3

		psrlq		mm3, 40
		psllq		mm6, 16
		por			mm3, mm6
		movd		[edx], mm1

		psrld		mm4, 16
		psrlq		mm5, 24
		por			mm5, mm4
		movd		[edx+4], mm3

		add			edx, 0x0c
		add			esi, 0x04
		cmp			esi, [Clip_Width]
		movd		[edx-4], mm5

		jl			convRGB24

		add			eax, [DOUBLE_WIDTH]
		add			ebx, [DOUBLE_WIDTH]
		add			ecx, [DOUBLE_WIDTH]
		sub			edx, [NINE_CLIP_WIDTH]
		mov			esi, 0x00
		sub			edi, 0x02
		cmp			edi, 0x00
		jg			convRGB24

		emms
	}
}

static void conv444toRGB24even(unsigned char *py, unsigned char *pu, unsigned char *pv, unsigned char *dst)
{
	py += Coded_Picture_Width + CLIP_STEP;
	pu += Coded_Picture_Width + CLIP_STEP;
	pv += Coded_Picture_Width + CLIP_STEP;
	dst += RGB_DOWN2;

	BFB = 1;

	__asm
	{
		mov			eax, [py]
		mov			ebx, [pu]
		mov			ecx, [pv]
		mov			edx, [dst]
		mov			edi, [Clip_Height]
		mov			esi, 0x00
		pxor		mm0, mm0

convRGB24:
		movd		mm1, [eax+esi]
		movd		mm3, [ebx+esi]
		punpcklbw	mm1, mm0
		punpcklbw	mm3, mm0
		movd		mm5, [ecx+esi]
		punpcklbw	mm5, mm0
		movq		mm7, [mmmask_0128]
		psubw		mm3, mm7
		psubw		mm5, mm7

		psubw		mm1, [RGB_Offset]
		movq		mm2, mm1
		movq		mm7, [mmmask_0001]
		punpcklwd	mm1, mm7
		punpckhwd	mm2, mm7
		movq		mm7, [RGB_Scale]
		pmaddwd		mm1, mm7
		pmaddwd		mm2, mm7

		movq		mm4, mm3
		punpcklwd	mm3, mm0
		punpckhwd	mm4, mm0
		movq		mm7, [RGB_CBU]
		pmaddwd		mm3, mm7
		pmaddwd		mm4, mm7
		paddd		mm3, mm1
		paddd		mm4, mm2
		psrld		mm3, 13
		psrld		mm4, 13
		packuswb	mm3, mm0
		packuswb	mm4, mm0

		movq		mm6, mm5
		punpcklwd	mm5, mm0
		punpckhwd	mm6, mm0
		movq		mm7, [RGB_CRV]
		pmaddwd		mm5, mm7
		pmaddwd		mm6, mm7
		paddd		mm5, mm1
		paddd		mm6, mm2
		psrld		mm5, 13
		psrld		mm6, 13
		packuswb	mm5, mm0
		packuswb	mm6, mm0

		punpcklbw	mm3, mm5
		punpcklbw	mm4, mm6
		movq		mm5, mm3
		movq		mm6, mm4
		psrlq		mm5, 16
		psrlq		mm6, 16
		por			mm3, mm5
		por			mm4, mm6

		movd		mm5, [ebx+esi]
		movd		mm6, [ecx+esi]
		punpcklbw	mm5, mm0
		punpcklbw	mm6, mm0
		movq		mm7, [mmmask_0128]
		psubw		mm5, mm7
		psubw		mm6, mm7

		movq		mm7, mm6
		punpcklwd	mm6, mm5
		punpckhwd	mm7, mm5		
		movq		mm5, [RGB_CGX]
		pmaddwd		mm6, mm5
		pmaddwd		mm7, mm5
		paddd		mm6, mm1
		paddd		mm7, mm2

		psrld		mm6, 13
		psrld		mm7, 13
		packuswb	mm6, mm0
		packuswb	mm7, mm0

		punpcklbw	mm3, mm6
		punpcklbw	mm4, mm7

		movq		mm1, mm3
		movq		mm5, mm4
		movq		mm6, mm4

		psrlq		mm1, 32
		psllq		mm1, 24
		por			mm1, mm3

		psrlq		mm3, 40
		psllq		mm6, 16
		por			mm3, mm6
		movd		[edx], mm1

		psrld		mm4, 16
		psrlq		mm5, 24
		por			mm5, mm4
		movd		[edx+4], mm3

		add			edx, 0x0c
		add			esi, 0x04
		cmp			esi, [Clip_Width]
		movd		[edx-4], mm5

		jl			convRGB24

		add			eax, [DOUBLE_WIDTH]
		add			ebx, [DOUBLE_WIDTH]
		add			ecx, [DOUBLE_WIDTH]
		sub			edx, [NINE_CLIP_WIDTH]
		mov			esi, 0x00
		sub			edi, 0x02
		cmp			edi, 0x00
		jg			convRGB24

		emms
	}
}

void RenderRGB24()
{
	SetDIBitsToDevice(hDC, 0, 0, Clip_Width, Clip_Height,
		0, 0, 0, Clip_Height, rgb24, (LPBITMAPINFO)lpbirgb, DIB_RGB_COLORS);
}

void RenderYUY2()
{
	int i;
	unsigned char *src, *dst;
	POINT point = {0, 0};

	if (IDirectDrawSurface_Lock(lpOverlay, NULL, &ddsd, 0, NULL)==DD_OK)
	{
		src = yuy2;
		dst = (unsigned char *)ddsd.lpSurface;

		for (i=0; i<Clip_Height; i++)
		{
			memcpy(dst, src, DOUBLE_CLIP_WIDTH);
			src += DOUBLE_CLIP_WIDTH;
			dst += ddsd.lPitch;
		}

		if (IDirectDrawSurface_Unlock(lpOverlay, NULL)==DD_OK)
		{
			SetRect(&orect, 0, 0, Coded_Picture_Width, Coded_Picture_Height);

			ClientToScreen(hWnd, &point);
			prect.left = point.x;
			prect.right = point.x + Coded_Picture_Width;
			prect.top = point.y;
			prect.bottom = point.y + Coded_Picture_Height;

			if (prect.left < 0)
			{
				orect.left = -prect.left;
				prect.left = 0;
			}

			if (prect.top < 0)
			{
				orect.top = -prect.top;
				prect.top = 0;
			}

			if (prect.right > GetSystemMetrics(SM_CXSCREEN))
			{
				orect.right = Coded_Picture_Width + GetSystemMetrics(SM_CXSCREEN) - prect.right;
				prect.right = GetSystemMetrics(SM_CXSCREEN);
			}

			if (prect.bottom > GetSystemMetrics(SM_CYSCREEN))
			{
				orect.bottom = Coded_Picture_Height + GetSystemMetrics(SM_CYSCREEN) - prect.bottom;
				prect.bottom = GetSystemMetrics(SM_CYSCREEN);
			}

			IDirectDrawSurface_UpdateOverlay(lpOverlay, &orect, lpPrimary, &prect,
				DDOVER_SHOW | DDOVER_DDFX | DDOVER_KEYDESTOVERRIDE, &ddofx);
		}
	}
}

int DetectVideoType(int frame, int trf)
{
	static int Old_TRF, Repeat_On, Repeat_Off, Repeat_Init;

	if (frame)
	{
		if ((trf & 3) == ((Old_TRF+1) & 3))
			FILM_Purity++;
		else
			NTSC_Purity++;
	}
	else
		Video_Type = FILM_Purity = NTSC_Purity = Repeat_On = Repeat_Off = Repeat_Init = 0; 

	Old_TRF = trf;

	if (trf & 1)
		Repeat_On ++;
	else
		Repeat_Off ++;

	if (Repeat_Init)
	{
		if (Repeat_Off-Repeat_On == 5)
		{
			Repeat_Off = Repeat_On = 0;
			return 0;
		}
		else if (Repeat_On-Repeat_Off == 5)
		{
			Repeat_Off = Repeat_On = 0;
			return 2;
		}
	}
	else
	{
		if (Repeat_Off-Repeat_On == 3)
		{
			Repeat_Off = Repeat_On = 0;
			Repeat_Init = 1;
			return 0;
		}
		else if (Repeat_On-Repeat_Off == 3)
		{
			Repeat_Off = Repeat_On = 0;
			Repeat_Init = 1;
			return 2;
		}
	}

	return 1;
}

static void AVIKill()
{
	AVI_Init = 1; frame_count = 0;

	if (ps)
		AVIStreamClose(ps);

	if (psCompressed)
		AVIStreamClose(psCompressed);

	if (pfile)
		AVIFileClose(pfile);

	if (Stop_Flag || Fault_Flag)
	{
		AVIFileExit();

		if (Store_Flag!=STORE_RGB24)
			ICSeqCompressFrameEnd(pcompvars);

		ICCompressorFree(pcompvars);
	}
}

static int gcd(int x, int y)
{
	int t;

	while (y != 0) {
		t = x % y; x = y; y = t;
	}

	return x;
}
