//-----------------------------------------------------------------------------
//
// ControlMessage.cpp - Used by ControlPort to store
//                      incoming and outgoing messages.
//
//    Copyright (C) 2004  Mark D. Collier/Gregory L. Holt
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier                                 - 12/01/2006   v1.1
//                   Mark D. Collier/Gregory L. Holt  -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "ControlMessage.h"
#include "util.h"

#define malloc mymalloc
#define free   myfree
#define strdup mystrdup
#define strndup mystrndup

ControlMessage::ControlMessage( ControlPort *  aPort,
                                int            aConnectionId,
                                int            aSocketIndex,
                                const char *   aText )
{
    mPort         = aPort;
    mConnectionId = aConnectionId;
    mSocketIndex  = aSocketIndex;
    
    // 
    //  By definition, an object of ControlMessage class can never have a
    //  NULL text pointer member, however, it can point to a NUL string.
    //  If the code instantiating the object passes a NULL text pointer,
    //  then a NUL string is allocated and attached to the object.
    // 
    
    if ( !aText )
    {
        mText = strdup ( "" );
    }
    else
    {
    
        //
        //  The input string could be a constant literal or occupying dynamic memory.
        //  It is the responsibility of the code instantiating the new ControlMessage
        //  object to manage the memory occupied by the input string. The input
        //  string is constant as far as this constructor is concerned.
        //
        //  The input string is duplicated and attached to the new ControlMessage
        //  object, so that an object of this class need not be concerned about the type
        //  of the input string.
        //
    
        mText = strdup( aText );
    }
}


ControlMessage::~ControlMessage( void )
{   
    //
    // every ControlMessage has a text string, even if it is the NUL string
    //
    
    free( mText );
}

//
//  Copy Constructor
//

ControlMessage::ControlMessage( const ControlMessage& controlMessage )
{
    mPort         = controlMessage.mPort;
    mConnectionId = controlMessage.mConnectionId;
    mSocketIndex  = controlMessage.mSocketIndex;
    
    // 
    //  By definition, an object of ControlMessage class can never have a
    //  NULL text pointer member, however, it can point to a NUL string.
    // 

    mText         = strdup( controlMessage.mText );    
}

//
//  Overload Assignment Operator (i.e. operator= )
//

ControlMessage& ControlMessage::operator=( const ControlMessage& rhs )
{
    //
    //  Check for self-assignment!
    //
    
    if ( this == &rhs )
    {
        return *this;           // Self assignment! Just return the original object.
    }
    
    //
    //  We really are copying over an existing ControlMessage object different from the 
    //  ControlMessage object specified on the right-hand-side (i.e. rhs)
    //
    
    mPort         = rhs.mPort;
    mConnectionId = rhs.mConnectionId;
    mSocketIndex  = rhs.mSocketIndex;
    
    // 
    //  By definition, an object of ControlMessage class can never have a
    //  NULL text pointer member, however, it can point to a NUL string.
    // 
    
    free( mText );                 // Discard the lhs object's existing text string
    mText = strdup ( rhs.mText );  // Duplicate the text string from the rhs object.
    
    return *this;
}
    

char *  ControlMessage::GetText( void )
{
    //
    // every ControlMessage has a text string, even if it is the NUL string
    //

    return strdup( mText );
}


int  ControlMessage::GetTextLength( void )
{
    //
    // every ControlMessage has a text string, even if it is the NUL string
    //

    return strlen( mText );
}


void  ControlMessage::SetText( char *  value )
{
    //
    // every ControlMessage has a text string, even if it is the NUL string
    //

    free( mText );
    mText = strdup( value );
}


void  ControlMessage::SetText( char *  aText, char *  aAppendedText )
{
    char *  cp;
    int     textLength = 0;
    int     appendedTextLength = 0;
    int     textLengthCombined = 0;

    free( mText );  // every ControlMessage has a text string, even if it is the NUL string
    
    //
    //  If both input text pointers are NULL, then simply set the object's
    //  text string to the NUL string.
    //
    
    if ( aText == NULL && aAppendedText == NULL )
    {
        mText = strdup( "" );
        return;
    }
    
    if ( aText )
    {
        textLength = strlen( aText );
    }
    
    if ( aAppendedText ) 
    {
        appendedTextLength = strlen( aAppendedText );
    }
    
    textLengthCombined = textLength + appendedTextLength;
    
    if ( !textLengthCombined )
    {
        mText = strdup( "" );
        return;
    }        

    cp = ( char * )malloc( textLengthCombined + 3 );
    if ( !cp )
    {
        ReportError( "out of memory", __FILE__, __LINE__ );
    }
    memcpy( cp, aText, textLength );
    memcpy( cp + textLength, aAppendedText, appendedTextLength );
    cp[textLengthCombined    ] = '\r';
    cp[textLengthCombined + 1] = '\n';
    cp[textLengthCombined + 2] = '\0';

    mText = cp;
}


int  ControlMessage::GetConnectionId( void )
{
    return mConnectionId;
}


void  ControlMessage::SetConnectionId( int  aId )
{
    mConnectionId = aId;
}


int  ControlMessage::GetSocketIndex( void )
{
    return mSocketIndex;
}


void  ControlMessage::SetSocketIndex( int  aIndex )
{
    mSocketIndex = aIndex;
}

