//-----------------------------------------------------------------------------
//
//  ControlPort.cpp - Handles connections to the control
//                   TCP port, accepting, processing, delivering,
//                   and sending ControlMessages. The control
//                   port is the "command line interface" of the
//                   sip_rogue daemon.
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#include <sys/time.h>
#include <arpa/inet.h>
#include <errno.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/select.h>
#include <sys/socket.h>
#include <time.h>
#include <unistd.h>
#include <sched.h>

#include "util.h"
#include "RtpHandler.h"
#include "SipUdpPort.h"
#include "SipDispatcher.h"
#include "SipRegistrarConnector.h"
#include "SipEndPoint.h"
#include "SipProxyEndPoint.h"
#include "SipRegistrar.h"
#include "SipCall.h"
#include "ControlMessage.h"
#include "ControlPort.h"


#define MAX_RECV_SIZE   65536

#define malloc  mymalloc
#define free    myfree
#define strdup  mystrdup
#define strndup mystrndup

ControlPort::ControlPort( void )
{
    mAddress                 = INADDR_ANY;
    mPort                    = htons( 6060 );
    mMaxConnections          = 16;
    mSockets                 = new controlSocket[mMaxConnections];
    mConnections             = new controlConnection[mMaxConnections];
    mIncomingQueues          = new queue[mMaxConnections];
    mOutgoingQueues          = new queue[mMaxConnections];
    mNetworkStopFlag         = 0;
    mProcessStopFlag         = 0;
    mCreationTime            = time( NULL );
    pthread_mutex_init( &mAutoIdMutex, NULL );
    mAutoIdCounter           = 0;
}


ControlPort::~ControlPort( void )
{
    delete[] mSockets;
    delete[] mConnections;
    delete[] mIncomingQueues;
    delete[] mOutgoingQueues;
    pthread_mutex_destroy( &mAutoIdMutex );
}


void   ControlPort::SetBoundAddress( in_addr_t  aValue )
{
    mAddress = aValue;
}


void   ControlPort::SetBoundPort( in_port_t  aValue )
{
    mPort = aValue;
}

//-----------------------------------------------------------------------------
//
//   NetworkRun()
//
//   This member function is intended to be run in a
//   separate thread from the ProcessRun() member
//   function.
//
//-----------------------------------------------------------------------------

ControlPort::Error ControlPort::NetworkRun( void )
{
    char *              cp;
    char *              cp2;
    char *              cp3;
    char *              receiveBuffer;
    ControlMessage *    message;
    fd_set              fdSetRead;
    fd_set              fdSetWrite;
    int                 length;
    int                 length2;
    int                 newFileDescriptor;
    int                 receiveLength;
    int                 selectStatus;
    int                 sendLength;
    int                 socketFileDescriptor;
    int                 socketIndex;
    int                 socketsHighestFileDescriptor;
    long int            timeoutUsec;
    socklen_t           socketAddressLength;
    struct sockaddr_in  localAddress;
    struct sockaddr_in  remoteAddress;
    struct timeval      timeout;
    int                 ix;
    ControlPort::Error  error;

    //
    //  Prepare a sip_rogue server side control TCP socket and
    //  bind it to the telnet IP addr/port designated by the 
    //  sip_rogue operator when sip_rogue is executed.
    //
    //  There can be many client connections to this socket
    //  (i.e. mMaxConnections).
    //
    
    receiveBuffer = ( char * )malloc( MAX_RECV_SIZE + 1 );
    if ( !receiveBuffer )
    {
        return ERROR_MEMORY;
    }

    socketFileDescriptor = socket( PF_INET, SOCK_STREAM, IPPROTO_TCP );
    if ( socketFileDescriptor == -1 )
    {
        return ERROR_SOCKET;
    }

    if ( fcntl( socketFileDescriptor, F_SETFL, O_NONBLOCK ) == -1 )
    {
        return ERROR_FCNTL;
    }

    localAddress.sin_family      = AF_INET;
    localAddress.sin_addr.s_addr = mAddress;
    localAddress.sin_port        = mPort;
    memset( &( localAddress.sin_zero ), '\0', 8 );

    ix = -1;
    setsockopt( socketFileDescriptor,
                SOL_SOCKET,
                SO_REUSEADDR,
                &ix,
                sizeof( ix ) );

    if ( bind( socketFileDescriptor,
                ( struct sockaddr * )&localAddress,
                sizeof( localAddress ) ) == -1 )
    {
        return ERROR_BIND;
    }
    
    //
    //  Demonstrate a willingness to accept connections on this socket
    //

    if ( listen( socketFileDescriptor, 0 ) == -1 )
    {
        return ERROR_LISTEN;
    }
    
    //
    //  Initialize the array of socket file descriptors (i.e. TCP control connections)
    //  to empty.
    //

    socketsHighestFileDescriptor = 0;
    for ( socketIndex = 0; socketIndex < mMaxConnections; socketIndex++ )
    {
        mSockets[socketIndex].FileDescriptor = -1;
    }

    timeoutUsec = 1000;  // initialize the socket select() timeout to 1 ms.
    
    //
    //  Main Network Loop.
    //
    //  Note: This loop repeats until a sip_rogue operator shutsdown
    //             command is received or a fatal error occurs.
    //

    while ( !mNetworkStopFlag )
    {
        socketAddressLength = sizeof( remoteAddress );
        newFileDescriptor = accept( socketFileDescriptor,
                                    ( struct sockaddr * )&remoteAddress,
                                    &socketAddressLength );
        
        //
        //  If accept returns -1 and it is due to a reason other than there are no
        //  pending connections received for the socket, then the NetworkRun
        //  main loop ends.
        //
                                    
        if ( newFileDescriptor == -1 &&
             errno != EAGAIN         &&
             errno != EWOULDBLOCK       )
        {
            return ERROR_ACCEPT;
        }

        if ( newFileDescriptor >= 0 )
        {
            //
            //  A new connection to the control socket has been received.
            //  Search for an available entry in the socket connection array.
            //  If an empty entry is found, copy the new connection parms
            //  into it.
            //
            
            for ( socketIndex = 0;
                  socketIndex < mMaxConnections;
                  socketIndex++ )
            {
                if ( mSockets[socketIndex].FileDescriptor == -1 )
                {
                    mSockets[socketIndex].FileDescriptor     =
                                                  newFileDescriptor;
                    mSockets[socketIndex].SourceAddress      =
                                                  localAddress.sin_addr.s_addr;
                    mSockets[socketIndex].SourcePort         =
                                                  localAddress.sin_port;
                    mSockets[socketIndex].DestinationAddress =
                                                  remoteAddress.sin_addr.s_addr;
                    mSockets[socketIndex].DestinationPort    =
                                                  remoteAddress.sin_port;
                    mSockets[socketIndex].Buffer             =
                                                  NULL;
                    mSockets[socketIndex].ConnectionId       =
                                                  -socketIndex - 1;
                    if ( newFileDescriptor > socketsHighestFileDescriptor )
                    {
                        socketsHighestFileDescriptor = newFileDescriptor;
                    }
                    break;
                }
            }
            
            if ( socketIndex >= mMaxConnections )
            {
                // FIXME: Report "All sockets full; try again later."
                close( newFileDescriptor );
            }
        }
        
        //
        //  The next two sets of operations are a prelude to a select() function
        //  call where we'd like the OS to inform us if input is available to be
        //  read from control connections or if waiting output can be written to
        //  control connections.
        //  
        //  Create a set of FileDescriptor ID's (connections) we like to
        //  check for waiting input.
        //

        FD_ZERO( &fdSetRead );
        for ( socketIndex = 0; socketIndex < mMaxConnections; socketIndex++ )
        {
            if ( mSockets[socketIndex].FileDescriptor >= 0 )
            {
                FD_SET( mSockets[socketIndex].FileDescriptor, &fdSetRead );
            }
        }

        //
        //  Create a set of FileDescriptor ID's (connections) for which we have
        //  waiting output. Prepare to check if we may write waiting data to
        //  any of those connections.
        //
        
        FD_ZERO( &fdSetWrite );
        for ( socketIndex = 0; socketIndex < mMaxConnections; socketIndex++ )
        {
            if ( mSockets[socketIndex].FileDescriptor >= 0
                 && mOutgoingQueues[socketIndex].HasNext() )
            {
                FD_SET( mSockets[socketIndex].FileDescriptor, &fdSetWrite );
            }
        }

        timeout.tv_sec  = 0;
        timeout.tv_usec = timeoutUsec;
        
        //
        //  Wait up to the timeout limit for any of the control connections
        //  to signal waiting input data, or for any of the control connections we
        //  have waiting output data to signal they may be written.
        //
        //  Note: The timeout ranges from 1 ms to 100 ms in accordance with
        //             an algorithm appearing later in this loop.
        //

        selectStatus = select( socketsHighestFileDescriptor + 1,
                               &fdSetRead,
                               &fdSetWrite,
                               NULL,
                               &timeout );

        if ( selectStatus == -1 )
        {
            return ERROR_SELECT;
        }

        if ( selectStatus == 0 )
        {
            //
            //  No socket connections are ready to be read from,
            //  nor are any ready to be written to for which we
            //  have data waiting to be output.
            //
            //  For efficiency purposes we manipulate the 
            //  timeout for the next socket select() iteration.
            //  Gradually increase the timeout interval while
            //  we have no control I/O activity.  
            //  Range: 1 ms to 100 ms
            //
            
            if ( timeoutUsec < 100000 ) // 100 milliseconds
            {
                if ( timeoutUsec <= 1000 )
                {
                    timeoutUsec = 5000;
                }
                else if ( timeoutUsec <= 5000 )
                {
                    timeoutUsec = 25000;
                }
                else if ( timeoutUsec <= 25000 )
                {
                    timeoutUsec = 50000;
                }
                else if ( timeoutUsec <= 50000 )
                {
                    timeoutUsec = 100000;
                }
            }
            continue;  // jump to the top of the main loop
        }
        
        //
        //  At least one connection has data waiting to be read, or we have
        //  output data waiting for a connection that is ready to receive it.
        //

        timeoutUsec = 1000;  //  reset the I/O timeout to 1 ms for the next loop

        for ( socketIndex = 0; socketIndex < mMaxConnections; socketIndex++ )
        {
            if ( mSockets[socketIndex].FileDescriptor >= 0 )
            {
                if ( FD_ISSET( mSockets[socketIndex].FileDescriptor,
                               &fdSetRead ) )
                {
                    receiveLength = recv( mSockets[socketIndex].FileDescriptor,
                                          receiveBuffer,
                                          MAX_RECV_SIZE,
                                          0 );
                    if ( receiveLength < 1 )
                    {
                        //
                        //  Hmmm... something is wrong. We checked that there
                        //  was data avail to be read on this connection, but then
                        //  no data was produced when we tried to receive it.
                        //  Close the connection and drain any pending outgoing
                        //  messages for this connection.
                        //
                        
                        close( mSockets[socketIndex].FileDescriptor );
                        mSockets[socketIndex].FileDescriptor = -1;
                        while (   ( message = getNextOutgoing( socketIndex ) )
                               != NULL )
                        {
                            delete message;
                        }
                        continue; // jump to top of loop iterating through connection array
                    }
                    
                    //
                    //  We've received one or more bytes for the ith control connection
                    //
                    //  Convert the receive buffer to a string by appending a
                    //  terminating NUL character.
                    //
                    //  Note: Although it's extremely unlikely given the huge size
                    //             of the receive buffer relative to the length of sip_rogue
                    //             operator commands, make sure the byte we're setting
                    //             to NUL is not outside the receive buffer.
                    //
                    
                    if ( receiveLength == MAX_RECV_SIZE )
                    {
                        receiveBuffer[MAX_RECV_SIZE - 1] = '\0';
                    }
                    else
                    {
                        receiveBuffer[receiveLength] = '\0';
                    }
                    
                    //
                    //  Scan the receive buffer for a commands. Since this is
                    //  a stream socket, we don't know yet how many 
                    //  command lines or command line fragments we've
                    //  just received for the ith connection. For example,
                    //  we could have just received an entire command from
                    //  start to finish ending with CRLF. Or we could have
                    //  received only the first part of a command, or the last
                    //  part of a command ending with CRLF (i.e. having
                    //  received the first part earlier), or the last part of a prior
                    //  command ending with CRLF and the first part of
                    //  the next command, or multiple complete command
                    //  lines, or almost any combination - however infeasible
                    //  that might be.
                    //
                    //  Carve up multiple commands in the buffer into 
                    //  individual strings as needed.
                    //
                    
                    cp = receiveBuffer;
                    while ( *cp )
                    {
                        //
                        //  Scan for a CRLF. We might not find a CRLF.
                        //  
                        //  Note: '\r' = CR, '\n' = LF
                        //
                        
                        cp2 = cp;
                        while ( *cp2 )
                        {
                            if ( *cp2 == '\r' || *cp2 == '\n' )
                            {
                                break; // exit while loop
                            }
                            cp2++;
                        } // end while
                        
                        if ( *cp2 == '\r' || *cp2 == '\n' )
                        {
                            if ( *cp2 == '\r' )
                            {
                                *cp2 = '\0';
                                cp2++;
                            }
                            if ( *cp2 == '\n' )
                            {
                                *cp2 = '\0';
                                cp2++;
                            }
                            
                            //
                            //  We declare that a complete command has been received for
                            //  the ith control connection. We wish to append the command
                            //  data we've just received for the ith connection to any
                            //  previously received data for this command that has been
                            //  buffered and then submit the complete command to be
                            //  processed.
                            //
                            //  FIXME: We might have received only a CR and an LF
                            //                   is forthcoming.
                            //
                            
                            length = 0;                                
                            if ( mSockets[socketIndex].Buffer )
                            {
                                //
                                //  Have previously buffered the leading part of the
                                //  command whose end we have just received.
                                //
                                
                                length =
                                     strlen( mSockets[socketIndex].Buffer );
                                if ( length >= MAX_RECV_SIZE )
                                {
                                    length = MAX_RECV_SIZE - 1;
                                }
                            }
                            length2 = strlen( cp ); // rest of the cmd minus the CRLF
                            if ( length + length2 >= MAX_RECV_SIZE )
                            {
                                length2 = MAX_RECV_SIZE - length - 1;
                            }
                            
                            //
                            //  Allocate sufficient memory to hold the entire command
                            //  string. Copy in any leading part that had been previously
                            //  buffered, then copy in the remainder of the command
                            //  from the receive buffer.
                            //
                            //  Note: The command submitted to be processed excludes
                            //             the received CRLF
                            //
                            
                            cp3 = ( char * )malloc( length + length2 + 1 );
                            if ( !cp3 )
                            {
                                return ERROR_MEMORY;
                            }
                            if ( length )
                            {
                                memcpy( cp3,
                                        mSockets[socketIndex].Buffer,
                                        length );
                                free( mSockets[socketIndex].Buffer );
                                mSockets[socketIndex].Buffer = NULL;
                            }
                            if ( length2 )
                            {
                                memcpy( cp3 + length, cp, length2 );
                            }
                            cp3[length + length2] = '\0';
                            
                            //
                            //  Submit the completed control command
                            //
                            
                            addIncoming( socketIndex, 
                                         new ControlMessage(
                                                      this,
                                                      mSockets[socketIndex].
                                                          ConnectionId,
                                                      socketIndex,
                                                      cp3 ) );
                                                    
                            free( cp3 );
                        }
                        else
                        {
                            //
                            //  We've received only part of a control command at this
                            //  point on the ith control connection. Store the part we've
                            //  received. We should receive the remainder of the
                            //  command at some future point.
                            //
                            
                            length = 0;                                
                            if ( mSockets[socketIndex].Buffer )
                            {
                                length =
                                     strlen( mSockets[socketIndex].Buffer );
                            }
                            length2 = strlen( cp );
                            if ( length + length2 >= MAX_RECV_SIZE )
                            {
                                length2 = MAX_RECV_SIZE - length - 1;
                            }
                            
                            //
                            //  Allocate sufficient memory to hold portion of the command
                            //  received thus far. Copy in any leading part that had been
                            //  previously buffered, then append that part of the command
                            //  we've just received.
                            //
 
                            cp3 = ( char * )malloc( length + length2 + 1 );
                            if ( !cp3 )
                            {
                                return ERROR_MEMORY;
                            }
                            if ( length )
                            {
                                memcpy( cp3,
                                        mSockets[socketIndex].Buffer,
                                        length );
                                free( mSockets[socketIndex].Buffer );
                                mSockets[socketIndex].Buffer = NULL;
                            }
                            if ( length2 )
                            {
                                memcpy( cp3 + length, cp, length2 );
                            }
                            cp3[length + length2] = '\0';
                            mSockets[socketIndex].Buffer = cp3;
                        }
                                   // Advance primary receive buffer ptr past 
                        cp = cp2;  // the command data just processed.
                        
                    } // end while receive buffer data for ith control connection remaining
                } //  if a connection has input ready to be read
                
                //
                //  Earlier, we decided if we had output waiting for the ith control
                //  socket connection. If we did, the select statement will inform
                //  us if the connection is available to receive some data.
                //
                
                if ( FD_ISSET( mSockets[socketIndex].FileDescriptor,
                               &fdSetWrite ) )
                {
                    message = getNextOutgoing( socketIndex );
                    sendLength = message->GetTextLength();
                    if ( sendLength > 0 )
                    {
                        cp = message->GetText();
                        sendLength = send( mSockets[socketIndex].FileDescriptor,
                                           cp,
                                           sendLength,
                                           0 );
                        free( cp );
                        if ( sendLength < 0 )
                        {
                            //
                            //  Hmmm... something is wrong. Close the connection
                            //  and drain any pending outgoing messages for this
                            //  connection.
                            //

                            close( mSockets[socketIndex].FileDescriptor );
                            mSockets[socketIndex].FileDescriptor = -1;
                            delete message;
                            while ( ( message
                                       = getNextOutgoing( socketIndex ) )
                                      != NULL )
                            {
                                delete message;
                            }
                            continue;  // jump to top of loop iterating through connection array
                        }
                        else if ( sendLength == 0 )
                        {
                            //
                            //  Hmmmm.... tried to write to the connection and nothing
                            //  was accepted. Add the message back onto the head of the  
                            //  ith connection's pending outgoing message queue.
                            //
                            
                            error = preAddOutgoing( message );
                            if ( error != ERROR_NONE )
                            {
                                return error;
                            }
                        }
                        else if ( sendLength < message->GetTextLength() )
                        {
                            //
                            //  Only a fraction of the message we tried to send
                            //  was accepted by the socket. Add that part of the
                            //  message that was not accepted back onto the 
                            //  head of the pending outgoing message queue for
                            //  the ith connection.
                            //
                            
                            cp = message->GetText();
                            message->SetText( cp + sendLength );
                            free( cp );
                            error = preAddOutgoing( message );
                            if ( error != ERROR_NONE )
                            {
                                return error;
                            }
                        }
                        else
                        {
                            //  the message was sent.
                            delete message;
                        }
                    }
                    else
                    {
                        // the message is empty, delete the ControlMessage object
                        delete message;
                    }                        
                } //  if a connection for which we have output is ready to receive it                 
            } // if a connection exists for the ith entry in the connection arrary. 
        } // for each entry of the socket connection arrary
    } // end of main loop

    free( receiveBuffer );
    
    //
    //  Close any socket connections, drain incoming and outgoing control message queues
    //

    for ( socketIndex = -1; socketIndex < mMaxConnections; socketIndex++ )
    {
        if ( socketIndex >= 0 )
        {
            if ( mSockets[socketIndex].FileDescriptor >= 0 )
            {
                close( mSockets[socketIndex].FileDescriptor );
            }
            if ( mSockets[socketIndex].Buffer )
            {
                free( mSockets[socketIndex].Buffer );
            }
            
            //
            //  Drain any pending outgoing control messages for the ith connection
            //
            
            while ( ( message = getNextOutgoing( socketIndex ) ) != NULL )
            {
                delete message;
            }
        }
        
        //
        //  Drain any pending incoming control messages for the ith connection
        //

        while ( ( message = getNextIncoming( socketIndex ) ) != NULL )
        {
            delete message;
        }

    }

    close( socketFileDescriptor );

    return ERROR_NONE;
}


void  ControlPort::NetworkStop( void )
{
    mNetworkStopFlag = -1;
}

//-----------------------------------------------------------------------------
//
//   ProcessRun()
//
//   This member function is intended to be run in a
//   separate thread from the NetworkRun() member
//   function.
//
//-----------------------------------------------------------------------------

ControlPort::Error  ControlPort::ProcessRun( void )
{
    char *                          commandLine;
    char *                          commandLineHead;
    bool                            processedMessage;
    char *                          word;
    ControlMessage *                message;
    int                             socketIndex;
    int                             ix;
    unsigned long                   sleepTime;
    controlSipUdpPort *             ctlSipUdpPort;
    controlSipDispatcher *          ctlSipDispatcher;
    controlRtpHandler *             ctlRtpHandler;
    controlSipRegistrarConnector *  ctlSipRegistrarConnector;
    controlSipEndPoint *            ctlSipEndPoint;
    controlSipProxyEndPoint *       ctlSipProxyEndPoint;
    controlSipRegistrar *           ctlSipRegistrar;
    timeval                         now;
    timeval                         waitUntil;
    controlSipCall *                ctlSipCall;
    controlSipCall *                ctlSipCallPrevious;
    controlSipProxyEndPoint *       ctlSipProxyEndPointPrevious;

    gettimeofday( &waitUntil, NULL );
    waitUntil.tv_sec++;
    mProcessStopFlag = 0;
    sleepTime = 1000; // 1 millisecond
    while ( !mProcessStopFlag )
    {
        do // while ( processedMessage )
        {
            gettimeofday( &now, NULL );
            if (   now.tv_sec > waitUntil.tv_sec
                   || (   now.tv_sec == waitUntil.tv_sec
                       && now.tv_usec > waitUntil.tv_usec ) )
            {
                for ( ix = 0; ix < mMaxConnections; ix++ )
                {
                    ctlSipCallPrevious = NULL;
                    pthread_mutex_lock( &( mConnections[ix].CallsMutex ) );
                    ctlSipCall = mConnections[ix].Calls;
                    while ( ctlSipCall )
                    {
                        if (      ctlSipCall->Call->GetState()
                               == ctlSipCall->Call->STATE_TERMINATED
                            && strncmp( "auto", ctlSipCall->Name, 4 ) == 0 )
                        {
                            ctlSipCall->Call->GetEndPoint()->GetDispatcher()->
                                                 RemoveCall( ctlSipCall->Call );
                            // delete ctlSipCall->Call;
                            if ( ctlSipCallPrevious )
                            {
                                ctlSipCallPrevious->Next = ctlSipCall->Next;
                            }
                            else
                            {
                                mConnections[ix].Calls = ctlSipCall->Next;
                            }
                            // delete ctlSipCall;
                            if ( ctlSipCallPrevious )
                            {
                                ctlSipCall = ctlSipCallPrevious->Next;
                            }
                            else
                            {
                                ctlSipCall = mConnections[ix].Calls;
                            }
                        }
                        else
                        {
                            ctlSipCallPrevious = ctlSipCall;
                            ctlSipCall = ctlSipCall->Next;
                        }
                        sched_yield();
                    }
                    pthread_mutex_unlock( &( mConnections[ix].CallsMutex ) );

                    ctlSipProxyEndPointPrevious = NULL;
                    pthread_mutex_lock( &( mConnections[ix].ProxyEndPointsMutex ) );
                    ctlSipProxyEndPoint = mConnections[ix].ProxyEndPoints;
                    while ( ctlSipProxyEndPoint )
                    {
                        if (   ctlSipProxyEndPoint->ProxyEndPoint->GetControlPort() == NULL )
                        {
                            ctlSipProxyEndPoint->ProxyEndPoint->GetDispatcher()->RemoveProxyEndPoint( ctlSipProxyEndPoint->ProxyEndPoint );
                            if ( ctlSipProxyEndPointPrevious )
                            {
                                ctlSipProxyEndPointPrevious->Next = ctlSipProxyEndPoint->Next;
                            }
                            else
                            {
                                mConnections[ix].ProxyEndPoints = ctlSipProxyEndPoint->Next;
                            }
                            if ( ctlSipProxyEndPointPrevious )
                            {
                                ctlSipProxyEndPoint = ctlSipProxyEndPointPrevious->Next;
                            }
                            else
                            {
                                ctlSipProxyEndPoint = mConnections[ix].ProxyEndPoints;
                            }
                        }
                        else
                        {
                            ctlSipProxyEndPointPrevious = ctlSipProxyEndPoint;
                            ctlSipProxyEndPoint = ctlSipProxyEndPoint->Next;
                        }
                        sched_yield();
                    }
                    pthread_mutex_unlock( &( mConnections[ix].ProxyEndPointsMutex ) );
                }
                gettimeofday( &waitUntil, NULL );
                waitUntil.tv_sec++;
            }
            processedMessage = 0;
            for ( socketIndex = -1;
                  socketIndex < mMaxConnections;
                  socketIndex++ )
            {
                message = getNextIncoming( socketIndex );
                if ( message )
                {
                    processedMessage = -1;
                    sleepTime = 1000;
                    if ( message->GetConnectionId() < 0 && socketIndex >= 0 )
                    {
                        message->SetConnectionId(
                                           mSockets[socketIndex].ConnectionId );
                    }
                    commandLineHead = commandLine = message->GetText();
                    word = GetNextWord( &commandLine );
                    if ( !*word )
                    {
                        // Send blank lines back; sort of a ping.
                        message->SetText( "\r\n" );
                        addOutgoing( message );
                        message = NULL;
                    }
                    else if ( strcasecmp( word, "echo" ) == 0 )
                    {
                        processEcho( message, commandLine );
                        message = NULL;
                    }
                    else if ( strcasecmp( word, "help" ) == 0 )
                    {
                        processHelp( message, commandLine );
                        message = NULL;
                    }
                    else if ( strcasecmp( word, "connection" ) == 0 )
                    {
                        processConnection( socketIndex, message, commandLine );
                        message = NULL;
                    }
                    else if ( strcasecmp( word, "status" ) == 0 )
                    {
                        processStatus( message, commandLine );
                        message = NULL;
                    }
                    else if ( strcasecmp( word, "summary" ) == 0 )
                    {
                        processSummary( message, commandLine );
                        message = NULL;
                    }
                    else if ( strcasecmp( word, "create" ) == 0 )
                    {
                        processCreate( message, commandLine );
                        message = NULL;
                    }
                    else if ( strcasecmp( word, "delete" ) == 0 )
                    {
                        processDelete( message, commandLine );
                        message = NULL;
                    }
                    else if ( strcasecmp( word, "execute" ) == 0 )
                    {
                        processExecute( message, commandLine );
                        message = NULL;
                    }
                    else if ( strcasecmp( word, "issue" ) == 0 )
                    {
                        processIssue( message, commandLine );
                        message = NULL;
                    }
                    else if ( strcasecmp( word, "sleep" ) == 0 )
                    {
                        processSleep( message, commandLine );
                        message = NULL;
                    }
                    else if (   strcasecmp( word, "quit" ) == 0
                             || strcasecmp( word, "exit" ) == 0 )
                    {
                        delete message; // delete the quit or exit ControlMessage object
                        
                        close( mSockets[socketIndex].FileDescriptor );
                        mSockets[socketIndex].FileDescriptor = -1;
                        while ( ( message = getNextOutgoing( socketIndex ) )
                                != NULL )
                        {
                            delete message;
                        }
                        // message pointer is now == NULL
                    }
                    else if ( strcasecmp( word, "shutdown" ) == 0 )
                    {
                        processShutdown( message, commandLine );
                        message = NULL;
                    }
                    else
                    {
                        message->SetText( "Error: Unknown command: ",
                                          word );
                        addOutgoing( message );
                        message = NULL;
                    }
                    if ( message )
                    {
                        delete message;
                    }
                    if ( commandLine )
                    {
                        free( commandLineHead );
                    }
                }
            }
            sched_yield();
        }
        while ( processedMessage );

        if ( sleepTime < 100000 ) // 100 milliseconds
        {
            if ( sleepTime <= 1000 )
            {
                sleepTime = 5000;
            }
            else if ( sleepTime <= 5000 )
            {
                sleepTime = 25000;
            }
            else if ( sleepTime <= 25000 )
            {
                sleepTime = 50000;
            }
            else if ( sleepTime <= 50000 )
            {
                sleepTime = 100000;
            }
        }
        else if ( mNetworkStopFlag )
        {
            mProcessStopFlag = -1;
        }
        sched_yield();
        usleep( sleepTime );
    }

    for ( ix = 0; ix < mMaxConnections; ix++ )
    {
        pthread_mutex_lock( &( mConnections[ix].CallsMutex ) );
        while ( ( ctlSipCall = mConnections[ix].Calls ) != NULL )
        {
            mConnections[ix].Calls = ctlSipCall->Next;
            // delete ctlSipCall;
        }
        pthread_mutex_unlock( &( mConnections[ix].CallsMutex ) );
        while (   ( ctlSipEndPoint = mConnections[ix].EndPoints )
               != NULL )
        {
            mConnections[ix].EndPoints = ctlSipEndPoint->Next;
            // delete ctlSipEndPoint;
        }
        pthread_mutex_lock( &( mConnections[ix].ProxyEndPointsMutex ) );
        while (   ( ctlSipProxyEndPoint = mConnections[ix].ProxyEndPoints )
               != NULL )
        {
            mConnections[ix].ProxyEndPoints = ctlSipProxyEndPoint->Next;
            // delete ctlSipProxyEndPoint;
        }
        pthread_mutex_unlock( &( mConnections[ix].ProxyEndPointsMutex ) );
        while (   ( ctlSipRegistrar = mConnections[ix].Registrars )
               != NULL )
        {
            mConnections[ix].Registrars = ctlSipRegistrar->Next;
            // delete ctlSipRegistrar;
        }
        while (   ( ctlRtpHandler = mConnections[ix].RtpHandlers )
               != NULL )
        {
            mConnections[ix].RtpHandlers = ctlRtpHandler->Next;
            // delete ctlRtpHandler;
        }
        while ( (  ctlSipRegistrarConnector
                 = mConnections[ix].SipRegistrarConnectors ) != NULL )
        {
            mConnections[ix].SipRegistrarConnectors =
                                                 ctlSipRegistrarConnector->Next;
            // delete ctlSipRegistrarConnectors;
        }
        while (   ( ctlSipDispatcher = mConnections[ix].Dispatchers )
               != NULL )
        {
            mConnections[ix].Dispatchers = ctlSipDispatcher->Next;
            // delete ctlSipDispatcher;
        }
        while ( ( ctlSipUdpPort = mConnections[ix].Ports ) != NULL )
        {
            mConnections[ix].Ports = ctlSipUdpPort->Next;
            // delete ctlSipUdpPort;
        }
    }

    return ERROR_NONE;
}


void  ControlPort::ProcessStop( void )
{
    mProcessStopFlag = -1;
}


bool  ControlPort::AddSipCall( int  aConnectionId, SipCall *  aCall )
{
    controlSipCall *  ctl;

    ctl = new controlSipCall();
    ctl->Next = NULL;
    ctl->Call = aCall;
    ctl->Name = ( char * )malloc( 13 );
    if ( !ctl->Name )
    {
        // delete ctl;
        return 0;
    }
    snprintf( ctl->Name, 13, "auto%x", nextAutoId() );
    pthread_mutex_lock( &( mConnections[aConnectionId].CallsMutex ) );
    ctl->Next = mConnections[aConnectionId].Calls;
    mConnections[aConnectionId].Calls = ctl;
    pthread_mutex_unlock( &( mConnections[aConnectionId].CallsMutex ) );

    return 1;
}


bool  ControlPort::AddSipProxyEndPoint( int                 aConnectionId,
                                        SipProxyEndPoint *  aProxyEndPoint )
{
    controlSipProxyEndPoint *  ctl;

    ctl = new controlSipProxyEndPoint();
    ctl->Next = NULL;
    ctl->ProxyEndPoint = aProxyEndPoint;
    ctl->Name = ( char * )malloc( 13 );
    if ( !ctl->Name )
    {
        // delete ctl;
        return 0;
    }
    snprintf( ctl->Name, 13, "auto%x", nextAutoId() );
    pthread_mutex_lock( &( mConnections[aConnectionId].ProxyEndPointsMutex ) );
    ctl->Next = mConnections[aConnectionId].ProxyEndPoints;
    mConnections[aConnectionId].ProxyEndPoints = ctl;
    pthread_mutex_unlock( &( mConnections[aConnectionId].ProxyEndPointsMutex ) );

    return 1;
}


ControlMessage *  ControlPort::getNextIncoming( int  aSocketIndex )
{
    if ( aSocketIndex < 0 )
    {
        return  mBackgroundIncomingQueue.GetNext();
    }
    else
    {
        return  mIncomingQueues[aSocketIndex].GetNext();
    }
}


ControlPort::Error  ControlPort::addIncoming( int               aSocketIndex,
                                              ControlMessage *  aMessage )
{
    if ( aSocketIndex < 0 )
    {
        return mBackgroundIncomingQueue.Add( aMessage );
    }
    else
    {
        return mIncomingQueues[aSocketIndex].Add( aMessage );
    }
}


ControlPort::Error  ControlPort::preAddIncoming( int               aSocketIndex,
                                                 ControlMessage *  aMessage )
{
    if ( aSocketIndex < 0 )
    {
        return mBackgroundIncomingQueue.PreAdd( aMessage );
    }
    else
    {
        return mIncomingQueues[aSocketIndex].PreAdd( aMessage );
    }
}


ControlMessage *  ControlPort::getNextOutgoing( int  aSocketIndex )
{
    if ( aSocketIndex < 0 )
    {
        return NULL;
    }
    else
    {
        return  mOutgoingQueues[aSocketIndex].GetNext();
    }
}


ControlPort::Error ControlPort::addOutgoing( ControlMessage *  aMessage )
{
    int    socketIndex;
    Error  error;

    error = ERROR_NONE;
    
    socketIndex = aMessage->GetSocketIndex();

    if ( socketIndex >= 0              &&
         socketIndex < mMaxConnections    )
    {
        if ( mSockets[socketIndex].FileDescriptor >= 0 &&
             mSockets[socketIndex].ConnectionId
               == aMessage->GetConnectionId() )
        {
            error = mOutgoingQueues[socketIndex].Add( aMessage );
        }
        else
        {
            error = ERROR_MSG_FOR_NONEXISTENT_CONNECTION;
        }
    }
    else
    {
        error = ERROR_SOCKIDX_OUT_OF_RANGE;
    }

    return error;
}


ControlPort::Error ControlPort::preAddOutgoing( ControlMessage *  aMessage )
{
    int    socketIndex;
    Error  error;

    error = ERROR_NONE;
    
    socketIndex = aMessage->GetSocketIndex();

    if ( socketIndex >= 0              &&
         socketIndex < mMaxConnections    )
    {
        if ( mSockets[socketIndex].FileDescriptor >= 0 &&
             mSockets[socketIndex].ConnectionId
               == aMessage->GetConnectionId() )
        {
            error = mOutgoingQueues[socketIndex].PreAdd( aMessage );
        }
        else
        {
            error = ERROR_MSG_FOR_NONEXISTENT_CONNECTION;
        }
    }
    else
    {
        error = ERROR_SOCKIDX_OUT_OF_RANGE;
    }

    return error;
}


void  ControlPort::processEcho( ControlMessage *  aMessage,
                                char *            aCommandLine )
{
    aMessage->SetText( aCommandLine, "" );
    addOutgoing( aMessage );
}


void  ControlPort::processHelp( ControlMessage *  aMessage,
                                char *            aCommandLine )
{
    aMessage->SetText(
        "Begin Help Text\r\n"
        "Please see README file for now.\r\n"
        "End Help Text\r\n"
    );
    addOutgoing( aMessage );
}


void  ControlPort::processConnection( int               aSocketIndex,
                                      ControlMessage *  aMessage,
                                      char *            aCommandLine )
{
    char *  cp;
    char *  word;
    int     ix;
    int     iy;

    word = GetNextWord( &aCommandLine );
    if ( *word )
    {
        ix = strtol( word, &cp, 16 );
        if ( *cp || ix < 0 || ix >= mMaxConnections )
        {
            aMessage->SetText( "Error: Invalid connection id: ", word );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
        else if ( mSockets[aSocketIndex].ConnectionId == ix )
        {
            ix = -1;
            aMessage->SetText( "Info: Reselected same connection: ", word );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
        else
        {
            for ( iy = 0; iy < mMaxConnections; iy++ )
            {
                if (   mSockets[iy].FileDescriptor >= 0
                    && mSockets[iy].ConnectionId == ix )
                {
                    ix = -1;
                    aMessage->SetText( "Error: Connection in use: ", word );
                    addOutgoing( aMessage );
                    aMessage = NULL;
                    break;
                }
            }
            if ( ix >= 0 )
            {
                mSockets[aSocketIndex].ConnectionId = ix;
                aMessage->SetConnectionId( ix );
                aMessage->SetText( "Info: Connection set to: ", word );
                addOutgoing( aMessage );
                aMessage = NULL;
            }
        }
    }
    else if ( ( word = ( char * )malloc(16) ) == NULL )
    {
        ReportError( "out of memory", __FILE__, __LINE__ );
    }
    else
    {
        if (   aMessage->GetConnectionId() >= 0
            && aMessage->GetConnectionId() < mMaxConnections )
        {
            snprintf( word, 16, "%02X", aMessage->GetConnectionId() );
        }
        else
        {
            strcpy( word, "none" );
        }
        aMessage->SetText( "Info: Connection id is: ", word );
        addOutgoing( aMessage );
        aMessage = NULL;
        free( word );
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::processStatus( ControlMessage *  aMessage,
                                  char *            aCommandLine )
{
    bool                    connected;
    char *                  cp;
    char *                  word;
    controlSipCall *        ctlSipCall;
    controlSipDispatcher *  ctlSipDispatcher;
    controlRtpHandler *     ctlRtpHandler;
    controlSipRegistrarConnector *     ctlSipRegistrarConnector;
    controlSipEndPoint *    ctlSipEndPoint;
    controlSipProxyEndPoint *    ctlSipProxyEndPoint;
    controlSipRegistrar *   ctlSipRegistrar;
    controlSipUdpPort *     ctlSipUdpPort;
    int                     ix;
    int                     iy;
    int                     iz;
    int                     totalSipCalls;
    int                     totalSipDispatchers;
    int                     totalRtpHandlers;
    int                     totalSipRegistrarConnectors;
    int                     totalSipEndPoints;
    int                     totalSipProxyEndPoints;
    int                     totalSipRegistrars;
    int                     totalObjectsOverall;
    int                     totalSipUdpPorts;

    word = GetNextWord( &aCommandLine );
    if ( strcasecmp( word, "of" ) == 0 )
    {
        processStatusOf( aMessage, aCommandLine );
        aMessage = NULL;
    }
    else if ( *word )
    {
        aMessage->SetText( "Error: Syntax error at: ", word );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    #define STATUS_SIZE 1900
    else if ( ( cp = ( char * )malloc( STATUS_SIZE ) ) == NULL )
    {
        ReportError( "out of memory", __FILE__, __LINE__ );
    }
    else
    {
        ix = 0;
        iy = time( NULL ) - mCreationTime;
        ix += snprintf( cp + ix,
                        STATUS_SIZE - ix,
                        "Begin General Status\r\n"
                        "Uptime: %d %02d:%02d:%02d\r\n",
                        iy / 86400,
                        iy % 86400 / 3600,
                        iy % 3600 / 60,
                        iy % 60 );
        if ( aMessage->GetConnectionId() < 0 )
        {
            ix += snprintf( cp + ix, STATUS_SIZE - ix, "Connection: none\r\n" );
        }
        else
        {
            ix += snprintf( cp + ix,
                            STATUS_SIZE - ix,
                            "Connection: %02X\r\n",
                            aMessage->GetConnectionId() );
        }
        totalObjectsOverall = 0;
        totalSipUdpPorts    = 0;
        totalSipDispatchers = 0;
        totalRtpHandlers    = 0;
        totalSipRegistrarConnectors    = 0;
        totalSipEndPoints   = 0;
        totalSipProxyEndPoints   = 0;
        totalSipRegistrars   = 0;
        totalSipCalls       = 0;
        for ( iy = 0; iy < mMaxConnections; iy++ )
        {
            ctlSipUdpPort = mConnections[iy].Ports;
            while ( ctlSipUdpPort )
            {
                totalObjectsOverall++;
                totalSipUdpPorts++;
                ctlSipUdpPort = ctlSipUdpPort->Next;
                sched_yield();
            }
            ctlSipDispatcher = mConnections[iy].Dispatchers;
            while ( ctlSipDispatcher )
            {
                totalObjectsOverall++;
                totalSipDispatchers++;
                ctlSipDispatcher = ctlSipDispatcher->Next;
                sched_yield();
            }
            ctlRtpHandler = mConnections[iy].RtpHandlers;
            while ( ctlRtpHandler )
            {
                totalObjectsOverall++;
                totalRtpHandlers++;
                ctlRtpHandler = ctlRtpHandler->Next;
                sched_yield();
            }
            ctlSipRegistrarConnector = mConnections[iy].SipRegistrarConnectors;
            while ( ctlSipRegistrarConnector )
            {
                totalObjectsOverall++;
                totalSipRegistrarConnectors++;
                ctlSipRegistrarConnector = ctlSipRegistrarConnector->Next;
                sched_yield();
            }
            ctlSipEndPoint = mConnections[iy].EndPoints;
            while ( ctlSipEndPoint )
            {
                totalObjectsOverall++;
                totalSipEndPoints++;
                ctlSipEndPoint = ctlSipEndPoint->Next;
                sched_yield();
            }
            pthread_mutex_lock( &( mConnections[iy].ProxyEndPointsMutex ) );
            ctlSipProxyEndPoint = mConnections[iy].ProxyEndPoints;
            while ( ctlSipProxyEndPoint )
            {
                totalObjectsOverall++;
                totalSipProxyEndPoints++;
                ctlSipProxyEndPoint = ctlSipProxyEndPoint->Next;
                sched_yield();
            }
            pthread_mutex_unlock( &( mConnections[iy].ProxyEndPointsMutex ) );
            ctlSipRegistrar = mConnections[iy].Registrars;
            while ( ctlSipRegistrar )
            {
                totalObjectsOverall++;
                totalSipRegistrars++;
                ctlSipRegistrar = ctlSipRegistrar->Next;
                sched_yield();
            }
            pthread_mutex_lock( &( mConnections[iy].CallsMutex ) );
            ctlSipCall = mConnections[iy].Calls;
            while ( ctlSipCall )
            {
                totalObjectsOverall++;
                totalSipCalls++;
                ctlSipCall = ctlSipCall->Next;
                sched_yield();
            }
            pthread_mutex_unlock( &( mConnections[iy].CallsMutex ) );
        }
        ix += snprintf( cp + ix,
                        STATUS_SIZE - ix,
                        "Total Objects: %d\r\n",
                        totalObjectsOverall );
        if ( totalSipUdpPorts )
        {
            ix += snprintf( cp + ix,
                            STATUS_SIZE - ix,
                            "Total SipUdpPort Objects: %d\r\n",
                            totalSipUdpPorts );
        }
        if ( totalSipDispatchers )
        {
            ix += snprintf( cp + ix,
                            STATUS_SIZE - ix,
                            "Total SipDispatcher Objects: %d\r\n",
                            totalSipDispatchers );
        }
        if ( totalRtpHandlers )
        {
            ix += snprintf( cp + ix,
                            STATUS_SIZE - ix,
                            "Total RtpHandler Objects: %d\r\n",
                            totalRtpHandlers );
        }
        if ( totalSipRegistrarConnectors )
        {
            ix += snprintf( cp + ix,
                            STATUS_SIZE - ix,
                            "Total SipRegistrarConnector Objects: %d\r\n",
                            totalSipRegistrarConnectors );
        }
        if ( totalSipEndPoints )
        {
            ix += snprintf( cp + ix,
                            STATUS_SIZE - ix,
                            "Total SipEndPoint Objects: %d\r\n",
                            totalSipEndPoints );
        }
        if ( totalSipProxyEndPoints )
        {
            ix += snprintf( cp + ix,
                            STATUS_SIZE - ix,
                            "Total SipProxyEndPoint Objects: %d\r\n",
                            totalSipProxyEndPoints );
        }
        if ( totalSipRegistrars )
        {
            ix += snprintf( cp + ix,
                            STATUS_SIZE - ix,
                            "Total SipRegistrar Objects: %d\r\n",
                            totalSipRegistrars );
        }
        if ( totalSipCalls )
        {
            ix += snprintf( cp + ix,
                            STATUS_SIZE - ix,
                            "Total SipCall Objects: %d\r\n",
                            totalSipCalls );
        }
        ix += snprintf( cp + ix,
                        STATUS_SIZE - ix,
                        "Begin Connections Report\r\n"
                        "ID   OBJECTS  |  ID   OBJECTS  |  ID   OBJECTS  |  "
                        "ID   OBJECTS\r\n" );
        for ( iy = 0; iy < mMaxConnections; iy++ )
        {
            connected = false;
            for ( iz = 0; iz < mMaxConnections; iz++ )
            {
                if (   mSockets[iz].FileDescriptor >= 0
                    && mSockets[iz].ConnectionId == iy )
                {
                    connected = true;
                }
            }
            totalObjectsOverall = 0;
            ctlSipUdpPort = mConnections[iy].Ports;
            while ( ctlSipUdpPort )
            {
                totalObjectsOverall++;
                ctlSipUdpPort = ctlSipUdpPort->Next;
                sched_yield();
            }
            ctlSipDispatcher = mConnections[iy].Dispatchers;
            while ( ctlSipDispatcher )
            {
                totalObjectsOverall++;
                ctlSipDispatcher = ctlSipDispatcher->Next;
                sched_yield();
            }
            ctlRtpHandler = mConnections[iy].RtpHandlers;
            while ( ctlRtpHandler )
            {
                totalObjectsOverall++;
                ctlRtpHandler = ctlRtpHandler->Next;
                sched_yield();
            }
            ctlSipRegistrarConnector = mConnections[iy].SipRegistrarConnectors;
            while ( ctlSipRegistrarConnector )
            {
                totalObjectsOverall++;
                ctlSipRegistrarConnector = ctlSipRegistrarConnector->Next;
                sched_yield();
            }
            ctlSipEndPoint = mConnections[iy].EndPoints;
            while ( ctlSipEndPoint )
            {
                totalObjectsOverall++;
                ctlSipEndPoint = ctlSipEndPoint->Next;
                sched_yield();
            }
            pthread_mutex_lock( &( mConnections[iy].ProxyEndPointsMutex ) );
            ctlSipProxyEndPoint = mConnections[iy].ProxyEndPoints;
            while ( ctlSipProxyEndPoint )
            {
                totalObjectsOverall++;
                ctlSipProxyEndPoint = ctlSipProxyEndPoint->Next;
                sched_yield();
            }
            pthread_mutex_unlock( &( mConnections[iy].ProxyEndPointsMutex ) );
            ctlSipRegistrar = mConnections[iy].Registrars;
            while ( ctlSipRegistrar )
            {
                totalObjectsOverall++;
                ctlSipRegistrar = ctlSipRegistrar->Next;
                sched_yield();
            }
            pthread_mutex_lock( &( mConnections[iy].CallsMutex ) );
            ctlSipCall = mConnections[iy].Calls;
            while ( ctlSipCall )
            {
                totalObjectsOverall++;
                ctlSipCall = ctlSipCall->Next;
                sched_yield();
            }
            pthread_mutex_unlock( &( mConnections[iy].CallsMutex ) );
            ix += snprintf( cp + ix,
                            STATUS_SIZE - ix,
                            "%02X%s",
                            iy,
                              aMessage->GetConnectionId() == iy
                            ? "-*-"
                            : connected ? " * " : "   " );
            if ( totalObjectsOverall )
            {
                ix += snprintf( cp + ix,
                                STATUS_SIZE - ix,
                                "%7d",
                                totalObjectsOverall );
            }
            else
            {
                ix += snprintf( cp + ix, STATUS_SIZE - ix, "       " );
            }
            if ( iy % 4 == 3 )
            {
                ix += snprintf( cp + ix, STATUS_SIZE - ix, "\r\n" );
            }
            else
            {
                ix += snprintf( cp + ix, STATUS_SIZE - ix, "  |  " );
            }

        }
        ix += snprintf( cp + ix,
                        STATUS_SIZE - ix,
                        " *  Has a user connected currently\r\n"
                        "-*- The connection currently in use on this socket\r\n"
                        "End Connections Report\r\n"
                        "End General Status\r\n" );
        if ( ix >= STATUS_SIZE )
        {
            ReportError( "exceeded max status size", __FILE__, __LINE__ );
            strcpy( cp,
                    "Error: Exceeded max status buffer size\r\n" );
        }
        aMessage->SetText( cp );
        addOutgoing( aMessage );
        aMessage = NULL;
        free( cp );
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::processStatusOf( ControlMessage *  aMessage,
                                    char *            aCommandLine )
{
    char *                  name;
    controlSipUdpPort *     ctlSipUdpPort;
    controlSipDispatcher *  ctlSipDispatcher;
    controlRtpHandler    *  ctlRtpHandler;
    controlSipRegistrarConnector    *  ctlSipRegistrarConnector;
    controlSipEndPoint *    ctlSipEndPoint;
    controlSipProxyEndPoint *    ctlSipProxyEndPoint;
    controlSipRegistrar *   ctlSipRegistrar;
    controlSipCall *        ctlSipCall;

    name = GetNextWord( &aCommandLine );
    if ( !*name )
    {
        aMessage->SetText(
                        "Error: Name required with \"status of\" command\r\n" );
        addOutgoing( aMessage );
        return;
    }
    else if ( aMessage->GetConnectionId() )
    {
        aMessage->SetText(
                  "Error: Connection required with \"status of\" command\r\n" );
        addOutgoing( aMessage );
        return;
    }

    ctlSipUdpPort = mConnections[aMessage->GetConnectionId()].Ports;
    while ( ctlSipUdpPort )
    {
        if ( strcasecmp( name, ctlSipUdpPort->Name ) == 0 )
        {
            break;
        }
        ctlSipUdpPort = ctlSipUdpPort->Next;
        sched_yield();
    }
    if ( ctlSipUdpPort )
    {
        processStatusOfSipUdpPort( aMessage, aCommandLine, ctlSipUdpPort );
        return;
    }

    ctlSipDispatcher = mConnections[aMessage->GetConnectionId()].Dispatchers;
    while ( ctlSipDispatcher )
    {
        if ( strcasecmp( name, ctlSipDispatcher->Name ) == 0 )
        {
            break;
        }
        ctlSipDispatcher = ctlSipDispatcher->Next;
        sched_yield();
    }
    if ( ctlSipDispatcher )
    {
        processStatusOfSipDispatcher( aMessage,
                                      aCommandLine,
                                      ctlSipDispatcher );
        return;
    }

    ctlRtpHandler = mConnections[aMessage->GetConnectionId()].RtpHandlers;
    while ( ctlRtpHandler )
    {
        if ( strcasecmp( name, ctlRtpHandler->Name ) == 0 )
        {
            break;
        }
        ctlRtpHandler = ctlRtpHandler->Next;
        sched_yield();
    }
    if ( ctlRtpHandler )
    {
        processStatusOfRtpHandler( aMessage, aCommandLine, ctlRtpHandler );
        return;
    }

    ctlSipRegistrarConnector =
               mConnections[aMessage->GetConnectionId()].SipRegistrarConnectors;
    while ( ctlSipRegistrarConnector )
    {
        if ( strcasecmp( name, ctlSipRegistrarConnector->Name ) == 0 )
        {
            break;
        }
        ctlSipRegistrarConnector = ctlSipRegistrarConnector->Next;
        sched_yield();
    }
    if ( ctlSipRegistrarConnector )
    {
        processStatusOfSipRegistrarConnector( aMessage,
                                              aCommandLine,
                                              ctlSipRegistrarConnector );
        return;
    }

    ctlSipEndPoint = mConnections[aMessage->GetConnectionId()].EndPoints;
    while ( ctlSipEndPoint )
    {
        if ( strcasecmp( name, ctlSipEndPoint->Name ) == 0 )
        {
            break;
        }
        ctlSipEndPoint = ctlSipEndPoint->Next;
        sched_yield();
    }
    if ( ctlSipEndPoint )
    {
        processStatusOfSipEndPoint( aMessage, aCommandLine, ctlSipEndPoint );
        return;
    }

    pthread_mutex_lock(
                    &( mConnections[aMessage->GetConnectionId()].ProxyEndPointsMutex ) );
    ctlSipProxyEndPoint = mConnections[aMessage->GetConnectionId()].ProxyEndPoints;
    while ( ctlSipProxyEndPoint )
    {
        if ( strcasecmp( name, ctlSipProxyEndPoint->Name ) == 0 )
        {
            break;
        }
        ctlSipProxyEndPoint = ctlSipProxyEndPoint->Next;
        sched_yield();
    }
    pthread_mutex_unlock(
                    &( mConnections[aMessage->GetConnectionId()].ProxyEndPointsMutex ) );
    if ( ctlSipProxyEndPoint )
    {
        processStatusOfSipProxyEndPoint( aMessage, aCommandLine, ctlSipProxyEndPoint );
        return;
    }

    ctlSipRegistrar = mConnections[aMessage->GetConnectionId()].Registrars;
    while ( ctlSipRegistrar )
    {
        if ( strcasecmp( name, ctlSipRegistrar->Name ) == 0 )
        {
            break;
        }
        ctlSipRegistrar = ctlSipRegistrar->Next;
        sched_yield();
    }
    if ( ctlSipRegistrar )
    {
        processStatusOfSipRegistrar( aMessage, aCommandLine, ctlSipRegistrar );
        return;
    }

    pthread_mutex_lock(
                    &( mConnections[aMessage->GetConnectionId()].CallsMutex ) );
    ctlSipCall = mConnections[aMessage->GetConnectionId()].Calls;
    while ( ctlSipCall )
    {
        if ( strcasecmp( name, ctlSipCall->Name ) == 0 )
        {
            break;
        }
        ctlSipCall = ctlSipCall->Next;
        sched_yield();
    }
    pthread_mutex_unlock(
                    &( mConnections[aMessage->GetConnectionId()].CallsMutex ) );
    if ( ctlSipCall )
    {
        processStatusOfSipCall( aMessage, aCommandLine, ctlSipCall );
        return;
    }

    if ( aMessage )
    {
        aMessage->SetText( "Error: Could not find object named: ", name );
        addOutgoing( aMessage );
        return;
    }

}


void  ControlPort::processStatusOfSipUdpPort(
                                           ControlMessage *     aMessage,
                                           char *               aCommandLine,
                                           controlSipUdpPort *  aCtlSipUdpPort )
{
    SipUdpPort *  sipUdpPort;
    char *        cp;
    int           ix;

    sipUdpPort = aCtlSipUdpPort->Port;

    #define SIP_UDP_PORT_STATUS_SIZE 1300
    cp = ( char * )malloc( SIP_UDP_PORT_STATUS_SIZE );
    if ( !cp )
    {
        ReportError( "out of memory", __FILE__, __LINE__ );
        // delete aMessage;
        return;
    }

    ix = time( NULL ) - sipUdpPort->GetCreationTime();
    snprintf( cp,
              SIP_UDP_PORT_STATUS_SIZE,
              "Begin Status of SipUdpPort %s\r\n"
              "Uptime: %d %02d:%02d:%02d\r\n"
              "IP: %d.%d.%d.%d\r\n"
              "Port: %d\r\n"
              "SIP Messages Sent: %lu\r\n"
              "SIP Messages Received: %lu\r\n"
              "SIP Messages Queued to Send: %u\r\n"
              "Responding Sources Queued to Send: %u\r\n"
              "Requesting Sources Queued to Send: %u\r\n"
              "Retransmitting Sources Queued to Send: %u\r\n"
              "SIP Messages Queued to Dispatch: %u\r\n"
              "Most SIP Messages Queued to Send: %u\r\n"
              "Most Responding Sources Queued to Send: %u\r\n"
              "Most Requesting Sources Queued to Send: %u\r\n"
              "Most Retransmitting Sources Queued to Send: %u\r\n"
              "Most SIP Messages Queued to Dispatch: %u\r\n"
              "SIP Bytes Sent: %llu\r\n"
              "SIP Bytes Received: %llu\r\n"
              "End Status of SipUdpPort %s\r\n",
              aCtlSipUdpPort->Name,
              ix / 86400,
              ix % 86400 / 3600,
              ix % 3600 / 60,
              ix % 60,
              sipUdpPort->GetBoundAddress()       & 0xff,
              sipUdpPort->GetBoundAddress() >>  8 & 0xff,
              sipUdpPort->GetBoundAddress() >> 16 & 0xff,
              sipUdpPort->GetBoundAddress() >> 24 & 0xff,
              ntohs( sipUdpPort->GetBoundPort() ),
              sipUdpPort->GetMessagesSent(),
              sipUdpPort->GetMessagesReceived(),
              sipUdpPort->GetOutgoingQueueQueued(),
              sipUdpPort->GetRespondingUdpSourceQueueQueued(),
              sipUdpPort->GetRequestingUdpSourceQueueQueued(),
              sipUdpPort->GetRetransmittingUdpSourceQueueQueued(),
              sipUdpPort->GetIncomingQueueQueued(),
              sipUdpPort->GetOutgoingQueueMostQueued(),
              sipUdpPort->GetRespondingUdpSourceQueueMostQueued(),
              sipUdpPort->GetRequestingUdpSourceQueueMostQueued(),
              sipUdpPort->GetRetransmittingUdpSourceQueueMostQueued(),
              sipUdpPort->GetIncomingQueueMostQueued(),
              sipUdpPort->GetBytesSent(),
              sipUdpPort->GetBytesReceived(),
              aCtlSipUdpPort->Name );

    aMessage->SetText( cp );
    addOutgoing( aMessage );
    free( cp );
}


void  ControlPort::processStatusOfSipDispatcher(
                                     ControlMessage *        aMessage,
                                     char *                  aCommandLine,
                                     controlSipDispatcher *  aCtlSipDispatcher )
{
    aMessage->SetText( "Error: SipDispatcher status not implemented yet.\r\n" );
    addOutgoing( aMessage );
}


void  ControlPort::processStatusOfRtpHandler(
                                           ControlMessage *     aMessage,
                                           char *               aCommandLine,
                                           controlRtpHandler *  aCtlRtpHandler )
{
    aMessage->SetText( "Error: RtpHandler status not implemented yet.\r\n" );
    addOutgoing( aMessage );
}


void  ControlPort::processStatusOfSipRegistrarConnector(
                     ControlMessage *                aMessage,
                     char *                          aCommandLine,
                     controlSipRegistrarConnector *  aCtlSipRegistrarConnector )
{
    aMessage->SetText(
               "Error: SipRegistrarConnector status not implemented yet.\r\n" );
    addOutgoing( aMessage );
}


void  ControlPort::processStatusOfSipEndPoint(
                                         ControlMessage *      aMessage,
                                         char *                aCommandLine,
                                         controlSipEndPoint *  aCtlSipEndPoint )
{
    aMessage->SetText( "Error: SipEndPoint status not implemented yet.\r\n" );
    addOutgoing( aMessage );
}


void  ControlPort::processStatusOfSipProxyEndPoint(
                                         ControlMessage *      aMessage,
                                         char *                aCommandLine,
                                         controlSipProxyEndPoint *  aCtlSipProxyEndPoint )
{
    aMessage->SetText( "Error: SipProxyEndPoint status not implemented yet.\r\n" );
    addOutgoing( aMessage );
}


void  ControlPort::processStatusOfSipRegistrar(
                                         ControlMessage *       aMessage,
                                         char *                 aCommandLine,
                                         controlSipRegistrar *  aCtlSipRegistrar )
{
    aMessage->SetText( "Error: SipRegistrar status not implemented yet.\r\n" );
    addOutgoing( aMessage );
}


void  ControlPort::processStatusOfSipCall( ControlMessage *  aMessage,
                                           char *            aCommandLine,
                                           controlSipCall *  aCtlSipCall )
{
    aMessage->SetText( "Error: SipCall status not implemented yet.\r\n" );
    addOutgoing( aMessage );
}


void  ControlPort::processSummary( ControlMessage *  aMessage,
                                   char *            aCommandLine )
{
    char *            word;

    word = GetNextWord( &aCommandLine );
    if ( strcasecmp( word, "of" ) == 0 )
    {
        word = GetNextWord( &aCommandLine );
        if ( strcasecmp( word, "sipUdpPort" ) == 0 )
        {
            processSummaryOfSipUdpPort( aMessage, aCommandLine );
            aMessage = NULL;
        }
        else if ( strcasecmp( word, "sipDispatcher" ) == 0 )
        {
            processSummaryOfSipDispatcher( aMessage, aCommandLine );
            aMessage = NULL;
        }
        else if ( strcasecmp( word, "rtphandler" ) == 0 )
        {
            processSummaryOfRtpHandler( aMessage, aCommandLine );
            aMessage = NULL;
        }
        else if ( strcasecmp( word, "sipregistrarconnector" ) == 0 )
        {
            processSummaryOfSipRegistrarConnector( aMessage, aCommandLine );
            aMessage = NULL;
        }
        else if ( strcasecmp( word, "sipEndPoint" ) == 0 )
        {
            processSummaryOfSipEndPoint( aMessage, aCommandLine );
            aMessage = NULL;
        }
        else if ( strcasecmp( word, "sipProxyEndPoint" ) == 0 )
        {
            processSummaryOfSipProxyEndPoint( aMessage, aCommandLine );
            aMessage = NULL;
        }
        else if ( strcasecmp( word, "sipRegistrar" ) == 0 )
        {
            processSummaryOfSipRegistrar( aMessage, aCommandLine );
            aMessage = NULL;
        }
        else if ( strcasecmp( word, "sipCall" ) == 0 )
        {
            processSummaryOfSipCall( aMessage, aCommandLine );
            aMessage = NULL;
        }
        else
        {
            aMessage->SetText( "Error: Syntax error at: ", word );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
    }
    else
    {
        aMessage->SetText( "Error: Syntax error at: ", word );
        addOutgoing( aMessage );
        aMessage = NULL;
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::processSummaryOfSipUdpPort( ControlMessage *  aMessage,
                                               char *            aCommandLine )
{
    aMessage->SetText(
                   "Error: Summary of SipUdpPort is not implemented yet.\r\n" );
    addOutgoing( aMessage );
}


void  ControlPort::processSummaryOfSipDispatcher(
                                                ControlMessage *  aMessage,
                                                char *            aCommandLine )
{
    aMessage->SetText(
                   "Error: Summary of SipDispatcher not implemented yet.\r\n" );
    addOutgoing( aMessage );
}


void  ControlPort::processSummaryOfRtpHandler( ControlMessage *  aMessage,
                                               char *            aCommandLine )
{
    aMessage->SetText(
                   "Error: Summary of RtpHandler not implemented yet.\r\n" );
    addOutgoing( aMessage );
}


void  ControlPort::processSummaryOfSipRegistrarConnector(
                                                ControlMessage *  aMessage,
                                                char *            aCommandLine )
{
    aMessage->SetText(
           "Error: Summary of SipRegistrarConnector not implemented yet.\r\n" );
    addOutgoing( aMessage );
}


void  ControlPort::processSummaryOfSipEndPoint( ControlMessage *  aMessage,
                                                char *            aCommandLine )
{
    aMessage->SetText(
                  "Error: Summary of SipEndPoint is not implemented yet.\r\n" );
    addOutgoing( aMessage );
}


void  ControlPort::processSummaryOfSipProxyEndPoint( ControlMessage *  aMessage,
                                                char *            aCommandLine )
{
    aMessage->SetText(
                  "Error: Summary of SipProxyEndPoint is not implemented yet.\r\n" );
    addOutgoing( aMessage );
}


void  ControlPort::processSummaryOfSipRegistrar( ControlMessage *  aMessage,
                                                char *            aCommandLine )
{
    aMessage->SetText(
                  "Error: Summary of SipRegistrar is not implemented yet.\r\n" );
    addOutgoing( aMessage );
}


void  ControlPort::processSummaryOfSipCall( ControlMessage *  aMessage,
                                            char *            aCommandLine )
{
    char *            cp;
    controlSipCall *  ctlSipCall;
    int               activeCallStateTotals[SipCall::STATE_TERMINATED + 2];
    int               ix;
    int               iy;
    int               mallocSize;
    int               totalActiveCalls;
    int               totalInactiveCalls;

    if ( aMessage->GetConnectionId() < 0 )
    {
        aMessage->SetText(
               "Error: Connection required to generate a sipCall summary\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else
    {
        mallocSize = 600 + SipCall::STATE_TERMINATED * 100;
        if ( ( cp = ( char * )malloc( mallocSize ) ) == NULL )
        {
            ReportError( "out of memory", __FILE__, __LINE__ );
        }
        else
        {
            ix = 0;
            ix += snprintf( cp + ix,
                            mallocSize - ix,
                            "Begin Summary of SipCall\r\n" );
            totalActiveCalls   = 0;
            totalInactiveCalls = 0;
            for ( iy = 0; iy <= SipCall::STATE_TERMINATED + 1; iy++ )
            {
                activeCallStateTotals[iy] = 0;
            }
            pthread_mutex_lock(
                    &( mConnections[aMessage->GetConnectionId()].CallsMutex ) );
            ctlSipCall = mConnections[aMessage->GetConnectionId()].Calls;
            while ( ctlSipCall )
            {
                if (   ctlSipCall->Call->GetState()
                    == SipCall::STATE_TERMINATED )
                {
                    totalInactiveCalls++;
                }
                else
                {
                    totalActiveCalls++;
                }
                if (   ctlSipCall->Call->GetState() >= 0
                    &&    ctlSipCall->Call->GetState()
                       <= SipCall::STATE_TERMINATED )
                {
                    activeCallStateTotals[ctlSipCall->Call->GetState()]++;
                }
                else
                {
                    activeCallStateTotals[SipCall::STATE_TERMINATED + 1]++;
                }
                ctlSipCall = ctlSipCall->Next;
                sched_yield();
            }
            pthread_mutex_unlock(
                    &( mConnections[aMessage->GetConnectionId()].CallsMutex ) );
            ix += snprintf( cp + ix,
                            mallocSize - ix,
                            "Total Objects: %d\r\n"
                            "Total Active Objects: %d\r\n"
                            "Total Inactive Objects: %d\r\n",
                            totalActiveCalls + totalInactiveCalls,
                            totalActiveCalls,
                            totalInactiveCalls );
            for ( iy = 0; iy <= SipCall::STATE_TERMINATED; iy++ )
            {
                if ( activeCallStateTotals[iy] )
                {
                    ix += snprintf( cp + ix,
                                    mallocSize - ix,
                                    "Total Objects in State of %s: %d\r\n",
                                    SipCall::GetNameForState(
                                                     ( SipCall::State )iy ),
                                    activeCallStateTotals[iy] );
                }
            }
            if ( activeCallStateTotals[SipCall::STATE_TERMINATED + 1] )
            {
                snprintf(
                         cp + ix,
                         mallocSize - ix,
                         "Total Objects in an Unknown State: %d\r\n",
                         activeCallStateTotals[SipCall::STATE_TERMINATED + 1] );
            }
            snprintf( cp + ix,
                      mallocSize - ix,
                      "End Summary of SipCall\r\n" );
            if ( ix >= mallocSize )
            {
                ReportError( "exceeded max summary size",
                             __FILE__, __LINE__ );
                strcpy( cp, "Error: Exceeded max status buffer size\r\n" );
            }
            aMessage->SetText( cp );
            addOutgoing( aMessage );
            aMessage = NULL;
            free( cp );
        }
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::processCreate( ControlMessage *  aMessage,
                                  char *            aCommandLine )
{
    char *                  word;

    word = GetNextWord( &aCommandLine );
    if ( strcasecmp( word, "script" ) == 0 )
    {
        createScript( aMessage, aCommandLine );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "sipUdpPort" ) == 0 )
    {
        createSipUdpPort( aMessage, aCommandLine );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "sipDispatcher" ) == 0 )
    {
        createSipDispatcher( aMessage, aCommandLine );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "rtphandler" ) == 0 )
    {
        createRtpHandler( aMessage, aCommandLine );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "sipregistrarconnector" ) == 0 )
    {
        createSipRegistrarConnector( aMessage, aCommandLine );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "sipEndPoint" ) == 0 )
    {
        createSipEndPoint( aMessage, aCommandLine );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "sipProxyEndPoint" ) == 0 )
    {
        createSipProxyEndPoint( aMessage, aCommandLine );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "sipregistrar" ) == 0 )
    {
        createSipRegistrar( aMessage, aCommandLine );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "sipCall" ) == 0 )
    {
        createSipCall( aMessage, aCommandLine );
        aMessage = NULL;
    }
    else
    {
        aMessage->SetText( "Error: Don't know how to create: ", word );
        addOutgoing( aMessage );
        aMessage = NULL;
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::processDelete( ControlMessage *  aMessage,
                                  char *            aCommandLine )
{
    char *                             word;
    controlSipCall *                   ctlSipCall;
    controlSipCall *                   ctlSipCallNext;
    controlSipCall *                   ctlSipCallPrevious;
    controlSipDispatcher *             ctlSipDispatcher;
    controlSipDispatcher *             ctlSipDispatcherNext;
    controlSipDispatcher *             ctlSipDispatcherPrevious;
    controlRtpHandler *                ctlRtpHandler;
    controlRtpHandler *                ctlRtpHandlerNext;
    controlRtpHandler *                ctlRtpHandlerPrevious;
    controlSipRegistrarConnector *     ctlSipRegistrarConnector;
    controlSipRegistrarConnector *     ctlSipRegistrarConnectorNext;
    controlSipRegistrarConnector *     ctlSipRegistrarConnectorPrevious;
    controlSipEndPoint *               ctlSipEndPoint;
    controlSipEndPoint *               ctlSipEndPointNext;
    controlSipEndPoint *               ctlSipEndPointPrevious;
    controlSipProxyEndPoint *               ctlSipProxyEndPoint;
    controlSipProxyEndPoint *               ctlSipProxyEndPointNext;
    controlSipProxyEndPoint *               ctlSipProxyEndPointPrevious;
    controlSipRegistrar *              ctlSipRegistrar;
    controlSipRegistrar *              ctlSipRegistrarNext;
    controlSipRegistrar *              ctlSipRegistrarPrevious;
    controlSipUdpPort *                ctlSipUdpPort;
    controlSipUdpPort *                ctlSipUdpPortNext;
    controlSipUdpPort *                ctlSipUdpPortPrevious;
    int                                connectionId;

    connectionId = aMessage->GetConnectionId();

    if ( !*( word = GetNextWord( &aCommandLine ) ) )
    {
        aMessage->SetText( "Error: Must give name of object to delete\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( connectionId < 0 )
    {
        aMessage->SetText(
                      "Error: Connection required to // delete an object\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else
    {
        ctlSipUdpPortPrevious = NULL;
        ctlSipUdpPort = mConnections[aMessage->GetConnectionId()].Ports;
        while ( ctlSipUdpPort )
        {
            if ( strcasecmp( ctlSipUdpPort->Name, word ) == 0 )
            {
                break;
            }
            ctlSipUdpPort = ctlSipUdpPort->Next;
            sched_yield();
        }
        if ( ctlSipUdpPort )
        {
            ctlSipUdpPortNext = ctlSipUdpPort->Next;
            deleteSipUdpPort( aMessage, ctlSipUdpPort, aCommandLine );
            if ( ctlSipUdpPortPrevious )
            {
                ctlSipUdpPortPrevious->Next = ctlSipUdpPortNext;
            }
            else
            {
                mConnections[aMessage->GetConnectionId()].Ports =
                                                              ctlSipUdpPortNext;
            }
            aMessage = NULL;
        }

        if ( aMessage )
        {
            ctlSipDispatcherPrevious = NULL;
            ctlSipDispatcher = 
                          mConnections[aMessage->GetConnectionId()].Dispatchers;
            while ( ctlSipDispatcher )
            {
                if ( strcasecmp( ctlSipDispatcher->Name, word ) == 0 )
                {
                    break;
                }
                ctlSipDispatcher = ctlSipDispatcher->Next;
                sched_yield();
            }
            if ( ctlSipDispatcher )
            {
                ctlSipDispatcherNext = ctlSipDispatcher->Next;
                deleteSipDispatcher( aMessage, ctlSipDispatcher, aCommandLine );
                if ( ctlSipDispatcherPrevious )
                {
                    ctlSipDispatcherPrevious->Next = ctlSipDispatcherNext;
                }
                else
                {
                    mConnections[aMessage->GetConnectionId()].Dispatchers =
                                                           ctlSipDispatcherNext;
                }
                aMessage = NULL;
            }
        }

        if ( aMessage )
        {
            ctlRtpHandlerPrevious = NULL;
            ctlRtpHandler =
                          mConnections[aMessage->GetConnectionId()].RtpHandlers;
            while ( ctlRtpHandler )
            {
                if ( strcasecmp( ctlRtpHandler->Name, word ) == 0 )
                {
                    break;
                }
                ctlRtpHandler = ctlRtpHandler->Next;
                sched_yield();
            }
            if ( ctlRtpHandler )
            {
                ctlRtpHandlerNext = ctlRtpHandler->Next;
                deleteRtpHandler( aMessage, ctlRtpHandler, aCommandLine );
                if ( ctlRtpHandlerPrevious )
                {
                    ctlRtpHandlerPrevious->Next = ctlRtpHandlerNext;
                }
                else
                {
                    mConnections[aMessage->GetConnectionId()].RtpHandlers =
                                                              ctlRtpHandlerNext;
                }
                aMessage = NULL;
            }
        }

        if ( aMessage )
        {
            ctlSipRegistrarConnectorPrevious = NULL;
            ctlSipRegistrarConnector =
               mConnections[aMessage->GetConnectionId()].SipRegistrarConnectors;
            while ( ctlSipRegistrarConnector )
            {
                if ( strcasecmp( ctlSipRegistrarConnector->Name, word ) == 0 )
                {
                    break;
                }
                ctlSipRegistrarConnector = ctlSipRegistrarConnector->Next;
                sched_yield();
            }
            if ( ctlSipRegistrarConnector )
            {
                ctlSipRegistrarConnectorNext = ctlSipRegistrarConnector->Next;
                deleteSipRegistrarConnector( aMessage,
                                             ctlSipRegistrarConnector,
                                             aCommandLine );
                if ( ctlSipRegistrarConnectorPrevious )
                {
                    ctlSipRegistrarConnectorPrevious->Next =
                                                   ctlSipRegistrarConnectorNext;
                }
                else
                {
                    mConnections[aMessage->GetConnectionId()].
                          SipRegistrarConnectors = ctlSipRegistrarConnectorNext;
                }
                aMessage = NULL;
            }
        }

        if ( aMessage )
        {
            ctlSipEndPointPrevious = NULL;
            ctlSipEndPoint = 
                            mConnections[aMessage->GetConnectionId()].EndPoints;
            while ( ctlSipEndPoint )
            {
                if ( strcasecmp( ctlSipEndPoint->Name, word ) == 0 )
                {
                    break;
                }
                ctlSipEndPoint = ctlSipEndPoint->Next;
                sched_yield();
            }
            if ( ctlSipEndPoint )
            {
                ctlSipEndPointNext = ctlSipEndPoint->Next;
                deleteSipEndPoint( aMessage, ctlSipEndPoint, aCommandLine );
                if ( ctlSipEndPointPrevious )
                {
                    ctlSipEndPointPrevious->Next = ctlSipEndPointNext;
                }
                else
                {
                    mConnections[aMessage->GetConnectionId()].EndPoints =
                                                             ctlSipEndPointNext;
                }
                aMessage = NULL;
            }
        }

        if ( aMessage )
        {
            ctlSipProxyEndPointPrevious = NULL;
            pthread_mutex_lock( &( mConnections[connectionId].ProxyEndPointsMutex ) );
            ctlSipProxyEndPoint = 
                            mConnections[aMessage->GetConnectionId()].ProxyEndPoints;
            while ( ctlSipProxyEndPoint )
            {
                if ( strcasecmp( ctlSipProxyEndPoint->Name, word ) == 0 )
                {
                    break;
                }
                ctlSipProxyEndPoint = ctlSipProxyEndPoint->Next;
                sched_yield();
            }
            if ( ctlSipProxyEndPoint )
            {
                ctlSipProxyEndPointNext = ctlSipProxyEndPoint->Next;
                deleteSipProxyEndPoint( aMessage, ctlSipProxyEndPoint, aCommandLine );
                if ( ctlSipProxyEndPointPrevious )
                {
                    ctlSipProxyEndPointPrevious->Next = ctlSipProxyEndPointNext;
                }
                else
                {
                    mConnections[aMessage->GetConnectionId()].ProxyEndPoints =
                                                             ctlSipProxyEndPointNext;
                }
                aMessage = NULL;
            }
            pthread_mutex_unlock( &( mConnections[connectionId].ProxyEndPointsMutex ) );
        }

        if ( aMessage )
        {
            ctlSipRegistrarPrevious = NULL;
            ctlSipRegistrar = 
                            mConnections[aMessage->GetConnectionId()].Registrars;
            while ( ctlSipRegistrar )
            {
                if ( strcasecmp( ctlSipRegistrar->Name, word ) == 0 )
                {
                    break;
                }
                ctlSipRegistrar = ctlSipRegistrar->Next;
                sched_yield();
            }
            if ( ctlSipRegistrar )
            {
                ctlSipRegistrarNext = ctlSipRegistrar->Next;
                deleteSipRegistrar( aMessage, ctlSipRegistrar, aCommandLine );
                if ( ctlSipRegistrarPrevious )
                {
                    ctlSipRegistrarPrevious->Next = ctlSipRegistrarNext;
                }
                else
                {
                    mConnections[aMessage->GetConnectionId()].Registrars =
                                                             ctlSipRegistrarNext;
                }
                aMessage = NULL;
            }
        }

        if ( aMessage )
        {
            ctlSipCallPrevious = NULL;
            pthread_mutex_lock( &( mConnections[connectionId].CallsMutex ) );
            ctlSipCall = mConnections[connectionId].Calls;
            while ( ctlSipCall )
            {
                if ( strcasecmp( ctlSipCall->Name, word ) == 0 )
                {
                    break;
                }
                ctlSipCallPrevious = ctlSipCall;
                ctlSipCall = ctlSipCall->Next;
                sched_yield();
            }
            if ( ctlSipCall )
            {
                ctlSipCallNext = ctlSipCall->Next;
                if ( deleteSipCall( aMessage, ctlSipCall, aCommandLine ) )
                {
                    if ( ctlSipCallPrevious )
                    {
                        ctlSipCallPrevious->Next = ctlSipCallNext;
                    }
                    else
                    {
                        mConnections[aMessage->GetConnectionId()].Calls =
                                                                 ctlSipCallNext;
                    }
                }
                aMessage = NULL;
            }
            pthread_mutex_unlock( &( mConnections[connectionId].CallsMutex ) );
        }

        if ( aMessage )
        {
            aMessage->SetText( "Error: Could not find object named: ", word );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::processExecute( ControlMessage *  aMessage,
                                   char *            aCommandLine )
{
    char *                name;
    char *                cp;
    char *                cp2;
    char *                command;
    controlMessageNode *  node;
    controlScript *       script;
    char *                parameters[9];
    int                   ix;
    bool                  inBackground;

    name = GetNextWord( &aCommandLine );
    if ( !*name )
    {
        aMessage->SetText(
                          "Error: Execute command requires a script name\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else
    {
        inBackground = false;
        parameters[0] = GetNextWord( &aCommandLine );
        parameters[1] = GetNextWord( &aCommandLine );
        if (   strcasecmp( parameters[0], "in" ) == 0
            && strcasecmp( parameters[1], "background" ) == 0 )
        {
            inBackground = true;
            parameters[0] = GetNextWord( &aCommandLine );
            parameters[1] = GetNextWord( &aCommandLine );
        }
        for ( ix = 2; ix < 9; ix++ )
        {
            parameters[ix] = GetNextWord( &aCommandLine );
        }
        for ( ix = 0; ix < 9; ix++ )
        {
            if ( !*parameters[ix] )
            {
                parameters[ix] = "NULL";
            }
        }
        if ( *( cp = GetNextWord( &aCommandLine ) ) )
        {
            aMessage->SetText( "Error: Too many parameters given at: ", cp );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
        else
        {
            script = mConnections[aMessage->GetConnectionId()].Scripts;
            while ( script )
            {
                if ( strcasecmp( script->Name, name ) == 0 )
                {
                    break;
                }
                script = script->Next;
            }
            if ( !script )
            {
                aMessage->SetText( "Error: Cannot find script named: ", name );
                addOutgoing( aMessage );
                aMessage = NULL;
            }
            else
            {
                node = script->Messages;
                while ( node )
                {
                    command = node->Message->GetText();
                    cp = command;
                    while ( ( cp = strchr( cp, '$' ) ) != NULL )
                    {
                        if ( cp[1] >= '1' && cp[1] <= '9' )
                        {
                            cp2 = ( char * )malloc(
                                               strlen( command )
                                             + strlen( parameters[cp[1] - '1'] )
                                             - 1 );
                            strncpy( cp2, command, cp - command );
                            strcpy( cp2 + ( cp - command ),
                                    parameters[cp[1] - '1'] );
                            strcpy(   cp2
                                    + ( cp - command )
                                    + strlen( parameters[cp[1] - '1'] ),
                                    cp + 2 );
                            cp =  cp2
                                + ( cp - command )
                                + strlen( parameters[cp[1] - '1'] );
                            free( command );
                            command = cp2;
                        }
                        else
                        {
                            cp++;
                        }
                    }
                    preAddIncoming( aMessage->GetSocketIndex(),
                                    new ControlMessage(
                                                   this,
                                                   aMessage->GetConnectionId(),
                                                       inBackground
                                                   ? -1
                                                   : aMessage->GetSocketIndex(),
                                                   command ) );
                    node = node->Next;
                }
                aMessage->SetText( "Info: Queued execution of script: ", name );
                addOutgoing( aMessage );
                aMessage = NULL;
            }
        }
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::processIssue( ControlMessage *  aMessage,
                                 char *            aCommandLine )
{
    char * word;
    
    controlSipCall *                 ctlSipCall;
    controlSipDispatcher *           ctlSipDispatcher;
    controlRtpHandler *              ctlRtpHandler;
    controlSipRegistrarConnector *   ctlSipRegistrarConnector;
    controlSipEndPoint *             ctlSipEndPoint;
    controlSipProxyEndPoint *        ctlSipProxyEndPoint;
    controlSipRegistrar *            ctlSipRegistrar;
    controlSipUdpPort *              ctlSipUdpPort;
    
    long microseconds;

    word = GetNextWord( &aCommandLine );
    if ( !*word )
    {
        aMessage->SetText( "Error: Issue command requires an object name\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( aMessage->GetConnectionId() < 0 )
    {
        aMessage->SetText(
                          "Error: Connection required to issue a command\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "with" ) == 0 )
    {
        if ( strcasecmp( word = GetNextWord( &aCommandLine ),
                         "interval" ) != 0 )
        {
            aMessage->SetText( "Error: Expected \"interval\", got: ", word );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
        else if ( !*( word = GetNextWord( &aCommandLine ) ) )
        {
            aMessage->SetText(
                    "Error: No microseconds specified after \"interval\"\r\n" );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
        else if ( ( microseconds = strtol( word, &word, 10 ) ) < 0 )
        {
            aMessage->SetText(
                    "Error: No microseconds specified after \"interval\"\r\n" );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
        else if ( *word )
        {
            aMessage->SetText(
               "Error: Invalid microseconds specified after \"interval\"\r\n" );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
        else if ( strcasecmp( word = GetNextWord( &aCommandLine ),
                              "all" ) != 0 )
        {
            aMessage->SetText( "Error: Expected \"all\", got: ", word );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
        else
        {
            processIssueAll( aMessage, aCommandLine, microseconds );
            aMessage = NULL;
        }
    }
    else if ( strcasecmp( word, "all" ) == 0 )
    {
        processIssueAll( aMessage, aCommandLine, 0 );
        aMessage = NULL;
    }
    else
    {
        //
        // An Issue command is being processed without the optional 
        //     [ With Interval <Microseconds> ]
        // and without the
        //    All
        // qualifier
        //
        // Now we must determine the nature of the <ObjectType>
        // (i.e. the object of the Issue command). The <ObjectType> must
        // be one of:
        //
        //  SipUdpPort
        //  SipDispatcher
        //  RtpHandler
        //  SipEndPoint
        //  SipRegistrarConnector
        //  SipProxyEndPoint
        //  SipCall
        //  SipRegistrar
        //
        
        ctlSipUdpPort = mConnections[aMessage->GetConnectionId()].Ports;
        while ( ctlSipUdpPort )
        {
            if ( strcasecmp( ctlSipUdpPort->Name, word ) == 0 )
            {
                break;
            }
            ctlSipUdpPort = ctlSipUdpPort->Next;
            sched_yield();
        }
        if ( ctlSipUdpPort )
        {
            issueSipUdpPort( aMessage, ctlSipUdpPort, aCommandLine );
            aMessage = NULL;
        }
        else
        {
            ctlSipDispatcher = 
                          mConnections[aMessage->GetConnectionId()].Dispatchers;
            while ( ctlSipDispatcher )
            {
                if ( strcasecmp( ctlSipDispatcher->Name, word ) == 0 )
                {
                    break;
                }
                ctlSipDispatcher = ctlSipDispatcher->Next;
                sched_yield();
            }
            if ( ctlSipDispatcher )
            {
                issueSipDispatcher( aMessage, ctlSipDispatcher, aCommandLine );
                aMessage = NULL;
            }
            else
            {
                ctlRtpHandler = 
                          mConnections[aMessage->GetConnectionId()].RtpHandlers;
                while ( ctlRtpHandler )
                {
                    if ( strcasecmp( ctlRtpHandler->Name, word ) == 0 )
                    {
                        break;
                    }
                    ctlRtpHandler = ctlRtpHandler->Next;
                    sched_yield();
                }
                if ( ctlRtpHandler )
                {
                    issueRtpHandler( aMessage, ctlRtpHandler, aCommandLine );
                    aMessage = NULL;
                }
                else
                {
                    ctlSipRegistrarConnector =
                                      mConnections[aMessage->GetConnectionId()].
                                                         SipRegistrarConnectors;
                    while ( ctlSipRegistrarConnector )
                    {
                        if ( strcasecmp( ctlSipRegistrarConnector->Name,
                                         word ) == 0 )
                        {
                            break;
                        }
                        ctlSipRegistrarConnector =
                                                 ctlSipRegistrarConnector->Next;
                        sched_yield();
                    }
                    if ( ctlSipRegistrarConnector )
                    {
                        issueSipRegistrarConnector( aMessage,
                                                    ctlSipRegistrarConnector,
                                                    aCommandLine );
                        aMessage = NULL;
                    }
                    else
                    {
                        ctlSipEndPoint = 
                            mConnections[aMessage->GetConnectionId()].EndPoints;
                        while ( ctlSipEndPoint )
                        {
                            if ( strcasecmp( ctlSipEndPoint->Name, word ) == 0 )
                            {
                                break;
                            }
                            ctlSipEndPoint = ctlSipEndPoint->Next;
                            sched_yield();
                        }
                        if ( ctlSipEndPoint )
                        {
                            issueSipEndPoint( aMessage,
                                              ctlSipEndPoint,
                                              aCommandLine );
                            aMessage = NULL;
                        }
                        else
                        {
                            pthread_mutex_lock( &( mConnections[aMessage->GetConnectionId()].ProxyEndPointsMutex ) );
                            ctlSipProxyEndPoint = 
                                mConnections[aMessage->GetConnectionId()].ProxyEndPoints;
                            while ( ctlSipProxyEndPoint )
                            {
                                if ( strcasecmp( ctlSipProxyEndPoint->Name, word ) == 0 )
                                {
                                    break;
                                }
                                ctlSipProxyEndPoint = ctlSipProxyEndPoint->Next;
                                sched_yield();
                            }
                            pthread_mutex_unlock( &( mConnections[aMessage->GetConnectionId()].ProxyEndPointsMutex ) );
                            if ( ctlSipProxyEndPoint )
                            {
                                issueSipProxyEndPoint( aMessage,
                                                  ctlSipProxyEndPoint,
                                                  aCommandLine );
                                aMessage = NULL;
                            }
                            else
                            {
                                pthread_mutex_lock(
                                       &( mConnections[aMessage->GetConnectionId()].
                                                                     CallsMutex ) );
                                ctlSipCall =
                                    mConnections[aMessage->GetConnectionId()].Calls;
                                while ( ctlSipCall )
                                {
                                    if ( strcasecmp( ctlSipCall->Name, word ) == 0 )
                                    {
                                        break;
                                    }
                                    ctlSipCall = ctlSipCall->Next;
                                    sched_yield();
                                }
                                pthread_mutex_unlock(
                                       &( mConnections[aMessage->GetConnectionId()].
                                                                     CallsMutex ) );
                                if ( ctlSipCall )
                                {
                                    issueSipCall( aMessage,
                                                  ctlSipCall,
                                                  aCommandLine );
                                    aMessage = NULL;
                                }
                                else
                                {
                                    ctlSipRegistrar = 
                                        mConnections[aMessage->GetConnectionId()].Registrars;
                                    while ( ctlSipRegistrar )
                                    {
                                        if ( strcasecmp( ctlSipRegistrar->Name, word ) == 0 )
                                        {
                                            break;
                                        }
                                        ctlSipRegistrar = ctlSipRegistrar->Next;
                                        sched_yield();
                                    }
                                    if ( ctlSipRegistrar )
                                    {
                                        issueSipRegistrar( aMessage,
                                                          ctlSipRegistrar,
                                                          aCommandLine );
                                        aMessage = NULL;
                                    }
                                    else
                                    {
                                        aMessage->SetText(
                                                 "Error: Could not find object named: ",
                                                 word );
                                        addOutgoing( aMessage );
                                        aMessage = NULL;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::processIssueAll( ControlMessage *  aMessage,
                                    char *            aCommandLine,
                                    long              aMicroseconds )
{
    char *              word;
    char *              cp;
    controlSipCall *    ctlSipCall;
    timeval             waitUntil;
    timeval             now;
    unsigned long long  offBy;
    unsigned long long  totalObjects;

    if ( !*( word = GetNextWord( &aCommandLine ) ) )
    {
        aMessage->SetText( "Error: Expected an object type, got none\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "sipudpport" ) == 0 )
    {
        aMessage->SetText(
                       "Error: Issue All SipUdpPort not implemented yet.\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "sipdispatcher" ) == 0 )
    {
        aMessage->SetText(
                    "Error: Issue All SipDispatcher not implemented yet.\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "rtphandler" ) == 0 )
    {
        aMessage->SetText(
                    "Error: Issue All RtpHandler not implemented yet.\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "sipregistrarconnector" ) == 0 )
    {
        aMessage->SetText(
            "Error: Issue All SipRegistrarConnector not implemented yet.\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "sipendpoint" ) == 0 )
    {
        aMessage->SetText(
                      "Error: Issue All SipEndPoint not implemented yet.\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "sipProxyEndpoint" ) == 0 )
    {
        aMessage->SetText(
                      "Error: Issue All SipProxyEndPoint not implemented yet.\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "sipregistrar" ) == 0 )
    {
        aMessage->SetText(
                      "Error: Issue All SipRegistrar not implemented yet.\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "sipcall" ) == 0 )
    {
        gettimeofday( &waitUntil, NULL );
        offBy = 0;
        totalObjects = 0;
        pthread_mutex_lock(
                    &( mConnections[aMessage->GetConnectionId()].CallsMutex ) );
        ctlSipCall = mConnections[aMessage->GetConnectionId()].Calls;
        while ( ctlSipCall )
        {
            gettimeofday( &now, NULL );
            while (   now.tv_sec < waitUntil.tv_sec
                   || (   now.tv_sec == waitUntil.tv_sec
                       && now.tv_usec < waitUntil.tv_usec ) )
            {
                sched_yield();
                gettimeofday( &now, NULL );
            }
            offBy +=  ( now.tv_sec - waitUntil.tv_sec ) * 1000000L
                    + ( now.tv_usec - waitUntil.tv_usec );
            totalObjects++;
            issueSipCall( new ControlMessage( this,
                                              aMessage->GetConnectionId(),
                                              aMessage->GetSocketIndex(),
                                              aCommandLine ),
                          ctlSipCall,
                          aCommandLine );
            ctlSipCall = ctlSipCall->Next;
            waitUntil.tv_usec += aMicroseconds;
            if ( waitUntil.tv_usec >= 1000000L )
            {
                waitUntil.tv_sec += waitUntil.tv_usec / 1000000L;
                waitUntil.tv_usec %= 1000000L;
            }
        }
        pthread_mutex_unlock(
                    &( mConnections[aMessage->GetConnectionId()].CallsMutex ) );
        cp = ( char * )malloc( 1024 );
        if ( totalObjects )
        {
            if ( aMicroseconds )
            {
                snprintf( cp,
                          1024,
                          "Info: Issue All SipCall completed. Average issue "
                          "was off by %.0lf microseconds (%.2lf%%).\r\n",
                          ( double )offBy / ( double )totalObjects,
                            ( double )offBy / ( double )totalObjects
                          / ( double )aMicroseconds * 100.0 );
            }
            else
            {
                snprintf( cp,
                          1024,
                          "Info: Issue All SipCall completed. Average issue "
                          "was off by %.0lf microseconds.\r\n",
                          ( double )offBy / ( double )totalObjects );
            }
        }
        else
        {
            snprintf( cp,
                      1024,
                      "Info: Issue All SipCall completed. No objects.\n" );
        }
        aMessage->SetText( cp );
        free( cp );
        addOutgoing( aMessage );
        aMessage = NULL;
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::processSleep( ControlMessage *  aMessage,
                                 char *            aCommandLine )
{
    char *   amount;
    char *   units;
    timeval  waitUntil;
    timeval  now;

    if ( !*( amount = GetNextWord( &aCommandLine ) ) )
    {
        aMessage->SetText( "Error: Sleep command requires an amount\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( !*( units = GetNextWord( &aCommandLine ) ) )
    {
        aMessage->SetText( "Error: Sleep command requires a unit type\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if (   strcasecmp( units, "microseconds" ) == 0
             || strcasecmp( units, "microsecond" ) == 0 )
    {
        gettimeofday( &waitUntil, NULL );
        waitUntil.tv_usec += strtol( amount, NULL, 10 );
        if ( waitUntil.tv_usec > 1000000L )
        {
            waitUntil.tv_sec += waitUntil.tv_usec / 1000000L;
            waitUntil.tv_usec %= 1000000L;
        }
        gettimeofday( &now, NULL );
        while (   now.tv_sec < waitUntil.tv_sec
               || (   now.tv_sec == waitUntil.tv_sec
                   && now.tv_usec < waitUntil.tv_usec ) )
        {
            sched_yield();
            gettimeofday( &now, NULL );
        }
    }
    else if (   strcasecmp( units, "milliseconds" ) == 0
             || strcasecmp( units, "millisecond" ) == 0 )
    {
        sched_yield();
        usleep( strtol( amount, NULL, 10 ) * 1000 );
    }
    else if (   strcasecmp( units, "seconds" ) == 0
             || strcasecmp( units, "second" ) == 0 )
    {
        sched_yield();
        sleep( strtol( amount, NULL, 10 ) );
    }
    else if (   strcasecmp( units, "minutes" ) == 0
             || strcasecmp( units, "minute" ) == 0 )
    {
        sched_yield();
        sleep( strtol( amount, NULL, 10 ) * 60 );
    }
    else
    {
        aMessage->SetText( "Error: Unknown units: ", units );
        addOutgoing( aMessage );
        aMessage = NULL;
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::processShutdown( ControlMessage *  aMessage,
                                    char *            aCommandLine )
{
    char *                  word;
    controlRtpHandler *     ctlRtpHandler;
    controlSipCall *        ctlSipCall;
    controlSipDispatcher *  ctlSipDispatcher;
    controlSipUdpPort *     ctlSipUdpPort;
    int                     ix;
    bool                    callInProgress;
    int                     elapsed;
    int                     connectionId;
    int                     socketIndex;

    connectionId = aMessage->GetConnectionId();
    socketIndex = aMessage->GetSocketIndex();

    word = GetNextWord( &aCommandLine );
    if ( strcasecmp( word, "now" ) == 0 )
    {
        aMessage->SetText( "Info: Shutdown in progress\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
        sched_yield();
        for ( ix = 0; ix < mMaxConnections; ix++ )
        {
            ctlRtpHandler = mConnections[ix].RtpHandlers;
            while ( ctlRtpHandler )
            {
                ctlRtpHandler->Handler->Stop();
                ctlRtpHandler = ctlRtpHandler->Next;
                sched_yield();
            }
            ctlSipDispatcher = mConnections[ix].Dispatchers;
            while ( ctlSipDispatcher )
            {
                ctlSipDispatcher->Dispatcher->Stop();
                ctlSipDispatcher = ctlSipDispatcher->Next;
                sched_yield();
            }
            ctlSipUdpPort = mConnections[ix].Ports;
            while ( ctlSipUdpPort )
            {
                ctlSipUdpPort->Port->Stop();
                ctlSipUdpPort = ctlSipUdpPort->Next;
                sched_yield();
            }
            // FIXME: Should join all these threads.
        }
        NetworkStop();
        ProcessStop();
    }
    else if ( *word )
    {
        aMessage->SetText( "Error: Unknown parameter for shutdown: ", word );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else
    {
        aMessage->SetText( "Info: Shutdown in progress\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
        callInProgress = true;
        elapsed = 0;
        while ( callInProgress && elapsed < 60 )
        {
            sched_yield();
            callInProgress = false;
            for ( ix = 0; ix < mMaxConnections; ix++ )
            {
                pthread_mutex_lock( &( mConnections[ix].CallsMutex ) );
                ctlSipCall = mConnections[ix].Calls;
                while ( ctlSipCall )
                {
                    switch ( ctlSipCall->Call->GetState() )
                    {
                        case SipCall::STATE_INVALID:
                        case SipCall::STATE_CREATED:
                        case SipCall::STATE_SEND_BYE:
                        case SipCall::STATE_RESEND_BYE:
                        case SipCall::STATE_SENT_BYE:
                        case SipCall::STATE_GOT_OK_FOR_BYE:
                        case SipCall::STATE_GOT_BYE:
                        case SipCall::STATE_SEND_OK_FOR_BYE:
                        case SipCall::STATE_TERMINATE:
                        case SipCall::STATE_TERMINATED:
                            break;
                        case SipCall::STATE_TIMER_FIRE:
                        case SipCall::STATE_GOT_MESSAGE:
                        case SipCall::STATE_SEND_INVITE:
                        case SipCall::STATE_RESEND_INVITE:
                        case SipCall::STATE_SENT_INVITE:
                        case SipCall::STATE_INVITE_GOT_PROVISIONAL:
                        case SipCall::STATE_INVITE_GOT_PROVISIONAL_WAIT:
                        case SipCall::STATE_GOT_OK_FOR_INVITE:
                        case SipCall::STATE_SEND_ACK_FOR_INVITE:
                        case SipCall::STATE_SEND_TRYING_FOR_INVITE:
                        case SipCall::STATE_SENT_TRYING_FOR_INVITE:
                        case SipCall::STATE_SEND_RINGING:
                        case SipCall::STATE_SENT_RINGING:
                        case SipCall::STATE_SEND_OK_FOR_INVITE:
                        case SipCall::STATE_RESEND_OK_FOR_INVITE:
                        case SipCall::STATE_SENT_OK_FOR_INVITE:
                            callInProgress = true;
                            break;
                        case SipCall::STATE_ESTABLISHED:
                            callInProgress = true;
                            ctlSipCall->Call->Disconnect();
                            break;
                    }
                    ctlSipCall = ctlSipCall->Next;
                }
                pthread_mutex_unlock( &( mConnections[ix].CallsMutex ) );
            }
            if ( callInProgress )
            {
                sleep( 1 );
            }
            elapsed++;
        }
        sched_yield();
        for ( ix = 0; ix < mMaxConnections; ix++ )
        {
            ctlRtpHandler = mConnections[ix].RtpHandlers;
            while ( ctlRtpHandler )
            {
                ctlRtpHandler->Handler->Stop();
                ctlRtpHandler = ctlRtpHandler->Next;
                sched_yield();
            }
            ctlSipDispatcher = mConnections[ix].Dispatchers;
            while ( ctlSipDispatcher )
            {
                ctlSipDispatcher->Dispatcher->Stop();
                ctlSipDispatcher = ctlSipDispatcher->Next;
                sched_yield();
            }
            ctlSipUdpPort = mConnections[ix].Ports;
            while ( ctlSipUdpPort )
            {
                ctlSipUdpPort->Port->Stop();
                ctlSipUdpPort = ctlSipUdpPort->Next;
                sched_yield();
            }
            // FIXME: Should join all these threads.
            addOutgoing( new ControlMessage( this,
                                             connectionId,
                                             socketIndex,
                                             "Info: Exiting process\r\n" ) );
        }
        NetworkStop();
        ProcessStop();
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::createScript( ControlMessage *     aMessage,
                                 char *               aCommandLine )
{
    char *                marker;
    char *                msgText;
    char *                word;
    ControlMessage *      msg;
    controlMessageNode *  newNode;
    controlScript *       script;

    word = GetNextWord( &aCommandLine );
    if ( !*word )
    {
        aMessage->SetText( "Error: Name required to create script\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( aMessage->GetConnectionId() < 0 )
    {
        aMessage->SetText(
                        "Error: Connection required to create sipUdpPort\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else
    {
        script = new controlScript();
        script->Name = strdup( word );

        msg = getNextIncoming( aMessage->GetSocketIndex() );
        while ( !msg )
        {
            sched_yield();
            usleep( 100000 );
            msg = getNextIncoming( aMessage->GetSocketIndex() );
        }
        marker = msg->GetText();
        // delete msg;

        script->Messages = NULL;
        newNode = NULL;
        while ( true )
        {
            msg = getNextIncoming( aMessage->GetSocketIndex() );
            while ( !msg )
            {
                sched_yield();
                usleep( 100000 );
                msg = getNextIncoming( aMessage->GetSocketIndex() );
            }
            msgText = msg->GetText();
            if ( strcmp( msgText, marker ) == 0 )
            {
                free( msgText );
                // delete msg;
                break;
            }
            free( msgText );
            newNode = ( controlMessageNode * )malloc(
                                                 sizeof( controlMessageNode ) );
            newNode->Message = msg;
            // We store the script as a backwards linked list, as we'll push
            // them onto the front of the queue during execution.
            newNode->Next = script->Messages;
            script->Messages = newNode;
        }

        free( marker );

        script->Next = mConnections[aMessage->GetConnectionId()].Scripts;
        mConnections[aMessage->GetConnectionId()].Scripts = script;
        aMessage->SetText( "Info: Script created: ", script->Name );
        addOutgoing( aMessage );
        aMessage = NULL;
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::createSipUdpPort( ControlMessage *     aMessage,
                                     char *               aCommandLine )
{
    char *               cp;
    char *               word;
    controlSipUdpPort *  ctlSipUdpPort;
    pthread_t            sipUdpPortThread;

    // FIXME: Check name.

    word = GetNextWord( &aCommandLine );
    if ( !*word )
    {
        aMessage->SetText( "Error: Name required to create sipUdpPort\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( aMessage->GetConnectionId() < 0 )
    {
        aMessage->SetText(
                        "Error: Connection required to create sipUdpPort\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else
    {
        ctlSipUdpPort = new controlSipUdpPort();
        ctlSipUdpPort->Next = NULL;
        if ( ( ctlSipUdpPort->Name = strdup( word ) ) == NULL )
        {
            // delete ctlSipUdpPort;
            ReportError( "out of memory", __FILE__, __LINE__ );
        }
        else
        {
            ctlSipUdpPort->Port = new SipUdpPort();
            ctlSipUdpPort->Port->SetBoundAddress( INADDR_ANY );
            ctlSipUdpPort->Port->SetBoundPort( htons( 5060 ) );

            word = GetNextWord( &aCommandLine );
            if ( *word )
            {
                if ( strcasecmp( word, "using" ) == 0 )
                {
                    word = GetNextWord( &aCommandLine );
                    cp = word;
                    while ( *cp && *cp != ':' )
                    {
                        cp++;
                    }
                    if ( *cp != ':' )
                    {
                        aMessage->SetText( "Error: Expected ip:port at: ",
                                          word );
                        addOutgoing( aMessage );
                        aMessage = NULL;
                    }
                    else
                    {
                        *cp = '\0';
                        cp++;
                        ctlSipUdpPort->Port->SetBoundAddress(
                                                            inet_addr( word ) );
                        ctlSipUdpPort->Port->SetBoundPort(
                                              htons( strtol( cp, NULL, 10 ) ) );
                    }
                }
                else
                {
                    aMessage->SetText( "Error: Bad syntax at: ", word );
                    addOutgoing( aMessage );
                    aMessage = NULL;
                }
            }

            if ( aMessage )
            {
                switch ( pthread_create( &sipUdpPortThread,
                                         NULL,
                                         ctlSipUdpPort->ThreadFunction,
                                         ctlSipUdpPort->Port ) )
                {
                    case 0:
                        ctlSipUdpPort->Next =
                            mConnections[aMessage->GetConnectionId()].Ports;
                        mConnections[aMessage->GetConnectionId()].Ports =
                                                                  ctlSipUdpPort;
                        aMessage->SetText( "Info: sipUdpPort created: ",
                                          ctlSipUdpPort->Name );
                        addOutgoing( aMessage );
                        aMessage = NULL;
                        ctlSipUdpPort = NULL;
                        break;
                    case EAGAIN:
                        ReportError( "too many threads", __FILE__, __LINE__ );
                        break;
                    case EINVAL:
                        ReportError( "invalid thread attributes",
                                      __FILE__, __LINE__ );
                        break;
                    case EPERM:
                        ReportError( "insufficient thread permissions",
                                      __FILE__, __LINE__ );
                        break;
                    default:
                        ReportError( "unknown thread error",
                                      __FILE__, __LINE__ );
                        break;
                }
            }
            if ( ctlSipUdpPort )
            {
                // delete ctlSipUdpPort->Port;
                free( ctlSipUdpPort->Name );
                // delete ctlSipUdpPort;
            }
        }
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::createSipDispatcher( ControlMessage *  aMessage,
                                        char *            aCommandLine )
{
    char *                  word;
    char *                  name;
    controlSipUdpPort *     ctlSipUdpPort;
    controlSipDispatcher *  ctlSipDispatcher;
    pthread_t               sipDispatcherThread;

    name = GetNextWord( &aCommandLine );
    if ( !*name )
    {
        aMessage->SetText( "Error: Name required to create sipDispatcher\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( aMessage->GetConnectionId() < 0 )
    {
        aMessage->SetText(
                     "Error: Connection required to create sipDispatcher\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else
    {
        word = GetNextWord( &aCommandLine );
        if ( *word && strcasecmp( word, "using" ) != 0 )
        {
            aMessage->SetText( "Error: Expected \"using\" at: ", word );
            addOutgoing( aMessage );
            return;
        }
        ctlSipUdpPort = NULL;
        if ( *word )
        {
            word = GetNextWord( &aCommandLine );
            if ( !*word )
            {
                aMessage->SetText(
                         "Error: Expected \"SipUdpPort\" after \"using\"\r\n" );
                addOutgoing( aMessage );
                return;
            }
            if ( strcasecmp( word, "sipUdpPort" ) != 0 )
            {
                aMessage->SetText( "Error: Expected \"SipUdpPort\" at: ",
                                   word );
                addOutgoing( aMessage );
                return;
            }
            word = GetNextWord( &aCommandLine );
            if ( !*word )
            {
                aMessage->SetText( "Error: Expected name of SipUdpPort\r\n" );
                addOutgoing( aMessage );
                return;
            }
            ctlSipUdpPort = mConnections[aMessage->GetConnectionId()].Ports;
            while ( ctlSipUdpPort )
            {
                if ( strcasecmp( word, ctlSipUdpPort->Name ) == 0 )
                {
                    break;
                }
                ctlSipUdpPort = ctlSipUdpPort->Next;
                sched_yield();
            }
        }
        if ( !ctlSipUdpPort )
        {
            ctlSipUdpPort = mConnections[aMessage->GetConnectionId()].Ports;
        }
        if ( !ctlSipUdpPort )
        {
            aMessage->SetText(
                  "Error: No SipUdpPorts to use with a new SipDispatcher\r\n" );
            addOutgoing( aMessage );
            return;
        }

        ctlSipDispatcher = new controlSipDispatcher();
        ctlSipDispatcher->Next = NULL;
        ctlSipDispatcher->Name = strdup( name );
        ctlSipDispatcher->Dispatcher = new SipDispatcher();
        ctlSipDispatcher->Dispatcher->SetUdpPort( ctlSipUdpPort->Port );

        switch ( pthread_create( &sipDispatcherThread,
                                 NULL,
                                 ctlSipDispatcher->ThreadFunction,
                                 ctlSipDispatcher->Dispatcher ) )
        {
            case 0:
                ctlSipDispatcher->Next =
                          mConnections[aMessage->GetConnectionId()].Dispatchers;
                mConnections[aMessage->GetConnectionId()].Dispatchers =
                                                               ctlSipDispatcher;
                aMessage->SetText( "Info: SipDispatcher created: ",
                                   ctlSipDispatcher->Name );
                addOutgoing( aMessage );
                aMessage = NULL;
                ctlSipDispatcher = NULL;
                break;
            case EAGAIN:
                ReportError( "too many threads", __FILE__, __LINE__ );
                // delete ctlSipDispatcher;
                break;
            case EINVAL:
                ReportError( "invalid thread attributes",
                              __FILE__, __LINE__ );
                // delete ctlSipDispatcher;
                break;
            case EPERM:
                ReportError( "insufficient thread permissions",
                              __FILE__, __LINE__ );
                // delete ctlSipDispatcher;
                break;
            default:
                ReportError( "unknown thread error",
                              __FILE__, __LINE__ );
                // delete ctlSipDispatcher;
                break;
        }
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::createRtpHandler( ControlMessage *     aMessage,
                                     char *               aCommandLine )
{
    char *               word;
    char *               name;
    controlRtpHandler *  ctlRtpHandler;
    pthread_t            rtpHandlerThread;

    name = GetNextWord( &aCommandLine );
    if ( !*name )
    {
        aMessage->SetText( "Error: Name required to create RtpHandler\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( aMessage->GetConnectionId() < 0 )
    {
        aMessage->SetText(
                        "Error: Connection required to create RtpHandler\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else
    {
        word = GetNextWord( &aCommandLine );
        if ( *word )
        {
            aMessage->SetText( "Error: Extra information at: ", word );
            addOutgoing( aMessage );
            return;
        }
        ctlRtpHandler = new controlRtpHandler();
        ctlRtpHandler->Next = NULL;
        ctlRtpHandler->Name = strdup( name );
        ctlRtpHandler->Handler = new RtpHandler();

        switch ( pthread_create( &rtpHandlerThread,
                                 NULL,
                                 ctlRtpHandler->ThreadFunction,
                                 ctlRtpHandler->Handler ) )
        {
            case 0:
                ctlRtpHandler->Next =
                          mConnections[aMessage->GetConnectionId()].RtpHandlers;
                mConnections[aMessage->GetConnectionId()].RtpHandlers =
                                                                  ctlRtpHandler;
                aMessage->SetText( "Info: RtpHandler created: ",
                                   ctlRtpHandler->Name );
                addOutgoing( aMessage );
                aMessage = NULL;
                ctlRtpHandler = NULL;
                break;
            case EAGAIN:
                ReportError( "too many threads", __FILE__, __LINE__ );
                // delete ctlRtpHandler;
                break;
            case EINVAL:
                ReportError( "invalid thread attributes",
                              __FILE__, __LINE__ );
                // delete ctlRtpHandler;
                break;
            case EPERM:
                ReportError( "insufficient thread permissions",
                              __FILE__, __LINE__ );
                // delete ctlRtpHandler;
                break;
            default:
                ReportError( "unknown thread error",
                              __FILE__, __LINE__ );
                // delete ctlRtpHandler;
                break;
        }
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::createSipRegistrarConnector( ControlMessage *  aMessage,
                                                char *            aCommandLine )
{
    char *                          name;
    char *                          word;
    char *                          cp;
    in_addr_t                       ip;
    in_port_t                       port_;
    controlSipRegistrarConnector *  ctlSipRegistrarConnector;
    controlSipDispatcher *          ctlSipDispatcher;

    ctlSipDispatcher = mConnections[aMessage->GetConnectionId()].Dispatchers;
    if ( aMessage->GetConnectionId() < 0 )
    {
        aMessage->SetText(
             "Error: Connection required to create SipRegistrarConnector\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( !*( name = GetNextWord( &aCommandLine ) ) )
    {
        aMessage->SetText(
                   "Error: Name required to create SipRegistrarConnector\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( !*( word = GetNextWord( &aCommandLine ) ) )
    {
        aMessage->SetText( "Error: Must specify ip, port, and domain to create "
                          "SipRegistrarConnector.\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "using" ) == 0 )
    {
        if ( strcasecmp( word = GetNextWord( &aCommandLine ), "dispatcher" ) )
        {
            aMessage->SetText( "Error: Expected \"dispatcher\", got: ", word );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
        else if ( !*( word = GetNextWord( &aCommandLine ) ) )
        {
            aMessage->SetText( "Error: Must specify name of dispatcher to "
                              "use.\r\n" );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
        else
        {
            ctlSipDispatcher =
                          mConnections[aMessage->GetConnectionId()].Dispatchers;
            while ( ctlSipDispatcher )
            {
                if ( strcasecmp( word, ctlSipDispatcher->Name ) == 0 )
                {
                    break;
                }
                ctlSipDispatcher = ctlSipDispatcher->Next;
                sched_yield();
            }
            if ( !ctlSipDispatcher )
            {
                aMessage->SetText(
                                  "Error: Could not find SipDispatcher named: ",
                                  word );
                addOutgoing( aMessage );
                aMessage = NULL;
            }
        }
    }

    ip    = INADDR_ANY;
    port_ = 5060;
    if ( !aMessage )
    {
        // NOOP
    }
    else if ( !ctlSipDispatcher )
    {
        aMessage->SetText( "Error: No SipDispatchers to use with a new "
                          "SipRegistrarConnector\r\n" );
        addOutgoing( aMessage );
        return;
    }
    else if ( strcasecmp( word, "to" ) )
    {
        aMessage->SetText( "Error: Expected \"to\" or \"using\", got: ", word );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( !*( word = GetNextWord( &aCommandLine ) ) )
    {
        aMessage->SetText( "Error: Must specify ip:port after \"to\".\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else
    {
        cp = word;
        while ( *cp && *cp != ':' )
        {
            cp++;
        }
        if ( *cp != ':' )
        {
            aMessage->SetText( "Error: Expected ip:port at: ", word );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
        else
        {
            *cp = '\0';
            cp++;
            ip    = inet_addr( word );
            port_ = htons( strtol( cp, NULL, 10 ) );
        }
    }

    if ( !aMessage )
    {
        // NOOP;
    }
    else if ( strcasecmp( word = GetNextWord( &aCommandLine ), "with" ) )
    {
        aMessage->SetText( "Error: Expected \"with\" at: ", word );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word = GetNextWord( &aCommandLine ), "the" ) )
    {
        aMessage->SetText( "Error: Expected \"the\" at: ", word );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word = GetNextWord( &aCommandLine ), "domain" ) )
    {
        aMessage->SetText( "Error: Expected \"domain\" at: ", word );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( !*( word = GetNextWord( &aCommandLine ) ) )
    {
        aMessage->SetText( "Error: Must specify the domain to register "
                          "with.\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else
    {
        ctlSipRegistrarConnector = new controlSipRegistrarConnector();
        ctlSipRegistrarConnector->Name = name;
        ctlSipRegistrarConnector->Connector = new SipRegistrarConnector();
        ctlSipRegistrarConnector->Connector->SetDispatcher(
                                                 ctlSipDispatcher->Dispatcher );
        ctlSipRegistrarConnector->Connector->SetAddress( ip );
        ctlSipRegistrarConnector->Connector->SetPort( port_ );
        ctlSipRegistrarConnector->Connector->SetDomain( strdup( word ) );
        ctlSipRegistrarConnector->Next =
               mConnections[aMessage->GetConnectionId()].SipRegistrarConnectors;
        mConnections[aMessage->GetConnectionId()].SipRegistrarConnectors =
                                                       ctlSipRegistrarConnector;
        aMessage->SetText( "Info: SipRegistrarConnector created: ", name );
        addOutgoing( aMessage );
        aMessage = NULL;
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::createSipEndPoint( ControlMessage *  aMessage,
                                      char *            aCommandLine )
{
    char *                          word;
    controlSipEndPoint *            ctlSipEndPoint;
    controlSipDispatcher *          ctlSipDispatcher;
    controlRtpHandler *             ctlRtpHandler;
    controlSipRegistrarConnector *  ctlSipRegistrarConnector;

    word = GetNextWord( &aCommandLine );
    if ( !*word )
    {
        aMessage->SetText( "Error: Name required to create sipEndPoint\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( aMessage->GetConnectionId() < 0 )
    {
        aMessage->SetText(
                       "Error: Connection required to create sipEndPoint\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else
    {
        ctlSipEndPoint = new controlSipEndPoint();
        ctlSipEndPoint->Next = NULL;
        if (   ( ctlSipEndPoint->Name = strdup( word ) )
                 == NULL )
        {
            // delete ctlSipEndPoint;
            ReportError( "out of memory", __FILE__, __LINE__ );
        }
        else
        {

            ctlSipDispatcher =
                          mConnections[aMessage->GetConnectionId()].Dispatchers;
            if ( !ctlSipDispatcher )
            {
                // delete ctlSipEndPoint;
                aMessage->SetText(
                 "Error: No SipDispatchers to use with a new SipEndPoint\r\n" );
                addOutgoing( aMessage );
                return;
            }
            ctlRtpHandler =
                          mConnections[aMessage->GetConnectionId()].RtpHandlers;
            if ( strcasecmp( GetNextWord( &aCommandLine ), "with" ) == 0 )
            {
                if (   strcasecmp( GetNextWord( &aCommandLine ), "rtphandler" )
                    == 0 )
                {
                    if (   strcasecmp( GetNextWord( &aCommandLine ), "none" )
                        == 0 )
                    {
                        ctlRtpHandler = NULL;
                    }
                }
            }
            ctlSipRegistrarConnector =
               mConnections[aMessage->GetConnectionId()].SipRegistrarConnectors;
            ctlSipEndPoint->EndPoint = new SipEndPoint();
            ctlSipEndPoint->EndPoint->SetUser( strdup( ctlSipEndPoint->Name ) );
            ctlSipEndPoint->EndPoint->SetControlPort( this );
            ctlSipEndPoint->EndPoint->SetControlPortConnectionId(
                                                  aMessage->GetConnectionId() );
            if ( ctlRtpHandler )
            {
                ctlSipEndPoint->EndPoint->SetRtpHandler(
                                                       ctlRtpHandler->Handler );
            }
            if ( ctlSipRegistrarConnector )
            {
                ctlSipEndPoint->EndPoint->SetRegistrarConnector(
                                          ctlSipRegistrarConnector->Connector );
            }
            ctlSipEndPoint->EndPoint->SetDispatcher(
                                                 ctlSipDispatcher->Dispatcher );
            ctlSipEndPoint->EndPoint->GetDispatcher()->AddEndPoint(
                                                     ctlSipEndPoint->EndPoint );
            ctlSipEndPoint->Next =
                            mConnections[aMessage->GetConnectionId()].EndPoints;
            mConnections[aMessage->GetConnectionId()].EndPoints =
                                                                 ctlSipEndPoint;
            aMessage->SetText( "Info: sipEndPoint created: ",
                              ctlSipEndPoint->Name );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::createSipProxyEndPoint( ControlMessage *  aMessage,
                                      char *            aCommandLine )
{
    char *                          name;
    char *                          toIdentifier;
    controlSipProxyEndPoint *            ctlSipProxyEndPoint;
    controlSipDispatcher *          ctlSipDispatcher;
    controlSipRegistrarConnector *  ctlSipRegistrarConnector;

    if ( aMessage->GetConnectionId() < 0 )
    {
        aMessage->SetText(
                       "Error: Connection required to create sipProxyEndPoint\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( !*( name = GetNextWord( &aCommandLine ) ) )
    {
        aMessage->SetText( "Error: Name required to create sipProxyEndPoint\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( "to", GetNextWord( &aCommandLine ) ) != 0 )
    {
        aMessage->SetText( "Error: Expected \"to\" after: ", name );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( !*( toIdentifier = GetNextWord( &aCommandLine ) ) )
    {
        aMessage->SetText( "Error: to <identifier> required to create sipProxyEndPoint\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else
    {
        ctlSipProxyEndPoint = new controlSipProxyEndPoint();
        ctlSipProxyEndPoint->Next = NULL;
        if (   ( ctlSipProxyEndPoint->Name = strdup( name ) )
                 == NULL )
        {
            // delete ctlSipProxyEndPoint;
            ReportError( "out of memory", __FILE__, __LINE__ );
        }
        else
        {

            ctlSipDispatcher =
                          mConnections[aMessage->GetConnectionId()].Dispatchers;
            if ( !ctlSipDispatcher )
            {
                // delete ctlSipProxyEndPoint;
                aMessage->SetText(
                 "Error: No SipDispatchers to use with a new SipProxyEndPoint\r\n" );
                addOutgoing( aMessage );
                return;
            }
            ctlSipRegistrarConnector =
               mConnections[aMessage->GetConnectionId()].SipRegistrarConnectors;
            ctlSipProxyEndPoint->ProxyEndPoint = new SipProxyEndPoint();
            ctlSipProxyEndPoint->ProxyEndPoint->SetUser( strdup( ctlSipProxyEndPoint->Name ) );
            ctlSipProxyEndPoint->ProxyEndPoint->SetTo( new SipIdentifier( toIdentifier ) );
            ctlSipProxyEndPoint->ProxyEndPoint->SetControlPort( this );
            ctlSipProxyEndPoint->ProxyEndPoint->SetControlPortConnectionId(
                                                  aMessage->GetConnectionId() );
            if ( ctlSipRegistrarConnector )
            {
                ctlSipProxyEndPoint->ProxyEndPoint->SetRegistrarConnector(
                                          ctlSipRegistrarConnector->Connector );
            }
            ctlSipProxyEndPoint->ProxyEndPoint->SetDispatcher(
                                                 ctlSipDispatcher->Dispatcher );
            ctlSipProxyEndPoint->ProxyEndPoint->GetDispatcher()->AddProxyEndPoint(
                                                     ctlSipProxyEndPoint->ProxyEndPoint );
            pthread_mutex_lock(
                    &( mConnections[aMessage->GetConnectionId()].ProxyEndPointsMutex ) );
            ctlSipProxyEndPoint->Next =
                            mConnections[aMessage->GetConnectionId()].ProxyEndPoints;
            mConnections[aMessage->GetConnectionId()].ProxyEndPoints =
                                                                 ctlSipProxyEndPoint;
            pthread_mutex_unlock(
                    &( mConnections[aMessage->GetConnectionId()].ProxyEndPointsMutex ) );
            aMessage->SetText( "Info: sipProxyEndPoint created: ",
                              ctlSipProxyEndPoint->Name );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::createSipRegistrar( ControlMessage *  aMessage,
                                       char *            aCommandLine )
{
    char *                          name;
    char *                          domain;
    controlSipRegistrar *           ctlSipRegistrar;
    controlSipDispatcher *          ctlSipDispatcher;

    if ( aMessage->GetConnectionId() < 0 )
    {
        aMessage->SetText(
                      "Error: Connection required to create sipRegistrar\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( !*( name = GetNextWord( &aCommandLine ) ) )
    {
        aMessage->SetText( "Error: Name required to create sipRegistrar\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( !*( domain = GetNextWord( &aCommandLine ) ) )
    {
        aMessage->SetText(
                          "Error: Domain required to create sipRegistrar\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else
    {
        ctlSipRegistrar = new controlSipRegistrar();
        ctlSipRegistrar->Next = NULL;
        if (   ( ctlSipRegistrar->Name = strdup( name ) ) == NULL )
        {
            // delete ctlSipRegistrar;
            ReportError( "out of memory", __FILE__, __LINE__ );
        }
        else
        {

            ctlSipDispatcher =
                          mConnections[aMessage->GetConnectionId()].Dispatchers;
            if ( !ctlSipDispatcher )
            {
                // delete ctlSipRegistrar;
                aMessage->SetText(
                "Error: No SipDispatchers to use with a new SipRegistrar\r\n" );
                addOutgoing( aMessage );
                return;
            }
            ctlSipRegistrar->Registrar = new SipRegistrar();
            ctlSipRegistrar->Registrar->SetDomain( strdup( domain ) );
            ctlSipRegistrar->Registrar->SetDispatcher(
                                                 ctlSipDispatcher->Dispatcher );
            ctlSipRegistrar->Registrar->GetDispatcher()->AddRegistrar(
                                                   ctlSipRegistrar->Registrar );
            ctlSipRegistrar->Next =
                           mConnections[aMessage->GetConnectionId()].Registrars;
            mConnections[aMessage->GetConnectionId()].Registrars =
                                                                ctlSipRegistrar;
            aMessage->SetText( "Info: sipRegistrar created: ",
                              ctlSipRegistrar->Name );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::createSipCall( ControlMessage *  aMessage,
                                  char *            aCommandLine )
{
    char *                sipIdentifier;
    char *                sipCallName;
    char *                sipEndPointName;
    char *                word;
    controlSipCall *      ctlSipCall;
    controlSipEndPoint *  ctlSipEndPoint;

    sipCallName     = NULL;
    sipEndPointName = NULL;
    sipIdentifier   = NULL;
    if ( aMessage->GetConnectionId() < 0 )
    {
        aMessage->SetText( "Error: Connection required to create sipCall\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( !*( sipCallName = GetNextWord( &aCommandLine ) ) )
    {
        aMessage->SetText( "Error: Name required to create sipCall\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( isNameInUse( sipCallName, aMessage->GetConnectionId() ) )
    {
        aMessage->SetText( "Error: Name already in use\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( "using", word = GetNextWord( &aCommandLine ) ) )
    {
        aMessage->SetText( "Error: Expecting \"using\" at: ", word );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( "end", word = GetNextWord( &aCommandLine ) ) )
    {
        aMessage->SetText( "Error: Expecting \"end\" at: ", word );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( "point", word = GetNextWord( &aCommandLine ) ) )
    {
        aMessage->SetText( "Error: Expecting \"point\" at: ", word );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( !*( sipEndPointName = GetNextWord( &aCommandLine ) ) )
    {
        aMessage->SetText(
                       "Error: End point name required to create sipCall\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( "to", word = GetNextWord( &aCommandLine ) ) )
    {
        aMessage->SetText( "Error: Expecting \"to\" at: ", word );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( !*( sipIdentifier = GetNextWord( &aCommandLine ) ) )
    {
        aMessage->SetText( "Error: URI required to create sipCall\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( !( ctlSipCall = new controlSipCall() ) )
    {
        ReportError( "out of memory", __FILE__, __LINE__ );
    }
    else if ( ( ctlSipCall->Name = strdup( sipCallName ) ) == NULL )
    {
        // delete ctlSipCall;
        ReportError( "out of memory", __FILE__, __LINE__ );
    }
    else if ( ( ctlSipCall->Call = new SipCall() ) == NULL )
    {
        free( ctlSipCall->Name );
        // delete ctlSipCall;
        ReportError( "out of memory", __FILE__, __LINE__ );
    }
    else
    {
        ctlSipCall->Next = NULL;
        ctlSipEndPoint = mConnections[aMessage->GetConnectionId()].EndPoints;
        while ( ctlSipEndPoint )
        {
            if ( strcasecmp( sipEndPointName, ctlSipEndPoint->Name ) == 0 )
            {
                break;
            }
            ctlSipEndPoint = ctlSipEndPoint->Next;
            sched_yield();
        }
        if ( !ctlSipEndPoint )
        {
            aMessage->SetText( "Error: Could not find sipEndPoint named: ",
                              sipEndPointName );
            addOutgoing( aMessage );
            aMessage = NULL;
            // delete ctlSipCall->Call;
            free( ctlSipCall->Name );
            // delete ctlSipCall;
        }
        else
        {
            ctlSipCall->Call->SetEndPoint( ctlSipEndPoint->EndPoint );
            ctlSipCall->Call->SetRemoteIdentifier(
                                 new SipIdentifier( strdup( sipIdentifier ) ) );
            ctlSipCall->Call->GetEndPoint()->GetDispatcher()->AddCall(
                                                             ctlSipCall->Call );
            pthread_mutex_lock(
                    &( mConnections[aMessage->GetConnectionId()].CallsMutex ) );
            ctlSipCall->Next = mConnections[aMessage->GetConnectionId()].Calls;
            mConnections[aMessage->GetConnectionId()].Calls = ctlSipCall;
            pthread_mutex_unlock(
                    &( mConnections[aMessage->GetConnectionId()].CallsMutex ) );
            aMessage->SetText( "Info: sipCall created: ", ctlSipCall->Name );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::deleteSipUdpPort( ControlMessage *     aMessage,
                                     controlSipUdpPort *  aCtlSipUdpPort,
                                     char *               aCommandLine )
{
    char *               name;
    name = strdup( aCtlSipUdpPort->Name );
    aCtlSipUdpPort->Port->Stop();
    // FIXME: join thread then // delete aCtlSipUdpPort->Port;
    // delete aCtlSipUdpPort;
    aMessage->SetText( "Info: Deleted sipUdpPort: ", name );
    addOutgoing( aMessage );
    free( name );
}


void  ControlPort::deleteSipDispatcher(
                                      ControlMessage *        aMessage,
                                      controlSipDispatcher *  aCtlSipDispatcher,
                                      char *                  aCommandLine )
{
    char *                  name;
    name = strdup( aCtlSipDispatcher->Name );
    aCtlSipDispatcher->Dispatcher->Stop();
    // FIXME: join thread then // delete aCtlSipDispatcher->Dispatcher;
    // delete aCtlSipDispatcher;
    aMessage->SetText( "Info: Deleted sipDispatcher: ", name );
    addOutgoing( aMessage );
    free( name );
}


void  ControlPort::deleteRtpHandler( ControlMessage *     aMessage,
                                     controlRtpHandler *  aCtlRtpHandler,
                                     char *               aCommandLine )
{
    char *                  name;
    name = strdup( aCtlRtpHandler->Name );
    aCtlRtpHandler->Handler->Stop();
    // FIXME: join thread then // delete aCtlRtpHandler->Handler;
    // delete aCtlRtpHandler;
    aMessage->SetText( "Info: Deleted RtpHandler: ", name );
    addOutgoing( aMessage );
    free( name );
}


void  ControlPort::deleteSipRegistrarConnector(
                      ControlMessage *                aMessage,
                      controlSipRegistrarConnector *  aCtlSipRegistrarConnector,
                      char *                          aCommandLine )
{
    char *                  name;
    name = strdup( aCtlSipRegistrarConnector->Name );
    // delete aCtlSipRegistrarConnector;
    aMessage->SetText( "Info: Deleted SipRegistrarConnector: ", name );
    addOutgoing( aMessage );
    free( name );
}


void  ControlPort::deleteSipEndPoint( ControlMessage *      aMessage,
                                      controlSipEndPoint *  aCtlSipEndPoint,
                                      char *                aCommandLine )
{
    char *                name;
    name = strdup( aCtlSipEndPoint->Name );
    aCtlSipEndPoint->EndPoint->GetDispatcher()->RemoveEndPoint(
                                                    aCtlSipEndPoint->EndPoint );
    // delete aCtlSipEndPoint->EndPoint;
    // delete aCtlSipEndPoint;
    aMessage->SetText( "Info: Deleted sipEndPoint: ", name );
    addOutgoing( aMessage );
    free( name );
}


void  ControlPort::deleteSipProxyEndPoint( ControlMessage *      aMessage,
                                      controlSipProxyEndPoint *  aCtlSipProxyEndPoint,
                                      char *                aCommandLine )
{
    char *                name;
    name = strdup( aCtlSipProxyEndPoint->Name );
    aCtlSipProxyEndPoint->ProxyEndPoint->GetDispatcher()->RemoveProxyEndPoint(
                                                    aCtlSipProxyEndPoint->ProxyEndPoint );
    // delete aCtlSipProxyEndPoint->ProxyEndPoint;
    // delete aCtlSipProxyEndPoint;
    aMessage->SetText( "Info: Deleted sipProxyEndPoint: ", name );
    addOutgoing( aMessage );
    free( name );
}


void  ControlPort::deleteSipRegistrar( ControlMessage *      aMessage,
                                      controlSipRegistrar *  aCtlSipRegistrar,
                                      char *                aCommandLine )
{
    char *                name;
    name = strdup( aCtlSipRegistrar->Name );
    aCtlSipRegistrar->Registrar->GetDispatcher()->RemoveRegistrar(
                                                    aCtlSipRegistrar->Registrar );
    // delete aCtlSipRegistrar->Registrar;
    // delete aCtlSipRegistrar;
    aMessage->SetText( "Info: Deleted sipRegistrar: ", name );
    addOutgoing( aMessage );
    free( name );
}


bool  ControlPort::deleteSipCall( ControlMessage *  aMessage,
                                  controlSipCall *  aCtlSipCall,
                                  char *            aCommandLine )
{
    char *  name;
    char *  word;
    if (   strcasecmp( word = GetNextWord( &aCommandLine ), "if" ) == 0
        && strcasecmp( word = GetNextWord( &aCommandLine ),
                       "terminated" ) == 0 )
    {
        if ( aCtlSipCall->Call->GetState() ==
                                           aCtlSipCall->Call->STATE_TERMINATED )
        {
            name = strdup( aCtlSipCall->Name );
            aCtlSipCall->Call->UpdateStateMachine( NULL,
                                          aCtlSipCall->Call->STATE_TERMINATED );
            aCtlSipCall->Call->GetEndPoint()->GetDispatcher()->RemoveCall(
                                                            aCtlSipCall->Call );
            // delete aCtlSipCall->Call;
            // delete aCtlSipCall;
            aMessage->SetText( "Info: Deleted sipCall: ", name );
            addOutgoing( aMessage );
            aMessage = NULL;
            free( name );
            return true;
        }
        else
        {
            aMessage->SetText(
                            "Info: Call not terminated, not deleting sipCall: ",
                            aCtlSipCall->Name );
            addOutgoing( aMessage );
            aMessage = NULL;
            return false;
        }
    }
    else
    {
        name = strdup( aCtlSipCall->Name );
        aCtlSipCall->Call->UpdateStateMachine( NULL,
                                          aCtlSipCall->Call->STATE_TERMINATED );
        aCtlSipCall->Call->GetEndPoint()->GetDispatcher()->RemoveCall(
                                                            aCtlSipCall->Call );
        // delete aCtlSipCall->Call;
        // delete aCtlSipCall;
        aMessage->SetText( "Info: Deleted sipCall: ", name );
        addOutgoing( aMessage );
        aMessage = NULL;
        free( name );
        return true;
    }

    return false;
}


void  ControlPort::issueSipUdpPort( ControlMessage *     aMessage,
                                    controlSipUdpPort *  aCtlSipUdpPort,
                                    char *               aCommandLine )
{
    char *  word;

    if ( !*( word = GetNextWord( &aCommandLine ) ) )
    {
        aMessage->SetText( "Error: No command given to issue.\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "hold" ) == 0 )
    {
        aCtlSipUdpPort->Port->SetHoldState( true );
        aMessage->SetText( "Info: Issued hold command to SipUdpPort: ",
                          aCtlSipUdpPort->Name );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "unhold" ) == 0 )
    {
        aCtlSipUdpPort->Port->SetHoldState( false );
        aMessage->SetText( "Info: Issued unhold command to SipUdpPort: ",
                          aCtlSipUdpPort->Name );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else
    {
        aMessage->SetText( "Error: Unknown SipUdpPort command: ", word );
        addOutgoing( aMessage );
        aMessage = NULL;
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::issueSipDispatcher(
                                      ControlMessage *        aMessage,
                                      controlSipDispatcher *  aCtlSipDispatcher,
                                      char *                  aCommandLine )
{
    char *  word;

    word = GetNextWord( &aCommandLine );
    if ( !*word )
    {
        aMessage->SetText( "Error: No command given to issue\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "set" ) == 0 )
    {
        word = GetNextWord( &aCommandLine );
        if ( !*word )
        {
            aMessage->SetText( "Error: No variable given to set\r\n" );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
        else if ( strcasecmp( word, "t1" ) == 0 )
        {
            word = GetNextWord( &aCommandLine );
            if ( !*word )
            {
                aMessage->SetText( "Error: No value given to set\r\n" );
                addOutgoing( aMessage );
                aMessage = NULL;
            }
            else
            {
                aCtlSipDispatcher->Dispatcher->SetT1( strtol( word,
                                                              NULL,
                                                              10 ) );
                aMessage->SetText( "Info: Set T1 of SipDispatcher: ",
                                  aCtlSipDispatcher->Name );
                addOutgoing( aMessage );
                aMessage = NULL;
            }
        }
        else if ( strcasecmp( word, "t2" ) == 0 )
        {
            word = GetNextWord( &aCommandLine );
            if ( !*word )
            {
                aMessage->SetText( "Error: No value given to set\r\n" );
                addOutgoing( aMessage );
                aMessage = NULL;
            }
            else
            {
                aCtlSipDispatcher->Dispatcher->SetT2( strtol( word,
                                                              NULL,
                                                              10 ) );
                aMessage->SetText( "Info: Set T2 of SipDispatcher: ",
                                  aCtlSipDispatcher->Name );
                addOutgoing( aMessage );
                aMessage = NULL;
            }
        }
        else if ( strcasecmp( word, "retransmissions" ) == 0 )
        {
            word = GetNextWord( &aCommandLine );
            if ( !*word )
            {
                aMessage->SetText( "Error: No value given to set\r\n" );
                addOutgoing( aMessage );
                aMessage = NULL;
            }
            else
            {
                aCtlSipDispatcher->Dispatcher->SetRetransmissionsDesired(
                                              strcasecmp( word, "true" ) == 0
                                           || strcasecmp( word, "yes" ) == 0
                                           || strcasecmp( word, "on" ) == 0
                                           || strcasecmp( word, "enabled" ) == 0
                                           || strcasecmp( word, "1" ) == 0
                                           || strcasecmp( word, "-1" ) == 0 );
                aMessage->SetText( "Info: Set Retransmissions of "
                                  "SipDispatcher: ",
                                  aCtlSipDispatcher->Name );
                addOutgoing( aMessage );
                aMessage = NULL;
            }
        }
        else if ( strcasecmp( word, "timeouts" ) == 0 )
        {
            word = GetNextWord( &aCommandLine );
            if ( !*word )
            {
                aMessage->SetText( "Error: No value given to set\r\n" );
                addOutgoing( aMessage );
                aMessage = NULL;
            }
            else
            {
                aCtlSipDispatcher->Dispatcher->SetTimeoutsDesired(
                                              strcasecmp( word, "true" ) == 0
                                           || strcasecmp( word, "yes" ) == 0
                                           || strcasecmp( word, "on" ) == 0
                                           || strcasecmp( word, "enabled" ) == 0
                                           || strcasecmp( word, "1" ) == 0
                                           || strcasecmp( word, "-1" ) == 0 );
                aMessage->SetText( "Info: Set Timeouts of SipDispatcher: ",
                                  aCtlSipDispatcher->Name );
                addOutgoing( aMessage );
                aMessage = NULL;
            }
        }
        else if ( strcasecmp( word, "provisionals" ) == 0 )
        {
            word = GetNextWord( &aCommandLine );
            if ( !*word )
            {
                aMessage->SetText( "Error: No value given to set\r\n" );
                addOutgoing( aMessage );
                aMessage = NULL;
            }
            else
            {
                aCtlSipDispatcher->Dispatcher->SetProvisionalsDesired(
                                              strcasecmp( word, "true" ) == 0
                                           || strcasecmp( word, "yes" ) == 0
                                           || strcasecmp( word, "on" ) == 0
                                           || strcasecmp( word, "enabled" ) == 0
                                           || strcasecmp( word, "1" ) == 0
                                           || strcasecmp( word, "-1" ) == 0 );
                aMessage->SetText( "Info: Set Provisionals of SipDispatcher: ",
                                  aCtlSipDispatcher->Name );
                addOutgoing( aMessage );
                aMessage = NULL;
            }
        }
        else
        {
            aMessage->SetText( "Error: Unknown variable name: ", word );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
    }
    else
    {
        aMessage->SetText( "Error: Unknown command issued to sipDispatcher: ",
                          word );
        addOutgoing( aMessage );
        aMessage = NULL;
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::issueRtpHandler( ControlMessage *        aMessage,
                                    controlRtpHandler *  aCtlRtpHandler,
                                    char *               aCommandLine )
{
    char *  word;

    word = GetNextWord( &aCommandLine );
    if ( !*word )
    {
        aMessage->SetText( "Error: No command given to issue\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else
    {
        aMessage->SetText( "Error: Unknown command issued to RtpHandler: ",
                           word );
        addOutgoing( aMessage );
        aMessage = NULL;
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::issueSipRegistrarConnector(
                      ControlMessage *                aMessage,
                      controlSipRegistrarConnector *  aCtlSipRegistrarConnector,
                      char *                          aCommandLine )
{
    char *  word;

    word = GetNextWord( &aCommandLine );
    if ( !*word )
    {
        aMessage->SetText( "Error: No command given to issue\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else
    {
        aMessage->SetText(
                     "Error: Unknown command issued to SipRegistrarConnector: ",
                     word );
        addOutgoing( aMessage );
        aMessage = NULL;
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::issueSipEndPoint( ControlMessage *      aMessage,
                                     controlSipEndPoint *  aCtlSipEndPoint,
                                     char *                aCommandLine )
{
    char *  cp;
    char *  range;
    char *  word;
    int     rangeFrom;
    int     rangeTo;
    bool    bToCaller;

    word = GetNextWord( &aCommandLine );
    if ( !*word )
    {
        aMessage->SetText( "Error: No command given to issue\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "accept" ) == 0 )
    {
        word = GetNextWord( &aCommandLine );
        if ( strcasecmp( word, "calls" ) != 0 )
        {
            aMessage->SetText(
                    "Error: Expected \"calls\" after \"accept\", got: ", word );
            addOutgoing( aMessage );
            return;
        }
        range = NULL;
        word  = GetNextWord( &aCommandLine );
        if ( *word )
        {
            if ( strcasecmp( word, "after" ) != 0 )
            {
                aMessage->SetText( "Error: Expected \"after\", got: ", word );
                addOutgoing( aMessage );
                return;
            }
            if (   strcasecmp( word = GetNextWord( &aCommandLine ), "ringing" )
                != 0 )
            {
                aMessage->SetText( "Error: Expected \"ringing\", got: ", word );
                addOutgoing( aMessage );
                return;
            }
            if (   strcasecmp( word = GetNextWord( &aCommandLine ), "for" )
                != 0 )
            {
                aMessage->SetText( "Error: Expected \"for\", got: ", word );
                addOutgoing( aMessage );
                return;
            }
            range = GetNextWord( &aCommandLine );
            if ( !*range )
            {
                aMessage->SetText( "Error: Expected a range, got: ", word );
                addOutgoing( aMessage );
                return;
            }
            if ( strcasecmp( word = GetNextWord( &aCommandLine ), "seconds" )
                != 0 )
            {
                aMessage->SetText( "Error: Expected \"seconds\", got: ", word );
                addOutgoing( aMessage );
                return;
            }
            cp = strchr( range, '-' );
            if ( cp )
            {
                *cp = '\0';
                cp++;
                rangeFrom = strtol( range, NULL, 10 );
                rangeTo   = strtol( cp,    NULL, 10 );
            }
            else
            {
                rangeFrom = rangeTo = strtol( range, NULL, 10 );
            }
            if ( rangeTo < rangeFrom )
            {
                aMessage->SetText( "Error: Invalid range specified.\r\n" );
                addOutgoing( aMessage );
                return;
            }
        }
        aCtlSipEndPoint->EndPoint->SetAcceptCalls( true );
        if ( range )
        {
            aCtlSipEndPoint->EndPoint->SetRingingMinimum( rangeFrom * 1000 );
            aCtlSipEndPoint->EndPoint->SetRingingMaximum( rangeTo * 1000 );
        }
        aMessage->SetText( "Info: Issued \"accept calls\" to sipEndPoint: ",
                          aCtlSipEndPoint->Name );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "relay" ) == 0 )
    {
        word = GetNextWord( &aCommandLine );
        if ( strcasecmp( word, "calls" ) != 0 )
        {
            aMessage->SetText(
                    "Error: Expected \"calls\" after \"relay\", got: ", word );
            addOutgoing( aMessage );
            return;
        }
        word = GetNextWord( &aCommandLine );
        if ( strcasecmp( word, "to" ) != 0 )
        {
            aMessage->SetText(
                    "Error: Expected \"to\" after \"calls\", got: ", word );
            addOutgoing( aMessage );
            return;
        }
        word = GetNextWord( &aCommandLine );
        if ( !*word )
        {
            aMessage->SetText(
                    "Error: Expected a SIP identifier after \"to\"\n" );
            addOutgoing( aMessage );
            return;
        }
        aCtlSipEndPoint->EndPoint->RelayCallsTo(
                                          new SipIdentifier( strdup( word ) ) );
        aMessage->SetText( "Info: Issued \"relay calls\" to sipEndPoint: ",
                          aCtlSipEndPoint->Name );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "tap" ) == 0 )
    {
        word = GetNextWord( &aCommandLine );
        if ( strcasecmp( word, "calls" ) != 0 )
        {
            aMessage->SetText(
                    "Error: Expected \"calls\" after \"tap\", got: ", word );
            addOutgoing( aMessage );
            return;
        }
        word = GetNextWord( &aCommandLine );
        if ( strcasecmp( word, "to" ) != 0 )
        {
            aMessage->SetText(
                    "Error: Expected \"to\" after \"calls\", got: ", word );
            addOutgoing( aMessage );
            return;
        }
        word = GetNextWord( &aCommandLine );
        if ( !*word )
        {
            aMessage->SetText(
                    "Error: Expected a SIP identifier after \"to\"\n" );
            addOutgoing( aMessage );
            return;
        }
        aCtlSipEndPoint->EndPoint->TapCallsTo(
                                          new SipIdentifier( strdup( word ) ) );
        aMessage->SetText( "Info: Issued \"tap calls\" to sipEndPoint: ",
                          aCtlSipEndPoint->Name );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "insertrtp" ) == 0 )
    {
        word = GetNextWord( &aCommandLine );
        if ( strcasecmp( word, "to" ) != 0 )
        {
            aMessage->SetText(
                    "Error: Expected \"to\" after \"insertrtp\", got: ", word );
            addOutgoing( aMessage );
            return;
        }
        word = GetNextWord( &aCommandLine );
        if ( !*word )
        {
            aMessage->SetText(
                    "Error: Expected a SIP identifier after \"to\"\n" );
            addOutgoing( aMessage );
            return;
        }
        //
        //  Note: the sip identifier (i.e. the <NamedUri>) is ignored in
        //             any case. Any non-whitespace character will do.
        //             It is presently required for backward compatibility
        //             to the VoIP Hacking Exposed book. It is the next keyword
        //             that is imperative (i.e. caller or callee).
        //
        word = GetNextWord( &aCommandLine );
        if ( !*word )
        {
            aMessage->SetText(
                    "Error: Expected \"caller\" or \"callee\" after "
                    "\"<NamedUri>\"\n" );
            addOutgoing( aMessage );
            return;
        }        
        bToCaller = false;        
        if ( strcasecmp( word, "caller" ) == 0 )
        {
            bToCaller = true;
        }
        else if ( strcasecmp( word, "callee" ) != 0 )
        {
            aMessage->SetText(
                    "Error: Expected \"caller\" or \"callee\" after "
                    "\"<NamedUri>\", got: ", word );
            addOutgoing( aMessage );
            return;
        }        
        word = GetNextWord( &aCommandLine );
        if ( !*word )
        {
            aMessage->SetText(
                    "Error: Expected audio file pathname after "
                    "\"caller/callee\"\n" );
            addOutgoing( aMessage );
            return;
        }
        if ( aCtlSipEndPoint->EndPoint->
                PrepAttackAudio( bToCaller, word, ATTACK_AUDIO_INSERT ) )
        {
            aMessage->SetText( "Info: Issued \"insertrtp\" to sipEndPoint: ",
                          aCtlSipEndPoint->Name );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
        else
        {
            aMessage->SetText( "Error: failed to load pre-recorded audio\n" );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
    }
    else if ( strcasecmp( word, "mixrtp" ) == 0 )
    {
        word = GetNextWord( &aCommandLine );
        if ( strcasecmp( word, "to" ) != 0 )
        {
            aMessage->SetText(
                    "Error: Expected \"to\" after \"mixrtp\", got: ", word );
            addOutgoing( aMessage );
            return;
        }
        word = GetNextWord( &aCommandLine );
        if ( !*word )
        {
            aMessage->SetText(
                    "Error: Expected a SIP identifier after \"to\"\n" );
            addOutgoing( aMessage );
            return;
        }
        //
        //  Note: the sip identifier (i.e. the <NamedUri>) is ignored in
        //             any case. Any non-whitespace character will do.
        //             It is presently required for backward compatibility
        //             to the VoIP Hacking Exposed book. It is the next keyword
        //             that is imperative (i.e. caller or callee).
        //
        word = GetNextWord( &aCommandLine );
        if ( !*word )
        {
            aMessage->SetText(
                    "Error: Expected \"caller\" or \"callee\" after "
                    "\"<NamedUri>\"\n" );
            addOutgoing( aMessage );
            return;
        }        
        bToCaller = false;        
        if ( strcasecmp( word, "caller" ) == 0 )
        {
            bToCaller = true;
        }
        else if ( strcasecmp( word, "callee" ) != 0 )
        {
            aMessage->SetText(
                    "Error: Expected \"caller\" or \"callee\" after "
                    "\"<NamedUri>\", got: ", word );
            addOutgoing( aMessage );
            return;
        }        
        word = GetNextWord( &aCommandLine );
        if ( !*word )
        {
            aMessage->SetText(
                    "Error: Expected audio file pathname after "
                    "\"caller/callee\"\n" );
            addOutgoing( aMessage );
            return;
        }
        
        if ( aCtlSipEndPoint->EndPoint->
                PrepAttackAudio( bToCaller, word, ATTACK_AUDIO_MIX ) )
        {
            aMessage->SetText( "Info: Issued \"mixrtp\" to sipEndPoint: ",
                          aCtlSipEndPoint->Name );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
        else
        {
            aMessage->SetText( "Error: failed to load pre-recorded audio\n" );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
    }
    else
    {
        aMessage->SetText( "Error: Unknown command issued to sipEndPoint: ",
                          aCtlSipEndPoint->Name );
        addOutgoing( aMessage );
        aMessage = NULL;
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::issueSipProxyEndPoint( ControlMessage *      aMessage,
                                     controlSipProxyEndPoint *  aCtlSipProxyEndPoint,
                                     char *                aCommandLine )
{
    char *  word;

    word = GetNextWord( &aCommandLine );
    if ( !*word )
    {
        aMessage->SetText( "Error: No command given to issue\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else
    {
        aMessage->SetText( "Error: Unknown command issued to sipProxyEndPoint: ",
                          word );
        addOutgoing( aMessage );
        aMessage = NULL;
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::issueSipRegistrar( ControlMessage *      aMessage,
                                     controlSipRegistrar *  aCtlSipRegistrar,
                                     char *                aCommandLine )
{
    char *  word;

    word = GetNextWord( &aCommandLine );
    if ( !*word )
    {
        aMessage->SetText( "Error: No command given to issue\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "randomize" ) == 0 )
    {
        aCtlSipRegistrar->Registrar->SetRandomization( true );
    }
    else
    {
        aMessage->SetText( "Error: Unknown command issued to sipRegistrar: ",
                          word );
        addOutgoing( aMessage );
        aMessage = NULL;
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


void  ControlPort::issueSipCall( ControlMessage *  aMessage,
                                 controlSipCall *  aCtlSipCall,
                                 char *            aCommandLine )
{
    char *  word;
    int     amount;

    word = GetNextWord( &aCommandLine );
    if ( !*word )
    {
        aMessage->SetText( "Error: No command given to issue\r\n" );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "connect" ) == 0 )
    {
        aCtlSipCall->Call->Connect();
        aMessage->SetText( "Info: Issued connect to sipCall: ",
                           aCtlSipCall->Name );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "disconnect" ) == 0 )
    {
        if (   strcasecmp( word = GetNextWord( &aCommandLine ), "if" ) == 0
            && strcasecmp( word = GetNextWord( &aCommandLine ),
                           "established" ) == 0 )
        {
            if (   aCtlSipCall->Call->GetState()
                == aCtlSipCall->Call->STATE_ESTABLISHED ) 
            {
                aCtlSipCall->Call->Disconnect();
                aMessage->SetText( "Info: Issued disconnect to sipCall: ",
                                  aCtlSipCall->Name );
                addOutgoing( aMessage );
                aMessage = NULL;
            }
            else
            {
                aMessage->SetText( "Info: Call not established, not issuing "
                                  "disconnect to sipCall: ",
                                  aCtlSipCall->Name );
                addOutgoing( aMessage );
                aMessage = NULL;
            }
        }
        else
        {
            aCtlSipCall->Call->Disconnect();
            aMessage->SetText( "Info: Issued disconnect to sipCall: ",
                              aCtlSipCall->Name );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
    }
    else if ( strcasecmp( word, "shell" ) == 0 )
    {
        aCtlSipCall->Call->SendShell( aCommandLine );
        aMessage->SetText( "Info: Issued shell command to sipCall: ",
                          aCtlSipCall->Name );
        addOutgoing( aMessage );
        aMessage = NULL;
    }
    else if ( strcasecmp( word, "rtp" ) == 0 )
    {
        word = GetNextWord( &aCommandLine );
        if ( strcasecmp( word, "burst" ) == 0 )
        {
            if ( strcasecmp( GetNextWord( &aCommandLine ), "for" ) != 0 )
            {
                aMessage->SetText( "Error: Expected \"for\" at: ", word );
                addOutgoing( aMessage );
                aMessage = NULL;
            }
            else
            {
                amount = strtol( GetNextWord( &aCommandLine ), NULL, 10 );
                word = GetNextWord( &aCommandLine );
                if (   strcasecmp( word, "second" ) != 0
                    && strcasecmp( word, "seconds" ) != 0 )
                {
                    aMessage->SetText( "Error: Expected \"seconds\" at: ",
                                      word );
                    addOutgoing( aMessage );
                    aMessage = NULL;
                }
                else
                {
                    aCtlSipCall->Call->SendRtpBurst( amount );
                    aMessage->SetText( "Info: Informed sipCall to send RTP "
                                      "burst: ",
                                      aCtlSipCall->Name );
                    addOutgoing( aMessage );
                    aMessage = NULL;
                }
            }
        }
        else if ( strcasecmp( word, "jitter" ) == 0 )
        {
            if ( strcasecmp( GetNextWord( &aCommandLine ), "for" ) != 0 )
            {
                aMessage->SetText( "Error: Expected \"for\" at: ", word );
                addOutgoing( aMessage );
                aMessage = NULL;
            }
            else
            {
                amount = strtol( GetNextWord( &aCommandLine ), NULL, 10 );
                word = GetNextWord( &aCommandLine );
                if (   strcasecmp( word, "second" ) != 0
                    && strcasecmp( word, "seconds" ) != 0 )
                {
                    aMessage->SetText( "Error: Expected \"seconds\" at: ",
                                      word );
                    addOutgoing( aMessage );
                    aMessage = NULL;
                }
                else
                {
                    aCtlSipCall->Call->SendRtpJitter( amount );
                    aMessage->SetText( "Info: Informed sipCall to send RTP "
                                      "jitter: ",
                                      aCtlSipCall->Name );
                    addOutgoing( aMessage );
                    aMessage = NULL;
                }
            }
        }
        else if ( strcasecmp( word, "uutest" ) == 0 )
        {
            aCtlSipCall->Call->SendRtpUuTest();
            aMessage->SetText( "Info: Informed sipCall to send an RTP "
                               "UU Test: ",
                               aCtlSipCall->Name );
            addOutgoing( aMessage );
            aMessage = NULL;
        }
    }
    else
    {
        aMessage->SetText( "Error: Unknown command issued to sipCall: ", word );
        addOutgoing( aMessage );
        aMessage = NULL;
    }

    if ( aMessage )
    {
        // delete aMessage;
    }
}


unsigned int  ControlPort::nextAutoId( void )
{
    int rv;
    pthread_mutex_lock( &mAutoIdMutex );
    rv = mAutoIdCounter;
    mAutoIdCounter++;
    pthread_mutex_unlock( &mAutoIdMutex );
    return rv;
}


bool  ControlPort::isNameInUse( char *  aName, int  aConnectionId )
{
    controlSipUdpPort *     ctlSipUdpPort;
    controlSipDispatcher *  ctlSipDispatcher;
    controlRtpHandler *     ctlRtpHandler;
    controlSipRegistrarConnector *     ctlSipRegistrarConnector;
    controlSipEndPoint *    ctlSipEndPoint;
    controlSipProxyEndPoint *    ctlSipProxyEndPoint;
    controlSipRegistrar *    ctlSipRegistrar;
    controlSipCall *        ctlSipCall;

    ctlSipUdpPort = mConnections[aConnectionId].Ports;
    while ( ctlSipUdpPort )
    {
        if ( strcasecmp( aName, ctlSipUdpPort->Name ) == 0 )
        {
            return true;
        }
        ctlSipUdpPort = ctlSipUdpPort->Next;
        sched_yield();
    }

    ctlSipDispatcher = mConnections[aConnectionId].Dispatchers;
    while ( ctlSipDispatcher )
    {
        if ( strcasecmp( aName, ctlSipDispatcher->Name ) == 0 )
        {
            return true;
        }
        ctlSipDispatcher = ctlSipDispatcher->Next;
        sched_yield();
    }

    ctlRtpHandler = mConnections[aConnectionId].RtpHandlers;
    while ( ctlRtpHandler )
    {
        if ( strcasecmp( aName, ctlRtpHandler->Name ) == 0 )
        {
            return true;
        }
        ctlRtpHandler = ctlRtpHandler->Next;
        sched_yield();
    }

    ctlSipRegistrarConnector =
                             mConnections[aConnectionId].SipRegistrarConnectors;
    while ( ctlSipRegistrarConnector )
    {
        if ( strcasecmp( aName, ctlSipRegistrarConnector->Name ) == 0 )
        {
            return true;
        }
        ctlSipRegistrarConnector = ctlSipRegistrarConnector->Next;
        sched_yield();
    }

    ctlSipEndPoint = mConnections[aConnectionId].EndPoints;
    while ( ctlSipEndPoint )
    {
        if ( strcasecmp( aName, ctlSipEndPoint->Name ) == 0 )
        {
            return true;
        }
        ctlSipEndPoint = ctlSipEndPoint->Next;
        sched_yield();
    }

    pthread_mutex_lock( &( mConnections[aConnectionId].ProxyEndPointsMutex ) );
    ctlSipProxyEndPoint = mConnections[aConnectionId].ProxyEndPoints;
    while ( ctlSipProxyEndPoint )
    {
        if ( strcasecmp( aName, ctlSipProxyEndPoint->Name ) == 0 )
        {
            pthread_mutex_unlock( &( mConnections[aConnectionId].ProxyEndPointsMutex ) );
            return true;
        }
        ctlSipProxyEndPoint = ctlSipProxyEndPoint->Next;
        sched_yield();
    }
    pthread_mutex_unlock( &( mConnections[aConnectionId].ProxyEndPointsMutex ) );

    ctlSipRegistrar = mConnections[aConnectionId].Registrars;
    while ( ctlSipRegistrar )
    {
        if ( strcasecmp( aName, ctlSipRegistrar->Name ) == 0 )
        {
            return true;
        }
        ctlSipRegistrar = ctlSipRegistrar->Next;
        sched_yield();
    }

    pthread_mutex_lock( &( mConnections[aConnectionId].CallsMutex ) );
    ctlSipCall = mConnections[aConnectionId].Calls;
    while ( ctlSipCall )
    {
        if ( strcasecmp( aName, ctlSipCall->Name ) == 0 )
        {
            pthread_mutex_unlock( &( mConnections[aConnectionId].CallsMutex ) );
            return true;
        }
        ctlSipCall = ctlSipCall->Next;
        sched_yield();
    }
    pthread_mutex_unlock( &( mConnections[aConnectionId].CallsMutex ) );

    return false;
}


ControlPort::queue::queue( void )
{
    pthread_mutex_init( &mMutex, NULL );
    mHead = NULL;
    mTail = NULL;
}


ControlPort::queue::~queue( void )
{
    pthread_mutex_destroy( &mMutex );
}


bool ControlPort::queue::HasNext( void )
{
    return mHead != NULL;
}


ControlMessage *  ControlPort::queue::GetNext( void )
{
    ControlMessage *      message;
    controlMessageNode *  messageNode;

    pthread_mutex_lock( &mMutex );
    messageNode = mHead;
    if ( messageNode )
    {
        mHead = messageNode->Next;
        messageNode->Next = NULL;
        if ( !mHead )
        {
            mTail = NULL;
        }
    }
    pthread_mutex_unlock( &mMutex );
    if ( messageNode )
    {
        message = messageNode->Message;
        delete messageNode;
        return message;
    }
    return NULL;
}

//
//  Add a message to the tail of the queue object.
//
//  Actually, the messages objects themselves are not directly inserted into
//  the queue. The message objects are attached to a ControlMessageNode
//  object. It is the ControlMessageNode object which is linked into the queue.
//
//

ControlPort::Error  ControlPort::queue::Add( ControlMessage *  aMessage )
{
    controlMessageNode *  messageNode;

    messageNode = new controlMessageNode();
    if ( !messageNode )
    {
        return ERROR_MEMORY;
    }
    messageNode->Next    = NULL;
    messageNode->Message = aMessage;

    pthread_mutex_lock( &mMutex );
    if ( mTail )
    {
        mTail->Next = messageNode;
        mTail       = messageNode;
    }
    else
    {
        mHead = mTail = messageNode;
    }
    pthread_mutex_unlock( &mMutex );

    return ERROR_NONE;
}


//
//  Add a message to the head of the queue object.
//
//  Actually, the messages objects themselves are not directly inserted into
//  the queue. The message objects are attached to a ControlMessageNode
//  object. It is the ControlMessageNode object which is linked into the queue.
//

ControlPort::Error  ControlPort::queue::PreAdd( ControlMessage *  aMessage )
{
    controlMessageNode *  messageNode;

    messageNode = new controlMessageNode();
    if ( !messageNode )
    {
        return ERROR_MEMORY;
    }
    messageNode->Next    = NULL;
    messageNode->Message = aMessage;

    pthread_mutex_lock( &mMutex );
    messageNode->Next = mHead;
    mHead = messageNode;
    pthread_mutex_unlock( &mMutex );

    return ERROR_NONE;
}


ControlPort::controlSocket::controlSocket( void )
{
    FileDescriptor = -1;
    Buffer         = NULL;
}


ControlPort::controlSocket::~controlSocket( void )
{
}


ControlPort::controlConnection::controlConnection( void )
{
    Ports       = NULL;
    Dispatchers = NULL;
    RtpHandlers = NULL;
    SipRegistrarConnectors = NULL;
    EndPoints   = NULL;
    ProxyEndPoints   = NULL;
    Registrars   = NULL;
    Calls       = NULL;
    Scripts     = NULL;
    pthread_mutex_init( &CallsMutex, NULL );
    pthread_mutex_init( &ProxyEndPointsMutex, NULL );
}


ControlPort::controlConnection::~controlConnection( void )
{
    pthread_mutex_destroy( &CallsMutex );
    pthread_mutex_destroy( &ProxyEndPointsMutex );
}


void *  ControlPort__controlSipUdpPort__ThreadFunction( void *  aParms )
{
    switch ( ( ( SipUdpPort * )aParms )->Run() )
    {
        case SipUdpPort::ERROR_NONE:
            break;
        case SipUdpPort::ERROR_SOCKET:
            fprintf( stderr, "sipUdpPort socket error\n" );
            return NULL;
        case SipUdpPort::ERROR_BIND:
            fprintf( stderr, "sipUdpPort bind error\n" );
            return NULL;
        case SipUdpPort::ERROR_SELECT:
            fprintf( stderr, "sipUdpPort select error\n" );
            return NULL;
        case SipUdpPort::ERROR_RECVFROM:
            fprintf( stderr, "sipUdpPort recvfrom error\n" );
            return NULL;
        case SipUdpPort::ERROR_MEMORY:
            fprintf( stderr, "sipUdpPort memory error\n" );
            return NULL;
        case SipUdpPort::ERROR_SENDTO:
            fprintf( stderr, "sipUdpPort sendto error\n" );
            return NULL;
        case SipUdpPort::ERROR_SENDTO_ALL_BYTES_NOT_SENT:
            fprintf( stderr, "sipUdpPort sendto all bytes not sent error\n" );
            return NULL;
        case SipUdpPort::ERROR_BAD_OUTGOING_MESSAGE_TYPE:
            fprintf( stderr, "sipUdpPort bad outgoing message error\n" );
            return NULL;
        case SipUdpPort::ERROR_OUTGOING_MESSAGE_TOO_LARGE:
            fprintf( stderr, "sipUdpPort outgoing message to large error\n" );
            return NULL;
        default:
            fprintf( stderr, "Unknown sipUdpPort error\n" );
            return NULL;
    }
    return NULL;
}


ControlPort::controlScript::controlScript( void )
{
    Name = NULL;
}


ControlPort::controlScript::~controlScript( void )
{
    if ( Name )
    {
        free( Name );
    }
}


ControlPort::controlSipUdpPort::controlSipUdpPort( void )
{
    Name = NULL;
    ThreadFunction = ControlPort__controlSipUdpPort__ThreadFunction;
}


ControlPort::controlSipUdpPort::~controlSipUdpPort( void )
{
    if ( Name )
    {
        free( Name );
    }
}


void *  ControlPort__controlSipDispatcher__ThreadFunction( void *  aParms )
{
    ( ( SipDispatcher * )aParms )->Run();
    return NULL;
}


ControlPort::controlSipDispatcher::controlSipDispatcher( void )
{
    Name = NULL;
    ThreadFunction = ControlPort__controlSipDispatcher__ThreadFunction;
}


ControlPort::controlSipDispatcher::~controlSipDispatcher( void )
{
    if ( Name )
    {
        free( Name );
    }
}


void *  ControlPort__controlRtpHandler__ThreadFunction( void *  aParms )
{
    ( ( RtpHandler * )aParms )->Run();
    return NULL;
}


ControlPort::controlRtpHandler::controlRtpHandler( void )
{
    Name = NULL;
    ThreadFunction = ControlPort__controlRtpHandler__ThreadFunction;
}


ControlPort::controlRtpHandler::~controlRtpHandler( void )
{
    if ( Name )
    {
        free( Name );
    }
}


ControlPort::controlSipRegistrarConnector::controlSipRegistrarConnector( void )
{
    Name = NULL;
}


ControlPort::controlSipRegistrarConnector::~controlSipRegistrarConnector( void )
{
    if ( Name )
    {
        free( Name );
    }
}


ControlPort::controlSipEndPoint::controlSipEndPoint( void )
{
    Name = NULL;
}


ControlPort::controlSipEndPoint::~controlSipEndPoint( void )
{
    if ( Name )
    {
        free( Name );
    }
}


ControlPort::controlSipProxyEndPoint::controlSipProxyEndPoint( void )
{
    Name = NULL;
}


ControlPort::controlSipProxyEndPoint::~controlSipProxyEndPoint( void )
{
    if ( Name )
    {
        free( Name );
    }
}


ControlPort::controlSipRegistrar::controlSipRegistrar( void )
{
    Name = NULL;
}


ControlPort::controlSipRegistrar::~controlSipRegistrar( void )
{
    if ( Name )
    {
        free( Name );
    }
}


ControlPort::controlSipCall::controlSipCall( void )
{
    Name = NULL;
}


ControlPort::controlSipCall::~controlSipCall( void )
{
    if ( Name )
    {
        free( Name );
    }
}

