//-----------------------------------------------------------------------------
//
//  ControlPort.h - Handles connections to the control
//                   TCP port, accepting, processing, delivering,
//                   and sending ControlMessages. The control
//                   port is the "command line interface" of the
//                   sip_rogue daemon.
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#ifndef CONTROL_PORT_H
#define CONTROL_PORT_H

#include <netinet/in.h>
#include <pthread.h>
#include <time.h>

#include "RtpHandler.h"
#include "SipUdpPort.h"
#include "SipDispatcher.h"
#include "SipRegistrarConnector.h"
#include "SipEndPoint.h"
#include "SipProxyEndPoint.h"
#include "SipRegistrar.h"
#include "SipCall.h"
#include "ControlMessage.h"

class  ControlPort
{
    public:

        typedef enum
        {
            ERROR_NONE = 0,
            ERROR_MEMORY,
            ERROR_SOCKET,
            ERROR_FCNTL,
            ERROR_BIND,
            ERROR_LISTEN,
            ERROR_ACCEPT,
            ERROR_SELECT,
            ERROR_MSG_FOR_NONEXISTENT_CONNECTION,
            ERROR_SOCKIDX_OUT_OF_RANGE,
            ERROR_UNKNOWN
        }  Error;

        ControlPort( void );
        ~ControlPort( void );

        void   SetBoundAddress( in_addr_t  aAddress );
        void   SetBoundPort( in_port_t  aPort );

        Error  NetworkRun( void );
        void   NetworkStop( void );

        Error  ProcessRun( void );
        void   ProcessStop( void );

        bool   AddSipCall( int  aConnectionId, SipCall *  aCall );
        bool   AddSipProxyEndPoint( int                 aConnectionId,
                                    SipProxyEndPoint *  aProxyEndPoint );


    private:

        class  controlMessageNode
        {
            public:
                controlMessageNode *  Next;
                ControlMessage *      Message;
        };


        class  queue
        {
            public:
                queue( void );
                ~queue( void );

                bool              HasNext( void );
                ControlMessage *  GetNext( void );
                Error             Add( ControlMessage *  aMessage );
                Error             PreAdd( ControlMessage *  aMessage );

            private:
                pthread_mutex_t       mMutex;
                controlMessageNode *  mHead;
                controlMessageNode *  mTail;
        };


        class  controlSocket
        {
            public:
                controlSocket( void );
                ~controlSocket( void );

                int        FileDescriptor;
                in_addr_t  SourceAddress;
                in_port_t  SourcePort;
                in_addr_t  DestinationAddress;
                in_port_t  DestinationPort;
                char *     Buffer;
                int        ConnectionId;  // -1 if no connection claimed yet
        };


        class  controlScript
        {
            public:
                controlScript( void );
                ~controlScript( void );

                controlScript *       Next;
                char *                Name;
                controlMessageNode *  Messages;
        };


        class  controlSipUdpPort
        {
            public:
                controlSipUdpPort( void );
                ~controlSipUdpPort( void );

                void *          ( *  ThreadFunction )( void *  aParms );
                controlSipUdpPort *  Next;
                char *               Name;
                SipUdpPort *         Port;
        };


        class  controlSipDispatcher
        {
            public:
                controlSipDispatcher( void );
                ~controlSipDispatcher( void );

                void *             ( *  ThreadFunction )( void *  aParms );
                controlSipDispatcher *  Next;
                char *                  Name;
                SipDispatcher *         Dispatcher;
        };


        class  controlRtpHandler
        {
            public:
                controlRtpHandler( void );
                ~controlRtpHandler( void );

                void *          ( *  ThreadFunction )( void *  aParms );
                controlRtpHandler *  Next;
                char *               Name;
                RtpHandler *         Handler;
        };


        class  controlSipRegistrarConnector
        {
            public:
                controlSipRegistrarConnector( void );
                ~controlSipRegistrarConnector( void );

                controlSipRegistrarConnector *  Next;
                char *                          Name;
                SipRegistrarConnector *         Connector;
        };


        class  controlSipEndPoint
        {
            public:
                controlSipEndPoint( void );
                ~controlSipEndPoint( void );

                controlSipEndPoint *  Next;
                char *                Name;
                SipEndPoint *         EndPoint;
        };


        class  controlSipProxyEndPoint
        {
            public:
                controlSipProxyEndPoint( void );
                ~controlSipProxyEndPoint( void );

                controlSipProxyEndPoint *  Next;
                char *                Name;
                SipProxyEndPoint *         ProxyEndPoint;
        };


        class  controlSipRegistrar
        {
            public:
                controlSipRegistrar( void );
                ~controlSipRegistrar( void );

                controlSipRegistrar *  Next;
                char *                 Name;
                SipRegistrar *         Registrar;
        };


        class  controlSipCall
        {
            public:
                controlSipCall( void );
                ~controlSipCall( void );

                controlSipCall *  Next;
                char *            Name;
                SipCall *         Call;
        };


        class  controlConnection
        {
            public:
                controlConnection( void );
                ~controlConnection( void );

                controlScript *                 Scripts;
                controlSipUdpPort *             Ports;
                controlSipDispatcher *          Dispatchers;
                controlRtpHandler *             RtpHandlers;
                controlSipRegistrarConnector *  SipRegistrarConnectors;
                controlSipEndPoint *            EndPoints;
                controlSipProxyEndPoint *       ProxyEndPoints;
                pthread_mutex_t                 ProxyEndPointsMutex;
                controlSipRegistrar *           Registrars;
                pthread_mutex_t                 CallsMutex;
                controlSipCall *                Calls;
        };


        ControlMessage *  getNextIncoming( int  aSocketIndex );
        
        Error             addIncoming( int               aSocketIndex,
                                       ControlMessage *  aMessage );
        
        Error             preAddIncoming( int               aSocketIndex,
                                          ControlMessage *  aMessage );
        
        ControlMessage *  getNextOutgoing( int aConnectionId );
        
        Error             addOutgoing( ControlMessage *  aMessage );
        
        Error             preAddOutgoing( ControlMessage *  aMessage );

        void              processEcho( ControlMessage *  aMessage,
                                       char *            aCommandLine );

        void              processHelp( ControlMessage *  aMessage,
                                       char *            aCommandLine );

        void              processConnection( int               aSocketIndex,
                                             ControlMessage *  aMessage,
                                             char *            aCommandLine );

        void              processStatus( ControlMessage *  aMessage,
                                         char *            aCommandLine );
        void              processStatusOf( ControlMessage *  aMessage,
                                           char *            aCommandLine );
        void              processStatusOfSipUdpPort(
                                          ControlMessage *     aMessage,
                                          char *               aCommandLine,
                                          controlSipUdpPort *  aCtlSipUdpPort );
        void              processStatusOfSipDispatcher(
                                    ControlMessage *        aMessage,
                                    char *                  aCommandLine,
                                    controlSipDispatcher *  aCtlSipDispatcher );
        void              processStatusOfRtpHandler(
                                             ControlMessage *     aMessage,
                                             char *               aCommandLine,
                                             controlRtpHandler *  cRtpHandler );
        void              processStatusOfSipRegistrarConnector(
                                                 ControlMessage *  aMessage,
                                                 char *            aCommandLine,
                                                 controlSipRegistrarConnector *
                                                    aCtlSipRegistrarConnector );
        void              processStatusOfSipEndPoint(
                                        ControlMessage *      aMessage,
                                        char *                aCommandLine,
                                        controlSipEndPoint *  aCtlSipEndPoint );
        void              processStatusOfSipProxyEndPoint(
                                        ControlMessage *      aMessage,
                                        char *                aCommandLine,
                                        controlSipProxyEndPoint *  aCtlSipProxyEndPoint );
        void              processStatusOfSipRegistrar(
                                        ControlMessage *       aMessage,
                                        char *                 aCommandLine,
                                        controlSipRegistrar *  aCtlSipRegistrar );
        void              processStatusOfSipCall(
                                                ControlMessage *  aMessage,
                                                char *            aCommandLine,
                                                controlSipCall *  aCtlSipCall );

        void              processSummary( ControlMessage *  aMessage,
                                          char *  aCommandLine );
        void              processSummaryOfSipUdpPort(
                                               ControlMessage *  aMessage,
                                               char *            aCommandLine );
        void              processSummaryOfSipDispatcher(
                                               ControlMessage *  aMessage,
                                               char *            aCommandLine );
        void              processSummaryOfRtpHandler(
                                               ControlMessage *  aMessage,
                                               char *            aCommandLine );
        void              processSummaryOfSipRegistrarConnector(
                                               ControlMessage *  aMessage,
                                               char *            aCommandLine );
        void              processSummaryOfSipEndPoint(
                                               ControlMessage *  aMessage,
                                               char *            aCommandLine );
        void              processSummaryOfSipProxyEndPoint(
                                               ControlMessage *  aMessage,
                                               char *            aCommandLine );
        void              processSummaryOfSipRegistrar(
                                               ControlMessage *  aMessage,
                                               char *            aCommandLine );
        void              processSummaryOfSipCall(
                                               ControlMessage *  aMessage,
                                               char *            aCommandLine );

        void              processCreate( ControlMessage *  aMessage,
                                         char *            aCommandLine );

        void              processDelete( ControlMessage *  aMessage,
                                         char *            aCommandLine );

        void              processExecute( ControlMessage *  aMessage,
                                          char *            aCommandLine );

        void              processIssue( ControlMessage *  aMessage,
                                        char *            aCommandLine );

        void              processIssueAll( ControlMessage *  aMessage,
                                           char *            aCommandLine,
                                           long              aMicroseconds );

        void              processSleep( ControlMessage *  aMessage,
                                        char *            aCommandLine );

        void              processShutdown( ControlMessage *  aMessage,
                                           char *            aCommandLine );


        void              createScript( ControlMessage *  aMessage,
                                        char *            aCommandLine );
        void              createSipUdpPort( ControlMessage *  aMessage,
                                            char *            aCommandLine );
        void              createSipDispatcher( ControlMessage *  aMessage,
                                               char *            aCommandLine );
        void              createRtpHandler( ControlMessage *  aMessage,
                                            char *            aCommandLine );
        void              createSipRegistrarConnector(
                                               ControlMessage *  aMessage,
                                               char *            aCommandLine );
        void              createSipEndPoint( ControlMessage *  aMessage,
                                             char *            aCommandLine );
        void              createSipProxyEndPoint( ControlMessage *  aMessage,
                                             char *            aCommandLine );
        void              createSipRegistrar( ControlMessage *  aMessage,
                                             char *            aCommandLine );
        void              createSipCall( ControlMessage *  aMessage,
                                         char *            aCommandLine );


        void              deleteSipUdpPort( ControlMessage *     aMessage,
                                            controlSipUdpPort *  aCtlSipUdpPort,
                                            char *               aCommandLine );
        void              deleteSipDispatcher(
                                      ControlMessage *        aMessage,
                                      controlSipDispatcher *  aCtlSipDispatcher,
                                      char *                  aCommandLine );
        void              deleteRtpHandler( ControlMessage *     aMessage,
                                            controlRtpHandler *  cRtpHandler,
                                            char *               aCommandLine );
        void              deleteSipRegistrarConnector(
                                            ControlMessage *  aMessage,
                                            controlSipRegistrarConnector *
                                                      aCtlSipRegistrarConnector,
                                            char *            aCommandLine );
        void              deleteSipEndPoint(
                                          ControlMessage *      aMessage,
                                          controlSipEndPoint *  aCtlSipEndPoint,
                                          char *                aCommandLine );
        void              deleteSipProxyEndPoint(
                                          ControlMessage *      aMessage,
                                          controlSipProxyEndPoint *  aCtlSipProxyEndPoint,
                                          char *                aCommandLine );
        void              deleteSipRegistrar(
                                          ControlMessage *       aMessage,
                                          controlSipRegistrar *  aCtlSipRegistrar,
                                          char *                 aCommandLine );
        bool              deleteSipCall( ControlMessage *  aMessage,
                                         controlSipCall *  aCtlSipCall,
                                         char *            aCommandLine );


        void              issueSipUdpPort( ControlMessage *     aMessage,
                                           controlSipUdpPort *  aCtlSipUdpPort,
                                           char *               aCommandLine );
        void              issueSipDispatcher(
                                      ControlMessage *        aMessage,
                                      controlSipDispatcher *  aCtlSipDispatcher,
                                      char *                  aCommandLine );
        void              issueRtpHandler( ControlMessage *     aMessage,
                                           controlRtpHandler *  cRtpHandler,
                                           char *               aCommandLine );
        void              issueSipRegistrarConnector(
                                            ControlMessage *  aMessage,
                                            controlSipRegistrarConnector *
                                                      aCtlSipRegistrarConnector,
                                            char *            aCommandLine );
        void              issueSipEndPoint(
                                          ControlMessage *      aMessage,
                                          controlSipEndPoint *  aCtlSipEndPoint,
                                          char *                aCommandLine );
        void              issueSipProxyEndPoint(
                                          ControlMessage *      aMessage,
                                          controlSipProxyEndPoint *  aCtlSipProxyEndPoint,
                                          char *                aCommandLine );
        void              issueSipRegistrar(
                                          ControlMessage *       aMessage,
                                          controlSipRegistrar *  aCtlSipRegistrar,
                                          char *                 aCommandLine );
        void              issueSipCall( ControlMessage *  aMessage,
                                        controlSipCall *  aCtlSipCall,
                                        char *            aCommandLine );

        unsigned int      nextAutoId( void );

        bool              isNameInUse( char *  aName, int  aConnectionId );


        in_addr_t           mAddress;
        in_port_t           mPort;
        int                 mMaxConnections;
        controlSocket *     mSockets;
        controlConnection * mConnections;
        queue               mBackgroundIncomingQueue;
        queue *             mIncomingQueues;
        queue *             mOutgoingQueues;
        bool                mNetworkStopFlag;
        bool                mProcessStopFlag;
        time_t              mCreationTime;
        pthread_mutex_t     mAutoIdMutex;
        unsigned int        mAutoIdCounter;

};

#endif

