//-----------------------------------------------------------------------------
//
// RtpHandler.h - Handler all RTP/RTCP traffic.
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#include <sys/time.h>
#include <stdlib.h>
#include <pthread.h>

#include "jrtplib/rtpsession.h"
#include "jrtplib/rtppacket.h"

#include "RtpHandler.h"


RtpHandler::RtpHandler( void )
{
    pthread_mutex_init( &mSessionsMutex, NULL );
    mSessions = NULL;
    mStopFlag = false;
}


RtpHandler::~RtpHandler( void )
{
    sessionNode *  node;
    sessionNode *  nodeNext;
    pthread_mutex_lock( &mSessionsMutex );
    node = mSessions;
    while ( node )
    {
        nodeNext = node->Next;
        delete node;
        node = nodeNext;
    }
    mSessions = NULL;
    pthread_mutex_unlock( &mSessionsMutex );
    pthread_mutex_destroy( &mSessionsMutex );
}


void  RtpHandler::Run( void )
{
    sessionNode *  node;
    RTPPacket *    packet;
    timeval        now;
    timeval        later;
    unsigned long  spacing;
    int            nodeCount;

    while ( !mStopFlag )
    {
        pthread_mutex_lock( &mSessionsMutex );
        nodeCount = 0;
        node      = mSessions;
        while ( node )
        {
            nodeCount++;
            node->EndPoint->GetSession()->PollData();
            if ( node->EndPoint->GetSession()->GotoFirstSourceWithData() )
            {
                do
                {
                    while ( ( packet = node->EndPoint->GetSession()->
                                         GetNextPacket() )
                            != NULL )
                    {
                        node->EndPoint->IncomingRtpPacket( packet );
                    }
                }
                while ( node->EndPoint->GetSession()->
                        GotoNextSourceWithData() );
            }
            gettimeofday( &now, NULL );
            if (   node->NextTransmission.tv_sec < now.tv_sec
                || (   node->NextTransmission.tv_sec == now.tv_sec
                    && node->NextTransmission.tv_usec <= now.tv_usec ) )
            {
                spacing = node->EndPoint->GetRtpSpacing();
                packet = node->EndPoint->GetNextRtpPacket();
                if ( packet )
                {
                    // Note: GLH had encoded the variable, spacing, as the last argument
                    //            in the invocation of SendPacket, however for G.711 the RTP
                    //            header timestamp is incremented by the payload length
                    //            (i.e. 160 bytes), not an actual time of 20 ms. I have hardcoded
                    //            the timestamp increment for G.711 u-law. -- MOB
                    //
                    
                    node->EndPoint->GetSession()->
                            SendPacket(
                                packet->GetPayload(),
                                packet->GetPayloadLength(),
                                ( unsigned char )
                                    (  node->EndPoint->GetUuTest() ? 20 : 0 ),
                                false,
                                ( unsigned long ) 160 /* G.711 timestamp inc */ );
                    delete packet;
                }
                later = node->EndPoint->GetBurstUntil();
                if (   later.tv_sec > now.tv_sec
                    || (   later.tv_sec == now.tv_sec
                        && later.tv_usec > now.tv_usec ) )
                {
                    spacing /= 2;
                }
                later = node->EndPoint->GetJitterUntil();
                if (   later.tv_sec > now.tv_sec
                    || (   later.tv_sec == now.tv_sec
                        && later.tv_usec > now.tv_usec ) )
                {
                    spacing +=   ( random() % ( spacing * 100 ) )
                               - ( spacing * 50 );
                }
                node->NextTransmission.tv_usec += spacing * 1000;
                if ( node->NextTransmission.tv_usec >= 1000000 )
                {
                    node->NextTransmission.tv_sec++;
                    node->NextTransmission.tv_usec -= 1000000;
                }
            }
            sched_yield();
            node = node->Next;
        }
        pthread_mutex_unlock( &mSessionsMutex );
        if ( nodeCount <= 0 )
        {
            usleep( 10 );
        }
        else
        {
            sched_yield();
        }
    }
}


void   RtpHandler::Stop( void )
{
    mStopFlag = true;
}


void   RtpHandler::AddEndPoint( RtpEndPoint *  aValue )
{
    sessionNode *  node;
    sessionNode *  nodePrevious;
    pthread_mutex_lock( &mSessionsMutex );
    // First, remove the session if already added.
    nodePrevious = NULL;
    node         = mSessions;
    while ( node )
    {
        if ( node->EndPoint == aValue )
        {
            if ( nodePrevious )
            {
                nodePrevious->Next = node->Next;
            }
            else
            {
                mSessions = node->Next;
            }
            delete node;
            break;
        }
        nodePrevious = node;
        node         = node->Next;
    }
    // Next, add the session.
    node = new sessionNode();
    node->EndPoint = aValue;
    gettimeofday( &node->NextTransmission, NULL );
    node->Next    = mSessions;
    mSessions      = node;
    pthread_mutex_unlock( &mSessionsMutex );
}


void   RtpHandler::RemoveEndPoint( RtpEndPoint *  aValue )
{
    sessionNode *  node;
    sessionNode *  nodePrevious;
    pthread_mutex_lock( &mSessionsMutex );
    nodePrevious = NULL;
    node         = mSessions;
    while ( node )
    {
        if ( node->EndPoint == aValue )
        {
            if ( nodePrevious )
            {
                nodePrevious->Next = node->Next;
            }
            else
            {
                mSessions = node->Next;
            }
            delete node;
            break;
        }
        nodePrevious = node;
        node         = node->Next;
    }
    pthread_mutex_unlock( &mSessionsMutex );
}

