//-----------------------------------------------------------------------------
//
// SdpMessage.cpp - Encapsulates SDP messages.
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "util.h"
#include "SdpMessage.h"

#define malloc  mymalloc
#define free    myfree
#define strdup  mystrdup
#define strndup mystrndup

SdpMessage::SdpMessage( void )
{
    mVersion           = 0;
    mOrigin            = NULL;
    mName              = NULL;
    mConnectionData    = NULL;
    mMediaDescriptions = NULL;
}


// aPayload contents will be duplicated within SdpMessage.
SdpMessage::SdpMessage( char *  aPayload )
{
    char *              line;
    char *              lines;
    char *              nextLine;
    MediaDescription *  lastMediaDescription;

    mVersion             = 0;
    mOrigin              = NULL;
    mName                = NULL;
    mConnectionData      = NULL;
    mMediaDescriptions   = NULL;
    lastMediaDescription = NULL;

    lines    = strdup( aPayload );
    nextLine = lines;
    
    while ( nextLine && *nextLine )
    {
        line = nextLine;

        if ( line[0] == 'm' && line[1] == '=' )
        {
            do
            {
                nextLine = strchr( nextLine + 1, '\n' );
            }
            while ( nextLine && ( nextLine[1] != 'm' || nextLine[2] != '=' ) );
        }
        else
        {
            nextLine = strchr( line, '\n' );
        }

        if ( nextLine )
        {
            if ( nextLine[-1] == '\r' )
            {
                nextLine[-1] = '\0';
            }
            else
            {
                *nextLine = '\0';
            }
            nextLine++;
        }

        if ( *line && line[1] == '=' )
        {
            if ( *line == 'v' )
            {
                mVersion = strtol( line + 2, NULL, 10 );
            }
            else if ( *line == 'o' )
            {
                if ( mOrigin )
                {
                    delete mOrigin;
                }
                mOrigin = new Origin( line );
            }
            else if ( *line == 's' )
            {
                mName = strdup( line + 2 );
            }
            else if ( *line == 'c' )
            {
                if ( mConnectionData )
                {
                    delete mConnectionData;
                }
                mConnectionData = new ConnectionData( line );
            }
            else if ( *line == 'm' )
            {
                if ( !lastMediaDescription )
                {
                    mMediaDescriptions = new MediaDescription( line );
                    lastMediaDescription = mMediaDescriptions;
                }
                else
                {
                    lastMediaDescription->SetNext(
                                                 new MediaDescription( line ) );
                    lastMediaDescription = lastMediaDescription->GetNext();
                }
            }
        }
    }

    if ( lines )
    {
        free( lines );
    }
}


SdpMessage::SdpMessage( int                 aVersion,
                        Origin *            aOrigin,
                        char *              aName,
                        ConnectionData *    aConnectionData,
                        MediaDescription *  aMediaDescriptions )
{
    mVersion           = aVersion;
    mOrigin            = aOrigin;
    mName              = aName;
    mConnectionData    = aConnectionData;
    mMediaDescriptions = aMediaDescriptions;
}


SdpMessage::~SdpMessage( void )
{
    MediaDescription *  md;

    if ( mOrigin )
    {
        delete mOrigin;
    }
    if ( mName )
    {
        free( mName );
    }
    if ( mConnectionData )
    {
        delete mConnectionData;
    }
    while ( mMediaDescriptions )
    {
        md = mMediaDescriptions->GetNext();
        delete mMediaDescriptions;
        mMediaDescriptions = md;
    }
}


int  SdpMessage::GetVersion( void )
{
    return mVersion;
}


// Do not delete return value.
SdpMessage::Origin *  SdpMessage::GetOrigin( void )
{
    return mOrigin;
}


// Do not free return value.
char *  SdpMessage::GetName( void )
{
    return mName;
}


// Do not delete return value.
SdpMessage::ConnectionData *  SdpMessage::GetConnectionData( void )
{
    return mConnectionData;
}


// Do not delete return value(s).
SdpMessage::MediaDescription *  SdpMessage::GetMediaDescriptions( void )
{
    return mMediaDescriptions;
}


// Caller must free return value.
char *  SdpMessage::GetPacket( void )
{
    char *              connectionDataLine;
    char *              mediaDescriptionLines;
    char *              originLine;
    char *              packet;
    int                 ix;
    int                 packetSize;
    MediaDescription *  mediaDescription;

    connectionDataLine = mConnectionData ? mConnectionData->GetSdpLine() : NULL;
    originLine         = mOrigin ? mOrigin->GetSdpLine() : NULL;
    if ( !originLine )
    {
        originLine = strdup( "o=- 0 0 IN IP4 0.0.0.0\r\n" );
    }
    packet             = NULL;
    packetSize         = 0;
    ix                 = 0;

    while ( ix >= packetSize )
    {
        if ( packet )
        {
            free( packet );
        }
        packetSize += 1024;
        packet = ( char * )malloc( packetSize );
        ix = snprintf( packet,
                       packetSize,
                       "v=%d\r\n"
                       "%s"
                       "s=%s\r\n",
                       mVersion,
                       originLine,
                       mName );
        if ( ix >= packetSize )
        {
            continue;
        }
        if ( connectionDataLine )
        {
            ix += snprintf( packet + ix,
                            packetSize - ix,
                            "%s",
                            connectionDataLine );
            if ( ix >= packetSize )
            {
                continue;
            }
        }
        mediaDescription = mMediaDescriptions;
        while ( mediaDescription )
        {
            mediaDescriptionLines = mediaDescription->GetSdpLines();
            if ( mediaDescriptionLines )
            {
                ix += snprintf( packet + ix,
                                packetSize - ix,
                                "%s",
                                mediaDescriptionLines );
                if ( ix >= packetSize )
                {
                    continue;
                }
                free( mediaDescriptionLines );
            }
            mediaDescription = mediaDescription->GetNext();
        }
    }

    if ( originLine )
    {
        free( originLine );
    }
    if ( connectionDataLine )
    {
        free( connectionDataLine );
    }

    return packet;
}


SdpMessage::Origin::Origin( void )
{
    mUserName    = NULL;
    mSessionId   = 0;
    mVersion     = 0;
    mNetworkType = NULL;
    mAddressType = NULL;
    mAddress     = NULL;
}


// aSdpLine contents will be duplicated within Origin.
SdpMessage::Origin::Origin( char *  aSdpLine )
{
    char *  mallocedLine;
    char *  line;
    char *  word;

    mUserName    = NULL;
    mSessionId   = 0;
    mVersion     = 0;
    mNetworkType = NULL;
    mAddressType = NULL;
    mAddress     = NULL;

    mallocedLine = line = strdup( aSdpLine );

    if ( line && line[0] == 'o' && line[1] == '=' )
    {
        line += 2;
        word = GetNextWord( &line );
        if ( *word )
        {
            mUserName = strdup( word );
            word = GetNextWord( &line );
            if ( *word )
            {
                mSessionId = strtol( word, NULL, 10 );
                word = GetNextWord( &line );
                if ( *word )
                {
                    mVersion = strtol( word, NULL, 10 );
                    word = GetNextWord( &line );
                    if ( *word )
                    {
                        mNetworkType = strdup( word );
                        word = GetNextWord( &line );
                        if ( *word )
                        {
                            mAddressType = strdup( word );
                            word = GetNextWord( &line );
                            if ( *word )
                            {
                                mAddress = strdup( word );
                            }
                        }
                    }
                }
            }
        }
    }

    if ( mallocedLine )
    {
        free( mallocedLine );
    }
}


// Ensure aUserName, mNetworkType, mAddressType, and mAddress are all malloced
// or NULL, as they will be freed.
SdpMessage::Origin::Origin( char *  aUserName,
                            int     aSessionId,
                            int     aVersion,
                            char *  aNetworkType,
                            char *  aAddressType,
                            char *  aAddress )
{
    mUserName    = aUserName;
    mSessionId   = aSessionId;
    mVersion     = aVersion;
    mNetworkType = aNetworkType;
    mAddressType = aAddressType;
    mAddress     = aAddress;
}


SdpMessage::Origin::~Origin( void )
{
    if ( mUserName )
    {
        free( mUserName );
    }
    if ( mNetworkType )
    {
        free( mNetworkType );
    }
    if ( mAddressType )
    {
        free( mAddressType );
    }
    if ( mAddress )
    {
        free( mAddress );
    }
}


// Do not free return value.
char *  SdpMessage::Origin::GetUserName( void )
{
    return mUserName;
}


int  SdpMessage::Origin::GetSessionId( void )
{
    return mSessionId;
}


int  SdpMessage::Origin::GetVersion( void )
{
    return mVersion;
}


// Do not free return value.
char *  SdpMessage::Origin::GetNetworkType( void )
{
    return mNetworkType;
}


// Do not free return value.
char *  SdpMessage::Origin::GetAddressType( void )
{
    return mAddressType;
}


// Do not free return value.
char *  SdpMessage::Origin::GetAddress( void )
{
    return mAddress;
}


// Caller must free return value.
char *  SdpMessage::Origin::GetSdpLine( void )
{
    char *  line;
    int     size;
    int     ix;

    size =  2/* o= */
          + ( mUserName ? strlen( mUserName ) : 1 /* dash */ ) + 1 /* space */
          + 1 /* session id is at least 1 digit */;
    for ( ix = mSessionId / 10; ix; ix /= 10 )
    {
        size++;
    }
    size += 1 /* space */ + 1 /* version is at least 1 digit */;
    for ( ix = mVersion / 10; ix; ix /= 10 )
    {
        size++;
    }
    size +=   1 /* space */
           + ( mNetworkType ? strlen( mNetworkType ) : 1 /* - */ ) + 1 /* sp */
           + ( mAddressType ? strlen( mAddressType ) : 1 /* - */ ) + 1 /* sp */
           + ( mAddress ? strlen( mAddress ) : 1 /* - */ )
           + 3 /* cr, lf, and null terminator */;

    line = ( char * )malloc( size );

    if ( line )
    {
        snprintf( line,
                  size,
                  "o=%s %d %d %s %s %s\r\n",
                  mUserName ? mUserName : "-",
                  mSessionId,
                  mVersion,
                  mNetworkType ? mNetworkType : "-",
                  mAddressType ? mAddressType : "-",
                  mAddress ? mAddress : "-" );
        line[size - 1] = '\0';
    }

    return line;
}


SdpMessage::ConnectionData::ConnectionData( void )
{
    mNetworkType = NULL;
    mAddressType = NULL;
    mAddress     = NULL;
}


// aSdpLine contents will be duplicated within ConnectionData.
SdpMessage::ConnectionData::ConnectionData( char *  aSdpLine )
{
    char *  mallocedLine;
    char *  line;
    char *  word;

    mNetworkType = NULL;
    mAddressType = NULL;
    mAddress     = NULL;

    mallocedLine = line = strdup( aSdpLine );

    if ( line && line[0] == 'c' && line[1] == '=' )
    {
        line += 2;
        word = GetNextWord( &line );
        if ( *word )
        {
            mNetworkType = strdup( word );
            word = GetNextWord( &line );
            if ( *word )
            {
                mAddressType = strdup( word );
                word = GetNextWord( &line );
                if ( *word )
                {
                    mAddress = strdup( word );
                }
            }
        }
    }

    if ( mallocedLine )
    {
        free( mallocedLine );
    }
}


// Ensure aNetworkType, aAddressType, and aAddress are all malloced or NULL, as
// they will be freed.
SdpMessage::ConnectionData::ConnectionData( char *  aNetworkType,
                                            char *  aAddressType,
                                            char *  aAddress )
{
    mNetworkType = aNetworkType;
    mAddressType = aAddressType;
    mAddress     = aAddress;
}


SdpMessage::ConnectionData::~ConnectionData( void )
{
    if ( mNetworkType )
    {
        free( mNetworkType );
    }
    if ( mAddressType )
    {
        free( mAddressType );
    }
    if ( mAddress )
    {
        free( mAddress );
    }
}


// Do not free return value.
char *  SdpMessage::ConnectionData::GetNetworkType( void )
{
    return mNetworkType;
}


// Do not free return value.
char *  SdpMessage::ConnectionData::GetAddressType( void )
{
    return mAddressType;
}


// Do not free return value.
char *  SdpMessage::ConnectionData::GetAddress( void )
{
    return mAddress;
}


// Caller must free return value.
char *  SdpMessage::ConnectionData::GetSdpLine( void )
{
    char *  line;
    int     size;

    size =  2 /* c= */
          + ( mNetworkType ? strlen( mNetworkType ) : 1 /* - */ ) + 1 /* sp */
          + ( mAddressType ? strlen( mAddressType ) : 1 /* - */ ) + 1 /* sp */
          + ( mAddress ? strlen( mAddress ) : 1 /* - */ )
          + 3 /* cr, lf, and null terminator */;

    line = ( char * )malloc( size );

    if ( line )
    {
        snprintf( line,
                  size,
                  "c=%s %s %s\r\n",
                  mNetworkType ? mNetworkType : "-",
                  mAddressType ? mAddressType : "-",
                  mAddress ? mAddress : "-" );
        line[size - 1] = '\0';
    }

    return line;
}


SdpMessage::MediaDescription::MediaDescription( void )
{
    mNext            = NULL;
    mType            = NULL;
    mPortSpec        = NULL;
    mProtocol        = NULL;
    mFormatsCount    = 0;
    mFormats         = NULL;
    mConnectionData  = NULL;
}


// aSdpLines contents will be duplicated within MediaDescription.
SdpMessage::MediaDescription::MediaDescription( char *  aSdpLines )
{
    char *  line;
    char *  lines;
    char *  word;
    char *  nextLine;
    char *  cp;
    char *  cp2;
    int     ix;

    mNext           = NULL;
    mType           = NULL;
    mPortSpec       = NULL;
    mProtocol       = NULL;
    mFormatsCount   = 0;
    mFormats        = NULL;
    mConnectionData = NULL;

    lines = strdup( aSdpLines );
    line  = lines;

    nextLine = strchr( line, '\n' );
    if ( nextLine )
    {
        if ( nextLine[-1] == '\r' )
        {
            nextLine[-1] = '\0';
        }
        else
        {
            *nextLine = '\0';
        }
        nextLine++;
    }
    
    if ( line[0] == 'm' && line[1] == '=' )
    {
        line += 2;
        word = GetNextWord( &line );
        if ( *word )
        {
            mType = strdup( word );
            word = GetNextWord( &line );
            if ( *word )
            {
                mPortSpec = strdup( word );
                word = GetNextWord( &line );
                if ( *word )
                {
                    mProtocol = strdup( word );
                    cp = cp2 = strdup( line );
                    mFormatsCount = 0;
                    while ( *( word = GetNextWord( &cp ) ) )
                    {
                        mFormatsCount++;
                    }
                    free( cp2 );
                    if ( mFormatsCount == 0 )
                    {
                        mFormats = NULL;
                    }
                    else
                    {
                        mFormats = ( char ** )malloc(  sizeof( char * )
                                                    * mFormatsCount );
                        for ( ix = 0; ix < mFormatsCount; ix++ )
                        {
                            mFormats[ix] = strdup( GetNextWord( &line ) );
                        }
                    }
                }
            }
        }
    }

    line = nextLine;
    ix   = 0;

    while ( nextLine && *nextLine )
    {
        line     = nextLine;
        nextLine = strchr( line, '\n' );
        if ( nextLine )
        {
            if ( nextLine[-1] == '\r' )
            {
                nextLine[-1] = '\0';
            }
            else
            {
                *nextLine = '\0';
            }
            nextLine++;
        }

        if ( *line && line[1] == '=' )
        {
            if ( *line == 'm' )
            {
                break;
            }
            else if ( *line == 'c' )
            {
                if ( mConnectionData )
                {
                    delete mConnectionData;
                }
                mConnectionData = new SdpMessage::ConnectionData( line );
            }
        }
    }
    
    if ( lines )
    {
        free( lines );
    }
}


// Ensure aType, aProtocol, and aFormats are malloced or NULL, as they will be
// freed. Ensure aConnectionData is newed or NULL, as it will be deleted.
SdpMessage::MediaDescription::MediaDescription(
                                             char *            aType,
                                             char *            aPortSpec,
                                             char *            aProtocol,
                                             int               aFormatsCount,
                                             char **           aFormats,
                                             ConnectionData *  aConnectionData )
{
    mNext            = NULL;
    mType            = aType;
    mPortSpec        = aPortSpec;
    mProtocol        = aProtocol;
    mFormatsCount    = aFormatsCount;
    mFormats         = aFormats;
    mConnectionData  = aConnectionData;
}


SdpMessage::MediaDescription::~MediaDescription( void )
{
    int ix;

    if ( mType )
    {
        free( mType );
    }
    if ( mProtocol )
    {
        free( mProtocol );
    }
    if ( mFormats )
    {
        for ( ix = 0; ix < mFormatsCount; ix++ )
        {
            if ( mFormats[ix] )
            {
                free( mFormats[ix] );
            }
        }
        free( mFormats );
    }
    if ( mConnectionData )
    {
        delete mConnectionData;
    }
}


SdpMessage::MediaDescription *  SdpMessage::MediaDescription::GetNext( void )
{
    return mNext;
}


void  SdpMessage::MediaDescription::SetNext( MediaDescription *  aValue )
{
    mNext = aValue;
}


// Do not free return values.
char *  SdpMessage::MediaDescription::GetType( void )
{
    return mType;
}


// Do not free return values.
char *  SdpMessage::MediaDescription::GetPortSpec( void )
{
    return mPortSpec;
}


int  SdpMessage::MediaDescription::GetPortIP4Value( void )
{
    return strtol( mPortSpec, NULL, 10 );
}


int  SdpMessage::MediaDescription::GetPortIP4Count( void )
{
    char *  cp;
    cp = strchr( mPortSpec, '/' );
    return cp ? strtol( cp + 1, NULL, 10 ) : 1;
}


// Do not delete return value.
char *  SdpMessage::MediaDescription::GetProtocol( void )
{
    return mProtocol;
}


int  SdpMessage::MediaDescription::GetFormatsCount( void )
{
    return mFormatsCount;
}


// Do not free return value.
char **  SdpMessage::MediaDescription::GetFormats( void )
{
    return mFormats;
}


// Do not delete return value.
SdpMessage::ConnectionData *  SdpMessage::MediaDescription::GetConnectionData(
                                                                          void )
{
    return mConnectionData;
}


// Caller must free return value. Contains trailing newline.
char *  SdpMessage::MediaDescription::GetSdpLines( void )
{
    char *  connectionDataLine;
    char *  lines;
    int     ix;
    int     iy;
    int     size;

    if ( mConnectionData )
    {
        connectionDataLine = mConnectionData->GetSdpLine();
    }
    else
    {
        connectionDataLine = NULL;
    }

    size =  2/* m= */
          + ( mType ? strlen( mType ) : 1 /* dash */ ) + 1 /* space */
          + ( mPortSpec ? strlen( mPortSpec ) : 1 /* dash */ ) + 1 /* space */
          + ( mProtocol ? strlen( mProtocol ) : 1 /* dash */ );
    for ( ix = 0; ix < mFormatsCount; ix++ )
    {
        size += 1 /* space */ + strlen( mFormats[ix] );
    }
    size +=  2 /* cr, lf */
           + ( connectionDataLine ? strlen( connectionDataLine ) : 0 )
           + 3 /* cr, lf, null */;

    lines = ( char * )malloc( size );

    if ( lines )
    {
        iy = snprintf( lines,
                       size,
                       "m=%s %s %s",
                       mType ? mType : "-",
                       mPortSpec ? mPortSpec : "-",
                       mProtocol ? mProtocol : "-" );
        for ( ix = 0; ix < mFormatsCount; ix++ )
        {
            iy += snprintf( lines + iy, size - iy, " %s", mFormats[ix] );
        }
        snprintf( lines + iy,
                  size - iy,
                  "\r\n%s",
                  connectionDataLine ? connectionDataLine : "" );
        lines[size - 1] = '\0';
    }

    return lines;
}

