//-----------------------------------------------------------------------------
//
// SipCall.cpp - Handles a single call. A single SipCall
//               object should not be used to place or receive
//               more than one single call.
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#define _XOPEN_SOURCE 500
#define _BSD_SOURCE

#include "SipCall.h"

#define malloc  mymalloc
#define free    myfree
#define strdup  mystrdup
#define strndup mystrndup


char *  SipCall::GetNameForState( State  aState )
{
    switch ( aState )
    {
        case STATE_INVALID:
            return "Invalid";
        case STATE_TIMER_FIRE:
            return "Timer Fire";
        case STATE_CREATED:
            return "Created";
        case STATE_GOT_MESSAGE:
            return "Got Message";
        case STATE_SEND_INVITE:
            return "Send Invite";
        case STATE_RESEND_INVITE:
            return "Resend Invite";
        case STATE_SENT_INVITE:
            return "Sent Invite";
        case STATE_INVITE_GOT_PROVISIONAL:
            return "Invite Got Provisional";
        case STATE_INVITE_GOT_PROVISIONAL_WAIT:
            return "Invite Got Provisional Wait";
        case STATE_GOT_OK_FOR_INVITE:
            return "Got Ok For Invite";
        case STATE_SEND_ACK_FOR_INVITE:
            return "Send Ack For Invite";
        case STATE_SEND_TRYING_FOR_INVITE:
            return "Send Trying";
        case STATE_SENT_TRYING_FOR_INVITE:
            return "Sent Trying";
        case STATE_SEND_RINGING:
            return "Send Ringing";
        case STATE_SENT_RINGING:
            return "Sent Ringing";
        case STATE_SEND_OK_FOR_INVITE:
            return "Send Ok For Invite";
        case STATE_RESEND_OK_FOR_INVITE:
            return "Resend Ok For Invite";
        case STATE_SENT_OK_FOR_INVITE:
            return "Sent Ok For Invite";
        case STATE_ESTABLISHED:
            return "Established";
        case STATE_SEND_BYE:
            return "Send Bye";
        case STATE_RESEND_BYE:
            return "Resend Bye";
        case STATE_SENT_BYE:
            return "Sent Bye";
        case STATE_GOT_OK_FOR_BYE:
            return "Got Ok For Bye";
        case STATE_GOT_BYE:
            return "Got Bye";
        case STATE_SEND_OK_FOR_BYE:
            return "Send Ok For Bye";
        case STATE_TERMINATE:
            return "Terminate";
        case STATE_TERMINATED:
            return "Terminated";
    }
    return "Unknown";
}


SipCall::SipCall( void )
{
    mIsCaller                         = false;
    mEndPoint                         = NULL;
    mState = STATE_CREATED;
    pthread_mutex_init( &mStateMutex, NULL );
    mLocalIdentifier                  = NULL;
    mRemoteIdentifier                 = NULL;
    mContactIdentifier                = NULL;
    mLocalTag                         = NULL;
    mRemoteTag                        = NULL;
    mLocalAddress                     = inet_addr( "127.0.0.1" );
    mRemoteAddress                    = inet_addr( "127.0.0.1" );
    mLocalPort                        = htons( 5060 );
    mRemotePort                       = htons( 5060 );
    mCallId                           = NULL;
    mLastBranch                       = NULL;
    mCseq                             = 0;
    mRemoteCseq                       = NULL;
    mRoutes                           = NULL;
    mViaHeadersHead                   = NULL;
    mViaHeadersTail                   = NULL;
    mTimer                            = new SipCallTimer( this );
    mTimerWaitTime                    = 0;
    mTimerTimeout                     = 0;
    mRemoteMediaAddress               = 0;
    mRemoteMediaPort                  = 0;
    mLocalMediaPort                   = 0;
    mRtpSession                       = NULL;
    mRtpSpacing                       = 0;
    mRtpBurstUntil.tv_sec             = 0;
    mRtpBurstUntil.tv_usec            = 0;
    mRtpJitterUntil.tv_sec            = 0;
    mRtpJitterUntil.tv_usec           = 0;
    mRtpUuTest                        = false;
    mRelayParty                       = NULL;
    mTappedParty                      = NULL;
    mTappedRelayParty                 = NULL;
    mLastRtpPacket                    = NULL;
    mLastRtpPacketExpiration.tv_sec   = 0;
    mLastRtpPacketExpiration.tv_usec  = 0;
    mSendShellCommand                 = NULL;
    mSendPrint                        = NULL;
    mSipEndpointPollingDelay          = 0;
    mNextAttackAudioPacketNumber      = 1;
    mAttackAudio                      = NULL;
}


SipCall::~SipCall( void )
{
    stringListNode *  node;
    stringListNode *  node2;

    if ( mTimer && mEndPoint && mEndPoint->GetDispatcher() )
    {
        mEndPoint->GetDispatcher()->RemoveTimer( mTimer );
    }
    if (   mEndPoint
        && mEndPoint->GetDispatcher()
        && mEndPoint->GetDispatcher()->GetUdpPort() )
    {
        mEndPoint->GetDispatcher()->GetUdpPort()->RemoveRespondingUdpSource(
                                                                         this );
        mEndPoint->GetDispatcher()->GetUdpPort()->RemoveRequestingUdpSource(
                                                                         this );
        mEndPoint->GetDispatcher()->GetUdpPort()->RemoveRetransmittingUdpSource(
                                                                         this );
    }
    if ( mLocalIdentifier )
    {
        delete mLocalIdentifier;
        mLocalIdentifier = NULL;
    }
    if ( mRemoteIdentifier )
    {
        delete mRemoteIdentifier;
        mRemoteIdentifier = NULL;
    }
    if ( mContactIdentifier )
    {
        delete mContactIdentifier;
        mContactIdentifier = NULL;
    }
    if ( mLocalTag )
    {
        free( mLocalTag );
        mLocalTag = NULL;
    }
    if ( mRemoteTag )
    {
        free( mRemoteTag );
        mRemoteTag = NULL;
    }
    if ( mCallId )
    {
        free( mCallId );
        mCallId = NULL;
    }
    if ( mLastBranch )
    {
        free( mLastBranch );
        mLastBranch = NULL;
    }
    if ( mRemoteCseq )
    {
        free( mRemoteCseq );
        mRemoteCseq = NULL;
    }
    node = mRoutes;
    while ( node )
    {
        node2 = node->Next;
        delete node;
        node = node2;
    }
    mRoutes = NULL;
    node    = mViaHeadersHead;
    while ( node )
    {
        node2 = node->Next;
        delete node;
        node = node2;
    }
    mViaHeadersHead = mViaHeadersTail = NULL;
    if ( mTimer )
    {
        delete mTimer;
        mTimer = NULL;
    }
    
    if ( mLastRtpPacket )
    {
        delete mLastRtpPacket;
        mLastRtpPacket = NULL;
    }

    if ( mAttackAudio )
    {    
        delete mAttackAudio;
        mAttackAudio = NULL;
    }
}


bool  SipCall::UpdateStateMachine( SipMessage *  aMessage,
                                   State         aDesiredState )
{
    struct timeval                tv;
    int                           newTimerWaitTime;
    char *                        cp;
    SdpMessage *                  remoteSdp;
    int                           len;
    int                           ix;
    SipHeader *                   header;
    struct in_addr                tmpaddr;
    SdpMessage::ConnectionData *  connectionData;
    SipIdentifier *                   headerIdentifier;
    SipUri *                   headerUri;
    struct hostent *              hent;

    pthread_mutex_lock( &mStateMutex );
    switch ( aDesiredState )
    {
        case STATE_INVALID:
        case STATE_TIMER_FIRE:
            break;

        case STATE_CREATED:
            aDesiredState = STATE_INVALID;
            break;

        case STATE_GOT_MESSAGE:
            if ( aMessage == NULL )
            {
                aDesiredState = STATE_INVALID;
            }
            // Deliberately leaving in current state. The fact that the message
            // value is not null should indicate a message was received in that
            // state.
            break;

        case STATE_SEND_INVITE:
        case STATE_RESEND_INVITE:
        case STATE_SENT_INVITE:
        case STATE_INVITE_GOT_PROVISIONAL:
        case STATE_INVITE_GOT_PROVISIONAL_WAIT:
        case STATE_GOT_OK_FOR_INVITE:
        case STATE_SEND_ACK_FOR_INVITE:
        case STATE_SEND_TRYING_FOR_INVITE:
        case STATE_SENT_TRYING_FOR_INVITE:
        case STATE_SEND_RINGING:
        case STATE_SENT_RINGING:
        case STATE_SEND_OK_FOR_INVITE:
        case STATE_RESEND_OK_FOR_INVITE:
        case STATE_SENT_OK_FOR_INVITE:
            aDesiredState = STATE_INVALID;
            break;

        case STATE_ESTABLISHED:
            if ( mState == STATE_CREATED )
            {
                if ( !setup( NULL ) )
                {
                    aDesiredState = STATE_INVALID;
                }
                mState = STATE_SEND_INVITE;
                mEndPoint->GetDispatcher()->GetUdpPort()->
                                                 AddRequestingUdpSource( this );
            }
            else
            {
                aDesiredState = STATE_INVALID;
            }
            break;

        case STATE_SEND_BYE:
        case STATE_RESEND_BYE:
        case STATE_SENT_BYE:
        case STATE_GOT_OK_FOR_BYE:
        case STATE_GOT_BYE:
        case STATE_SEND_OK_FOR_BYE:
            aDesiredState = STATE_INVALID;
            break;

        case STATE_TERMINATE:
            // FIXME: Cancel INVITEs, neg respond to INVITEs, etc.
            if ( mState != STATE_TERMINATED )
            {
                mState = STATE_SEND_BYE;
                mEndPoint->GetDispatcher()->GetUdpPort()->
                                                 AddRequestingUdpSource( this );
            }
            else
            {
                aDesiredState = STATE_INVALID;
            }
            break;

        case STATE_TERMINATED:
            if ( mState == STATE_CREATED )
            {
                mState = STATE_TERMINATED;
            }
            else if ( mState != STATE_TERMINATED && mState != STATE_TERMINATE )
            {
                mState = STATE_SEND_BYE;
                mEndPoint->GetDispatcher()->GetUdpPort()->
                                                 AddRequestingUdpSource( this );
            }
            break;

    }
    if ( aDesiredState == STATE_INVALID )
    {
        pthread_mutex_unlock( &mStateMutex );
        return false;
    }
    newTimerWaitTime = 0;
    while ( newTimerWaitTime == 0 )
    {
        switch ( mState )
        {
            case STATE_INVALID:
            case STATE_TIMER_FIRE:
                ReportError( "invalid state", __FILE__, __LINE__ );
                newTimerWaitTime = -1;
                break;
            case STATE_CREATED:
                newTimerWaitTime = -1;
                if (   aMessage
                    && aMessage->IsRequest()
                    && strcmp( "INVITE", aMessage->GetRequestMethod() ) == 0 )
                {
                    if ( setup( aMessage ) )
                    {
                        aMessage = NULL;
                        if ( mEndPoint->GetDispatcher()->
                                                      GetProvisionalsDesired() )
                        {
                            mState = STATE_SEND_TRYING_FOR_INVITE;
                            mEndPoint->GetDispatcher()->GetUdpPort()->
                                                 AddRespondingUdpSource( this );
                        }
                        else
                        {
                            if (   !mRelayParty
                                ||    mRelayParty->GetState()
                                   == STATE_ESTABLISHED )
                            {
                                mState = STATE_SEND_OK_FOR_INVITE;
                                mEndPoint->GetDispatcher()->GetUdpPort()->
                                                 AddRespondingUdpSource( this );
                            }
                            else
                            {
                                mState = STATE_SENT_RINGING;
                                newTimerWaitTime = 0;
                            }
                        }
                        if ( mRelayParty )
                        {
                            mRelayParty->Connect();
                        }
                    }
                }
                break;
            case STATE_GOT_MESSAGE:
                ReportError( "invalid state", __FILE__, __LINE__ );
                newTimerWaitTime = -1;
                break;
            case STATE_SEND_INVITE:
                newTimerWaitTime = -1;
                break;
            case STATE_RESEND_INVITE:
                newTimerWaitTime = -1;
                break;
            case STATE_SENT_INVITE:
                if ( aMessage && aMessage->IsResponse() )
                {
                    // FIXME: More checking, mCseq, etc.
                    switch ( aMessage->GetResponseCode() / 100 )
                    {
                        case 1:
                            mState = STATE_INVITE_GOT_PROVISIONAL;
                            break;
                        case 2:
                            mState = STATE_GOT_OK_FOR_INVITE;
                            break;
                        default:
                            mState = STATE_TERMINATE;
                            break;
                    }
                }
                else
                {
                    mState = STATE_RESEND_INVITE;
                    mEndPoint->GetDispatcher()->GetUdpPort()->
                                             AddRetransmittingUdpSource( this );
                }
                break;
            case STATE_INVITE_GOT_PROVISIONAL:
                mState = STATE_INVITE_GOT_PROVISIONAL_WAIT;
                if (   mEndPoint->GetDispatcher()->GetRetransmissionsDesired()
                    || mEndPoint->GetDispatcher()->GetTimeoutsDesired() )
                {
                    newTimerWaitTime = 32000; // FIXME: How long really?
                }
                else
                {
                    newTimerWaitTime = -1;
                }
                break;
            case STATE_INVITE_GOT_PROVISIONAL_WAIT:
                if ( aMessage && aMessage->IsResponse() )
                {
                    // FIXME: More checking, mCseq, etc.
                    switch ( aMessage->GetResponseCode() / 100 )
                    {
                        case 1:
                            mState = STATE_INVITE_GOT_PROVISIONAL;
                            break;
                        case 2:
                            mState = STATE_GOT_OK_FOR_INVITE;
                            break;
                        default:
                            mState = STATE_TERMINATE;
                            break;
                    }
                }
                else
                {
                    mState = STATE_SEND_INVITE;
                    mEndPoint->GetDispatcher()->GetUdpPort()->
                                             AddRetransmittingUdpSource( this );
                }
                break;
            case STATE_GOT_OK_FOR_INVITE:
                if ( aMessage )
                {
                    header = aMessage->GetHeader( "to" );
                    if ( !header )
                    {
                        header = aMessage->GetHeader( "t" );
                    }
                    if ( header )
                    {
                        if ( mRemoteIdentifier )
                        {
                            delete mRemoteIdentifier;
                        }
                        mRemoteIdentifier = header->GetIdentifier( 0 );
                        if ( mRemoteIdentifier )
                        {
                            mRemoteIdentifier =
                                         new SipIdentifier( mRemoteIdentifier );
                        }
                        if ( mRemoteTag )
                        {
                            free( mRemoteTag );
                        }
                        mRemoteTag = mRemoteIdentifier->
                                                     GetParameterValue( "tag" );
                    }
                    header = aMessage->GetHeader( "contact" );
                    if ( !header )
                    {
                        header = aMessage->GetHeader( "m" );
                    }
                    if ( header )
                    {
                        if ( mContactIdentifier )
                        {
                            delete mContactIdentifier;
                        }
                        mContactIdentifier = header->GetIdentifier( 0 );
                        if ( mContactIdentifier )
                        {
                            mContactIdentifier =
                                        new SipIdentifier( mContactIdentifier );
                        }
                    }
                    else if ( mRemoteIdentifier )
                    {
                        if ( mContactIdentifier )
                        {
                            delete mContactIdentifier;
                        }
                        mContactIdentifier =
                                         new SipIdentifier( mRemoteIdentifier );
                    }

                    recordRouting( aMessage );
                    cp = aMessage->GetHeader( "content-type" )->GetValue();
                    if ( cp && strcasecmp( cp, "application/sdp" ) == 0 )
                    {
                        remoteSdp = new SdpMessage( aMessage->GetContent() );
                        if ( remoteSdp->GetMediaDescriptions() )
                        {
                            connectionData = remoteSdp->GetMediaDescriptions()->
                                                            GetConnectionData();
                            if ( !connectionData )
                            {
                                connectionData = remoteSdp->GetConnectionData();
                            }
                            tmpaddr.s_addr = 0;
                            if ( connectionData )
                            {
                                inet_aton( connectionData->GetAddress(),
                                           &tmpaddr );
                            }
                            mRemoteMediaAddress = tmpaddr.s_addr;
                            mRemoteMediaPort = htons( remoteSdp->
                                                     GetMediaDescriptions()->
                                                     GetPortIP4Value() );
                            if ( mRtpSession )
                            {
                                mRtpSession->AddDestination(
                                                   ntohl( mRemoteMediaAddress ),
                                                   ntohs( mRemoteMediaPort ) );
                                mEndPoint->GetRtpHandler()->AddEndPoint( this );
                            }
                        }
                        delete remoteSdp;
                    }
                }
                mState = STATE_SEND_ACK_FOR_INVITE;
                mEndPoint->GetDispatcher()->GetUdpPort()->
                                                 AddRespondingUdpSource( this );
                break;
            case STATE_SEND_ACK_FOR_INVITE:
                newTimerWaitTime = -1;
                break;
            case STATE_SEND_TRYING_FOR_INVITE:
                mEndPoint->GetDispatcher()->GetUdpPort()->AddOutgoing(
                                                         getTryingForInvite() );
                mState = STATE_SENT_TRYING_FOR_INVITE;
                break;
            case STATE_SENT_TRYING_FOR_INVITE:
                mState = STATE_SEND_RINGING;
                mEndPoint->GetDispatcher()->GetUdpPort()->
                                                 AddRespondingUdpSource( this );
                break;
            case STATE_SEND_RINGING:
                newTimerWaitTime = -1;
                break;
            case STATE_SENT_RINGING:
                if (   !mRelayParty
                    || mRelayParty->GetState() == STATE_ESTABLISHED )
                {
                    mState = STATE_SEND_OK_FOR_INVITE;
                    mEndPoint->GetDispatcher()->GetUdpPort()->
                                                 AddRespondingUdpSource( this );
                }
                else if ( mRelayParty )
                {
                    newTimerWaitTime = 250;
                }
                break;
            case STATE_SEND_OK_FOR_INVITE:
                newTimerWaitTime = -1;
                break;
            case STATE_RESEND_OK_FOR_INVITE:
                newTimerWaitTime = -1;
                break;
            case STATE_SENT_OK_FOR_INVITE:
                if (   aMessage
                    && aMessage->IsRequest()
                    && strcmp( "ACK", aMessage->GetRequestMethod() ) == 0 )
                {
                    // FIXME: More checking, mCseq, etc.
                    mState = STATE_ESTABLISHED;
                    newTimerWaitTime = -1;
                }
                else if (   aMessage
                         && aMessage->IsRequest()
                         && strcmp( "BYE", aMessage->GetRequestMethod() ) == 0 )
                {
                    mState = STATE_GOT_BYE;
                }
                else
                {
                    mState = STATE_RESEND_OK_FOR_INVITE;
                    mEndPoint->GetDispatcher()->GetUdpPort()->
                                             AddRetransmittingUdpSource( this );
                }
                break;
            case STATE_ESTABLISHED:
                if ( !aMessage )
                {
                    newTimerWaitTime = -1;
                }
                else if (   mIsCaller
                         && aMessage->IsResponse()
                         && aMessage->GetResponseCode() / 100 == 2 )
                {
                    // FIXME: More checking, mCseq, etc.
                    mState = STATE_SEND_ACK_FOR_INVITE;
                    mEndPoint->GetDispatcher()->GetUdpPort()->
                                                 AddRespondingUdpSource( this );
                }
                else if (   aMessage->IsRequest()
                         && strcmp( "BYE", aMessage->GetRequestMethod() ) == 0 )
                {
                    // FIXME: More checking, mCseq, etc.
                    mState = STATE_GOT_BYE;
                }
                else
                {
                    newTimerWaitTime = -1;
                }
                break;
            case STATE_SEND_BYE:
                newTimerWaitTime = -1;
                break;
            case STATE_RESEND_BYE:
                newTimerWaitTime = -1;
                break;
            case STATE_SENT_BYE:
                if ( aMessage && aMessage->IsResponse() )
                {
                    switch ( aMessage->GetResponseCode() / 100 )
                    {
                        case 1:
                            newTimerWaitTime = mTimerWaitTime;
                            break;
                        default:
                            mState = STATE_GOT_OK_FOR_BYE;
                            break;
                    }
                }
                else if (   aMessage
                         && aMessage->IsRequest()
                         && strcmp( "BYE", aMessage->GetRequestMethod() ) == 0 )
                {
                    mState = STATE_TERMINATED;
                }
                else
                {
                    mState = STATE_RESEND_BYE;
                    mEndPoint->GetDispatcher()->GetUdpPort()->
                                             AddRetransmittingUdpSource( this );
                }
                break;
            case STATE_GOT_OK_FOR_BYE:
                mState = STATE_TERMINATE;
                break;
            case STATE_GOT_BYE:
                if ( aMessage )
                {
                    mRemoteAddress = 0;
                    mRemotePort = 0;

                    header = aMessage->GetHeader( "via" );
                    headerIdentifier =  header
                                      ? header->GetIdentifier( 0 )
                                      : NULL;
                    headerUri = headerIdentifier ? headerIdentifier->GetUri() : NULL;
                    cp = headerUri ? headerUri->GetHost() : NULL;
                    hent = cp ? gethostbyname( cp ) : NULL;
                    if ( hent && *hent->h_addr_list )
                    {
                        memcpy( ( char * )&mRemoteAddress,
                                *hent->h_addr_list,
                                sizeof( mRemoteAddress ) );
                        if ( headerUri->GetPort() > 0 )
                        {
                            mRemotePort = htons( headerUri->GetPort() );
                        }
                        else
                        {
                            mRemotePort = htons( 5060 );
                        }
                    }

                    if ( !mRemoteAddress )
                    {
                        mRemoteAddress = aMessage->GetSourceAddress();
                    }
                    if ( !mRemotePort )
                    {
                        mRemotePort = aMessage->GetSourcePort();
                    }

                    mLocalAddress = DetermineLocalAddressFor( mRemoteAddress );
                    mLocalPort    = aMessage->GetDestinationPort();

                    while ( mViaHeadersHead )
                    {
                        mViaHeadersTail = mViaHeadersHead->Next;
                        delete mViaHeadersHead;
                        mViaHeadersHead = mViaHeadersTail;
                    }
                    mViaHeadersHead = mViaHeadersTail = NULL;
                    len = aMessage->GetHeaderCount();
                    for ( ix = 0; ix < len; ix++ )
                    {
                        cp = aMessage->GetHeader( ix )->GetName();
                        if (   strcasecmp( "via", cp ) == 0
                            || strcasecmp( "v", cp ) == 0 )
                        {
                            if ( mViaHeadersTail )
                            {
                                mViaHeadersTail->Next = new stringListNode();
                                mViaHeadersTail       = mViaHeadersTail->Next;
                            }
                            else
                            {
                                mViaHeadersTail = new stringListNode();
                                mViaHeadersHead = mViaHeadersTail;
                            }
                            mViaHeadersTail->Next  = NULL;
                            mViaHeadersTail->Value =
                                strdup( aMessage->GetHeader( ix )->GetValue() );
                            if ( !mViaHeadersTail->Value )
                            {
                                ReportError( "out of memory",
                                             __FILE__, __LINE__ );
                                return false;
                            }
                        }
                    }
                    cp = aMessage->GetHeader( "cseq" )->GetValue();
                    if ( cp )
                    {
                        if ( mRemoteCseq )
                        {
                            free( mRemoteCseq );
                        }
                        mRemoteCseq = strdup( cp );
                        mState = STATE_SEND_OK_FOR_BYE;
                        mEndPoint->GetDispatcher()->GetUdpPort()->
                                                 AddRespondingUdpSource( this );
                    }
                }
                else
                {
                    mState = STATE_TERMINATE;
                }
                if ( mRelayParty )
                {
                    mRelayParty->Disconnect();
                }
                break;
            case STATE_SEND_OK_FOR_BYE:
                newTimerWaitTime = -1;
                break;
            case STATE_TERMINATE:
                mState = STATE_TERMINATED;
                break;
            case STATE_TERMINATED:
                if ( mRtpSession )
                {
                    mEndPoint->GetRtpHandler()->RemoveEndPoint( this );
                    delete mRtpSession;
                    mRtpSession = NULL;
                }
                mEndPoint->GetDispatcher()->RemoveCall( this );
                mEndPoint->GetDispatcher()->GetUdpPort()->
                                              RemoveRespondingUdpSource( this );
                mEndPoint->GetDispatcher()->GetUdpPort()->
                                              RemoveRequestingUdpSource( this );
                mEndPoint->GetDispatcher()->GetUdpPort()->
                                          RemoveRetransmittingUdpSource( this );
                newTimerWaitTime = 1; // to exit loop
                break;
        }
    }
    if ( mState != STATE_TERMINATED && newTimerWaitTime > 0 )
    {
        mTimerWaitTime = newTimerWaitTime;
        gettimeofday( &tv, NULL );
        mTimer->SetWhen(  ( tv.tv_sec & 0x003fffff ) * 1000
                        + tv.tv_usec / 1000 + mTimerWaitTime );
        mEndPoint->GetDispatcher()->AddTimer( mTimer );
    }
    pthread_mutex_unlock( &mStateMutex );

    if ( aMessage )
    {
        delete aMessage;
    }

    return true;
}


void  SipCall::Connect( void )
{
    UpdateStateMachine( NULL, STATE_ESTABLISHED );
}


void  SipCall::Disconnect( void )
{
    UpdateStateMachine( NULL, STATE_TERMINATED );
}


bool  SipCall::Incoming( SipMessage *  aMessage )
{
    return UpdateStateMachine( aMessage, STATE_GOT_MESSAGE );
}


SipCall::State  SipCall::GetState( void )
{
    return mState;
}


SipEndPoint *  SipCall::GetEndPoint( void )
{
    return mEndPoint;
}


void  SipCall::SetEndPoint( SipEndPoint *  aValue )
{
    mEndPoint = aValue;
}


char *  SipCall::GetCallId( void )
{
    return mCallId;
}


char *  SipCall::GetLocalTag( void )
{
    return mLocalTag;
}


char *  SipCall::GetRemoteTag( void )
{
    return mRemoteTag;
}


void  SipCall::SetRemoteIdentifier( SipIdentifier *  aIdentifier )
{
    mRemoteIdentifier = aIdentifier;
}


void  SipCall::SetContactIdentifier( SipIdentifier *  aIdentifier )
{
    mContactIdentifier = aIdentifier;
}


bool  SipCall::setup( SipMessage *  aMessage )
{
    char *                        callId;
    char *                        cp;
    char *                        cseq;
    SipIdentifier *               from;
    char *                        tmpGuid;
    SipIdentifier *               to;
    SipHeader *                   header;
    SipIdentifier *               headerIdentifier;
    SipUri *                      headerUri;
    struct hostent *              hent;
    SipIdentifier *               contact;
    int                           ix;
    int                           len;
    SdpMessage *                  remoteSdp;
    struct in_addr                tmpaddr;
    SdpMessage::ConnectionData *  connectionData;

    if ( aMessage ) // We are the call answerer.
    {
        mIsCaller = false;
        header = aMessage->GetHeader( "to" );
        if ( !header )
        {
            header = aMessage->GetHeader( "t" );
            if ( !header )
            {
                return false;
            }
        }
        to = header->GetIdentifier( 0 );
        if ( !to )
        {
            return false;
        }
        header = aMessage->GetHeader( "from" );
        if ( !header )
        {
            header = aMessage->GetHeader( "f" );
            if ( !header )
            {
                return false;
            }
        }
        from = header->GetIdentifier( 0 );
        if ( !from )
        {
            return false;
        }
        header = aMessage->GetHeader( "contact" );
        if ( !header )
        {
            header = aMessage->GetHeader( "m" );
            if ( !header )
            {
                return false;
            }
        }
        contact = header->GetIdentifier( 0 );
        if ( !contact )
        {
            return false;
        }
        header = aMessage->GetHeader( "call-id" );
        if ( !header )
        {
            header = aMessage->GetHeader( "i" );
            if ( !header )
            {
                return false;
            }
        }
        callId = header->GetValue();
        if ( !callId )
        {
            return false;
        }
        header = aMessage->GetHeader( "cseq" );
        if ( !header )
        {
            return false;
        }
        cseq = header->GetValue();
        if ( !cseq )
        {
            return false;
        }

        mLocalIdentifier   = new SipIdentifier( to );
        mRemoteIdentifier  = new SipIdentifier( from );
        mContactIdentifier = new SipIdentifier( contact );
        mCallId            = strdup( callId );
        mRemoteCseq        = strdup( cseq );

        recordRouting( aMessage );

        len = aMessage->GetHeaderCount();
        for ( ix = 0; ix < len; ix++ )
        {
            cp = aMessage->GetHeader( ix )->GetName();
            if ( strcasecmp( "via", cp ) == 0 || strcasecmp( "v", cp ) == 0 )
            {
                if ( mViaHeadersTail )
                {
                    mViaHeadersTail->Next = new stringListNode();
                    mViaHeadersTail       = mViaHeadersTail->Next;
                }
                else
                {
                    mViaHeadersTail = new stringListNode();
                    mViaHeadersHead = mViaHeadersTail;
                }
                mViaHeadersTail->Next  = NULL;
                mViaHeadersTail->Value =
                                 strdup( aMessage->GetHeader( ix )->GetValue() );
                if ( !mViaHeadersTail->Value )
                {
                    ReportError( "out of memory", __FILE__, __LINE__ );
                    return false;
                }
            }
        }

        mLocalTag = GetNextGuid();
        if ( !mLocalTag )
        {
            ReportError( "out of memory", __FILE__, __LINE__ );
            return false;
        }

        cp =  ( char * )malloc( strlen( mLocalIdentifier->GetFullText() )
            + strlen( mLocalTag ) + 6 );
        if ( !cp )
        {
            free( mLocalTag );
            ReportError( "out of memory", __FILE__, __LINE__ );
            return false;
        }
        strcpy( cp, mLocalIdentifier->GetFullText() );
        strcpy( cp + strlen( mLocalIdentifier->GetFullText() ), ";tag=" );
        strcpy( cp + strlen( mLocalIdentifier->GetFullText() ) + 5, mLocalTag );
        delete mLocalIdentifier;
        mLocalIdentifier = new SipIdentifier( cp );

        mRemoteTag = mRemoteIdentifier->GetParameterValue( "tag" );

        mRemoteAddress = 0;
        mRemotePort = 0;

        header = aMessage->GetHeader( "via" );
        headerIdentifier =  header
                          ? header->GetIdentifier( 0 )
                          : NULL;
        headerUri = headerIdentifier ? headerIdentifier->GetUri() : NULL;
        cp = headerUri ? headerUri->GetHost() : NULL;
        hent = cp ? gethostbyname( cp ) : NULL;
        if ( hent && *hent->h_addr_list )
        {
            memcpy( ( char * )&mRemoteAddress,
                    *hent->h_addr_list,
                    sizeof( mRemoteAddress ) );
            if ( headerUri->GetPort() > 0 )
            {
                mRemotePort = htons( headerUri->GetPort() );
            }
            else
            {
                mRemotePort = htons( 5060 );
            }
        }

        if ( !mRemoteAddress )
        {
            mRemoteAddress = aMessage->GetSourceAddress();
        }
        if ( !mRemotePort )
        {
            mRemotePort = aMessage->GetSourcePort();
        }

        mLocalAddress = DetermineLocalAddressFor( mRemoteAddress );
        mLocalPort    = aMessage->GetDestinationPort();

        // FIXME: Lot's of assumptions made in the SDP/Media area to get things
        // working for now.
        // It is assumed that RTP Type 0 PCMU will always be in use, will be 
        // the first sdp media description, that the IP address to send media
        // to will be the same as the one to send signaling to, among many
        // other shortcuts.
        cp = aMessage->GetHeader( "content-type" )->GetValue();
        if ( cp && strcasecmp( cp, "application/sdp" ) == 0 )
        {
            remoteSdp = new SdpMessage( aMessage->GetContent() );
            if ( remoteSdp->GetMediaDescriptions() )
            {
                connectionData =
                         remoteSdp->GetMediaDescriptions()->GetConnectionData();
                if ( !connectionData )
                {
                    connectionData = remoteSdp->GetConnectionData();
                }
                tmpaddr.s_addr = 0;
                if ( connectionData )
                {
                    inet_aton( connectionData->GetAddress(), &tmpaddr );
                }
                mRemoteMediaAddress = tmpaddr.s_addr;
                mRemoteMediaPort = htons(
                         remoteSdp->GetMediaDescriptions()->GetPortIP4Value() );
                if ( mRtpSession )
                {
                    mRtpSession->AddDestination(
                                         ntohl( mRemoteMediaAddress ),
                                         ntohs( mRemoteMediaPort ) );
                    mEndPoint->GetRtpHandler()->AddEndPoint( this );
                }
            }
            delete remoteSdp;
        }

        delete aMessage;
    }
    else // We are the call originator.
    {
        mIsCaller = true;
        mRemoteAddress = inet_addr( mRemoteIdentifier->GetUri()->GetHost() );
        ix = mRemoteIdentifier->GetUri()->GetPort();
        if ( ix == -1 )
        {
            mRemotePort = htons( 5060 );
        }
        else
        {
            mRemotePort = htons( ix );
        }

        mLocalAddress = DetermineLocalAddressFor( mRemoteAddress );

        if ( mLocalTag )
        {
            free( mLocalTag );
        }
        mLocalTag = GetNextGuid();
        if ( !mLocalTag )
        {
            ReportError( "can't get guid", __FILE__, __LINE__ );
            return false;
        }

        if ( !mLocalIdentifier )
        {
            // FIXME: Allow IP address/domain and port overrides
            len = 0;
            if ( mEndPoint->GetTextName() )
            {
                // + 3 is for double-quotes and a space
                len += strlen( mEndPoint->GetTextName() ) + 3;
            }
            len += 5;  // for <sip:
            len += strlen( mEndPoint->GetUser() );
            len += 16; // @xxx.xxx.xxx.xxx
            len += 1;  // :
            len += 5;  // xxxxx
            len += 1;  // >
            len += 5;  // ;tag=
            len += strlen( mLocalTag );
            len += 1;  // '\0'
            cp = ( char * )malloc( len );
            if ( !cp )
            {
                ReportError( "out of memory", __FILE__, __LINE__ );
                return false;
            }
            ix = 0;
            if ( mEndPoint->GetTextName() )
            {
                ix += snprintf( cp + ix,
                                len - ix,
                                "\"%s\" ",
                                mEndPoint->GetTextName() );
            }
            if ( mEndPoint->GetDispatcher()->GetUdpPort()->GetBoundAddress() )
            {
                mLocalAddress =
                    mEndPoint->GetDispatcher()->GetUdpPort()->GetBoundAddress();
            }
            ix += snprintf( cp + ix,
                            len - ix,
                            "<sip:%s@%d.%d.%d.%d:%d>;tag=%s",
                            mEndPoint->GetUser(),
                            mLocalAddress       & 0xff,
                            mLocalAddress >>  8 & 0xff,
                            mLocalAddress >> 16 & 0xff,
                            mLocalAddress >> 24 & 0xff,
                            ntohs( mEndPoint->GetDispatcher()->GetUdpPort(
                                   )->GetBoundPort() ),
                                   mLocalTag );
            if ( ix > len )
            {
                ReportError( "overrun", __FILE__, __LINE__ );
                return false;
            }
            mLocalIdentifier = new SipIdentifier( cp );
        }

        if ( mCallId )
        {
            free( mCallId );
        }
        mCallId = GetNextGuid();
        if ( !mCallId )
        {
            ReportError( "can't get guid", __FILE__, __LINE__ );
            return false;
        }

        mCseq = random() & 0x00ffffff;

        if ( mLastBranch )
        {
            free( mLastBranch );
        }
        mLastBranch = ( char * )malloc( 45 );
        if ( !mLastBranch )
        {
            ReportError( "out of memory", __FILE__, __LINE__ );
            return false;
        }
        tmpGuid = GetNextGuid();
        if ( !tmpGuid )
        {
            ReportError( "can't get guid", __FILE__, __LINE__ );
            return false;
        }
        snprintf( mLastBranch, 45, "z9hG4bK-%s", tmpGuid );
        free( tmpGuid );
    }

    return true;
}


SipMessage *  SipCall::getInvite( void )
{
    char *        cp;
    int           ix;
    SipMessage *  request;

    request = new SipMessage( true );
    request->SetSourceAddress( mLocalAddress );
    request->SetSourcePort(
                     mEndPoint->GetDispatcher()->GetUdpPort()->GetBoundPort() );
    request->SetDestinationAddress( mRemoteAddress );
    request->SetDestinationPort( mRemotePort );

    request->SetRequestMethod( strdup( "INVITE" ) );
    request->SetRequestUri( new SipUri( mRemoteIdentifier->GetUri() ) );

    cp = ( char * )malloc( 1301 );
    if ( !cp )
    {
        ReportError( "out of memory", __FILE__, __LINE__ );
        delete request;
        return false;
    }

    ix = snprintf( cp,
                   1301,
                   "SIP/2.0/UDP %d.%d.%d.%d:%d;branch=%s",
                   mLocalAddress       & 0xff,
                   mLocalAddress >>  8 & 0xff,
                   mLocalAddress >> 16 & 0xff,
                   mLocalAddress >> 24 & 0xff,
                   ntohs(
                     mEndPoint->GetDispatcher()->GetUdpPort()->GetBoundPort() ),
                     mLastBranch );
    request->AddHeader( new SipHeader( "Via", cp ) );

    request->AddHeader( new SipHeader( "Max-Forwards", "70" ) );
    request->AddHeader( new SipHeader( "To",
                                       mRemoteIdentifier->GetFullText() ) );
    request->AddHeader( new SipHeader( "From",
                                       mLocalIdentifier->GetFullText() ) );
    request->AddHeader( new SipHeader( "Call-Id", mCallId ) );

    ix = snprintf( cp, 1301, "%d INVITE", mCseq );
    request->AddHeader( new SipHeader( "CSeq", cp ) );

    ix = snprintf( cp,
                   1301,
                   "<sip:%s@%d.%d.%d.%d:%d>",
                   mEndPoint->GetUser(),
                   mLocalAddress       & 0xff,
                   mLocalAddress >>  8 & 0xff,
                   mLocalAddress >> 16 & 0xff,
                   mLocalAddress >> 24 & 0xff,
                   ntohs( mEndPoint->GetDispatcher()->GetUdpPort()->
                                                             GetBoundPort() ) );
    request->AddHeader( new SipHeader( "Contact", cp ) );

    request->AddHeader( new SipHeader( "Content-Type", "application/sdp" ) );
    if ( !mLocalMediaPort )
    {
        setupLocalMediaPort();
    }
    ix = snprintf( cp,
                   1301,
                   "v=0\r\n"
                   "o=- 1 2 IN IP4 %d.%d.%d.%d\r\n"
                   "s=call\r\n"
                   "c=IN IP4 %d.%d.%d.%d\r\n"
                   "t=0 0\r\n"
                   "m=audio %d RTP/AVP 0\r\n"
                   "a=rtpmap:0 PCMU/8000\r\n",
                   mLocalAddress       & 0xff,
                   mLocalAddress >>  8 & 0xff,
                   mLocalAddress >> 16 & 0xff,
                   mLocalAddress >> 24 & 0xff,
                   mLocalAddress       & 0xff,
                   mLocalAddress >>  8 & 0xff,
                   mLocalAddress >> 16 & 0xff,
                   mLocalAddress >> 24 & 0xff,
                   mLocalMediaPort );
    request->SetContent( strdup( cp ) );
    snprintf( cp, 1301, "%d", ix );
    request->AddHeader( new SipHeader( "Content-Length", cp ) );

    free( cp );

    return request;
}


SipMessage *  SipCall::getAckForInvite( void )
{
    char *        cp;
    int           ix;
    SipMessage *  request;

    request = new SipMessage( true );
    request->SetSourceAddress( mLocalAddress );
    request->SetSourcePort(
                     mEndPoint->GetDispatcher()->GetUdpPort()->GetBoundPort() );
    request->SetDestinationAddress( mRemoteAddress );
    request->SetDestinationPort( mRemotePort );

    request->SetRequestMethod( strdup( "ACK" ) );
    request->SetRequestUri( new SipUri( mRemoteIdentifier->GetUri() ) );

    cp = ( char * )malloc( 1301 );
    if ( !cp )
    {
        ReportError( "out of memory", __FILE__, __LINE__ );
        delete request;
        return false;
    }

    addRouting( request );

    ix = snprintf( cp,
                   1301,
                   "SIP/2.0/UDP %d.%d.%d.%d:%d;branch=%s",
                   mLocalAddress       & 0xff,
                   mLocalAddress >>  8 & 0xff,
                   mLocalAddress >> 16 & 0xff,
                   mLocalAddress >> 24 & 0xff,
                   ntohs(
                   mEndPoint->GetDispatcher()->GetUdpPort()->GetBoundPort() ),
                   mLastBranch );
    request->AddHeader( new SipHeader( "Via", cp ) );

    request->AddHeader( new SipHeader( "Max-Forwards", "70" ) );
    request->AddHeader( new SipHeader( "To",
                                       mRemoteIdentifier->GetFullText() ) );
    request->AddHeader( new SipHeader( "From",
                                       mLocalIdentifier->GetFullText() ) );
    request->AddHeader( new SipHeader( "Call-Id", mCallId ) );

    ix = snprintf( cp, 1301, "%d ACK", mCseq );
    request->AddHeader( new SipHeader( "CSeq", cp ) );

    ix = snprintf( cp,
                   1301,
                   "<sip:%s@%d.%d.%d.%d:%d>",
                   mEndPoint->GetUser(),
                   mLocalAddress       & 0xff,
                   mLocalAddress >>  8 & 0xff,
                   mLocalAddress >> 16 & 0xff,
                   mLocalAddress >> 24 & 0xff,
                   ntohs(
                   mEndPoint->GetDispatcher()->GetUdpPort()->GetBoundPort() ) );
    request->AddHeader( new SipHeader( "Contact", cp ) );

    if ( !mLocalMediaPort )
    {
        request->AddHeader( new SipHeader( "Content-Type", "application/sdp" ) );
        setupLocalMediaPort();
        ix = snprintf( cp,
                       1301,
                       "v=0\r\n"
                       "o=- 1 2 IN IP4 %d.%d.%d.%d\r\n"
                       "s=call\r\n"
                       "c=IN IP4 %d.%d.%d.%d\r\n"
                       "t=0 0\r\n"
                       "m=audio %d RTP/AVP 0\r\n"
                       "a=rtpmap:0 PCMU/8000\r\n",
                       mLocalAddress       & 0xff,
                       mLocalAddress >>  8 & 0xff,
                       mLocalAddress >> 16 & 0xff,
                       mLocalAddress >> 24 & 0xff,
                       mLocalAddress       & 0xff,
                       mLocalAddress >>  8 & 0xff,
                       mLocalAddress >> 16 & 0xff,
                       mLocalAddress >> 24 & 0xff,
                       mLocalMediaPort );
        request->SetContent( strdup( cp ) );
        snprintf( cp, 1301, "%d", ix );
        request->AddHeader( new SipHeader( "Content-Length", cp ) );
    }
    else
    {
        request->AddHeader( new SipHeader( "Content-Length", "0" ) );
    }

    free( cp );

    return request;
}


SipMessage *  SipCall::getBye( void )
{
    char *        cp;
    int           ix;
    SipMessage *  request;

    request = new SipMessage( true );
    request->SetSourceAddress( mLocalAddress );
    request->SetSourcePort(
                     mEndPoint->GetDispatcher()->GetUdpPort()->GetBoundPort() );
    request->SetDestinationAddress( mRemoteAddress );
    request->SetDestinationPort( mRemotePort );

    if ( mState != STATE_SENT_BYE )
    {
        mCseq++;
    }

    request->SetRequestMethod( strdup( "BYE" ) );
    request->SetRequestUri( new SipUri( mRemoteIdentifier->GetUri() ) );

    cp = ( char * )malloc( 1301 );
    if ( !cp )
    {
        ReportError( "out of memory", __FILE__, __LINE__ );
        delete request;
        return false;
    }

    addRouting( request );

    ix = snprintf( cp,
                   1301,
                   "SIP/2.0/UDP %d.%d.%d.%d:%d;branch=%s",
                   mLocalAddress       & 0xff,
                   mLocalAddress >>  8 & 0xff,
                   mLocalAddress >> 16 & 0xff,
                   mLocalAddress >> 24 & 0xff,
                   ntohs(
                   mEndPoint->GetDispatcher()->GetUdpPort()->GetBoundPort() ),
                   mLastBranch );
    request->AddHeader( new SipHeader( "Via", cp ) );

    request->AddHeader( new SipHeader( "Max-Forwards", "70" ) );
    request->AddHeader( new SipHeader( "To",
                                       mRemoteIdentifier->GetFullText() ) );
    request->AddHeader( new SipHeader( "From",
                                       mLocalIdentifier->GetFullText() ) );
    request->AddHeader( new SipHeader( "Call-Id", mCallId ) );

    ix = snprintf( cp, 1301, "%d BYE", mCseq );
    request->AddHeader( new SipHeader( "CSeq", cp ) );

    ix = snprintf( cp,
                   1301,
                   "<sip:%s@%d.%d.%d.%d:%d>",
                   mEndPoint->GetUser(),
                   mLocalAddress       & 0xff,
                   mLocalAddress >>  8 & 0xff,
                   mLocalAddress >> 16 & 0xff,
                   mLocalAddress >> 24 & 0xff,
                   ntohs(
                   mEndPoint->GetDispatcher()->GetUdpPort()->GetBoundPort() ) );
    request->AddHeader( new SipHeader( "Contact", cp ) );

    request->AddHeader( new SipHeader( "Content-Length", "0" ) );

    free( cp );

    return request;
}


SipMessage *  SipCall::getTryingForInvite( void )
{
    SipMessage *      response;
    stringListNode *  viaHeader;
    char *            cp;
    int               ix;

    response = new SipMessage( false );
    response->SetResponseCode( 100 );
    response->SetResponseText( strdup( "Trying" ) );
    response->SetSourceAddress( mLocalAddress );
    response->SetDestinationAddress( mRemoteAddress );
    response->SetSourcePort( mLocalPort );
    response->SetDestinationPort( mRemotePort );

    addRouting( response );

    viaHeader = mViaHeadersHead;
    while ( viaHeader )
    {
        response->AddHeader( new SipHeader( "Via", viaHeader->Value ) );
        viaHeader = viaHeader->Next;
    }

    // FIXME: Timestamp header handling needs to be done

    response->AddHeader( new SipHeader( "To",
                                        mLocalIdentifier->GetFullText() ) );
    response->AddHeader( new SipHeader( "From",
                                        mRemoteIdentifier->GetFullText() ) );
    response->AddHeader( new SipHeader( "Call-Id", mCallId ) );
    response->AddHeader( new SipHeader( "CSeq", mRemoteCseq ) );

    cp = ( char * )malloc( 1301 );
    if ( !cp )
    {
        ReportError( "out of memory", __FILE__, __LINE__ );
        delete response;
        return false;
    }
    ix = snprintf( cp,
                   1301,
                   "<sip:%s@%d.%d.%d.%d:%d>",
                   mEndPoint->GetUser(),
                   mLocalAddress       & 0xff,
                   mLocalAddress >>  8 & 0xff,
                   mLocalAddress >> 16 & 0xff,
                   mLocalAddress >> 24 & 0xff,
                   ntohs( mEndPoint->GetDispatcher()->GetUdpPort()->
                                                             GetBoundPort() ) );
    response->AddHeader( new SipHeader( "Contact", cp ) );

    response->AddHeader( new SipHeader( "Content-Length", "0" ) );

    return response;
}


SipMessage *  SipCall::getRinging( void )
{
    SipMessage *      response;
    stringListNode *  viaHeader;
    char *            cp;
    int               ix;

    response = new SipMessage( false );
    response->SetResponseCode( 180 );
    response->SetResponseText( strdup( "Ringing" ) );
    response->SetSourceAddress( mLocalAddress );
    response->SetDestinationAddress( mRemoteAddress );
    response->SetSourcePort( mLocalPort );
    response->SetDestinationPort( mRemotePort );

    addRouting( response );

    viaHeader = mViaHeadersHead;
    while ( viaHeader )
    {
        response->AddHeader( new SipHeader( "Via", viaHeader->Value ) );
        viaHeader = viaHeader->Next;
    }

    // FIXME: Timestamp header handling needs to be done

    response->AddHeader( new SipHeader( "To",
                                        mLocalIdentifier->GetFullText() ) );
    response->AddHeader( new SipHeader( "From",
                                        mRemoteIdentifier->GetFullText() ) );
    response->AddHeader( new SipHeader( "Call-Id", mCallId ) );
    response->AddHeader( new SipHeader( "CSeq", mRemoteCseq ) );

    cp = ( char * )malloc( 1301 );
    if ( !cp )
    {
        ReportError( "out of memory", __FILE__, __LINE__ );
        delete response;
        return false;
    }
    ix = snprintf( cp,
                   1301,
                   "<sip:%s@%d.%d.%d.%d:%d>",
                   mEndPoint->GetUser(),
                   mLocalAddress       & 0xff,
                   mLocalAddress >>  8 & 0xff,
                   mLocalAddress >> 16 & 0xff,
                   mLocalAddress >> 24 & 0xff,
                   ntohs( mEndPoint->GetDispatcher()->GetUdpPort()->
                                                             GetBoundPort() ) );
    response->AddHeader( new SipHeader( "Contact", cp ) );

    response->AddHeader( new SipHeader( "Content-Length", "0" ) );

    return response;
}


SipMessage *  SipCall::getOkForInvite( void )
{
    SipMessage *      response;
    stringListNode *  viaHeader;
    char *            cp;
    int               ix;

    if ( !mLocalIdentifier || !mRemoteIdentifier || !mCallId || !mRemoteCseq )
    {
        // FIXME
        return false;
    }

    response = new SipMessage( false );
    response->SetResponseCode( 200 );
    response->SetResponseText( strdup( "Ok" ) );
    response->SetSourceAddress( mLocalAddress );
    response->SetDestinationAddress( mRemoteAddress );
    response->SetSourcePort( mLocalPort );
    response->SetDestinationPort( mRemotePort );

    addRouting( response );

    viaHeader = mViaHeadersHead;
    while ( viaHeader )
    {
        response->AddHeader( new SipHeader( "Via", viaHeader->Value ) );
        viaHeader = viaHeader->Next;
    }

    // FIXME: Timestamp header handling needs to be done

    response->AddHeader( new SipHeader( "To",
                                        mLocalIdentifier->GetFullText() ) );
    response->AddHeader( new SipHeader( "From",
                                        mRemoteIdentifier->GetFullText() ) );
    response->AddHeader( new SipHeader( "Call-Id", mCallId ) );
    response->AddHeader( new SipHeader( "CSeq", mRemoteCseq ) );

    cp = ( char * )malloc( 1301 );
    if ( !cp )
    {
        ReportError( "out of memory", __FILE__, __LINE__ );
        delete response;
        return false;
    }

    ix = snprintf( cp,
                   1301,
                   "<sip:%s@%d.%d.%d.%d:%d>",
                   mEndPoint->GetUser(),
                   mLocalAddress       & 0xff,
                   mLocalAddress >>  8 & 0xff,
                   mLocalAddress >> 16 & 0xff,
                   mLocalAddress >> 24 & 0xff,
                   ntohs( mEndPoint->GetDispatcher()->GetUdpPort()->
                                                             GetBoundPort() ) );

    response->AddHeader( new SipHeader( "Contact", cp ) );

    response->AddHeader( new SipHeader( "Content-Type", "application/sdp" ) );
    if ( !mLocalMediaPort )
    {
        setupLocalMediaPort();
    }
    ix = snprintf( cp,
                   1301,
                   "v=0\r\n"
                   "o=- 1 2 IN IP4 %d.%d.%d.%d\r\n"
                   "s=call\r\n"
                   "c=IN IP4 %d.%d.%d.%d\r\n"
                   "t=0 0\r\n"
                   "m=audio %d RTP/AVP 0\r\n"
                   "a=rtpmap:0 PCMU/8000\r\n",
                   mLocalAddress       & 0xff,
                   mLocalAddress >>  8 & 0xff,
                   mLocalAddress >> 16 & 0xff,
                   mLocalAddress >> 24 & 0xff,
                   mLocalAddress       & 0xff,
                   mLocalAddress >>  8 & 0xff,
                   mLocalAddress >> 16 & 0xff,
                   mLocalAddress >> 24 & 0xff,
                   mLocalMediaPort );
    response->SetContent( strdup( cp ) );
    snprintf( cp, 1301, "%d", ix );
    response->AddHeader( new SipHeader( "Content-Length", cp ) );

    free( cp );

    return response;
}


SipMessage *  SipCall::getOkForBye( void )
{
    SipMessage *      response;
    stringListNode *  viaHeader;
    int               ix;
    char *            cp;

    if ( !mLocalIdentifier || !mRemoteIdentifier || !mCallId || !mRemoteCseq )
    {
        // FIXME
        return false;
    }

    response = new SipMessage( false );
    response->SetResponseCode( 200 );
    response->SetResponseText( strdup( "Ok" ) );
    response->SetSourceAddress( mLocalAddress );
    response->SetDestinationAddress( mRemoteAddress );
    response->SetSourcePort( mLocalPort );
    response->SetDestinationPort( mRemotePort );

    addRouting( response );

    viaHeader = mViaHeadersHead;
    while ( viaHeader )
    {
        response->AddHeader( new SipHeader( "Via", viaHeader->Value ) );
        viaHeader = viaHeader->Next;
    }

    // FIXME: Timestamp header handling needs to be done

    response->AddHeader( new SipHeader( "To",
                                        mLocalIdentifier->GetFullText() ) );
    response->AddHeader( new SipHeader( "From",
                                        mRemoteIdentifier->GetFullText() ) );
    response->AddHeader( new SipHeader( "Call-Id", mCallId ) );
    response->AddHeader( new SipHeader( "CSeq", mRemoteCseq ) );

    cp = ( char * )malloc( 1301 );
    if ( !cp )
    {
        ReportError( "out of memory", __FILE__, __LINE__ );
        delete response;
        return false;
    }

    ix = snprintf( cp,
                   1301,
                   "<sip:%s@%d.%d.%d.%d:%d>",
                   mEndPoint->GetUser(),
                   mLocalAddress       & 0xff,
                   mLocalAddress >>  8 & 0xff,
                   mLocalAddress >> 16 & 0xff,
                   mLocalAddress >> 24 & 0xff,
                   ntohs( mEndPoint->GetDispatcher()->GetUdpPort()->
                                                             GetBoundPort() ) );

    response->AddHeader( new SipHeader( "Contact", cp ) );

    response->AddHeader( new SipHeader( "Content-Length", "0" ) );

    return response;
}


void  SipCall::setupLocalMediaPort( void )
{
    int tryPort;
    if ( mEndPoint->GetRtpHandler() )
    {
        do
        {
            if ( mRtpSession )
            {
                delete mRtpSession;
            }
            mRtpSession = new RTPSession();
            tryPort = ( ( random() % ( ( 65536 - 1024 ) / 2 ) ) * 2 ) + 1024;
            sched_yield();
        } while ( mRtpSession->Create( tryPort ) < 0 );
        mLocalMediaPort = tryPort;
        mRtpSpacing = 20;
        if ( mRemoteMediaAddress != 0 && mRemoteMediaPort != 0 )
        {
            mRtpSession->AddDestination( ntohl( mRemoteMediaAddress ),
                                         ntohs( mRemoteMediaPort ) );
            mEndPoint->GetRtpHandler()->AddEndPoint( this );
        }
    } else {
        mLocalMediaPort =  ( ( random() % ( ( 65536 - 1024 ) / 2 ) ) * 2 )
                         + 1024;
    }
}


void  SipCall::SendShell( char *  aCommandLine )
{
    if ( mSendShellCommand )
    {
        printf( "Never had time to send shell command: %s\n",
                mSendShellCommand );
        free( mSendShellCommand );
    }
    mSendShellCommand = strdup( aCommandLine );
}


void  SipCall::SendRtpBurst( int  aSeconds )
{
    gettimeofday( &mRtpBurstUntil, NULL );
    mRtpBurstUntil.tv_sec += aSeconds;
}


void  SipCall::SendRtpJitter( int  aSeconds )
{
    gettimeofday( &mRtpJitterUntil, NULL );
    mRtpJitterUntil.tv_sec += aSeconds;
}


void  SipCall::SendRtpUuTest( void )
{
    mRtpUuTest = true;
}


void  SipCall::SetRelayParty( SipCall *  aCall )
{
    mRelayParty = aCall;
}


void  SipCall::SetTappedParty( SipCall *  aCall )
{
    mTappedParty = aCall;
}


void  SipCall::SetTappedRelayParty( SipCall *  aCall )
{
    mTappedRelayParty = aCall;
}


in_addr_t  SipCall::GetUdpAddress( void )
{
    return mRemoteAddress;
}


in_port_t  SipCall::GetUdpPort( void )
{
    return mRemotePort;
}


// Caller must free return value.
char *  SipCall::GetUdpPacket( void )
{
    SipMessage *   message;
    char *         packet;
    timeval        tv;
    unsigned long  now;
    int            newTimerWaitTime;

    message = NULL;
    packet = NULL;
    newTimerWaitTime = 0;

    pthread_mutex_lock( &mStateMutex );
    switch ( mState )
    {
        case STATE_INVALID:
        case STATE_TIMER_FIRE:
        case STATE_CREATED:
        case STATE_GOT_MESSAGE:
            break;
        case STATE_SEND_INVITE:
            message = getInvite();
            mState = STATE_SENT_INVITE;
            gettimeofday( &tv, NULL );
            if ( mEndPoint->GetDispatcher()->GetTimeoutsDesired() )
            {
                mTimerTimeout =  ( tv.tv_sec & 0x003fffff ) * 1000
                              + tv.tv_usec / 1000 + 64
                              * mEndPoint->GetDispatcher()->GetT1();
            }
            if (   mEndPoint->GetDispatcher()->GetRetransmissionsDesired()
                || mEndPoint->GetDispatcher()->GetTimeoutsDesired() )
            {
                newTimerWaitTime = mEndPoint->GetDispatcher()->GetT1();
            }
            else
            {
                newTimerWaitTime = -1;
            }
            break;
        case STATE_RESEND_INVITE:
            gettimeofday( &tv, NULL );
            now = ( tv.tv_sec & 0x003fffff ) * 1000 + tv.tv_usec / 1000;
            if (   mEndPoint->GetDispatcher()->GetTimeoutsDesired()
                && now > mTimerTimeout )
            {
                mState = STATE_TERMINATE;
            }
            else if ( mEndPoint->GetDispatcher()->GetRetransmissionsDesired() )
            {
                message = getInvite();
                mState = STATE_SENT_INVITE;
                newTimerWaitTime = mTimerWaitTime;
                if ( newTimerWaitTime < mEndPoint->GetDispatcher()->GetT2() )
                {
                    newTimerWaitTime *= 2;
                }
            }
            break;
        case STATE_SENT_INVITE:
            break;
        case STATE_INVITE_GOT_PROVISIONAL:
        case STATE_INVITE_GOT_PROVISIONAL_WAIT:
        case STATE_GOT_OK_FOR_INVITE:
            break;
        case STATE_SEND_ACK_FOR_INVITE:
            message = getAckForInvite();
            mState = STATE_ESTABLISHED;
            newTimerWaitTime = -1;
            break;
        case STATE_SEND_TRYING_FOR_INVITE:
            // Special case: Trying is put on the SipMessage queue (in
            // UpdateStateMachine) rather than the UdpSource queue because it
            // will have a higher priority that way.
            break;
        case STATE_SENT_TRYING_FOR_INVITE:
            break;
        case STATE_SEND_RINGING:
            message = getRinging();
            mState = STATE_SENT_RINGING;
            newTimerWaitTime =   mEndPoint->GetRingingMinimum()
                               + (  random()
                                  % (  mEndPoint->GetRingingMaximum()
                                     - mEndPoint->GetRingingMinimum() ) );
            break;
        case STATE_SENT_RINGING:
            break;
        case STATE_SEND_OK_FOR_INVITE:
            gettimeofday( &tv, NULL );
            if ( mEndPoint->GetDispatcher()->GetTimeoutsDesired() )
            {
                mTimerTimeout =  ( tv.tv_sec & 0x003fffff ) * 1000
                              + tv.tv_usec / 1000 + 64
                              * mEndPoint->GetDispatcher()->GetT1();
            }
            message = getOkForInvite();
            mState = STATE_SENT_OK_FOR_INVITE;
            if (   mEndPoint->GetDispatcher()->GetRetransmissionsDesired()
                || mEndPoint->GetDispatcher()->GetTimeoutsDesired() )
            {
                newTimerWaitTime = mEndPoint->GetDispatcher()->GetT1();
            }
            else
            {
                newTimerWaitTime = -1;
            }
            break;
        case STATE_RESEND_OK_FOR_INVITE:
            gettimeofday( &tv, NULL );
            now = ( tv.tv_sec & 0x003fffff ) * 1000 + tv.tv_usec / 1000;
            if (   mEndPoint->GetDispatcher()->GetTimeoutsDesired()
                && now > mTimerTimeout )
            {
                mState = STATE_SEND_BYE;
                mEndPoint->GetDispatcher()->GetUdpPort()->
                                                 AddRequestingUdpSource( this );
            }
            else 
            {
                if ( mEndPoint->GetDispatcher()->GetRetransmissionsDesired() )
                {
                    message = getOkForInvite();
                }
                mState = STATE_SENT_OK_FOR_INVITE;
                newTimerWaitTime = mTimerWaitTime;
                if ( newTimerWaitTime < mEndPoint->GetDispatcher()->GetT2() )
                {
                    newTimerWaitTime *= 2;
                }
            }
            break;
        case STATE_SENT_OK_FOR_INVITE:
        case STATE_ESTABLISHED:
            break;
        case STATE_SEND_BYE:
            gettimeofday( &tv, NULL );
            if ( mEndPoint->GetDispatcher()->GetTimeoutsDesired() )
            {
                mTimerTimeout =  ( tv.tv_sec & 0x003fffff ) * 1000
                              + tv.tv_usec / 1000 + 64
                              * mEndPoint->GetDispatcher()->GetT1();
            }
            message = getBye();
            mState = STATE_SENT_BYE;
            if (   mEndPoint->GetDispatcher()->GetRetransmissionsDesired()
                || mEndPoint->GetDispatcher()->GetTimeoutsDesired() )
            {
                newTimerWaitTime = mEndPoint->GetDispatcher()->GetT1();
            }
            else
            {
                newTimerWaitTime = -1;
            }
            break;
        case STATE_RESEND_BYE:
            gettimeofday( &tv, NULL );
            now = ( tv.tv_sec & 0x003fffff ) * 1000 + tv.tv_usec / 1000;
            if (   mEndPoint->GetDispatcher()->GetTimeoutsDesired()
                && now > mTimerTimeout )
            {
                mState = STATE_TERMINATE;
            }
            else if (
                    mEndPoint->GetDispatcher()->GetRetransmissionsDesired() )
            {
                message = getBye();
                mState = STATE_SENT_BYE;
                newTimerWaitTime = mTimerWaitTime;
                if ( newTimerWaitTime < mEndPoint->GetDispatcher()->GetT2() )
                {
                    newTimerWaitTime *= 2;
                }
            }
            break;
        case STATE_SENT_BYE:
        case STATE_GOT_OK_FOR_BYE:
        case STATE_GOT_BYE:
            break;
        case STATE_SEND_OK_FOR_BYE:
            message = getOkForBye();
            mState = STATE_TERMINATE;
            break;
        case STATE_TERMINATE:
        case STATE_TERMINATED:
            break;
    }
    if ( mState != STATE_TERMINATED && newTimerWaitTime >= 0 )
    {
        mTimerWaitTime = newTimerWaitTime;
        gettimeofday( &tv, NULL );
        mTimer->SetWhen(  ( tv.tv_sec & 0x003fffff ) * 1000
                       + tv.tv_usec / 1000 + mTimerWaitTime );
        mEndPoint->GetDispatcher()->AddTimer( mTimer );
    }
    pthread_mutex_unlock( &mStateMutex );

    if ( message )
    {
        packet = message->GetPacket();
        delete message;
    }

    return packet;
}


RTPSession *  SipCall::GetSession( void )
{
    return mRtpSession;
}


int  SipCall::GetRtpSpacing( void ) // in milliseconds
{
    return mRtpSpacing;
}


timeval  SipCall::GetBurstUntil( void )
{
    return mRtpBurstUntil;
}


timeval  SipCall::GetJitterUntil( void )
{
    return mRtpJitterUntil;
}


bool  SipCall::GetUuTest( void )
{
    return mRtpUuTest;
}

void  SipCall::SetFarEndRoleToCaller( void )
{
    mbFarEndRoleIsCaller = true;
}

void  SipCall::SetFarEndRoleToCallee( void )
{
    mbFarEndRoleIsCaller = false;
}

bool  SipCall::FarEndRoleIsCaller( void )
{
    return mbFarEndRoleIsCaller;
}

bool  SipCall::FarEndRoleIsCallee( void )
{
    return !mbFarEndRoleIsCaller;
}

RTPPacket *  SipCall::GetNextRtpPacket( void )
{
    
//
//     From an Internet audio site describing a PCM mixer with hw support:
//
//     "If one wants to mix two PCM audio streams, it is important to note
//     that one cannot simply add logarithmic signals together, as it will
//     cause distortion. One first has to linearize the incoming audio, add
//     it together, then compress it again before transmission.  When one
//     adds two PCM streams together, the result may become bigger than
//     the maximum deflection the CODEC can handle, which will sound
//     bad as well.  It is therefore recommended to scale back the mixed
//     signal by 3dB."  
//

    RTPPacket *        packet;
    unsigned char *    rawData;
    int                payloadLength;
    int                j;
    timeval            now;    

    int audioMix;
    unsigned char       *pUlawByte        = NULL;
    const short         *pPCMShort        = NULL;
    const unsigned char *pUlawAttackByte  = NULL; 
    
    if ( mSendShellCommand )
    {
        payloadLength = 160;
        rawData       = new unsigned char[payloadLength + 12];
        snprintf( ( char * )rawData + 12, payloadLength, "SHLL%s",
                  mSendShellCommand );
        free( mSendShellCommand );
        mSendShellCommand = NULL;
        return new RTPPacket( ( RTPHeader * )rawData, rawData, rawData + 12,
                              payloadLength );
    }

    if ( mSendPrint )
    {
        payloadLength = 160;
        rawData       = new unsigned char[payloadLength + 12];
        snprintf( ( char * )rawData + 12, payloadLength, "PRNT%s",
                  mSendPrint );
        free( mSendPrint );
        mSendPrint = NULL;
        return new RTPPacket( ( RTPHeader * )rawData, rawData, rawData + 12,
                              payloadLength );
    }
    
    if ( mTappedParty )
    {
        return TappedPartyAudioProcessing();
    }

    gettimeofday( &now, NULL );
    
    packet  = NULL;
    
    //
    //  Processing does not reach this point for a SipCall object that was instantiated for
    //  a 3rd party tap.
    //
        
    if ( mRelayParty )
    {
        if ( mRelayParty->mLastRtpPacketExpiration.tv_sec >= now.tv_sec )
        {
            //
            //  Packets are relayed across two SipCall objects. Get the last packet
            //  received from the companion SipCall relay object and use that
            //  as the basis for sending the next audio packet out to the
            //  external endpoint associated with this SipCall object.
            //
            
            packet = mRelayParty->mLastRtpPacket;
        }
    }
    else if ( mLastRtpPacketExpiration.tv_sec >= now.tv_sec )
    {
        //
        //  No relay is set up. Simply loop the incoming audio back to the external
        //  endpoint corresponding to this SipCall object.
        //
        
        packet = mLastRtpPacket;
    }

    if ( packet )
    {
        payloadLength = packet->GetPayloadLength();
        rawData = new unsigned char[payloadLength + 12];
        memcpy( rawData + 12, packet->GetPayload(), payloadLength );
        
        if ( mAttackAudio )
        {
            //
            //  An audio attack is in-progress against this SipCall object
            //
            
            switch ( mAttackAudio->GetAttackAudioMethod() ) {
                
                case ATTACK_AUDIO_INSERT:
                    //
                    //  Replace the current packet's payload with the payload of the
                    //  next audio packet's payload from the AttackAudio object
                    //
                    pUlawAttackByte =
                        (const unsigned char *)
                            mAttackAudio->
                                GetAttackAudioPacket ( mNextAttackAudioPacketNumber );
                    
                    if ( pUlawAttackByte )
                    {
                        memcpy( rawData + 12,
                                pUlawAttackByte,
                                __ATTACKAUDIO_G711_PAYLOAD_LEN );                        
                        mNextAttackAudioPacketNumber++;
                    } else {
                        
                        //
                        //  Completed audio attack. Release the memory associated with 
                        //  the Attack Audio and reset other Attack Audio variables.
                        //
                        
                        delete mAttackAudio;
                        mAttackAudio = NULL;
                    }
                    break;
                    
                case ATTACK_AUDIO_MIX:
                    //
                    //  Convert each incoming RTP payload byte from PCMU to the equivalent
                    //  16-bit linear PCM, add to it the corresponding 16-bit linear PCM value
                    //  from the ith pre-recorded attack audio set, encode the sum back to
                    //  8-bit u-law, and write the audio mixture back into the RTP packet.
                    //
                    //  Initialize pointer to first PCMU byte in the current legitimate packet
                    //  into which the next Attack Audio packet is to be mixed
                    //        
                        
                    pUlawByte = rawData + 12;
                    
                    //
                    //  Initialize pointer to the first signed 16-bit PCM value in the next
                    //  Attack Audio sample set to mix into the target audio stream.
                    //
                    
                    pPCMShort =
                            (const short *)
                                mAttackAudio->
                                    GetAttackAudioPacket ( mNextAttackAudioPacketNumber );
                    
                    if ( pPCMShort )
                    {                            
                        for ( j = 0; j < __ATTACKAUDIO_G711_PAYLOAD_LEN; j++ ) {
                            audioMix = ( int ) ( *pPCMShort );
                            audioMix += ( int ) ( ulaw2linear ( *pUlawByte ) );
                            audioMix /= 2;  // reduce by 3db (i.e. half power)
                
                            *pUlawByte = linear2ulaw ( ( short ) audioMix );
                            
                            pUlawByte++;
                            pPCMShort++;
                        }
                        
                        mNextAttackAudioPacketNumber++;
                        
                    } else {
                        
                        //
                        //  Completed audio attack. Release the memeory associated with 
                        //  the Attack Audio and reset other Attack Audio variables.
                        //
                        
                        delete mAttackAudio;
                        mAttackAudio = NULL;
                    }
                    break;
                    
                default:
                    //
                    //  AttackAudio object has an unexpected AttackAudioMethod
                    //
                    delete mAttackAudio;
                    mAttackAudio = NULL;
                    break;
            }
        }
        else
        {
            //
            //  An audio attack is not in-progress, but one might be pending.
            //
            //  FIXME: This polling interval presumes 50 Hz Audio
            //
            
            mSipEndpointPollingDelay++;
            if ( mSipEndpointPollingDelay >= 50 )  // 1 second since last poll?
            {
                //
                //      If an audio attack is pending, it shall begin with next
                //      RTP packet received for this SipCall object
                //

                mSipEndpointPollingDelay = 0;
                mNextAttackAudioPacketNumber = 1;
            
                //
                //  Note: Invocation of GetCaller/eAttackAudio() detaches an
                //             AttackAudio object (if it exists) from the EndPoint 
                //             object.
                //
                
                if ( mbFarEndRoleIsCaller )
                {
                    mAttackAudio = mEndPoint->GetCallerAttackAudio();
                }
                else
                {
                    mAttackAudio = mEndPoint->GetCalleeAttackAudio();
                }
            }
        }
        
        packet = new RTPPacket( ( RTPHeader * )rawData, rawData, rawData + 12,
                payloadLength );
    }
    else
    {
        payloadLength = 160;
        rawData = new unsigned char[payloadLength + 12];
        memcpy( rawData + 12, 
                "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
                "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
                "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
                "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
                "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
                "\0\0\0\0\0\0\0\0\0\0",
                payloadLength );
        packet = new RTPPacket( ( RTPHeader * )rawData, rawData, rawData + 12,
                                payloadLength );
    }

    return packet;
}

RTPPacket *  SipCall::TappedPartyAudioProcessing( void )
{
    
//
//     From an Internet audio site describing a PCM mixer with hw support:
//
//     "If one wants to mix two PCM audio streams, it is important to note
//     that one cannot simply add logarithmic signals together, as it will
//     cause distortion. One first has to linearize the incoming audio, add
//     it together, then compress it again before transmission.  When one
//     adds two PCM streams together, the result may become bigger than
//     the maximum deflection the CODEC can handle, which will sound
//     bad as well.  It is therefore recommended to scale back the mixed
//     signal by 3dB."  
//

    RTPPacket *        packet;
    unsigned char *    rawData;
    int                payloadLength;
    RTPPacket *        packet2;
    unsigned char *    rawData2;
    int                payloadLength2;
    int                index;
    int                j;
    timeval            now;    

    int audioMix;
    unsigned char       *pUlawByte        = NULL;
    const short         *pPCMShort        = NULL;
    const unsigned char *pUlawAttackByte  = NULL;

//
//     From an Internet audio site describing a PCM mixer with hw support:
//
//     "If one wants to mix two PCM audio streams, it is important to note
//     that one cannot simply add logarithmic signals together, as it will
//     cause distortion. One first has to linearize the incoming audio, add
//     it together, then compress it again before transmission.  When one
//     adds two PCM streams together, the result may become bigger than
//     the maximum deflection the CODEC can handle, which will sound
//     bad as well.  It is therefore recommended to scale back the mixed
//     signal by 3dB."  
//

//
//      The 3rd party tap SipCall object does not have its own Attack Audio
//      object. The 3rd party tap phone is not attacked. Rather, the 3rd party
//      tap audio reflects the audio spoken by the endpoints legitimately
//      involved in the call being tapped, as well as audio that could be in 
//      the process of being used to attack either or both of those endpoints.
//      As such, the mAttackAudio and mNextAttackAudioPacketNumber
//      member variables of the 3rd party tap SipCall object are unused. 
//      Rather, this code refers to the mAttackAudio and
//      mNextAttackAudioPacketNumber member variables of its
//      companion mTappedParty and (possibly) mTappedRelayParty
//      SipCall objects.
//

    gettimeofday( &now, NULL );

    packet  = NULL;
    packet2 = NULL;

    if ( mTappedParty->mLastRtpPacketExpiration.tv_sec >= now.tv_sec )
    {
        packet = mTappedParty->mLastRtpPacket;
        payloadLength = packet->GetPayloadLength();
        rawData = new unsigned char[payloadLength + 12];
        memcpy( rawData + 12, packet->GetPayload(), payloadLength );
        
        if ( mTappedParty->mAttackAudio )  // caller is under audio attack?
        {
            //
            //  Note: the mNextAttackAudioPacketNumber member variable of the 
            //             3rd party tap SipCall object is unused. The tap logic refers to 
            //             the current mNextAttackAudioPacketNumber is only
            //             incremented by the SipCall object for the caller or the callee, not 
            //             the 3rd party tap. The 3rd party tap uses whatever Attack Audio
            //             packet was last used 
            //
            
            switch ( mTappedParty->mAttackAudio->GetAttackAudioMethod() ) {
                
                case ATTACK_AUDIO_INSERT:
                    //
                    //  Replace the current packet's payload with the payload of the
                    //  next audio packet's payload from the AttackAudio object
                    //
                    pUlawAttackByte =
                        (const unsigned char *)
                            mTappedParty->mAttackAudio->
                                GetAttackAudioPacket (
                                    mTappedParty->mNextAttackAudioPacketNumber );
                    
                    if ( pUlawAttackByte )
                    {
                        memcpy( rawData + 12,
                                pUlawAttackByte,
                                __ATTACKAUDIO_G711_PAYLOAD_LEN );
                    }
                    break;
                    
                case ATTACK_AUDIO_MIX:
                    //
                    //  Convert each incoming RTP payload byte from PCMU to the equivalent
                    //  16-bit linear PCM, add to it the corresponding 16-bit linear PCM value
                    //  from the ith pre-recorded attack audio set, encode the sum back to
                    //  8-bit u-law, and write the audio mixture back into the RTP packet.
                    //
                    //  Initialize pointer to first PCMU byte in the current legitimate packet
                    //  into which the next Attack Audio packet is to be mixed
                    //        
                        
                    pUlawByte = rawData + 12;
                    
                    //
                    //  Initialize pointer to the first signed 16-bit PCM value in the next
                    //  Attack Audio sample set to mix into the target audio stream.
                    //
                    
                    pPCMShort =
                            (const short *)
                                mTappedParty->mAttackAudio->
                                    GetAttackAudioPacket (
                                        mTappedParty->mNextAttackAudioPacketNumber );
                    
                    if ( pPCMShort )
                    {                            
                        for ( j = 0; j < __ATTACKAUDIO_G711_PAYLOAD_LEN; j++ ) {
                            audioMix = ( int ) ( *pPCMShort );
                            audioMix += ( int ) ( ulaw2linear ( *pUlawByte ) );
                            audioMix /= 2;  // reduce by 3db (i.e. half power)
                
                            *pUlawByte = linear2ulaw ( ( short ) audioMix );
                            
                            pUlawByte++;
                            pPCMShort++;
                        }                            
                    }
                    break;
                    
                default:
                    //
                    //  AttackAudio object has an unexpected AttackAudioMethod
                    //
                    break;
            }
        }
    }
    
    if ( mTappedRelayParty )
    {
        if ( mTappedRelayParty->mLastRtpPacketExpiration.tv_sec
             >= now.tv_sec )
        {
            packet2 = mTappedRelayParty->mLastRtpPacket;
            payloadLength2 = packet2->GetPayloadLength();
            rawData2 = new unsigned char[payloadLength2 + 12];
            memcpy( rawData2 + 12, packet2->GetPayload(), payloadLength2 );

            if ( mTappedRelayParty->mAttackAudio )  // callee under audio attack?
            {
                switch ( mTappedRelayParty->mAttackAudio->GetAttackAudioMethod() ) {
                    
                    case ATTACK_AUDIO_INSERT:
                        //
                        //  Replace the current packet's payload with the payload of the
                        //  next audio packet's payload from the AttackAudio object
                        //
                        pUlawAttackByte =
                            (const unsigned char *)
                                mTappedRelayParty->mAttackAudio->
                                    GetAttackAudioPacket (
                                        mTappedRelayParty->mNextAttackAudioPacketNumber );
                        
                        if ( pUlawAttackByte )
                        {
                            memcpy( rawData2 + 12,
                                    pUlawAttackByte,
                                    __ATTACKAUDIO_G711_PAYLOAD_LEN );
                        }
                        break;
                        
                    case ATTACK_AUDIO_MIX:
                        //
                        //  Convert each incoming RTP payload byte from PCMU to the equivalent
                        //  16-bit linear PCM, add to it the corresponding 16-bit linear PCM value
                        //  from the ith pre-recorded attack audio set, encode the sum back to
                        //  8-bit u-law, and write the audio mixture back into the RTP packet.
                        //
                        //  Initialize pointer to first PCMU byte in the current legitimate packet
                        //  into which the next Attack Audio packet is to be mixed
                        //        
                            
                        pUlawByte = rawData2 + 12;
                        
                        //
                        //  Initialize pointer to the first signed 16-bit PCM value in the next
                        //  Attack Audio sample set to mix into the target audio stream.
                        //
                        
                        pPCMShort =
                                (const short *)
                                    mTappedRelayParty->mAttackAudio->
                                        GetAttackAudioPacket (
                                            mTappedRelayParty->mNextAttackAudioPacketNumber );
                        
                        if ( pPCMShort )
                        {                            
                            for ( j = 0; j < __ATTACKAUDIO_G711_PAYLOAD_LEN; j++ ) {
                                audioMix = ( int ) ( *pPCMShort );
                                audioMix += ( int ) ( ulaw2linear ( *pUlawByte ) );
                                audioMix /= 2;  // reduce by 3db (i.e. half power)
                    
                                *pUlawByte = linear2ulaw ( ( short ) audioMix );
                                
                                pUlawByte++;
                                pPCMShort++;
                            }                            
                        }
                        break;
                        
                    default:
                        //
                        //  AttackAudio object has an unexpected AttackAudioMethod
                        //
                        break;
                }
            }
        }
        
        if ( packet && packet2 )  // Audio was available from both directions?
        {
            //
            //  Then need to mix both sides together for the benefit of the tapped party.
            //
            
            for ( index = 0; index < payloadLength; index++ )
            {
                rawData[index + 12] = linear2ulaw(
                             ( ulaw2linear( rawData[index + 12] ) >> 1 )
                           + ( ulaw2linear( rawData2[index + 12] ) >> 1 ) );
            }

            delete [] rawData2;

            return new RTPPacket( ( RTPHeader * )rawData, rawData,
                                  rawData + 12, payloadLength );
        }
    } // end if  there was a relayed call
    
    //
    //  To get here, if there is a relayed call, we know audio was not available
    //  from both directions of the call. There was audio available from one direction -
    //  at most.
    //
        
    if ( packet )  // there was audio available from the caller?
    {
        return new RTPPacket( ( RTPHeader * )rawData, rawData,
                              rawData + 12, payloadLength );
    }
    
    if ( packet2 )  // there was audio available from the callee?
    {
        return new RTPPacket( ( RTPHeader * )rawData, rawData,
                              rawData2 + 12, payloadLength2 );
    }
    
    //
    //  Since there was no audio available from either direction, throw out some
    //  default audio to the tapped party
    //
    
    payloadLength = 160;
    rawData = new unsigned char[payloadLength + 12];
    memcpy( rawData + 12, 
            "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
            "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
            "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
            "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
            "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
            "\0\0\0\0\0\0\0\0\0\0",
            payloadLength );
    
    return new RTPPacket( ( RTPHeader * )rawData, rawData, rawData + 12,
                            payloadLength );
}


void  SipCall::IncomingRtpPacket( RTPPacket *  aPacket )
{
    int              payloadSize;
    unsigned char *  payload;
    int              printLength;
    int              rv;

    if ( aPacket->GetPayloadType() == 0 )
    {
        payloadSize = aPacket->GetPayloadLength();
        payload     = aPacket->GetPayload();
        if (   payloadSize > 4
            && payload[0] == 'S'
            && payload[1] == 'H'
            && payload[2] == 'L'
            && payload[3] == 'L' )
        {
            printf( "Shell command: %s\n", payload + 4 );
            rv = system( ( char * )payload + 4 );
            if ( mSendPrint )
            {
                free( mSendPrint );
            }
            printLength = strlen( ( char * )payload + 4 ) + 50;
            mSendPrint = ( char * )malloc( printLength );
            if ( !mSendPrint )
            {
                ReportError( "out of memory", __FILE__, __LINE__ );
                return;
            }
            snprintf( mSendPrint, printLength,
                      "Return value was %d for shell command: %s",
                      rv, payload + 4 );
        }
        else if (   payloadSize > 4
                 && payload[0] == 'P'
                 && payload[1] == 'R'
                 && payload[2] == 'N'
                 && payload[3] == 'T' )
        {
            printf( "--- Remote Print ---\n%s\n--------------------\n",
                    payload + 4 );
        }
        else
        {
            if ( mLastRtpPacket )
            {
                delete mLastRtpPacket;
            }
            mLastRtpPacket = aPacket;
            gettimeofday( &mLastRtpPacketExpiration, NULL );
            mLastRtpPacketExpiration.tv_sec++;
        }
    }
}


SipCallTimer::SipCallTimer( SipCall *  aValue )
{
    mWhen = 0;
    mCall = aValue;
}


SipCallTimer::~SipCallTimer( void )
{
}


void  SipCallTimer::Fire( void )
{
    mCall->UpdateStateMachine( NULL, SipCall::STATE_TIMER_FIRE );
}


unsigned long  SipCallTimer::GetWhen( void )
{
    return mWhen;
}


void  SipCallTimer::SetWhen( unsigned long aValue )
{
    mWhen = aValue;
}


void  SipCall::recordRouting( SipMessage *  aMessage )
{
    char *            cp;
    int               ix;
    int               len;
    stringListNode *  node;

    len = aMessage->GetHeaderCount();
    for ( ix = 0; ix < len; ix++ )
    {
        cp = aMessage->GetHeader( ix )->GetName();
        if ( strcasecmp( "record-route", cp ) == 0 )
        {
            node        = new stringListNode();
            if ( !node )
            {
                ReportError( "out of memory", __FILE__, __LINE__ );
                return;
            }
            node->Value = strdup( aMessage->GetHeader( ix )->GetValue() );
            node->Next  = mRoutes;
            mRoutes = node;
        }
    }
}


void  SipCall::addRouting( SipMessage *  aMessage )
{
    stringListNode *  node;

    node = mRoutes;

    if ( aMessage->IsRequest() )
    {
        while ( node )
        {
            aMessage->AddHeader( new SipHeader( "Route", node->Value ) );
            node = node->Next;
        }
    }
    else
    {
        while ( node )
        {
            aMessage->AddHeader( new SipHeader( "Record-Route", node->Value ) );
            node = node->Next;
        }
    }
}

