//-----------------------------------------------------------------------------
//
// SipDispatcher.h - Pulls messages from a SipUdpPort
//                   and delivers them to registered SipCalls,
//                   SipEndPoints, and special undispatched
//                   message handlers. As a last resort, the
//                   dispatcher will respond with an error
//                   message to completely undispatched
//                   messages. The SipDispatcher also provides
//                   a timer facility allowing objects such as
//                   SipCalls to update their state at a particular
//                   time.
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#ifndef SIP_DISPATCHER_H
#define SIP_DISPATCHER_H

#include <pthread.h>

class  SipCall;
class  SipEndPoint;
class  SipProxyEndPoint;
class  SipRegistrar;
class  SipMessage;
class  SipUdpPort;

class  SipDispatcher
{
    public:

        class  Timer
        {
            public:
                virtual ~Timer( void ) {};

                // When is in milliseconds. Now is formulated with:
                //   ( timeval.tv_sec & 0x003fffff ) * 1000
                // + timeval.tv_usec / 1000
                virtual unsigned long  GetWhen( void ) = 0;
                virtual void           Fire( void ) = 0;
        };

        class  UndispatchedHandler
        {
            public:
                virtual ~UndispatchedHandler( void ) {};

                virtual bool Incoming( SipMessage *  aMessage ) = 0;
        };

        SipDispatcher( void );
        ~SipDispatcher( void );

        void                Run( void );
        void                Stop( void );

        void                AddCall( SipCall *  aCall );
        void                RemoveCall( SipCall *  aCall );

        void                AddEndPoint( SipEndPoint *  aEndPoint );
        void                RemoveEndPoint( SipEndPoint *  aEndPoint );
        SipEndPoint *       GetEndPointByUser( char *  aUser );

        void                AddProxyEndPoint( SipProxyEndPoint *  aProxyEndPoint );
        void                RemoveProxyEndPoint( SipProxyEndPoint *  aProxyEndPoint );
        SipProxyEndPoint *  GetProxyEndPointByUser( char *  aUser );

        void                AddRegistrar( SipRegistrar *  aRegistrar );
        void                RemoveRegistrar( SipRegistrar *  aRegistrar );

        // The function referenced should return true of the message has been
        // handled and deleted; otherwise, false should be returned and the
        // message will be sent to other handlers.
        void                AddUndispatchedHandler( UndispatchedHandler *  aValue );
        void                RemoveUndispatchedHandler( UndispatchedHandler *
                                                                             aValue );

        void                AddTimer( Timer *  aTimer );
        void                RemoveTimer( Timer *  aTimer );

        SipUdpPort *        GetUdpPort( void );
        void                SetUdpPort( SipUdpPort *  aPort );

        int                 GetT1( void );
        void                SetT1( int  aMilliseconds );
        int                 GetT2( void );
        void                SetT2( int  aMilliseconds );

        bool                GetRetransmissionsDesired( void );
        void                SetRetransmissionsDesired( bool  aValue );
        bool                GetTimeoutsDesired( void );
        void                SetTimeoutsDesired( bool  aValue );
        bool                GetProvisionalsDesired( void );
        void                SetProvisionalsDesired( bool  aValue );


    private:

        class  callNode
        {
            public:
                callNode *  Next;
                SipCall *   Call;
        };

        class  endPointNode
        {
            public:
                endPointNode *  Next;
                SipEndPoint *   EndPoint;
        };

        class  proxyEndPointNode
        {
            public:
                proxyEndPointNode *  Next;
                SipProxyEndPoint *   ProxyEndPoint;
        };

        class  registrarNode
        {
            public:
                registrarNode *  Next;
                SipRegistrar *   Registrar;
        };

        class  undispatchedNode
        {
            public:
                undispatchedNode *     Next;
                UndispatchedHandler *  Handler;
        };

        class  timerNode
        {
            public:
                timerNode *  Next;
                Timer *      Value;
        };

        SipUdpPort *        mUdpPort;
        pthread_mutex_t     mCallListMutex;
        callNode *          mCallList;
        pthread_mutex_t     mEndPointListMutex;
        endPointNode *      mEndPointList;
        pthread_mutex_t     mProxyEndPointListMutex;
        proxyEndPointNode *      mProxyEndPointList;
        pthread_mutex_t     mRegistrarListMutex;
        registrarNode *     mRegistrarList;
        pthread_mutex_t     mUndispatchedListMutex;
        undispatchedNode *  mUndispatchedList;
        pthread_mutex_t     mTimerListMutex;
        timerNode *         mTimerList;
        unsigned int        mTimerListSize;
        char                mStopFlag;
        int                 mSipRfcT1; // milliseconds
        int                 mSipRfcT2; // milliseconds
        bool                mRetransmissionsDesired;
        bool                mTimeoutsDesired;
        bool                mProvisionalsDesired;
};

#endif

