//-----------------------------------------------------------------------------
//
// SipEndPoint.cpp - Handles a single end point, placing
//                   and receiving calls by creating new SipCall
//                   objects.
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#include "SipEndPoint.h"

#define malloc  mymalloc
#define free    myfree
#define strdup  mystrdup
#define strndup mystrndup

SipEndPoint::SipEndPoint( void )
{
    mUser                               = NULL;
    mTextName                           = NULL;
    mDispatcher                         = NULL;
    mRegistrarConnector                 = NULL;
    mRtpHandler                         = NULL;
    mControlPort                        = NULL;
    mControlPortConnectionId            = -1;
    mRingingMinimum                     = 2000;
    mRingingMaximum                     = 5000;
    mAcceptCalls                        = false;
    mRelayCallsTo                       = NULL;
    mTapCallsTo                         = NULL;
    mCallerAttackAudio                  = NULL;
    mCalleeAttackAudio                  = NULL;        

}

SipEndPoint::~SipEndPoint( void )
{
    if ( mUser )
    {
        myfree( mUser );
        mUser = NULL;
    }
    if ( mTextName )
    {
        myfree( mTextName );
        mTextName = NULL;
    }
    if ( mCallerAttackAudio )
    {
        delete mCallerAttackAudio;
        mCallerAttackAudio = NULL;
    }
    if ( mCalleeAttackAudio )
    {
        delete mCalleeAttackAudio;
        mCalleeAttackAudio = NULL;
    }    
}


char *  SipEndPoint::GetUser( void )
{
    return mUser;
}


void  SipEndPoint::SetUser( char *  aValue )
{
    if ( mUser )
    {
        myfree( mUser );
    }
    mUser = aValue;
}


char *  SipEndPoint::GetTextName( void )
{
    return mTextName;
}


void  SipEndPoint::SetTextName( char *  aValue )
{
    if ( mTextName )
    {
        myfree( mTextName );
    }
    mTextName = aValue;
}


SipDispatcher *  SipEndPoint::GetDispatcher( void )
{
    return mDispatcher;
}


void  SipEndPoint::SetDispatcher( SipDispatcher *  aValue )
{
    mDispatcher = aValue;
}


SipRegistrarConnector *  SipEndPoint::GetRegistrarConnector( void )
{
    return mRegistrarConnector;
}


void  SipEndPoint::SetRegistrarConnector( SipRegistrarConnector *  aValue )
{
    if ( mRegistrarConnector )
    {
        mRegistrarConnector->RemoveEndPoint( this );
    }
    mRegistrarConnector = aValue;
    if ( mRegistrarConnector )
    {
        mRegistrarConnector->AddEndPoint( this );
    }
}


RtpHandler *  SipEndPoint::GetRtpHandler( void )
{
    return mRtpHandler;
}


void  SipEndPoint::SetRtpHandler( RtpHandler *  aValue )
{
    mRtpHandler = aValue;
}


ControlPort *  SipEndPoint::GetControlPort( void )
{
    return mControlPort;
}


void  SipEndPoint::SetControlPort( ControlPort *  aValue )
{
    mControlPort = aValue;
}


int  SipEndPoint::GetControlPortConnectionId( void )
{
    return mControlPortConnectionId;
}


void  SipEndPoint::SetControlPortConnectionId( int aId )
{
    mControlPortConnectionId = aId;
}


unsigned int  SipEndPoint::GetRingingMinimum( void )
{
    return mRingingMinimum;
}


void  SipEndPoint::SetRingingMinimum( unsigned int  aMilliseconds )
{
    mRingingMinimum = aMilliseconds;
}


unsigned int  SipEndPoint::GetRingingMaximum( void )
{
    return mRingingMaximum;
}


void  SipEndPoint::SetRingingMaximum( unsigned int  aMilliseconds )
{
    mRingingMaximum = aMilliseconds;
}


void  SipEndPoint::SetAcceptCalls( bool aValue )
{
    mAcceptCalls = aValue;
}


bool  SipEndPoint::Incoming( struct SipMessage *  aMessage )
{
    SipCall *  call;
    SipCall *  relayCall;
    SipCall *  tapCall;

    if ( aMessage->IsRequest() )
    {
        if (   mAcceptCalls
            && strcmp( "INVITE", aMessage->GetRequestMethod() ) == 0 )
        {
            call = new SipCall();
            call->SetEndPoint( this );
            call->SetFarEndRoleToCaller();
            GetDispatcher()->AddCall( call );
            if ( GetControlPort() )
            {
                GetControlPort()->AddSipCall( GetControlPortConnectionId(),
                                              call );
            }
            relayCall = NULL;
            if ( mRelayCallsTo )
            {
                relayCall = new SipCall();
                relayCall->SetEndPoint( this );
                relayCall->SetRemoteIdentifier(
                                           new SipIdentifier( mRelayCallsTo ) );
                relayCall->SetRelayParty( call );
                relayCall->SetFarEndRoleToCallee();
                call->SetRelayParty( relayCall );
                GetDispatcher()->AddCall( relayCall );
                if ( GetControlPort() )
                {
                    GetControlPort()->AddSipCall( GetControlPortConnectionId(),
                                                  relayCall );
                }
            }
            if ( mTapCallsTo )
            {
                tapCall = new SipCall();
                tapCall->SetEndPoint( this );
                tapCall->SetRemoteIdentifier(
                                             new SipIdentifier( mTapCallsTo ) );
                tapCall->SetTappedParty( call );
                if ( relayCall )
                {
                    tapCall->SetTappedRelayParty( relayCall );
                }
                GetDispatcher()->AddCall( tapCall );
                if ( GetControlPort() )
                {
                    GetControlPort()->AddSipCall( GetControlPortConnectionId(),
                                                  tapCall );
                }
                tapCall->Connect();
            }
            if ( !call->Incoming( aMessage ) )
            {
                delete aMessage;
            }
            return true;
        }
    }
    return false;
}

void  SipEndPoint::RelayCallsTo( SipIdentifier *  aIdentifier )
{
    mRelayCallsTo = aIdentifier;
}


void  SipEndPoint::TapCallsTo( SipIdentifier *  aIdentifier )
{
    mTapCallsTo = aIdentifier;
}

AttackAudio * SipEndPoint::GetCalleeAttackAudio( void )
{
    AttackAudio *pAudioBuf = NULL;
    
    //  Note: There are possible synchronization problems that
    //             remain to be addressed, since the AttackAudio object
    //             is initially created and may be deleted by a different
    //             thread. A SipCall object might seek to delete the
    //             AttackAudio object at about the same time this
    //             method is being invoked.
    
    //
    //  If an AttackAudio object is attached, the AttackAudio object is detached
    //  from this object and a pointer to it is returned to the object invoking
    //  this method. i.e. it is the ultimate responsibility of the object invoking 
    //  this method to delete the AttackAudio object when it is no longer required.
    //

    if ( mCalleeAttackAudio )
    {
        pAudioBuf = mCalleeAttackAudio;
        mCalleeAttackAudio = NULL;
    }
    
    return pAudioBuf;
}

AttackAudio * SipEndPoint::GetCallerAttackAudio( void )
{
    AttackAudio *pAudioBuf = NULL;
    
    //  Note: There are possible synchronization problems that
    //             remain to be addressed, since the AttackAudio object
    //             is initially created and may be deleted by a different
    //             thread. A SipCall object might seek to delete the
    //             AttackAudio object at about the same time this
    //             method is being invoked.
    
    //
    //  If an AttackAudio object is attached, the AttackAudio object is detached
    //  from this object and a pointer to it is returned to the object invoking
    //  this method. i.e. it is the ultimate responsibility of the object invoking 
    //  this method to delete the AttackAudio object when it is no longer required.
    //

    if ( mCallerAttackAudio )
    {
        pAudioBuf = mCallerAttackAudio;
        mCallerAttackAudio = NULL;
    }
    
    return pAudioBuf;
}

bool  SipEndPoint::PrepAttackAudio( bool bToCaller,
                                    char * psInputAudioFile,
                                    AttackAudioMethod attackAudioMethod )
{

//
//  Start by deleting any existing Audio file that might already
//  be attached to the SipEndPoint object for the specified target
//  of the "issue" command (i.e. either the caller or the callee).
//  This deletion is performed regardless of whether the new
//  command's audio file complies  with the restrictions
//  imposed by this object.
//
//  Note: It's possible another audio attack is already underway
//             for this SipEndPoint against a caller and/or callee.
//             If that is the case, AttackAudio objects for those attacks
//             have already detached from this object and attached
//             to the appropriate SipCall object. In-progress
//             audio attacks are unaffected by the new command.
//
//             SipCall objects point back to the SipEndPoint
//             object for which the SipCall objects were created,
//             not vice versa. When the SipCall object completes
//             it's present audio attack, it'll check back to its
//             SipEndPoint object periodically to see if a new audio
//             attack is pending.
//
//  Note: There are possible synchronization problems that
//             remain to be addressed since the audio attack is
//             executed in a different thread. A SipCall object might
//             seek to detach an AttackAudio object through
//             methods supplied by this object at exactly the
//             same time the following code is being executed.
//
    
    if ( bToCaller ) {
        if ( mCallerAttackAudio ) {
            delete mCallerAttackAudio;
            mCallerAttackAudio = NULL;
        }
        
        //
        //  Create a new AttackAudio object and load it with the 
        //  pre-recorded audio contained within the audio file
        //  specified by the issue command.
        //
        
        mCallerAttackAudio = new AttackAudio( attackAudioMethod );
        if ( !mCallerAttackAudio->PrepAttackAudio( psInputAudioFile ) )
        {
            delete mCallerAttackAudio;
            mCallerAttackAudio = NULL;
            return false;
        }
    } else {
        if ( mCalleeAttackAudio ) {
            delete mCalleeAttackAudio;
            mCalleeAttackAudio = NULL;
        }
        
        //
        //  Create a new AttackAudio object and load it with the 
        //  pre-recorded audio contained within the audio file
        //  specified by the issue command.
        //

        mCalleeAttackAudio = new AttackAudio( attackAudioMethod );
        if ( !mCalleeAttackAudio->PrepAttackAudio( psInputAudioFile ) )
        {
            delete mCalleeAttackAudio;
            mCalleeAttackAudio = NULL;
            return false;
        }
    }

    return true;
    
}  //  end PrepAttackAudio
