//-----------------------------------------------------------------------------
//
// SipHeader.cpp - Class for a single SIP header line.
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "util.h"
#include "SipHeader.h"
#include "SipIdentifier.h"

#define malloc  mymalloc
#define free    myfree
#define strdup  mystrdup
#define strndup mystrndup

SipHeader::SipHeader( SipHeader *  aCopyFrom )
{
    identifierNode *  node;
    identifierNode *  newNode;
    identifierNode *  lastNewNode;

    mFullText        = strdup( aCopyFrom->mFullText );
    mName            = aCopyFrom->mName ? strdup( aCopyFrom->mName ) : NULL;
    mValue           = aCopyFrom->mValue ? strdup( aCopyFrom->mValue ) : NULL;
    mIdentifierCount = aCopyFrom->mIdentifierCount;
    mIdentifiers     = NULL;
    node = aCopyFrom->mIdentifiers;
    lastNewNode = NULL;
    while ( node )
    {
        newNode = new identifierNode();
        newNode->Value = new SipIdentifier( node->Value );
        if ( lastNewNode )
        {
            lastNewNode->Next = newNode;
        }
        else
        {
            mIdentifiers = newNode;
        }
        lastNewNode = newNode;
        node = node->Next;
    }
}


// Ensure aFullText is malloced, as it will be freed.
SipHeader::SipHeader( char *  aFullText )
{
    mFullText        = aFullText;
    mName            = NULL;
    mValue           = NULL;
    mIdentifierCount = -1;
    mIdentifiers     = NULL;
}


// aName and aValue don't need to be malloced, as they won't be freed.
SipHeader::SipHeader( char *  aName, char *  aValue )
{
    mFullText        = ( char * )malloc(  strlen( aName )
                                        + strlen( aValue ) + 3 );
    snprintf( mFullText, strlen( aName ) + strlen( aValue ) + 3, "%s: %s",
              aName, aValue );
    mName            = NULL;
    mValue           = NULL;
    mIdentifierCount = -1;
    mIdentifiers     = NULL;
}


SipHeader::~SipHeader( void )
{
    identifierNode *  node;

    if ( mFullText )
    {
        free( mFullText );
        mFullText = NULL;
    }
    if ( mName )
    {
        free( mName );
        mName = NULL;
    }
    if ( mValue )
    {
        mValue = NULL;
    }
    mIdentifierCount = -1;
    while ( mIdentifiers )
    {
        node = mIdentifiers;
        mIdentifiers = node->Next;
        node->Next = NULL;
        delete node->Value;
        delete node;
    }
}


char *  SipHeader::GetFullText( void )
{
    return mFullText;
}


char *  SipHeader::GetName( void )
{
    char *  endOfName;
    char *  startOfValue;

    if ( !mName )
    {
        endOfName = mFullText;
        while ( *endOfName && *endOfName != ':' )
        {
            endOfName++;
        }
        startOfValue = endOfName;
        endOfName--;
        while (   endOfName > mFullText
               && ( *endOfName == ' ' || *endOfName == '\t' ) )
        {
            endOfName--;
        }
        if ( *startOfValue ) {
            startOfValue++;
        }
        while (   *startOfValue
               && ( *startOfValue == ' ' || *startOfValue == '\t' ) )
        {
            startOfValue++;
        }
        mName  = strndup( mFullText, endOfName - mFullText + 1 );
        mValue = strdup( startOfValue );
    }
    return mName;
}


char *  SipHeader::GetValue( void )
{
    if ( !mValue )
    {
        GetName();
    }
    return mValue;
}


int  SipHeader::GetIdentifierCount( void )
{
    int               count;
    char *            name;
    char *            start;
    char *            end;
    char *            nextStart;
    char *            value;
    identifierNode *  firstNode;
    identifierNode *  lastNode;

    if ( mIdentifierCount < 0 )
    {
        count     = 0;
        firstNode = NULL;
        lastNode  = NULL;
        name      = GetName();
        if (   strcasecmp( name, "contact"      ) == 0
            || strcasecmp( name, "m"            ) == 0
            || strcasecmp( name, "from"         ) == 0
            || strcasecmp( name, "f"            ) == 0
            || strcasecmp( name, "to"           ) == 0
            || strcasecmp( name, "t"            ) == 0
            || strcasecmp( name, "reply-to"     ) == 0
            || strcasecmp( name, "route"        ) == 0
            || strcasecmp( name, "record-route" ) == 0 
            || strcasecmp( name, "via"          ) == 0
            || strcasecmp( name, "v"            ) == 0 )
        {
            start = GetValue();
            if (   strcasecmp( name, "via" ) == 0
                || strcasecmp( name, "v"   ) == 0 )
            {
                while ( *start && *start != ' ' )
                {
                    start++;
                }
                while ( *start && *start == ' ' )
                {
                    start++;
                }
            }
            while ( *start )
            {
                count++;
                end = start;
                while ( *end && *end != ',' )
                {
                    if ( *end == '"' )
                    {
                        end++;
                        while ( *end && *end != '"' )
                        {
                            end++;
                        }
                        if ( *end )
                        {
                            end++;
                        }
                    }
                    else if ( *end == '<' )
                    {
                        end++;
                        while ( *end && *end != '>' )
                        {
                            end++;
                        }
                        if ( *end )
                        {
                            end++;
                        }
                    }
                    else
                    {
                        end++;
                    }
                }
                nextStart = end;
                if ( *end )
                {
                    end--;
                    nextStart++;
                }
                while ( end > start && ( *end == ' ' || *end == '\t' ) )
                {
                    end--;
                }
                if (   strcasecmp( name, "via" ) == 0
                    || strcasecmp( name, "v"   ) == 0 )
                {
                    value = ( char * )malloc( end - start + 10 );
                    memcpy( value, "sip:via@", 8 );
                    memcpy( value + 8, start, end - start + 1 );
                    value[end - start + 9] = '\0';
                }
                else
                {
                    value = strndup( start, end - start + 1 );
                }
                if ( lastNode )
                {
                    lastNode->Next = new identifierNode();
                    lastNode = lastNode->Next;
                }
                else
                {
                    firstNode = lastNode = new identifierNode();
                }
                lastNode->Next  = NULL;
                lastNode->Value = new SipIdentifier( value );
                start = nextStart;
                while ( *start && ( *start == ' ' || *start == '\t' ) )
                {
                    start++;
                }
            }
        }
        mIdentifierCount = count;
        mIdentifiers     = firstNode;
    }
    return mIdentifierCount;
}


SipIdentifier *  SipHeader::GetIdentifier( int  aIndex )
{
    identifierNode *  node;

    if ( mIdentifierCount < 0 )
    {
        GetIdentifierCount();
    }

    node = mIdentifiers;
    while ( node && aIndex > 0 )
    {
        node = node->Next;
        aIndex--;
    }

    return node ? node->Value : NULL;
}

