//-----------------------------------------------------------------------------
//
// SipMessage.h - Encapsulates SIP requests and responses.
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#ifndef SIP_MESSAGE_H
#define SIP_MESSAGE_H

#include <netinet/in.h>

#include "SipHeader.h"
#include "SipUri.h"

class  SipMessage
{
    public:

        SipMessage( bool aIsRequest );
        SipMessage( in_addr_t  aSourceAddress,
                    in_port_t  aSourcePort,
                    in_addr_t  aDestinationAddress,
                    in_port_t  aDestinationPort,
                    char *     aPayload );
        SipMessage( in_addr_t  aSourceAddress,
                    in_port_t  aSourcePort,
                    in_addr_t  aDestinationAddress,
                    in_port_t  aDestinationPort,
                    char *     aPayload,
                    int        aPayloadSize );
        ~SipMessage( void );

        char *       GetError( void );
        char *       GetErrorInfo( void );

        in_addr_t    GetSourceAddress( void );
        void         SetSourceAddress( in_addr_t  aAddr );

        in_port_t    GetSourcePort( void );
        void         SetSourcePort( in_port_t  aPort );

        in_addr_t    GetDestinationAddress( void );
        void         SetDestinationAddress( in_addr_t  aAddr );

        in_port_t    GetDestinationPort( void );
        void         SetDestinationPort( in_port_t  aPort );

        bool         IsRequest( void );
        bool         IsResponse( void );

        int          GetMajorVersion( void );
        int          GetMinorVersion( void );

        char *       GetRequestMethod( void );

        // Ensure aMethod is malloced, as it will be freed.
        void         SetRequestMethod( char *  aMethod );

        SipUri *     GetRequestUri( void );

        // Note that aUri will be deleted.
        void         SetRequestUri( SipUri *  aUri );

        int          GetResponseCode( void );

        void         SetResponseCode( int  aCode );

        char *       GetResponseText( void );

        // Ensure aText is malloced, as it will be freed.
        void         SetResponseText( char *  aText );


        // Note that aHeader will eventually be deleted.
        void         AddHeader( SipHeader *  aHeader );

        // Note that aHeader will eventually be deleted.
        void         InsertHeader( SipHeader *  aHeader );

        // Removes only the first header with the given name. Returns true if a
        // header was removed.
        bool         RemoveHeader( char *  aName );

        int          GetHeaderCount( void );
        SipHeader *  GetHeader( int  aIndex );
        SipHeader *  GetHeader( char *  aName );

        int          GetContentSize( void );
        char *       GetContent( void );
        // Ensure content is malloced, as it will be freed.
        void         SetContent( char *  aContent );

        // Caller must free return value.
        char *       GetPacket( void );


    private:

        class  headerNode
        {
            public:
                headerNode *  Next;
                SipHeader *   Value;
        };

        void    constructorHelper( in_addr_t  aSourceAddress,
                                   in_port_t  aSourcePort,
                                   in_addr_t  aDestinationAddress,
                                   in_port_t  aDestinationPort,
                                   bool       aRequestFlag );

        // aPayloadSize may be -1 if aPayload is NULL terminated.
        void    constructorHelper( in_addr_t  aSourceAddress,
                                   in_port_t  aSourcePort,
                                   in_addr_t  aDestinationAddress,
                                   in_port_t  aDestinationPort,
                                   char *     aPayload,
                                   int        aPayloadSize );

        // Parses headers in aPayload, adds them to the list of headers for
        // this SipMessage object, and returns a pointer within aPayload just
        // beyond the headers.
        char *  addHeaders( char *  aPayload );

        char *        mError;              // Do not free
        char *        mErrorInfo;          // Do not free
        in_addr_t     mSourceAddress;
        in_addr_t     mDestinationAddress;
        in_port_t     mSourcePort;
        in_port_t     mDestinationPort;
        bool          mRequestFlag;
        int           mMajorVersion;
        int           mMinorVersion;
        headerNode *  mHeaderListHead;
        headerNode *  mHeaderListTail;
        char *        mContent;
        int           mContentSize;
        char *        mRequestMethod;
        int           mResponseCode;
        SipUri *      mRequestUri;
        char *        mResponseText;

};

#endif

