//-----------------------------------------------------------------------------
//
// SipRegistrarConnector.cpp - 
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>

#include "SipMessage.h"
#include "SipUdpPort.h"
#include "SipDispatcher.h"
#include "SipEndPoint.h"
#include "SipProxyEndPoint.h"
#include "SipRegistrarConnector.h"
#include "util.h"

#define malloc  mymalloc
#define free    myfree
#define strdup  mystrdup
#define strndup mystrndup

SipRegistrarConnector::SipRegistrarConnector( void )
{
    mDispatcher    = NULL;
    mLocalAddress  = 0;
    mRemoteAddress = 0;
    mRemotePort    = 0;
    mDomain        = NULL;
    mTimerWhen     = 0;
    pthread_mutex_init( &mEndPointsMutex, NULL );
    mEndPoints     = NULL;
    pthread_mutex_init( &mProxyEndPointsMutex, NULL );
    mProxyEndPoints     = NULL;
    mCallId        = GetNextGuid();
    if ( !mCallId )
    {
        ReportError( "can't get guid", __FILE__, __LINE__ );
        mCallId = NULL;
    }
    mCseq          = random() & 0x00ffffff;
}


SipRegistrarConnector::~SipRegistrarConnector( void )
{
    endPointNode *  node;
    proxyEndPointNode *  proxyNode;

    if ( mDomain )
    {
        free( mDomain );
    }
    pthread_mutex_lock( &mEndPointsMutex );
    while ( mEndPoints )
    {
        node       = mEndPoints;
        mEndPoints = node->Next;
        delete node;
    }
    pthread_mutex_unlock( &mEndPointsMutex );
    pthread_mutex_destroy( &mEndPointsMutex );
    pthread_mutex_lock( &mProxyEndPointsMutex );
    while ( mProxyEndPoints )
    {
        proxyNode       = mProxyEndPoints;
        mProxyEndPoints = proxyNode->Next;
        delete proxyNode;
    }
    pthread_mutex_unlock( &mProxyEndPointsMutex );
    pthread_mutex_destroy( &mProxyEndPointsMutex );
    if ( mCallId )
    {
        free( mCallId );
    }
}



SipDispatcher *  SipRegistrarConnector::GetDispatcher( void )
{
    return mDispatcher;
}


void  SipRegistrarConnector::SetDispatcher( SipDispatcher *  aValue )
{
    timeval  tv;

    if ( mDispatcher )
    {
        mDispatcher->RemoveUndispatchedHandler( this );
        mDispatcher->RemoveTimer( this );
    }
    mDispatcher = aValue;
    if ( mDispatcher )
    {
        mDispatcher->AddUndispatchedHandler( this );
        gettimeofday( &tv, NULL );
        // Fire in one second
        mTimerWhen =  ( tv.tv_sec & 0x003fffff ) * 1000 + tv.tv_usec / 1000
                    + 100;
        mDispatcher->AddTimer( this );
    }
}


in_addr_t  SipRegistrarConnector::GetAddress( void )
{
    return mRemoteAddress;
}


void  SipRegistrarConnector::SetAddress( in_addr_t  aValue )
{
    mRemoteAddress = aValue;
    mLocalAddress  = DetermineLocalAddressFor( mRemoteAddress );
}


in_port_t  SipRegistrarConnector::GetPort( void )
{
    return mRemotePort;
}


void  SipRegistrarConnector::SetPort( in_port_t  aValue )
{
    mRemotePort = aValue;
}


// Do not free return value.
char *  SipRegistrarConnector::GetDomain( void )
{
    return mDomain;
}


// Ensure aValue is malloced, as it will be freed.
void  SipRegistrarConnector::SetDomain( char *  aValue )
{
    if ( mDomain )
    {
        free( mDomain );
    }
    mDomain = aValue;
}


void  SipRegistrarConnector::AddEndPoint( SipEndPoint *  aValue )
{
    endPointNode *  node;
    timeval         tv;

    pthread_mutex_lock( &mEndPointsMutex );
    node           = new endPointNode();
    node->EndPoint = aValue;
    node->Next     = mEndPoints;
    mEndPoints     = node;
    gettimeofday( &tv, NULL );
    // Fire in one second
    mTimerWhen = ( tv.tv_sec & 0x003fffff ) * 1000 + tv.tv_usec / 1000 + 100;
    mDispatcher->AddTimer( this );
    pthread_mutex_unlock( &mEndPointsMutex );
}


void  SipRegistrarConnector::RemoveEndPoint( SipEndPoint *  aValue )
{
    endPointNode *  node;
    endPointNode *  nodePrevious;

    pthread_mutex_lock( &mEndPointsMutex );
    nodePrevious = NULL;
    node         = mEndPoints;
    while ( node )
    {
        if ( node->EndPoint == aValue )
        {
            if ( nodePrevious )
            {
                nodePrevious->Next = node->Next;
            }
            else
            {
                mEndPoints = node->Next;
            }
            delete node;
            break;
        }
        nodePrevious = node;
        node         = node->Next;
    }
    pthread_mutex_unlock( &mEndPointsMutex );
}


void  SipRegistrarConnector::AddProxyEndPoint( SipProxyEndPoint *  aValue )
{
    proxyEndPointNode *  node;
    timeval         tv;

    pthread_mutex_lock( &mProxyEndPointsMutex );
    node           = new proxyEndPointNode();
    node->ProxyEndPoint = aValue;
    node->Next     = mProxyEndPoints;
    mProxyEndPoints     = node;
    gettimeofday( &tv, NULL );
    // Fire in one second
    mTimerWhen = ( tv.tv_sec & 0x003fffff ) * 1000 + tv.tv_usec / 1000 + 100;
    mDispatcher->AddTimer( this );
    pthread_mutex_unlock( &mProxyEndPointsMutex );
}


void  SipRegistrarConnector::RemoveProxyEndPoint( SipProxyEndPoint *  aValue )
{
    proxyEndPointNode *  node;
    proxyEndPointNode *  nodePrevious;

    pthread_mutex_lock( &mProxyEndPointsMutex );
    nodePrevious = NULL;
    node         = mProxyEndPoints;
    while ( node )
    {
        if ( node->ProxyEndPoint == aValue )
        {
            if ( nodePrevious )
            {
                nodePrevious->Next = node->Next;
            }
            else
            {
                mProxyEndPoints = node->Next;
            }
            delete node;
            break;
        }
        nodePrevious = node;
        node         = node->Next;
    }
    pthread_mutex_unlock( &mProxyEndPointsMutex );
}


// When is in milliseconds. Now is formulated with:
//   ( timeval.tv_sec & 0x003fffff ) * 1000
// + timeval.tv_usec / 1000
unsigned long  SipRegistrarConnector::GetWhen( void )
{
    return mTimerWhen;
}


void  SipRegistrarConnector::Fire( void )
{
    timeval         tv;
    bool            actionOccurred;
    endPointNode *  node;
    proxyEndPointNode *  proxyNode;

    actionOccurred = false;

    pthread_mutex_lock( &mEndPointsMutex );
    gettimeofday( &tv, NULL );
    node = mEndPoints;
    while ( node && !actionOccurred )
    {
        if (   node->State == STATE_CREATED
            || (   node->State == STATE_REGISTERED
                && node->RegistrationExpires.tv_sec <= tv.tv_sec ) )
        {
            sendRegister( node );
            actionOccurred = true;
        }
        node = node->Next;
    }
    pthread_mutex_unlock( &mEndPointsMutex );

    pthread_mutex_lock( &mProxyEndPointsMutex );
    gettimeofday( &tv, NULL );
    proxyNode = mProxyEndPoints;
    while ( proxyNode && !actionOccurred )
    {
        if (   proxyNode->State == STATE_CREATED
            || (   proxyNode->State == STATE_REGISTERED
                && proxyNode->RegistrationExpires.tv_sec <= tv.tv_sec ) )
        {
            sendRegister( proxyNode );
            actionOccurred = true;
        }
        proxyNode = proxyNode->Next;
    }
    pthread_mutex_unlock( &mProxyEndPointsMutex );

    gettimeofday( &tv, NULL );
    if ( actionOccurred )
    {
        // Fire again in one second
        mTimerWhen =  ( tv.tv_sec & 0x003fffff ) * 1000 + tv.tv_usec / 1000
                    + 100;
    }
    else
    {
        // Fire again in one minute
        mTimerWhen =  ( tv.tv_sec & 0x003fffff ) * 1000 + tv.tv_usec / 1000
                    + 60000;
    }
    mDispatcher->AddTimer( this );
}


// Must already hold lock of mEndPointsMutex.
bool  SipRegistrarConnector::sendRegister(
                                   SipRegistrarConnector::endPointNode *  node )
{
    char *     message;
    char *     tmpGuid;
    in_addr_t  localAddress;

    localAddress = GetDispatcher()->GetUdpPort()->GetBoundAddress();
    if ( localAddress == INADDR_ANY )
    {
        localAddress = mLocalAddress;
    }

    if ( node->Branch )
    {
        free( node->Branch );
    }
    node->Branch = ( char * )malloc( 45 );
    if ( !node->Branch )
    {
        ReportError( "out of memory", __FILE__, __LINE__ );
        return false;
    }
    tmpGuid = GetNextGuid();
    if ( !tmpGuid )
    {
        ReportError( "can't get guid", __FILE__, __LINE__ );
        return false;
    }
    snprintf( node->Branch, 45, "z9hG4bK-%s", tmpGuid );
    free( tmpGuid );

    if ( node->Tag )
    {
        free( node->Tag );
    }
    node->Tag = GetNextGuid();
    if ( !node->Tag )
    {
        ReportError( "can't get guid", __FILE__, __LINE__ );
        return false;
    }

    node->CSeq = mCseq++;

    gettimeofday( &node->RegistrationExpires, NULL );
    node->RegistrationExpires.tv_sec += 3000;

    message = ( char * )malloc( 2048 );
    if ( !message )
    {
        ReportError( "out of memory", __FILE__, __LINE__ );
        return false;
    }

    snprintf( message,
              2048,
              "REGISTER sip:%s SIP/2.0\n"
              "Via: SIP/2.0/UDP %d.%d.%d.%d:%d;branch=%s\n"
              "Max-Forwards: 70\n"
              "To: <sip:%s@%s>\n"
              "From: <sip:%s@%s>;tag=%s\n"
              "Call-Id: %s\n"
              "CSeq: %d REGISTER\n"
              "Contact: <sip:%s@%d.%d.%d.%d:%d>\n"
              "Expires: 3600\n"
              "Content-Length: 0\n"
              "\n",
              GetDomain(),
              localAddress       & 0xff,
              localAddress >>  8 & 0xff,
              localAddress >> 16 & 0xff,
              localAddress >> 24 & 0xff,
              ntohs( mDispatcher->GetUdpPort()->GetBoundPort() ),
              node->Branch,
              node->EndPoint->GetUser(),
              GetDomain(),
              node->EndPoint->GetUser(),
              GetDomain(),
              node->Tag,
              mCallId,
              node->CSeq,
              node->EndPoint->GetUser(),
              localAddress       & 0xff,
              localAddress >>  8 & 0xff,
              localAddress >> 16 & 0xff,
              localAddress >> 24 & 0xff,
              ntohs( mDispatcher->GetUdpPort()->GetBoundPort() ) );

    mDispatcher->GetUdpPort()->AddOutgoing(
                       new SipMessage(
                                      localAddress,
                                      mDispatcher->GetUdpPort()->GetBoundPort(),
                                      mRemoteAddress,
                                      mRemotePort,
                                      message ) );

    node->State = STATE_REGISTER_SENT;

    return true;
}


// Must already hold lock of mProxyEndPointsMutex.
bool  SipRegistrarConnector::sendRegister(
                                   SipRegistrarConnector::proxyEndPointNode *  node )
{
    char *     message;
    char *     tmpGuid;
    in_addr_t  localAddress;

    localAddress = GetDispatcher()->GetUdpPort()->GetBoundAddress();
    if ( localAddress == INADDR_ANY )
    {
        localAddress = mLocalAddress;
    }

    if ( node->Branch )
    {
        free( node->Branch );
    }
    node->Branch = ( char * )malloc( 45 );
    if ( !node->Branch )
    {
        ReportError( "out of memory", __FILE__, __LINE__ );
        return false;
    }
    tmpGuid = GetNextGuid();
    if ( !tmpGuid )
    {
        ReportError( "can't get guid", __FILE__, __LINE__ );
        return false;
    }
    snprintf( node->Branch, 45, "z9hG4bK-%s", tmpGuid );
    free( tmpGuid );

    if ( node->Tag )
    {
        free( node->Tag );
    }
    node->Tag = GetNextGuid();
    if ( !node->Tag )
    {
        ReportError( "can't get guid", __FILE__, __LINE__ );
        return false;
    }

    node->CSeq = mCseq++;

    gettimeofday( &node->RegistrationExpires, NULL );
    node->RegistrationExpires.tv_sec += 3000;

    message = ( char * )malloc( 2048 );
    if ( !message )
    {
        ReportError( "out of memory", __FILE__, __LINE__ );
        return false;
    }

    snprintf( message,
              2048,
              "REGISTER sip:%s SIP/2.0\n"
              "Via: SIP/2.0/UDP %d.%d.%d.%d:%d;branch=%s\n"
              "Max-Forwards: 70\n"
              "To: <sip:%s@%s>\n"
              "From: <sip:%s@%s>;tag=%s\n"
              "Call-Id: %s\n"
              "CSeq: %d REGISTER\n"
              "Contact: <sip:%s@%d.%d.%d.%d:%d>\n"
              "Expires: 3600\n"
              "Content-Length: 0\n"
              "\n",
              GetDomain(),
              localAddress       & 0xff,
              localAddress >>  8 & 0xff,
              localAddress >> 16 & 0xff,
              localAddress >> 24 & 0xff,
              ntohs( mDispatcher->GetUdpPort()->GetBoundPort() ),
              node->Branch,
              node->ProxyEndPoint->GetUser(),
              GetDomain(),
              node->ProxyEndPoint->GetUser(),
              GetDomain(),
              node->Tag,
              mCallId,
              node->CSeq,
              node->ProxyEndPoint->GetUser(),
              localAddress       & 0xff,
              localAddress >>  8 & 0xff,
              localAddress >> 16 & 0xff,
              localAddress >> 24 & 0xff,
              ntohs( mDispatcher->GetUdpPort()->GetBoundPort() ) );

    mDispatcher->GetUdpPort()->AddOutgoing(
                       new SipMessage(
                                      localAddress,
                                      mDispatcher->GetUdpPort()->GetBoundPort(),
                                      mRemoteAddress,
                                      mRemotePort,
                                      message ) );

    node->State = STATE_REGISTER_SENT;

    return true;
}


bool  SipRegistrarConnector::Incoming( SipMessage *  aMessage )
{
    char *          callId;
    char *          cp;
    endPointNode *  node;
    proxyEndPointNode *  proxyNode;
    int             cseq;

    if ( aMessage->IsResponse() )
    {
        cp = aMessage->GetHeader( "cseq" )->GetValue();
        if ( cp && strstr( cp, " REGISTER" ) )
        {
            cseq = strtol( cp , NULL, 10 );
            callId = aMessage->GetHeader( "call-id" )->GetValue();
            if ( !callId )
            {
                callId = aMessage->GetHeader( "i" )->GetValue();
            }
            if ( callId )
            {
                pthread_mutex_lock( &mEndPointsMutex );
                node = mEndPoints;
                while ( node )
                {
                    if (   node->CSeq == cseq
                        && strcmp( callId, mCallId ) == 0 )
                    {
                        if ( node->State == STATE_REGISTER_SENT )
                        {
                            node->State = STATE_REGISTERED;
                        }
                        pthread_mutex_unlock( &mEndPointsMutex );
                        return true;
                    }
                    node = node->Next;
                }
                pthread_mutex_unlock( &mEndPointsMutex );

                pthread_mutex_lock( &mProxyEndPointsMutex );
                proxyNode = mProxyEndPoints;
                while ( proxyNode )
                {
                    if (   proxyNode->CSeq == cseq
                        && strcmp( callId, mCallId ) == 0 )
                    {
                        if ( proxyNode->State == STATE_REGISTER_SENT )
                        {
                            proxyNode->State = STATE_REGISTERED;
                        }
                        pthread_mutex_unlock( &mProxyEndPointsMutex );
                        return true;
                    }
                    proxyNode = proxyNode->Next;
                }
                pthread_mutex_unlock( &mProxyEndPointsMutex );
            }
        }
    }

    return false;
}


SipRegistrarConnector::endPointNode::endPointNode( void )
{
    Next     = NULL;
    EndPoint = NULL;
    State    = STATE_CREATED;
    gettimeofday( &RegistrationExpires, NULL );
    Branch   = NULL;
    Tag      = NULL;
    CSeq     = 0;
}


SipRegistrarConnector::endPointNode::~endPointNode( void )
{
    if ( Branch )
    {
        free( Branch );
    }
    if ( Tag )
    {
        free( Tag );
    }
}


SipRegistrarConnector::proxyEndPointNode::proxyEndPointNode( void )
{
    Next     = NULL;
    ProxyEndPoint = NULL;
    State    = STATE_CREATED;
    gettimeofday( &RegistrationExpires, NULL );
    Branch   = NULL;
    Tag      = NULL;
    CSeq     = 0;
}


SipRegistrarConnector::proxyEndPointNode::~proxyEndPointNode( void )
{
    if ( Branch )
    {
        free( Branch );
    }
    if ( Tag )
    {
        free( Tag );
    }
}

