//-----------------------------------------------------------------------------
//
// SipUri.cpp - Functions for working with SIP URIs.
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "SipUri.h"
#include "util.h"

#define malloc  mymalloc
#define free    myfree
#define strdup  mystrdup
#define strndup mystrndup

SipUri::SipUri( SipUri *  aCopyFrom )
{
    // FIXME: This code can take advantage of the work that might already be
    //        done in the object being copied.
    mFullText          = strdup( aCopyFrom->mFullText );
    mScheme            = NULL;
    mSchemeValue       = NULL;
    mAuthority         = NULL;
    mUser              = NULL;
    mPassword          = NULL;
    mHost              = NULL;
    mPort              = -2;
    mPath              = NULL;
    mQuery             = NULL;
    mStartOfParameters = NULL;
    mParameterCount    = -1;
    mParameters        = NULL;
    mStartOfHeaders    = NULL;
    mHeaderCount       = -1;
    mHeaders           = NULL;
}


// Ensure aFullText is malloced, as it will be freed.
SipUri::SipUri( char *  aFullText )
{
    mFullText          = aFullText;
    mScheme            = NULL;
    mSchemeValue       = NULL;
    mAuthority         = NULL;
    mUser              = NULL;
    mPassword          = NULL;
    mHost              = NULL;
    mPort              = -2;
    mPath              = NULL;
    mQuery             = NULL;
    mStartOfParameters = NULL;
    mParameterCount    = -1;
    mParameters        = NULL;
    mStartOfHeaders    = NULL;
    mHeaderCount       = -1;
    mHeaders           = NULL;
}


SipUri::~SipUri( void )
{
    nameValueNode *  node;

    if ( mFullText )
    {
        free( mFullText );
        mFullText = NULL;
    }
    if ( mScheme )
    {
        free( mScheme );
        mScheme = NULL;
    }
    if ( mSchemeValue )
    {
        free( mSchemeValue );
        mSchemeValue = NULL;
    }
    if ( mAuthority )
    {
        free( mAuthority );
        mAuthority = NULL;
    }
    if ( mUser )
    {
        free( mUser );
        mUser = NULL;
    }
    if ( mPassword )
    {
        free( mPassword );
        mPassword = NULL;
    }
    if ( mHost )
    {
        free( mHost );
        mHost = NULL;
    }
    mPort = -2;
    if ( mPath )
    {
        free( mPath );
        mPath = NULL;
    }
    if ( mQuery )
    {
        free( mQuery );
        mQuery = NULL;
    }
    mStartOfParameters = NULL;
    mParameterCount    = -1;
    while ( mParameters )
    {
        node = mParameters;
        mParameters = node->Next;
        node->Next = NULL;
        free( node->Name );
        free( node->Value );
        delete node;
    }
    mStartOfHeaders = NULL;
    mHeaderCount    = -1;
    while ( mHeaders )
    {
        node = mHeaders;
        mHeaders = node->Next;
        node->Next = NULL;
        free( node->Name );
        free( node->Value );
        delete node;
    }
}


char *  SipUri::GetFullText( void )
{
    return mFullText;
}


char *  SipUri::GetScheme( void )
{
    char *  endOfScheme;

    if ( !mScheme )
    {
        endOfScheme = mFullText;
        while ( *endOfScheme && *endOfScheme != ':' )
        {
            endOfScheme++;
        }
        if ( !*endOfScheme )
        {
            mScheme      = strdup( "sip" );
            mSchemeValue = strdup( mFullText );
        }
        else
        {
            mScheme      = strndup( mFullText, endOfScheme - mFullText );
            mSchemeValue = strdup( endOfScheme + 1 );
        }
    }

    return mScheme;
}


char *  SipUri::GetSchemeValue( void )
{
    if ( !mSchemeValue )
    {
        GetScheme();
    }
    return mSchemeValue;
}


char *  SipUri::GetAuthority( void )
{
    char *  startOfAuthority;
    char *  endOfAuthority;

    if ( !mAuthority )
    {
        startOfAuthority = GetSchemeValue();
        if ( startOfAuthority[0] == '/' && startOfAuthority[1] == '/' )
        {
            startOfAuthority += 2;
            endOfAuthority = startOfAuthority;
            while ( *endOfAuthority && *endOfAuthority != '/' )
            {
                endOfAuthority++;
            }
            mAuthority = strndup( startOfAuthority,
                                  endOfAuthority - startOfAuthority );
        }
        else
        {
            mAuthority = strdup( "" );
        }
    }

    return mAuthority;
}


char *  SipUri::GetUser( void )
{
    char *  startOfUser;
    char *  endOfUser;
    char *  endOfPassword;

    if ( !mUser )
    {
        startOfUser = GetAuthority();
        if ( !*startOfUser )
        {
            startOfUser = GetSchemeValue();
        }
        if ( !*startOfUser || *startOfUser == '/' )
        {
            mUser     = strdup( "" );
            mPassword = strdup( "" );
        }
        else
        {
            endOfPassword = startOfUser;
            while ( *endOfPassword && *endOfPassword != '@' )
            {
                endOfPassword++;
            }
            if ( !*endOfPassword )
            {
                mUser     = strdup( "" );
                mPassword = strdup( "" );
            }
            else
            {
                endOfUser = startOfUser;
                while (   *endOfUser && *endOfUser != '/' && *endOfUser != '@'
                       && *endOfUser != ':' )
                {
                    endOfUser++;
                }
                mUser = strndup( startOfUser, endOfUser - startOfUser );
                if ( *endOfUser == ':' )
                {
                    mPassword = strndup( endOfUser + 1,
                                         endOfPassword - endOfUser - 1 );
                }
                else
                {
                    mPassword = strdup( "" );
                }
            }
        }
    }

    return mUser;
}


char *  SipUri::GetPassword( void )
{
    if ( !mPassword )
    {
        GetUser();
    }
    return mPassword;
}


char *  SipUri::GetHost( void )
{
    char *  startOfHost;
    char *  endOfHost;

    if ( !mHost )
    {
        startOfHost = GetAuthority();
        if ( !*startOfHost )
        {
            if (   strcmp( GetScheme(), "sip" ) == 0
                || strcmp( GetScheme(), "sips" ) == 0 )
            {
                startOfHost = GetSchemeValue();
            }
            else
            {
                mHost = strdup( "" );
                mPort = -1;
            }
        }
        if ( !mHost )
        {
            endOfHost = startOfHost;
            while ( *startOfHost && *startOfHost != '@' && *startOfHost != '/' )
            {
                startOfHost++;
            }
            if ( !*startOfHost || *startOfHost == '/' )
            {
                startOfHost = endOfHost;
            }
            else if ( *startOfHost == '@' )
            {
                startOfHost++;
            }
            else
            {
                mHost = strdup( "" );
                mPort = -1;
            }
            if ( !mHost )
            {
                endOfHost = startOfHost;
                while (   *endOfHost && *endOfHost != ':'
                       && *endOfHost != '/' && *endOfHost != ';' )
                {
                    endOfHost++;
                }
                mHost = strndup( startOfHost, endOfHost - startOfHost );
                if ( *endOfHost == ':' )
                {
                    mPort = strtol( endOfHost + 1, &mStartOfParameters, 10 );
                }
                else
                {
                    mStartOfParameters = endOfHost;
                    mPort = -1;
                }
                if (   strcmp( GetScheme(), "sip" ) != 0
                    && strcmp( GetScheme(), "sips" ) != 0 )
                {
                    mStartOfParameters = NULL;
                }
                else
                {
                    mStartOfHeaders = mStartOfParameters;
                    while ( *mStartOfHeaders && *mStartOfHeaders != '?' )
                    {
                        mStartOfHeaders++;
                    }
                }
            }
        }
    }

    return mHost;
}


// Returns -1 if not set.
int  SipUri::GetPort ( void )
{
    if ( mPort < -1 )
    {
        GetHost();
    }
    return mPort;
}


char *  SipUri::GetPath( void )
{
    char *  startOfPath;
    char *  endOfPath;

    if ( !mPath )
    {
        startOfPath = GetSchemeValue();
        if ( *startOfPath == '/' )
        {
            if ( startOfPath[1] == '/' )
            {
                startOfPath += 2;
                while ( *startOfPath && *startOfPath != '/' )
                {
                    startOfPath++;
                }
            }
            endOfPath = startOfPath;
            while ( *endOfPath && *endOfPath != '?' )
            {
                endOfPath++;
            }
            mPath  = strndup( startOfPath, endOfPath - startOfPath );
            if ( *endOfPath == '?' )
            {
                endOfPath++;
            }
            mQuery = strdup( endOfPath );
        }
        else
        {
            mPath  = strdup( "" );
            mQuery = strdup( "" );
        }
    }

    return mPath;
}


char *  SipUri::GetQuery( void )
{
    if ( !mQuery )
    {
        GetPath();
    }
    return mQuery;
}


int  SipUri::GetParameterCount( void )
{
    char *           startOfName;
    char *           endOfName;
    char *           endOfValue;
    nameValueNode *  node;
    nameValueNode *  lastNode;

    if ( mParameterCount < 0 )
    {
        if ( !mStartOfParameters )
        {
            GetHost();
        }

        mParameterCount = 0;
        lastNode        = NULL;
        startOfName     = mStartOfParameters;
        while ( startOfName && *startOfName == ';' )
        {
            startOfName++;
            if ( !*startOfName )
            {
                break;
            }
            endOfName = startOfName + 1;
            while ( *endOfName && *endOfName != '=' && *endOfName != ';' )
            {
                endOfName++;
            }
            node = new nameValueNode();
            node->Next = NULL;
            if ( *endOfName )
            {
                node->Name = strndup( startOfName, endOfName - startOfName );
            }
            else
            {
                node->Name = strdup( startOfName );
            }
            if ( *endOfName == '=' )
            {
                endOfValue = endOfName + 1;
                while ( *endOfValue && *endOfValue != ';' )
                {
                    endOfValue++;
                }
                if ( *endOfValue )
                {
                    node->Value = strndup( endOfName + 1,
                                           endOfValue - endOfName - 1 );
                }
                else
                {
                    node->Value = strdup( endOfName + 1 );
                }
                startOfName = endOfValue;
            }
            else if ( *endOfName == ';' )
            {
                node->Value = strdup( "" );
                startOfName = endOfName;
            }
            else
            {
                node->Value = strdup( "" );
                startOfName = endOfName;
            }
            if ( lastNode )
            {
                lastNode->Next = node;
                lastNode = node;
            }
            else
            {
                mParameters = lastNode = node;
            }
            mParameterCount++;
        }
    }

    return mParameterCount;
}


char *  SipUri::GetParameterName( int  aIndex )
{
    nameValueNode *  node;
    
    if ( mParameterCount < 0 )
    {
        GetParameterCount();
    }

    node = mParameters;
    while ( node && aIndex > 0 )
    {
        node = node->Next;
        aIndex--;
    }

    return node ? node->Name : ( char * )"";
}


char *  SipUri::GetParameterValue( int  aIndex )
{
    nameValueNode *  node;
    
    if ( mParameterCount < 0 )
    {
        GetParameterCount();
    }

    node = mParameters;
    while ( node && aIndex > 0 )
    {
        node = node->Next;
        aIndex--;
    }

    return node ? node->Value : ( char * )"";
}


char *  SipUri::GetParameterValue( const char *  aName )
{
    nameValueNode *  node;

    if ( mParameterCount < 0 )
    {
        GetParameterCount();
    }

    node = mParameters;
    while ( node )
    {
        if ( strcasecmp( node->Name, aName ) == 0 )
        {
            return node->Value;
        }
        node = node->Next;
    }

    return "";
}


int  SipUri::GetHeaderCount( void )
{
    char *           startOfName;
    char *           endOfName;
    char *           endOfValue;
    nameValueNode *  node;
    nameValueNode *  lastNode;

    if ( mHeaderCount < 0 )
    {
        if ( !mStartOfHeaders )
        {
            GetHost();
        }

        mHeaderCount = 0;
        lastNode     = NULL;
        startOfName  = mStartOfHeaders;
        while (   startOfName
               && ( *startOfName == '&'
                   || (   startOfName == mStartOfHeaders
                       && *startOfName == '?' ) ) )
        {
            startOfName++;
            if ( !*startOfName )
            {
                break;
            }
            endOfName = startOfName + 1;
            while ( *endOfName && *endOfName != '=' && *endOfName != '&' )
            {
                endOfName++;
            }
            node = new nameValueNode();
            node->Next = NULL;
            if ( *endOfName )
            {
                node->Name = strndup( startOfName, endOfName - startOfName );
            }
            else
            {
                node->Name = strdup( startOfName );
            }
            if ( *endOfName == '=' )
            {
                endOfValue = endOfName + 1;
                while ( *endOfValue && *endOfValue != '&' )
                {
                    endOfValue++;
                }
                if ( *endOfValue )
                {
                    node->Value = strndup( endOfName + 1,
                                           endOfValue - endOfName - 1 );
                }
                else
                {
                    node->Value = strdup( endOfName + 1 );
                }
                startOfName = endOfValue;
            }
            else if ( *endOfName == '&' )
            {
                node->Value = strdup( "" );
                startOfName = endOfName;
            }
            else
            {
                node->Value = strdup( "" );
                startOfName = endOfName;
            }
            if ( lastNode )
            {
                lastNode->Next = node;
                lastNode = node;
            }
            else
            {
                mHeaders = lastNode = node;
            }
            mHeaderCount++;
        }
    }

    return mHeaderCount;
}


char *  SipUri::GetHeaderName( int  aIndex )
{
    nameValueNode *  node;
    
    if ( mHeaderCount < 0 )
    {
        GetHeaderCount();
    }

    node = mHeaders;
    while ( node && aIndex > 0 )
    {
        node = node->Next;
        aIndex--;
    }

    return node ? node->Name : ( char * )"";
}


char *  SipUri::GetHeaderValue( int  aIndex )
{
    nameValueNode *  node;
    
    if ( mHeaderCount < 0 )
    {
        GetHeaderCount();
    }

    node = mHeaders;
    while ( node && aIndex > 0 )
    {
        node = node->Next;
        aIndex--;
    }

    return node ? node->Value : ( char * )"";
}


char *  SipUri::GetHeaderValue( const char *  aName )
{
    nameValueNode *  node;

    if ( mHeaderCount < 0 )
    {
        GetHeaderCount();
    }

    node = mHeaders;
    while ( node )
    {
        if ( strcasecmp( node->Name, aName ) == 0 )
        {
            return node->Value;
        }
        node = node->Next;
    }

    return "";
}

