#include <sys/stat.h>
#include <sys/signal.h>
#include <sys/fault.h>
#include <sys/syscall.h>
#include <sys/procfs.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <fcntl.h>
#include <neds/ninfile.h>
#include "qcdr_perf.h"

DefNormalClass(cQCD_PERF_LOG)

cQCD_PERF_LOG *cQCD_PERF_LOG::m_instance = NULL;
cQCD_MUTEX cQCD_PERF_LOG::m_mutex("perf_log_mutex");

cQCD_PERF_LOG* cQCD_PERF_LOG::Instance()
{
	if (!m_instance) {
		cQCD_SLOCK lock(m_mutex);
		if (!m_instance)
			m_instance = new cQCD_PERF_LOG;
	}
	return m_instance;
}

cQCD_PERF_LOG::cQCD_PERF_LOG() :
	cQTHREAD("perf_log", QPOS_PRIO_2),
	m_log_file_fd(-1),
	m_log_file(NULL),
	m_proc_fd(-1),
	m_run(true)
{
	char proc_name[64];
	long ticks_per_sec;
	
	gettimeofday(&m_start_time, NULL);

	if (NINLocal()->GetBOOLSetting("QCDR_Perf", "PerfMonitoring", true)) {
		m_log_file_name = NINLocal()->GetSetting("QCDR_Perf", "PerfFileName", cSTR("qcdr_perf.txt"));
		m_max_log_size = NINLocal()->GetDWORDSetting("QCDR_Perf", "PerfMaxLogSize", 0, 2000000000, 0);
		
		m_log_file_fd = open(m_log_file_name, O_WRONLY | O_CREAT | O_TRUNC | O_APPEND | 0644);
		if (m_log_file_fd < 0) {
			cerr << "error fd" << endl;
			exit(1);
		}
		m_log_file = fdopen(m_log_file_fd, "a");
		if (m_log_file_fd < 0 || !m_log_file) {
			cerr << "cannot open file " << m_log_file_name << endl;
			Assert1(0);
			exit(1);
		}
		sprintf(proc_name, "/proc/%05d", getpid());
		if ((m_proc_fd = open(proc_name, O_RDONLY)) < 0) {
			cerr << "cannot open file " << proc_name << endl;
			Assert1(0);
			exit(1);
		}
	} else
		m_run = false;
}

cQCD_PERF_LOG::~cQCD_PERF_LOG()
{
	m_run = false;
	Join();
	if (m_proc_fd >= 0)
		close(m_proc_fd);
	if (m_log_file)
		fclose(m_log_file);
}

static void timeval_sub(struct timeval *tv, struct timeval *tv_sub)
{
	tv->tv_sec -= tv_sub->tv_sec;
	if (tv->tv_usec < tv_sub->tv_usec) {
		tv->tv_usec += 1000000;
		tv->tv_sec--;
	}
	tv->tv_usec -= tv_sub->tv_usec;
}

static void rusage_sub(struct rusage *ru, struct rusage *ru_sub)
{
	timeval_sub(&ru->ru_utime, &ru_sub->ru_utime);
	timeval_sub(&ru->ru_stime, &ru_sub->ru_stime);
	ru->ru_maxrss -= ru_sub->ru_maxrss;
	ru->ru_ixrss -= ru_sub->ru_ixrss;
	ru->ru_idrss -= ru_sub->ru_idrss;
	ru->ru_isrss -= ru_sub->ru_isrss;
	ru->ru_minflt -= ru_sub->ru_minflt;
	ru->ru_majflt -= ru_sub->ru_majflt;
	ru->ru_nswap -= ru_sub->ru_nswap;
	ru->ru_inblock -= ru_sub->ru_inblock;
	ru->ru_oublock -= ru_sub->ru_oublock;
	ru->ru_msgsnd -= ru_sub->ru_msgsnd;
	ru->ru_msgrcv -= ru_sub->ru_msgrcv;
	ru->ru_nsignals -= ru_sub->ru_nsignals;
	ru->ru_nvcsw -= ru_sub->ru_nvcsw;
	ru->ru_nivcsw -= ru_sub->ru_nivcsw;
}

static long ptok(long val)
{
	static long page_size_kb = 0;
	
	if (!page_size_kb)
		page_size_kb = getpagesize() / 1024;
	return val * page_size_kb;
}

void cQCD_PERF_LOG::Main()
{
	const int sleep_interval = NINLocal()->GetDWORDSetting("QCDR_Perf", "PerfCheckpointIntvl", 0, 1000000000, 5000);

	while (m_run) {
		CheckLogSize();
		WriteLogEntry();
		CurrentThread()->Sleep(sleep_interval);
	}
}

void cQCD_PERF_LOG::CheckLogSize()
{
	struct stat stat_buf;
	
	if (m_max_log_size && fstat(m_log_file_fd, &stat_buf) == 0) {
		if (stat_buf.st_size > m_max_log_size)
			ftruncate(m_log_file_fd, 0);
	}
}

void cQCD_PERF_LOG::WriteLogEntry()
{
	time_t cur_time;
	
	time(&cur_time);
	fprintf(m_log_file, "\n\n%s", ctime(&cur_time));
	ElapsedTimeEntry();
	WriteStatMonEntry();
	WritePerfValEntry();
	fprintf(m_log_file, "Resource Usage:\n");
	WriteProcEntry();
	WriteRUsageEntry();
	fflush(m_log_file);
}

void cQCD_PERF_LOG::ElapsedTimeEntry()
{
	struct timeval elapsed;
	
	gettimeofday(&elapsed, NULL);
	timeval_sub(&elapsed, &m_start_time);
	
	fprintf(m_log_file, "Elapsed Time %ld.%03ld\n", elapsed.tv_sec, elapsed.tv_usec / 1000);
}

void cQCD_PERF_LOG::WriteProcEntry()
{
	struct prstatus prstat;
	struct prpsinfo prinfo;
	long val;
	
	ioctl(m_proc_fd, PIOCSTATUS, &prstat);
	ioctl(m_proc_fd, PIOCPSINFO, &prinfo);

	/* sometimes happen that it return > 1000000000 in then tv_nsec field */
	prstat.pr_utime.tv_sec += prstat.pr_utime.tv_nsec / 1000000000;
	prstat.pr_utime.tv_nsec %= 1000000000;
	prstat.pr_stime.tv_sec += prstat.pr_stime.tv_nsec / 1000000000;
	prstat.pr_stime.tv_nsec %= 1000000000;
		
	fprintf(m_log_file, "\tuser time used                 %8ld.%03lds\n",
		     prstat.pr_utime.tv_sec, prstat.pr_utime.tv_nsec / 1000000);
	fprintf(m_log_file, "\tsystem time used               %8ld.%03lds\n",
		     prstat.pr_stime.tv_sec, prstat.pr_stime.tv_nsec / 1000000);
	fprintf(m_log_file, "\tnumber of threads              %8ld\n",
		     prstat.pr_nthreads);
	
	val = ptok(prinfo.pr_size);
	fprintf(m_log_file, "\tprocess image size	      %8ld.%03ldMB\n", val / 1024, val % 1024);
	val = ptok(prinfo.pr_rssize);
	fprintf(m_log_file, "\tprocess rss		      %8ld.%03ldMB\n", val / 1024, val % 1024);
}

void cQCD_PERF_LOG::WriteRUsageEntry()
{
	struct rusage usage;
	
	getrusage(RUSAGE_SELF, &usage);
	
#if 0
	p += sprintf(p, "\tmaximum resident set size      %8ldKB, change %8ldKB\n",
		     ptok(usage.ru_maxrss), ptok(rusage_change->ru_maxrss));

	p += sprintf(p, "\tmem usage data+stack+text      %8ldKB, change %8ldKB\n",
		     ticks == 0 ? 0 : ptok(usage.ru_ixrss) / ticks + ptok(usage.ru_idrss) / ticks + ptok(usage.ru_isrss) / ticks,
		     ptok(rusage_change->ru_ixrss) + ptok(rusage_change->ru_idrss) + ptok(rusage_change->ru_isrss));

	p += sprintf(p, "\t  integral shared memory size  %8ldKB, change %8ldKB\n",
		     ticks == 0 ? 0 : ptok(usage.ru_ixrss) / ticks, 
		     ptok(rusage_change->ru_ixrss));
	p += sprintf(p, "\t  integral unshared data size  %8ldKB, change %8ldKB\n",
		     ticks == 0 ? 0 : ptok(usage.ru_idrss) / ticks, 
		     ptok(rusage_change->ru_idrss));
	p += sprintf(p, "\t  integral unshared stack size %8ldKB, change %8ldKB\n",
		     ticks == 0 ? 0 : ptok(usage.ru_isrss) / ticks, 
		     ptok(rusage_change->ru_isrss));
#endif
	
	fprintf(m_log_file, "\tpage reclaims (without I/O)    %8ld\n", usage.ru_minflt);
	fprintf(m_log_file, "\tpage faults   (require I/O)    %8ld\n", usage.ru_majflt);
#if 0
	fprintf(m_log_file, "\tswaps                          %8ld\n", usage.ru_nswap);
#endif
	fprintf(m_log_file, "\tblock input operations         %8ld\n", usage.ru_inblock);
	fprintf(m_log_file, "\tblock output operations        %8ld\n", usage.ru_oublock);
	fprintf(m_log_file, "\tmessages sent                  %8ld\n", usage.ru_msgsnd);
	fprintf(m_log_file, "\tmessages received              %8ld\n", usage.ru_msgrcv);
#if 0
	fprintf(m_log_file, "\tsignals received               %8ld\n", usage.ru_nsignals);
#endif
	fprintf(m_log_file, "\tvoluntary context switches     %8ld\n", usage.ru_nvcsw);
	fprintf(m_log_file, "\tinvoluntary context switches   %8ld\n", usage.ru_nivcsw);
}


/*
 * 
 * PerfVal
 * 
 */
void cQCD_PERF_LOG::InsertPerfVal(cQCD_PERF_VAL *pv)
{
	cQCD_SLOCK lock(m_mutex);
	m_perf_val.push_back(pv);
}


void cQCD_PERF_LOG::RemovePerfVal(cQCD_PERF_VAL *pv)
{
	list<cQCD_PERF_VAL *>::iterator li;
	
	cQCD_SLOCK lock(m_mutex);
	if ((li = find(m_perf_val.begin(), m_perf_val.end(), pv)) != m_perf_val.end())
		m_perf_val.erase(li);
}

void cQCD_PERF_LOG::WritePerfValEntry()
{
	list<cQCD_PERF_VAL *>::iterator li;
	long idle_milsec;

	cQCD_SLOCK lock(m_mutex);
	for (li = m_perf_val.begin(); li != m_perf_val.end(); li++) {
		idle_milsec = (*li)->GetIdleTime();
		fprintf(m_log_file, "\t%-20s idle time = %ld.%03ld\n", (const char *) (*li)->GetName(), idle_milsec / 1000, idle_milsec % 1000);
	}
}

/*
 * Stat Mon related functions
 */
void cQCD_PERF_LOG::Register(cQCD_STAT_MON *sm)
{
	cQCD_SLOCK lock(m_mutex);
	m_stat_list.push_back(sm);
}

void cQCD_PERF_LOG::Unregister(cQCD_STAT_MON *sm)
{
	list<cQCD_STAT_MON *>::iterator li;
	
	cQCD_SLOCK lock(m_mutex);
	for (li = m_stat_list.begin(); li != m_stat_list.end(); li++) {
		if (*li == sm) {
			m_stat_list.erase(li);
			return;
		}
	}
}

void cQCD_PERF_LOG::WriteStatMonEntry()
{
	list<cQCD_STAT_MON *>::iterator li;
	char label[1024];
	
	cQCD_SLOCK lock(m_mutex);
	for (li = m_stat_list.begin(); li != m_stat_list.end(); li++) {
		sprintf(label, "\t%-20s ", (const char *) (*li)->GetName());
		fprintf(m_log_file, "%s", (const char *) (*li)->StatMon(label));
	}
}

/*
 * 
 * cQCD_PERF_VAL
 * 
 */
cQCD_PERF_VAL::cQCD_PERF_VAL(const cSTR& name):
	m_objname(name),
	m_idle_milsec(0),
	m_state(ST_NEW),
	m_mutex("perf_val")
{
	memset(&m_start_time, 0, sizeof(m_start_time));
	memset(&m_st_time, 0, sizeof(m_st_time));
}

void cQCD_PERF_VAL::Start()
{
	cQCD_SLOCK lock(m_mutex);
	Assert1(m_state == ST_NEW);
	gettimeofday(&m_start_time, NULL);
	m_state = ST_OPEN;
}

void cQCD_PERF_VAL::Stop()
{
	cQCD_SLOCK lock(m_mutex);
	Assert1(m_state == ST_OPEN);
	gettimeofday(&m_st_time, NULL);
	m_state = ST_CLOSED;
}

void cQCD_PERF_VAL::StartIdleTime()
{
	cQCD_SLOCK lock(m_mutex);
	Assert1(m_state == ST_OPEN);
	gettimeofday(&m_st_time, NULL);
	m_state = ST_IDLE;
}

void cQCD_PERF_VAL::StopIdleTime()
{
	struct timeval cur_time;
	
	cQCD_SLOCK lock(m_mutex);
	Assert1(m_state == ST_IDLE);
	m_state = ST_OPEN;
	gettimeofday(&cur_time, NULL);
	timeval_sub(&cur_time, &m_st_time);
	m_idle_milsec += cur_time.tv_sec * 1000 + cur_time.tv_usec / 1000;
}

long cQCD_PERF_VAL::GetIdleTime()
{
	struct timeval cur_time;
	long retval;
	
	cQCD_SLOCK lock(m_mutex);
	retval = m_idle_milsec;
	if (m_state == ST_IDLE) {
		gettimeofday(&cur_time, NULL);
		timeval_sub(&cur_time, &m_st_time);
		retval += cur_time.tv_sec * 1000 + cur_time.tv_usec / 1000;
	}
	return retval;
}

