#include <linux/version.h>
#include <linux/config.h>
#include <linux/string.h>
#include <linux/module.h>
#include <linux/miscdevice.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/kbd_ll.h>
#ifdef CONFIG_PROC_FS
#include <linux/proc_fs.h>
#endif
#include <linux/smp_lock.h>
#include <asm/segment.h>
#include <linux/cuecat_driver.h>
#include <linux/cuecat_lib.h>
#include <linux/cuecat_at2xt.h>

#if !defined CONFIG_CUECAT_PCKBD && \
    !defined CONFIG_CUECAT_PSMOUSE && \
    !defined CONFIG_CUECAT_USB
#error no CueCat source defined !!!
#endif



/* Definitions */
#define BARCODE_BUFFER_SIZE 4096

#define MAX_CUECAT_SOURCES 3
#define SHUNT_SCANCODE 1
#define SCANCODES_TIMEOUT HZ/10

#ifdef CONFIG_CUECAT_PCKBD_ENABLED_BY_DEFAULT
#define DEFAULT_DRIVER_PCKBD_STARTUP_ENABLED 1
#else
#define DEFAULT_DRIVER_PCKBD_STARTUP_ENABLED 0
#endif

#ifdef CONFIG_CUECAT_PSMOUSE_ENABLED_BY_DEFAULT
#define DEFAULT_DRIVER_PSMOUSE_STARTUP_ENABLED 1
#else
#define DEFAULT_DRIVER_PSMOUSE_STARTUP_ENABLED 0
#endif

#ifdef CONFIG_CUECAT_USB_ENABLED_BY_DEFAULT
#define DEFAULT_DRIVER_USB_STARTUP_ENABLED 1
#else
#define DEFAULT_DRIVER_USB_STARTUP_ENABLED 0
#endif



/* Function prototypes */
#ifdef CONFIG_CUECAT_MODULE
#if defined CONFIG_CUECAT_PCKBD || defined CONFIG_CUECAT_PSMOUSE
extern int (*cuecat_pckeyb_scancode_processor)(unsigned char,int);
#endif
#ifdef CONFIG_CUECAT_USB
extern int (*cuecat_usbkbd_scancode_processor)(unsigned char,int);
#endif
#endif
extern inline void handle_mouse_event(unsigned char scancode);
static int cuecat_open(struct inode *inode,struct file *filp);
int cuecat_read(struct file *filp,char *buf,size_t count,loff_t *ppos);
static int cuecat_release(struct inode *inode,struct file *filp);
int barcode_decoder_timeout(void *unused);
#ifdef CONFIG_PROC_FS
void create_cuecat_proc_entry(void);
static int cuecat_write_proc(struct file *file,
                             const char *buffer,
                             unsigned long count,
                             void *data);
static int cuecat_read_proc(char *buffer,
                            char **start,
                            off_t offset,
                            int size,
                            int *eof,
                            void *data);
#endif
void flush_scancode_buffer(int scancode_source);



/* Types */
/* current driver information pertaining one CueCat */
struct cuecat_info
{
  char driver_present;

  char driver_enabled;
  unsigned char scancode_buffer[CUECAT_MAX_SCANCODE_SEQUENCE];
  int nb_scancodes_in_buffer;
  char cuecat_id[CUECAT_DEVICE_ID_LEN+1];		/* Cuecat device ID */
  char barcode_type[CUECAT_BARCODE_TYPE_LEN+1];
  char barcode[256];

  unsigned long last_scancode_arrival_time;
  char last_global_barcode_decoder_state;

  struct barcode_decoder_state barcode_decoder_state;
};



/* Global vars */
struct file_operations cuecat_fops=
{
  NULL,
  cuecat_read,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  cuecat_open,
  NULL,
  cuecat_release,
  NULL,
  NULL
};
static struct miscdevice cuecat_miscdevice=
{
  CUECAT_MINOR,
  "cuecat",
  &cuecat_fops
};

char nb_opens;						/* Nb processes opening
							   the corresponding
							   /dev/cuecat */
struct wait_queue *wq;					/* Process wait queue */

char halt_thread=0;

struct cuecat_info driver_info[MAX_CUECAT_SOURCES];
const unsigned char at2xt[128]=AT2XT_VALS;

char barcode_buffer[BARCODE_BUFFER_SIZE];
int barcode_buffer_bytes_available;
#ifdef CONFIG_PROC_FS
int procfs_cmd_state=0;
int procfs_cmd_driverno=0;
#endif



/* Functions */
#ifdef CONFIG_CUECAT_MODULE
int init_module(void)
#else
int cuecat_init(void)
#endif
{
  int i;

  printk("CueCat barcode reader driver\n");
  
  /* Register the driver */
  misc_register(&cuecat_miscdevice);

  for(i=0;i<MAX_CUECAT_SOURCES;i++)
  {
    /* Mark all drivers absent for now */
    driver_info[i].driver_present=0;

    /* Make sure all the scancode buffers starts empty */
    driver_info[i].nb_scancodes_in_buffer=0;

    /* Initialize the scancode arrival time markers */
    driver_info[i].last_scancode_arrival_time=jiffies;

    /* Initialize the last barcode decoder states */
    driver_info[i].last_global_barcode_decoder_state=
                                                NO_BARCODE_DECODING_IN_PROGRESS;

    /* Reset the barcode decoder states */
    cuecat_reset_barcode_decoder(&(driver_info[i].barcode_decoder_state));
  }

  /* Default states of the different drivers */
  driver_info[CUECAT_PCKBD].driver_enabled=DEFAULT_DRIVER_PCKBD_STARTUP_ENABLED;
  driver_info[CUECAT_PSMOUSE].driver_enabled=
                                         DEFAULT_DRIVER_PSMOUSE_STARTUP_ENABLED;
  driver_info[CUECAT_USB].driver_enabled=DEFAULT_DRIVER_USB_STARTUP_ENABLED;

  /* Mark present the drivers that have been chosen at compilation */
#ifdef CONFIG_CUECAT_PCKBD
  driver_info[CUECAT_PCKBD].driver_present=1;
  printk("  PC/AT CueCat barcode reader driver ready\n");
#endif
#ifdef CONFIG_CUECAT_PSMOUSE
  driver_info[CUECAT_PSMOUSE].driver_present=1;
  printk("  PS/2 CueCat barcode reader driver ready\n");
#endif
#ifdef CONFIG_CUECAT_USB
  driver_info[CUECAT_USB].driver_present=1;
  printk("  USB CueCat barcode reader driver ready\n");
#endif

  /* No barcode data available yet */
  barcode_buffer_bytes_available=0;

  /* No process waiting yet */
  wq=NULL;

  /* No /dev/cuecat open yet */
  nb_opens=0;

  /* Create a /proc filesystem entry */
#ifdef CONFIG_PROC_FS
  create_cuecat_proc_entry();
#endif

  /* Start the timeout thread */
  halt_thread=0;
  kernel_thread(barcode_decoder_timeout,NULL,0);

  /* If we're a module, tell the keyboard driver stub that we're here */
#ifdef CONFIG_CUECAT_MODULE
#if defined CONFIG_CUECAT_PCKBD || defined CONFIG_CUECAT_PSMOUSE
  if(driver_info[CUECAT_PCKBD].driver_present ||
     driver_info[CUECAT_PSMOUSE].driver_present)
    cuecat_pckeyb_scancode_processor=cuecat_process_scancode;
#endif
#if defined CONFIG_CUECAT_USB
  if(driver_info[CUECAT_USB].driver_present)
    cuecat_usbkbd_scancode_processor=cuecat_process_scancode;
#endif
#endif

  return(0);
}



/* If we're a module, unregister the device driver too */
#ifdef CONFIG_CUECAT_MODULE
void cleanup_module(void)
{
  /* Tell the keyboard driver stub that we're history */
#if defined CONFIG_CUECAT_PCKBD || defined CONFIG_CUECAT_PSMOUSE
  if(driver_info[CUECAT_PCKBD].driver_present ||
     driver_info[CUECAT_PSMOUSE].driver_present)
    cuecat_pckeyb_scancode_processor=NULL;
#endif
#if defined CONFIG_CUECAT_USB
  if(driver_info[CUECAT_USB].driver_present)
    cuecat_usbkbd_scancode_processor=NULL;
#endif

  /* Allow the timeout thread to stop itself */
  halt_thread=1;
  current->state=TASK_INTERRUPTIBLE;
  schedule_timeout(SCANCODES_TIMEOUT*5);		/* Paranoia timeout */

  printk("Removing CueCat barcode reader driver\n");

  /* Remove the proc filesystem entries */
#ifdef CONFIG_PROC_FS
  remove_proc_entry("scanners/cuecat",0);
  remove_proc_entry("scanners",0);
#endif

  /* Unregister the device driver */
  misc_deregister(&cuecat_miscdevice);
}
#endif



/* Open a device file */
int cuecat_open(struct inode *inode,struct file *filp)
{
  int minor=MINOR(inode->i_rdev);

  /* Is it our minor ? */
  if(minor!=CUECAT_MINOR)
    return(-ENODEV);

  /* We don't allow more than one process to open the device file */
  if(nb_opens)
    return(-EBUSY);

  nb_opens++;
  MOD_INC_USE_COUNT;

  return(0);
}



/* Close a device file */
int cuecat_release(struct inode *inode,struct file *filp)
{
  int minor=MINOR(inode->i_rdev);

  /* Is it our minor ? */
  if(minor!=CUECAT_MINOR)
    return(-ENODEV);

  nb_opens--;
  MOD_DEC_USE_COUNT;

  return(0);
}



/* Read from a device file */
int cuecat_read(struct file *filp,char *buf,size_t count,loff_t *ppos)
{
  int bytes_to_send;
  int minor=MINOR(filp->f_dentry->d_inode->i_rdev);
  int i;

  /* Is it our minor ? */
  if(minor!=CUECAT_MINOR)
    return(-ENODEV);

  /* Can't seek (pread) on this device */
  if (ppos != &filp->f_pos)
    return -ESPIPE;

  /* Put the calling process to sleep if the barcode buffer is empty */
  if(!barcode_buffer_bytes_available)
    interruptible_sleep_on(&wq);

  /* The process is awake, let's send as much as we can of the barcode buffer */
  bytes_to_send=barcode_buffer_bytes_available;
  if(count<bytes_to_send)
    bytes_to_send=count;
  memcpy(buf,barcode_buffer,bytes_to_send);
  for(i=bytes_to_send;i<=barcode_buffer_bytes_available;i++)
    barcode_buffer[i-bytes_to_send]=barcode_buffer[i];
  barcode_buffer_bytes_available-=bytes_to_send;

  return(bytes_to_send);
}



/* Process a scancode */
int cuecat_process_scancode(unsigned char scancode,int scancode_source)
{
  int i;

  /* If we don't know about the source of the scancode, ignore it (should we
     printk something instead ??) */
  if(scancode_source<0 || scancode_source>MAX_CUECAT_SOURCES ||
     !driver_info[scancode_source].driver_present)
    return(0);
  
  /* If the driver is disabled, simply exit */
  if(!driver_info[scancode_source].driver_enabled)
    return(0);

  /* Note the time of arrival of the new scancode */
  driver_info[scancode_source].last_scancode_arrival_time=jiffies;

  /* Purge the scancode buffer if we're out of space (it obviously can't be
     a barcode because it'd be already too long */
  if(driver_info[scancode_source].nb_scancodes_in_buffer==
     CUECAT_MAX_SCANCODE_SEQUENCE)
  {
    flush_scancode_buffer(scancode_source);

    /* Reset the barcode decoder, just to make sure */
    cuecat_reset_barcode_decoder(&(driver_info[scancode_source].
                                   barcode_decoder_state));
  }

  /* Store the new scancode in the buffer */
  driver_info[scancode_source].scancode_buffer[driver_info[scancode_source].
                                              nb_scancodes_in_buffer]=scancode;

  /* if coming from a PS/2 port, turn AT scancode to XT */
  if(scancode_source==CUECAT_PSMOUSE)
  {
    /* If this scancode is a key release modifier, shunt it right now */
    if(scancode==0xf0)
    {
      driver_info[scancode_source].nb_scancodes_in_buffer++;
      return(SHUNT_SCANCODE);
    }

    scancode=at2xt[scancode];
    if(driver_info[scancode_source].nb_scancodes_in_buffer &&
       driver_info[scancode_source].scancode_buffer[driver_info
       [scancode_source].nb_scancodes_in_buffer-1]==0xf0)
      scancode|=0x80;
  }

  driver_info[scancode_source].nb_scancodes_in_buffer++;

  /* Feed the new scancode to the barcode converter */
  switch(driver_info[scancode_source].last_global_barcode_decoder_state=
              cuecat_decode_barcode(&(driver_info[scancode_source].
                                      barcode_decoder_state),
                                      scancode,
                                      driver_info[scancode_source].cuecat_id,
                                      driver_info[scancode_source].barcode_type,
                                      driver_info[scancode_source].barcode))
  {
  case NO_BARCODE_DECODING_IN_PROGRESS:
    /* Purge the scancode buffer */
    flush_scancode_buffer(scancode_source);

    /* Do not let the scancode continue its voyage up the kernel (since it
       just got flushed) */
    return(SHUNT_SCANCODE);
    break;

  case POSSIBLE_BARCODE_DECODING_IN_PROGRESS:
    /* Do not let the scancode continue its voyage up the kernel */
    return(SHUNT_SCANCODE);
    break;

    break;

  case BARCODE_DECODED:
    /* Is there enough space in the buffer ? */
    i=strlen(driver_info[scancode_source].cuecat_id)+
      strlen(driver_info[scancode_source].barcode_type)+
      strlen(driver_info[scancode_source].barcode)+
      11;					/* length of "BARCODE:,,\n" */
    if(barcode_buffer_bytes_available+i<BARCODE_BUFFER_SIZE)
    {	/* Yes, we can store the new barcode */
      strcat(barcode_buffer,"BARCODE:");
      strcat(barcode_buffer,driver_info[scancode_source].cuecat_id);
      strcat(barcode_buffer,",");
      strcat(barcode_buffer,driver_info[scancode_source].barcode_type);
      strcat(barcode_buffer,",");
      strcat(barcode_buffer,driver_info[scancode_source].barcode);
      strcat(barcode_buffer,"\n");
      barcode_buffer_bytes_available+=i;
    }

    /* We have consumed the scancode buffer */
    driver_info[scancode_source].nb_scancodes_in_buffer=0;

    /* Wake up a sleeping process */
    wake_up_interruptible(&wq);

    /* Do not let the scancode continue its voyage up the kernel */
    return(SHUNT_SCANCODE);
    break;
  }

  /* If the scancode source is the USB CueCat and the driver is enabled, always
     shunt the scancode */
  if(scancode_source==CUECAT_USB)
    return(SHUNT_SCANCODE);
  else
    return(0);
}



/* Thread attached to task 0 that takes care of making sure a slow keyboard
   input times the barcode decoder out */
int barcode_decoder_timeout(void *unused)
{
  int driverno;

  /* Initialize thread */
  siginitsetinv(&current->blocked,0);
  lock_kernel();
  /* The thread doesn't need any user-level access, so get rid of all our 
     resources */
  exit_mm(current);
  exit_files(current);
  exit_fs(current);
  /* Set a name for the thread */
  strcpy(current->comm,"kcctod");
  unlock_kernel();

  while(!halt_thread)
  {
    /* Check all CueCat sources */
    for(driverno=0;driverno<MAX_CUECAT_SOURCES;driverno++)
    {
      /* If the driver is not present or disabled, simply skip the scancdde
         checking */
      if(driver_info[driverno].driver_present &&
         driver_info[driverno].driver_enabled)
      {
        /* If there is a possible barcode decoding in progress and the delay
           between the last scancode and now is greater than the timeout value,
           flush the scancode buffer and reset the barcode decoder */
        if(driver_info[driverno].last_global_barcode_decoder_state==
                                       POSSIBLE_BARCODE_DECODING_IN_PROGRESS &&
           jiffies-driver_info[driverno].last_scancode_arrival_time>=
                                       SCANCODES_TIMEOUT)
        {
          flush_scancode_buffer(driverno);

          /* Reset the barcode decoder */
          cuecat_reset_barcode_decoder(&(driver_info[driverno].
                                         barcode_decoder_state));
          driver_info[driverno].last_global_barcode_decoder_state=
                                                NO_BARCODE_DECODING_IN_PROGRESS;
        }
      }

      current->state=TASK_INTERRUPTIBLE;
      schedule_timeout(SCANCODES_TIMEOUT);
    }
  }

  return(0);
}



/* Proc filesystem init */
#ifdef CONFIG_PROC_FS
void create_cuecat_proc_entry(void)
{
  struct proc_dir_entry *cuecat_proc_dir_entry;
  struct proc_dir_entry *cuecat_proc_entry;

  /* Create the proc filesystem entries */
  if(!(cuecat_proc_dir_entry=create_proc_entry("scanners",S_IFDIR,0)))
    return;
  if(!(cuecat_proc_entry=create_proc_entry("scanners/cuecat",S_IWUSR | S_IRUSR,
                                           0)))
  {
    remove_proc_entry("scanners",0);
    return;
  }

  /* Configure the proc entry */
  cuecat_proc_entry->write_proc=cuecat_write_proc;
  cuecat_proc_entry->read_proc=cuecat_read_proc;
}



/* Recover data written to the proc cuecat entry */
static int cuecat_write_proc(struct file *file,
                             const char *buffer,
                             unsigned long count,
                             void *data)
{
  int i;

  /* Decode the data coming from the process */
  i=0;
  while(i<count)
  {
    switch(buffer[i])
    {
    case 'c':
    case 'C':
      if(procfs_cmd_state<=1)			/* Where we waiting for a C ?*/
        procfs_cmd_state++;
      else
        procfs_cmd_state=0;			/* No C allowed after "CC" */
      break;

    case '=':
      if(procfs_cmd_state!=3)
        procfs_cmd_state=0;			/* No = allowed other than */
      else					/* after "CC?" */
        procfs_cmd_state++;
      break;

    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
      if(procfs_cmd_state==2)			/* CueCat driver number ? */
      {
        procfs_cmd_driverno=buffer[i]-'0';
        if(procfs_cmd_driverno>=0 && procfs_cmd_driverno<MAX_CUECAT_SOURCES)
          procfs_cmd_state++;
        else
          procfs_cmd_state=0;
        break;
      }

      if(procfs_cmd_state==4)			/* Driver state */
      {
        if(buffer[i]=='0')
        {
          /* Was the driver enabled before ? */
          if(driver_info[procfs_cmd_driverno].driver_enabled)
          {
            /* Disable the driver */
            driver_info[procfs_cmd_driverno].driver_enabled=0;

            /* Flush the scancode buffer */
            flush_scancode_buffer(procfs_cmd_driverno);

            /* Reset the barcode decoder */
            cuecat_reset_barcode_decoder(&(driver_info[procfs_cmd_driverno].
                                           barcode_decoder_state));
            driver_info[procfs_cmd_driverno].last_global_barcode_decoder_state=
                                                NO_BARCODE_DECODING_IN_PROGRESS;
          }
        }
        else
          if(buffer[i]=='1')
          {
            driver_info[procfs_cmd_driverno].driver_enabled=1;
          }
        procfs_cmd_state=0;
        break;
      }

      /* Any other decoder state is invalid */
      procfs_cmd_state=0;
      break;
    }

    i++;
  }
  return(count);
}



/* Return the status of the drivers */
static int cuecat_read_proc(char *page,
                            char **start,
                            off_t offset,
                            int count,
                            int *eof,
                            void *data)
{
  int len=0, i;
  off_t begin=0;

  for(i=0;i<MAX_CUECAT_SOURCES;i++)
  {
    if(driver_info[i].driver_present)
    {
      if(i<MAX_CUECAT_SOURCES-1)
        len+=sprintf(page+len,"CC%d=%d,",i,driver_info[i].driver_enabled);
      else
        len+=sprintf(page+len,"CC%d=%d\n",i,driver_info[i].driver_enabled);
    }

    /* If we've written more than requested, quit */
    if(len+begin>offset+count)
      break;

    /* If the process requested to start after this point, don't bother saving*/
    if(len+begin<offset)
    {
      begin+=len;
      len=0;
    }
  }

  /* If we reached the end, set EOF */
  if(i>=MAX_CUECAT_SOURCES)
    *eof=1;

  /* If the process requested an offset past the end, return length 0 bytes */
  if(offset>=len+begin)
    return 0;

  *start=page+(offset-begin);

  return((count<begin+len-offset)?count:begin+len-offset);
}
#endif



/* Purge the scancode buffer */
void flush_scancode_buffer(int scancode_source)
{
  int i=0;

  while(i<driver_info[scancode_source].nb_scancodes_in_buffer)
  {
    switch(scancode_source)
    {
    case CUECAT_PCKBD:
      handle_scancode(driver_info[scancode_source].scancode_buffer[i],
                      !(driver_info[scancode_source].scancode_buffer[i]
                        & 0x80));
      break;
    case CUECAT_PSMOUSE:
      handle_mouse_event(driver_info[scancode_source].scancode_buffer[i]);
      break;
    }
    i++;
  }

  driver_info[scancode_source].nb_scancodes_in_buffer=0;
}
