/****************************************************************************\
**        Password Preprocesser v2.00 - Password Cracker Guess Mutator      **
** ======================================================================== **
**                Sourcefile: pwp.c -- Last Update: 09/20/91                **
** ======================================================================== **
**           Written By Doctor Dissector, Copyright (C) 1990, 1991          **
** ======================================================================== **
**                   LIMITED EDITION -- DO NOT DISTRIBUTE                   **
\****************************************************************************/

/*=[ License ]==============================================================*/

/*
** Password Preprocesser - Version 2.00
** Copyright (C) 1991 By Doctor Dissector
**
** This program is NOT free software BUT may be used without charge or
** payment in any form IF your copy is a "registered" distributed version.
** You may modify it as much as you please, however, you MAY NOT re-
** distribute it, in any shape or for: ie. modified OR unmodified,
** without the expressed written consent (ie. e-mail) of
** Doctor Dissector.
**
** This program was initially distributed in the hope that it will be
** useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/

/*=[ Disclaimer ]===========================================================*/

/*
**  The author of this package, Doctor Dissector, will not assume liability
**  for ANY of the effects of the use or distribution of this package.  If
**  you, at ANY time compile or use this package, you will assume full
**  liability for your own actions; Doctor Dissector can neither enforce the
**  lawful use of this package nor your actions before, during, or after
**  exposure to this package.  Doctor Dissector does NOT endorse your unlawful
**  use of this package to appropriate computer accounts not under your lawful
**  ownership.
**
**  The original intent of this package was to prove that Un*x accounts can be
**  easily, efficiently, and effectively cracked utilizing modified DES
**  encryption routines and proper statistical, mathematical, logical, and
**  programming skills.
*/

/*=[ Copyright ]============================================================*/

char pwp_c_msg[] = "Source: pwp.c, Copyright (C) 1990, 1991 By Doctor Dissector";

/*=[ Include Header Files ]=================================================*/

#include <stdio.h>
#include <string.h>
#include "pwp.h"

/*=[ Global Variables ]=====================================================*/

char progname[MAX_FILE];        /* calling name of the program */

int  double_w,                  /* double words?               0 = NO       */
     gecos,                     /* use GECOS fields?           0 = NO       */
     include,                   /* include sourcefile?         0 = NO       */
     kc,                        /* killer cracker format?      0 = NO       */
     logins,                    /* use login fields?           0 = NO       */
     lowercase,                 /* all lowercase?              0 = NO       */
     lowreverse,                /* lowercase & reverse?        0 = NO       */
     minlength,                 /* minimum length              0 = Inactive */
     nonalpha,                  /* minimum non-alpha chars     0 = Inactive */
     novowels,                  /* remove vowels?              0 = NO       */
     reverse,                   /* reverse?                    0 = NO       */
     upchar,                    /* make one char uppercase     0 = Inactive */
     uplast,                    /* last char uppercase?        0 = NO       */
     uppercase,                 /* all uppercase?              0 = NO       */
     upreverse,                 /* uppercase & reverse?        0 = NO       */
     translate;                 /* character/character xlate?  0 = Inactive */

char prefixes[MAX_TAIL],        /* prefix characters */
     suffixes[MAX_TAIL],        /* suffix characters */
     xchar[MAX_XCHAR][2];       /* xlate character table */

/*==========================================================================*/

/*
**     function: void print_usage(void)
**  description: displays program usage & options
**      returns: nothing
*/

void print_usage()
{
    printf("\n");
    printf("Password Preprocesser, Version 2.00 (9/20/91)\n");
    printf("Copyright (C) 1991, By Doctor Dissector\n");
    printf("*** DO NOT DISTRIBUTE THIS VERSION ***\n\n");
    printf("Brief:  Mutates a wordlist read from the standard input into other possible\n");
    printf("        guesses for use with password 'crackers' and outputs the result to\n");
    printf("        the standard output.\n\n");
    printf("Usage:  %s [[flag] ...] < [wordlist|pwfile] > [resultfile]\n\n", progname);
    printf("Flags:  -?                  display this usage menu\n");
    printf("        -Double             doubles the original word\n");
    printf("        -Gecos              create result words from GECOS fields of pwfile\n");
    printf("        -Help               display this usage menu\n");
    printf("        -Include            include original words into resultfile\n");
    printf("        -Kc                 save login/GECOS words in KC9 'single-crack' form\n");
    printf("        -LOGins             create result words from login fields of pwfile\n");
    printf("        -LOWErcase          convert original words to lowercase\n");
    printf("        -LOWReverse         convert original words to lowercase & reverse\n");
    printf("        -Minlength:<#>      min length <#> of result words\n");
    printf("        -NONalpha:<#>       min non-alphabetic chars <#> in result words\n");
    printf("        -NOVowels           remove vowels from original words\n");
    printf("        -Prefixes:<string>  prepend chars in <string> to original words\n");
    printf("        -Reverse            reverse original words\n");
    printf("        -Suffixes:<string>  append chars in <string> to original words\n");
    printf("        -Translate:<o><n>   translate char <o> in original words char <n>\n");
    printf("        -UPChar:<#>         convert char <#> in original words to uppercase\n");
    printf("        -UPLast             convert last char in original words to uppercase\n");
    printf("        -UPPercase          convert original words to uppercase\n");
    printf("        -UPReverse          convert original words to uppercase & reverse\n\n");
}

/*==========================================================================*/

/*
**     function: void init_vars(void)
**  description: pre-initialize global variables
**      returns: nothing
*/

void init_vars()
{
    progname[0]=0;
    double_w=0;
    gecos=0;
    include=0;
    kc=0;
    logins=0;
    lowercase=0;
    lowreverse=0;
    minlength=0;
    nonalpha=0;
    novowels=0;
    reverse=0;
    translate=0;
    upchar=0;
    uplast=0;
    uppercase=0;
    upreverse=0;
    prefixes[0]=0;
    suffixes[0]=0;
}

/*==========================================================================*/

/*
**     function: void eexit(int exitcode, char *desc1, char *desc2)
**  description: displays an exit/error message and exits with an exit
**               code of <exitcode>
**      returns: nothing
*/

void eexit(exitcode, desc1, desc2)
    int  exitcode;
    char *desc1, *desc2;
{
    fprintf(stderr, "\n%s: ", progname);
    if (desc1)
        fprintf(stderr, "%s", desc1);
    else
        fprintf(stderr, "abnormal program termination");
    if (desc2)
        fprintf(stderr, ": %s", desc2);
    fprintf(stderr, "\n");
    exit(exitcode);
}

/*==========================================================================*/

/*
**     function: int chstrnum(char xc, char *word)
**  description: counts the number occurrances of <xc> in <word>
**      returns: number of occurrances, 0 if none
*/

int chstrnum(xc, word)
    char xc, *word;
{
    register int i;

    for(i=0; (*word); word++)
        if (xc==(*word))
            i++;
    return(i);
}

/*==========================================================================*/

/*
**     function: char upcharacter(char c)
**  description: converts <c> to an uppercase equivalent
**      returns: uppercase <c> if any
*/

char upcharacter(c)
    register char c;
{
    return((c>='a') && (c<='z') ? c-32 : c);
}

/*==========================================================================*/

/*
**     function: char *lowcase(char *str)
**  description: converts <str> to its lowercase equivalent
**      returns: lowercase of <str>
*/

char *lowcase(str)
    register char *str;
{
    register char *ptr;
    static   char s[MAX_WORD];

    ptr=s;
    while(*str)
        (*ptr++)=(((*str)>='A') && ((*str)<='Z') ? (*str++)+32 : (*str++));
    *ptr=0;
    return(s);
}

/*==========================================================================*/

/*
**     function: char *upcase(char *str)
**  description: converts <str> to its uppercase equivalent
**      returns: uppercase of <str>
*/

char *upcase(str)
    register char *str;
{
    register char *ptr;
    static   char s[MAX_WORD];

    ptr=s;
    while(*str)
        (*ptr++)=(((*str)>='a') && ((*str)<='z') ? (*str++)-32 : (*str++));
    *ptr=0;
    return(s);
}

/*==========================================================================*/

/*
**     function: char *reverse_string(char *str)
**  description: reverse string <str>
**      returns: the reverse of <str>
*/

char *reverse_string(str)
    register char *str;
{
    register int  i, j;
    static   char s[MAX_WORD];

    i=strlen(str);
    s[i]=0;
    while(*str)
        s[--i]=(*str++);
    return(s);
}

/*==========================================================================*/

/*
**     function: char *do_lowercase(char *word)
**  description: creates lowercase equivalent of <word>
**      returns: lowercase equivalent of <word> or "\0" if both words match
*/

char *do_lowercase(word)
    char *word;
{
    static char s[MAX_WORD];

    strcpy(s, lowcase(word));
    if (include)
        if (!strcmp(s, word))
            s[0]=0;
    return(s);
}

/*==========================================================================*/

/*
**     function: char *do_uppercase(char *word)
**  description: creates uppercase equivalent of <word>
**      returns: uppercase equivalent of <word> or "\0" if both words match
*/

char *do_uppercase(word)
    char *word;
{
    static char s[MAX_WORD];

    strcpy(s, upcase(word));
    if (include)
        if (!strcmp(s, word))
            s[0]=0;
    return(s);
}

/*==========================================================================*/

/*
**     function: char *do_reverse(char *word)
**  description: reverses <word>
**      returns: reverse of <word> or "\0" if both words match
*/

char *do_reverse(word)
    char *word;
{
    static char s[MAX_WORD];

    strcpy(s, reverse_string(word));
    if (include)
        if (!strcmp(s, word))
            s[0]=0;
    return(s);
}

/*==========================================================================*/

/*
**     function: char *do_lowreverse(char *word)
**  description: converts <word> to lowercase & reverses it
**      returns: lowered/reversed <word> or "\0" if both words match
*/

char *do_lowreverse(word)
    char *word;
{
    static char s[MAX_WORD];

    strcpy(s, lowcase(reverse_string(word)));
    if (reverse)
        if (!strcmp(s, reverse_string(word)))
            s[0]=0;
    if ((lowercase) && (s[0]))
        if (!strcmp(s, lowcase(word)))
            s[0]=0;
    if ((include) && (s[0]))
        if (!strcmp(s, word))
            s[0]=0;
    return(s);
}

/*==========================================================================*/

/*
**     function: char *do_upreverse(char *word)
**  description: converts <word> to uppercase & reverses it
**      returns: uppered/reversed <word> or "\0" if both words match
*/

char *do_upreverse(word)
    char *word;
{
    static char s[MAX_WORD];

    strcpy(s, upcase(reverse_string(word)));
    if (reverse)
        if (!strcmp(s, reverse_string(word)))
            s[0]=0;
    if ((uppercase) && (s[0]))
        if (!strcmp(s, upcase(word)))
            s[0]=0;
    if ((include) && (s[0]))
        if (!strcmp(s, word))
            s[0]=0;
    return(s);
}

/*==========================================================================*/

/*
**     function: char *uplast_word(char *word)
**  description: converts the last character in <word> to uppercase
**      returns: converted <word>
*/

char *uplast_word(word)
    char *word;
{
    static char s[MAX_WORD];

    strcpy(s, word);
    s[strlen(word)-1]=upcharacter(word[strlen(word)-1]);
    return(s);
}

/*==========================================================================*/

/*
**     function: char *do_uplast(char *word)
**  description: converts the last character in <word> to uppercase
**      returns: converted <word> or "\0" if both words match
*/

char *do_uplast(word)
    char *word;
{
    static char s[MAX_WORD];

    strcpy(s, uplast_word(word));
    if (uppercase)
        if (!strcmp(s, upcase(word)))
            s[0]=0;
    if ((include) && (s[0]))
        if (!strcmp(s, word))
            s[0]=0;
    return(s);
}

/*==========================================================================*/

/*
**     function: char *do_upchar(char *word, int ch)
**  description: converts the character at index <ch-1> in <word> to uppercase
**      returns: converted <word> or "\0" if both words match
*/

char *do_upchar(word, ch)
    char *word;
          int  ch;
{
    static char s[MAX_WORD];

    strcpy(s, word);
    s[ch-1]=upcharacter(word[ch-1]);
    if (uppercase)
        if (!strcmp(s, upcase(word)))
            s[0]=0;
    if (uplast)
        if (!strcmp(s, uplast_word(word)))
            s[0]=0;
    if (strlen(word)<ch)
        s[0]=0;
    if ((include) && (s[0]))
        if (!strcmp(s, word))
            s[0]=0;
    return(s);
}

/*==========================================================================*/

/*
**     function: char *do_novowels(char *word)
**  description: removes vowels from <word>
**      returns: vowel-less <word> or "\0" if both words match
*/

char *do_novowels(word)
    char *word;
{
    register int  i, j;
    static   char s[MAX_WORD], c;

    for(i=0, j=0; word[i]; i++) {
        c=upcharacter(word[i]);
        if (!((c=='A') || (c=='E') || (c=='I') || (c=='O') || (c=='U')))
            s[j++]=word[i];
    }
    s[j]=0;
    if (include)
        if (!strcmp(s, word))
            s[0]=0;
    return(s);
}

/*==========================================================================*/

/*
**     function: char *do_double(char *word)
**  description: doubles the <word> if its length is less than 8
**      returns: doubled <word> or "\0" if <word> is longer than 7 chars
*/

char *do_double(word)
    char *word;
{
    static   char s[MAX_WORD];

    if (strlen(word)<8)
        sprintf(s, "%s%s", word, word);
    else
        s[0]=0;
    return(s);
}

/*==========================================================================*/

/*
**     function: char *do_prefixes(char *word, char prechar)
**  description: pre-pends the <word> with the character <prechar>
**      returns: pre-pended <word>
*/

char *do_prefixes(word, prechar)
    char *word, prechar;
{
    static char s[MAX_WORD];

    sprintf(s, "%c%s", prechar, word);
    return(s);
}

/*==========================================================================*/

/*
**     function: char *do_suffixes(char *word, char postchar)
**  description: post-pends the <word> with the character <word>'s length
**               is less than 8 characters
**      returns: post-pended <word> or "\0" if the <word> is 8 characters
**               or longer
*/

char *do_suffixes(word, postchar)
    char *word, postchar;
{
    static char s[MAX_WORD];

    if (strlen(word)<8)
        sprintf(s, "%s%c", word, postchar);
    else
        s[0]=0;
    return(s);
}

/*==========================================================================*/

/*
**     function: char *do_xlate_char(char *word, char *xc, int pos)
**  description: translates the <pos>th occurance xc[0] to xc[1]
**      returns: translated word
*/

char *do_xlate_char(word, xc, pos)
    char *word, *xc;
    int  pos;
{
    register int  i, j;
    static   char s[MAX_WORD];

    strcpy(s, word);
    for(i=0, j=0; (s[i]); i++) {
        if (s[i]==xc[0])
            if (++j>pos)
                break;
    }
    if (s[i])
        s[i]=xc[1];
    else
        s[0]=0;
    return(s);
}

/*==========================================================================*/

/*
**     function: char *do_xlate_all(char *word, char *xc)
**  description: translates all occurances of xc[0] to xc[1]
**      returns: translated <word>
*/

char *do_xlate_all(word, xc)
    char *word, *xc;
{
    register int  i;
    static   char s[MAX_WORD];

    for(i=0; word[i]; i++) {
        if (word[i]==xc[0])
            s[i]=xc[1];
        else
            s[i]=word[i];
    }
    s[i]=0;
    return(s);
}

/*==========================================================================*/

/*
**     function: int do_nonalpha(char *word)
**  description: counts the number of non-alphabetic characters in <word>
**      returns: number of non-alphabetic characters in <word>
*/

int do_nonalpha(word)
    char *word;
{
    register int i, okchars;

    for(i=0, okchars=0; word[i]; i++) {
        if (!(((word[i]>='A') && (word[i]<='Z')) ||
            ((word[i]>='a') && (word[i]<='z'))))
            okchars++;
    }
    return(okchars);
}

/*==========================================================================*/

/*
**     function: void do_result(char *word)
**  description: checks for minimum length of <word> and minimum
**               non-alphabetic characters in <word> before writing
**               the <word> to stdout
**      returns: nothing
*/

void do_result(word)
    char *word;
{
    if (!word[0])
        return;
    if (minlength)
        if (strlen(word)<minlength)
            return;
    if (nonalpha)
        if (do_nonalpha(word)<nonalpha)
            return;
    printf("%s\n", word);
}

/*==========================================================================*/

/*
**     function: void process_word(char *word)
**  description: processes one original <word> for mutation
**      returns: nothing
*/

void process_word(word)
    char *word;
{
    int i, j, k;

    if (include)
        do_result(word);
    if (lowercase)
        do_result(do_lowercase(word));
    if (uppercase)
        do_result(do_uppercase(word));
    if (reverse)
        do_result(do_reverse(word));
    if (lowreverse)
        do_result(do_lowreverse(word));
    if (upreverse)
        do_result(do_upreverse(word));
    if (uplast)
        do_result(do_uplast(word));
    if (upchar)
        do_result(do_upchar(word, upchar));
    if (novowels)
        do_result(do_novowels(word));
    if (double_w)
        do_result(do_double(word));
    if (prefixes[0]) {
        for(i=0; prefixes[i]; i++)
            do_result(do_prefixes(word, prefixes[i]));
    }
    if (suffixes[0]) {
        for(i=0; suffixes[i]; i++)
            do_result(do_suffixes(word, suffixes[i]));
    }
    if (translate) {
        for(i=0; i<translate; i++) {
            j=chstrnum(xchar[i][0], word);
            if (j) {
                for(k=0; k<j; k++)
                    do_result(do_xlate_char(word, &xchar[i], k));
                if (j>1)
                    do_result(do_xlate_all(word, &xchar[i]));
            }
        }
    }
}

/*==========================================================================*/

/*
**     function: int okgf(char gfc)
**  description: checks to see if the character is a GECOS delimiting char
**      returns: 1 if it is a delimiter, 0 if not
*/

int okgf(gfc)
    char gfc;
{
    if ((gfc!=',') && (gfc!=' ') && (gfc!='*') && (gfc!=':') && (gfc!='.') &&
        (gfc!='-') && (gfc!='/') && (gfc!=')') && (gfc!='('))
        return(1);
    return(0);
}

/*==========================================================================*/

/*
**     function: void process_line(char *line)
**  description: processes a single line of /etc/passwd information
**      returns: nothing
*/

void process_line(line)
    char *line;
{
    int  i, ok=0;
    char *login, *epw, *gec, *gecw;

    for(login=line; (*line!=':') && (*line); line++)
        ;
    if (*line) {
        *line=0;
        for(epw=++line; (*line!=':') && (*line); line++)
            ;
        if (*line) {
            *line=0;
            for(++line; (*line!=':') && (*line); line++)
                ;
            if (*line) {
                *line=0;
                for(++line; (*line!=':') && (*line); line++)
                    ;
                if (*line) {
                    *line=0;
                    for(gec=++line; (*line!=':') && (*line); line++)
                        ;
                    if (*line) {
                        *line=0;
                        ok=1;
                    }
                }
            }
        }
    }
    if (ok) {
        if (strlen(epw)>13)
            epw[13]=0;
        if ((epw[0]!='*') && (strlen(epw)==13) && (login[0]!='+')) {
            if (logins)
                process_word(login);
            if (gecos) {
                while(*gec) {
                    for(; (*gec) && (!okgf(*gec)); gec++)
                        ;
                    if (*gec) {
                        gecw=gec;
                        for(; (*gec) && (okgf(*gec)); gec++)
                            ;
                        if (*gec)
                            *gec++=0;
                        process_word(gecw);
                    }
                }
            }
            if (kc)
                printf("***!***\n");
        }
    }
}

/*==========================================================================*/

/*
**     function: void process_pwfile(void)
**  description: processes /etc/passwd file type info (login/GECOS)
**      returns: nothing
*/

void process_pwfile()
{
    char line[MAX_LINE];

    while(gets(line))
        process_line(line);
}


/*==========================================================================*/

/*
**     function: void process_wordlist(void)
**  description: processes each word from a wordlist
**      returns: nothing
*/

void process_wordlist()
{
    char word[MAX_WORD];

    while(gets(word)) {
        if (strcmp(word, "***!***"))
            process_word(word);
        else
            printf("***!***\n");
    }
}

/*==========================================================================*/

/*
**     function: int getcmd(char *name)
**  description: searches [cmdtab] for a matching command
**      returns: value assigned to each command in [cmdtab] or ERROR_FLAG
**               upon error
*/

int getcmd(name)
    char *name;
{
    struct cmd *c;

    for(c=cmdtab; c; c++) {
        if (!strncmp(name, c->cmd_name, c->unique)) {
            if (strlen(name)<=strlen(c->cmd_name)) {
                if (!strncmp(name, c->cmd_name, strlen(name)))
                    return(c->cmd_code);
            }
            return(ERROR_FLAG);
        }
    }
    return(ERROR_FLAG);
}

/*==========================================================================*/

/*
**     function: void process_arg(char *arg)
**  description: processes the value of one command line argument
**      returns: nothing
*/

void process_arg(arg)
    char *arg;
{
    int  result_code, i;
    char tail[MAX_TAIL];

    tail[0]=0;
    for(i=0; (arg[i]!=':') && (arg[i]); i++)
        ;
    if (arg[i]) {
        strcpy(tail, &arg[i+1]);
        arg[i]=0;
    }
    result_code=getcmd(lowcase(arg));
    switch(result_code) {
        case DOUBLE_FLAG:
            double_w=1;
            break;
        case GECOS_FLAG:
            gecos=1;
            break;
        case HELP_FLAG:
            print_usage();
            exit(0);
            break;
        case INCLUDE_FLAG:
            include=1;
            break;
        case KC_FLAG:
            kc=1;
            break;
        case LOGINS_FLAG:
            logins=1;
            break;
        case LOWERCASE_FLAG:
            lowercase=1;
            break;
        case LOWREVERSE_FLAG:
            lowreverse=1;
            break;
        case MINLENGTH_FLAG:
            minlength=atoi(tail);
            break;
        case NONALPHA_FLAG:
            nonalpha=atoi(tail);
            break;
        case NOVOWELS_FLAG:
            novowels=1;
            break;
        case PREFIXES_FLAG:
            strcpy(prefixes, tail);
            break;
        case REVERSE_FLAG:
            reverse=1;
            break;
        case SUFFIXES_FLAG:
            strcpy(suffixes, tail);
            break;
        case UPCHAR_FLAG:
            upchar=atoi(tail);
            break;
        case UPLAST_FLAG:
            uplast=1;
            break;
        case UPPERCASE_FLAG:
            uppercase=1;
            break;
        case UPREVERSE_FLAG:
            upreverse=1;
            break;
        case TRANSLATE_FLAG:
            if (strlen(tail)>1) {
                xchar[translate  ][0]=tail[0];
                xchar[translate++][1]=tail[1];
            }
            break;
        default:
            eexit(1, "invalid argument", arg);
            break;
    }
}

/*==========================================================================*/

/*
**     function: void main(int argc, char **argv)
**  description: main function
**      returns: nothing
*/

void main(argc, argv)
    int  argc;
    char **argv;
{
    char s[MAX_LINE];

    init_vars();
    strcpy(progname, *argv);
    if (argc==1) {
        print_usage();
        exit(0);
    }
    for(; --argc; ) {
        strcpy(s, *++argv);
        if ((s[0]=='-') || (s[0]=='/'))
            process_arg(&s[1]);
        else
            eexit(1, "invalid argument", s);
    }
    if ((kc) && (!gecos) && (!logins))
        eexit(1, "flag 'kc' must be used with 'logins' or 'gecos'", NULL);
    if ((gecos) || (logins))
        process_pwfile();
    else
        process_wordlist();
    exit(0);
}
