/*
Paros and its related class files.
Paros is an HTTP/HTTPS proxy for assessing web application security.
Copyright (C) 2003-2004 www.proofsecure.com

This program is free software; you can redistribute it and/or
modify it under the terms of the Clarified Artistic License
as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
Clarified Artistic License for more details.

You should have received a copy of the Clarified Artistic License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
package com.proofsecure.paros.network;

import java.net.URISyntaxException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class HttpRequestHeader extends HttpHeader {

	// method list
	public final static String OPTIONS	= "OPTIONS";
	public final static String GET 		= "GET";
	public final static String HEAD		= "HEAD";
	public final static String POST		= "POST";
	public final static String PUT		= "PUT";
	public final static String DELETE	= "DELETE";
	public final static String TRACE	= "TRACE";
	public final static String CONNECT	= "CONNECT";

	public final static String HOST = "Host";

	final static Pattern patternRequestLine
		= Pattern.compile(p_METHOD + p_SP + p_URI + p_SP + p_VERSION, Pattern.CASE_INSENSITIVE);
	protected final static Pattern patternHostHeader
		= Pattern.compile("([^:]+)\\s*:?\\s*(\\d*)");
	protected final static Pattern patternImage
		= Pattern.compile("\\.(jpg|jpeg|gif|tiff|tif|png)\\z", Pattern.CASE_INSENSITIVE);
	protected final static Pattern patternPartialRequestLine
		= Pattern.compile("\\A *(OPTIONS|GET|HEAD|POST|PUT|DELETE|TRACE|CONNECT)\\b", Pattern.CASE_INSENSITIVE);
		

	private String	mMethod = "";
	private String 	sUri = "";
	//private URI		mUri = null;
	private String	mHostName = "";
	private int		mHostPort = 80;
	private boolean	mIsSecure = false;
	private boolean	mIsAbsoluteUriRequired = false;
	
	public HttpRequestHeader() {
		mMethod = "";
		sUri = "";
		//mUri = null;
		mHostName = "";
		mHostPort = 80;
	}

    public HttpRequestHeader(String data) throws HttpMalformedHeaderException {
		this();
		setMessage(data);
    }

    public void setMessage(String data) throws HttpMalformedHeaderException {
		super.setMessage(data);
		try {
        	if (!parse()) {
        		mMalformedHeader = true;
        	}
    	} catch (Exception e) {
    		mMalformedHeader = true;
    	}

    	if (mMalformedHeader) {
    		throw new HttpMalformedHeaderException();
    	}

    }

    public String getMethod() {
        return mMethod;
    }
    
    public void setMethod(String method) {
    	mMethod = method.toUpperCase();
    	mStartLine = getMethod() + " " + getURI() + " " + getVersion();
    }
    	
	public String getURI() {
		return sUri;
	}
	
	public void setURI(String uri) {
		// ensure that Uri in header will not start with an invalid https header
		if (uri.startsWith(SCHEME_HTTPS)) {
			uri = SCHEME_HTTP + uri.substring(SCHEME_HTTPS.length());
			mIsSecure = true;
		} else if (uri.startsWith(SCHEME_HTTP)) {
			mIsSecure = false;
		}
		
		if (!uri.startsWith(SCHEME_HTTP)) {
			uri = SCHEME_HTTP + uri;
		}		
		
				
		
		try {
			sUri = HttpUtil.encodeURI(uri);
			mStartLine = getMethod() + " " + sUri + " " + getVersion();
		} catch (URISyntaxException e) {
			e.printStackTrace();
		}
		
		String hostHeader = getHostHeader(uri);
		parseHostName(hostHeader);
	}

	
	public String getURIPathQuery() {
		String result = sUri;	//mUri.getPath();
		int pos = result.indexOf("//");
		if (pos > -1) {
			pos = result.indexOf("/", pos+2);
			if (pos > -1) {
				result = result.substring(pos);
			} else {
				result = "";
			}
		}

		if (result.equals("")) {
			result = "/";
		}
		
		/*
		if (mUri.getQuery() != null) {
			result = result + "?" + mUri.getQuery();
		}
		*/

		return result;
	}
	
	public String getURIQuery() {
		int pos = sUri.indexOf("?");
		if (pos == -1) {
			return "";
		} else {
			return sUri.substring(pos+1);
		}
	}
			

	public String getURIHostPathQuery() {
		StringBuffer sb = new StringBuffer();
		
		if (getSecure()) {
			sb.append(SCHEME_HTTPS + getHostName());
			if (getHostPort() != 443)
			{
				sb.append(":" + getHostPort());
			}
		}
		else {
			sb.append(SCHEME_HTTP + getHostName());
			if (getHostPort() != 80) {
				sb.append(":" + getHostPort());
			}
		}

		String result = sb.append(getURIPathQuery()).toString();
		return result;
	}
	
	public String getURIHostPath() {
		String tmpUri = getURIHostPathQuery();
		int pos = tmpUri.indexOf("?");
		if (pos > -1) {
			tmpUri = tmpUri.substring(0, pos);
		}
		return tmpUri;
	}

	public boolean getSecure() {
		return mIsSecure;
	}
	
	public void setSecure(boolean isSecure) {
		mIsSecure = isSecure;
	}

	public void setVersion(String version) {
		mVersion = version.toUpperCase();
		mStartLine = getMethod() + " " + getURI() + " " + mVersion;
	}

	public int getContentLength() {
		if (content_length == -1) {
			return 0;
		}
		return content_length;
	}

    protected boolean parse() throws Exception {
    	    			
		Matcher matcher = patternRequestLine.matcher(mStartLine);
		if (!matcher.find()) {
			mMalformedHeader = true;
			return false;
		}
		
		mMethod = matcher.group(1);
		sUri	= matcher.group(2);

		mVersion	= matcher.group(3);
		
        if (!mVersion.equalsIgnoreCase(VERSION_HTTP10) && !mVersion.equalsIgnoreCase(VERSION_HTTP11)) {
        	mMalformedHeader = true;
        	return false;
        }

		String hostHeader = null;
		if (mMethod.equalsIgnoreCase(CONNECT)) {
			hostHeader = sUri;
		} else if (getHeader(HOST) != null) {
			hostHeader = getHeader(HOST);
		} else {
			hostHeader = getHostHeader(sUri);
		}

		parseHostName(hostHeader);

		// set mUri if sUri is a valid URI.  Otherwise build it.
		/*
		try {

			if (sUri.startsWith(URI_HTTP)) {
        		mUri = new URI(HttpUtil.encodeURI(sUri));
        	} else {
        		mUri = new URI(HttpUtil.encodeURI(URI_HTTP + buildHostPort(mHostName, mHostPort) + sUri));
        	}
        	//sUri = mUri.toString();
        } catch (Exception e) {
        	mUri = null;
        	e.printStackTrace();
        }
        */
        
		return true;
	}

	private String getHostHeader(String uri) {
		String result = null;
		String scheme = "";
		if (uri.startsWith(SCHEME_HTTP)) {
			scheme = SCHEME_HTTP;
		} else if (uri.startsWith(SCHEME_HTTPS)) {
			scheme = SCHEME_HTTPS;
		} else {
			return result;
		}
		
			
		int startOffset = scheme.length();
		int pos = uri.indexOf('/', startOffset);
		if (pos > -1) {
			result = sUri.substring(startOffset,pos);
		} else {
			result = sUri.substring(startOffset);
		}
		
		return result;
	}

	private void parseHostName(String hostHeader) {
		// no host header given but a valid host name already exist.  
		if (hostHeader == null) {
			return;
		}
		int pos = 0;
		if ((pos = hostHeader.indexOf(':', 2)) > -1) {
			mHostName = hostHeader.substring(0,pos).trim();
			try {
				mHostPort = Integer.parseInt(hostHeader.substring(pos+1));
			} catch (NumberFormatException e) {
			}
		} else {
			mHostName = hostHeader.trim();
		}

	}

	public String getHostName() {
		return mHostName;
	}
	
	public void setHostName(String hostName) {
		mHostName = hostName;
	}

	public int getHostPort() {
		return mHostPort;
	}
	
	public void setHostPort(int hostPort) {
		mHostPort = hostPort;
	}
	

	public String getAbsoluteURI() {

		String result = null;
		if (sUri.startsWith(SCHEME_HTTP)) {
			result = sUri;
		} else {
			result = SCHEME_HTTP + getHostName() + result;
		}
		return result;
	}

	public String toStringNoAbsoluteUri() {

		String result = null;
		if (sUri.startsWith(SCHEME_HTTP)) {
			int pos = sUri.indexOf('/',SCHEME_HTTP.length());
			if (pos > -1) {
				result = sUri.substring(pos);
			} else {
				result = "/";
			}
		} else {
			result = sUri;
		}

		result = mMethod + " " + result + " " + mVersion + mLineDelimiter;
		return result + mMsgHeader + mLineDelimiter;
	}

	public boolean isImage() {
		return (patternImage.matcher(getURIHostPath()).find());
	}

	public static boolean isRequestLine(String data) {
		
		return patternPartialRequestLine.matcher(data).find();
		
	}

	private String buildHostPort(String hostName, int  hostPort) {
		String result = hostName;
		
		if (hostPort != 443 && hostPort != 80) {
				result = result + ":" + hostPort;
		}

		return result;
	}
	
	/*
	To fix the absolute URL problem
	public String getStartLine() {
		String result = mMethod + " " + getAbsoluteURI() + " " + mVersion;
		return result;
	}
	*/
	

	
}