/*
Paros and its related class files.
Paros is an HTTP/HTTPS proxy for assessing web application security.
Copyright (C) 2003-2004 www.proofsecure.com

This program is free software; you can redistribute it and/or
modify it under the terms of the Clarified Artistic License
as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
Clarified Artistic License for more details.

You should have received a copy of the Clarified Artistic License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
package com.proofsecure.paros.scan;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Hashtable;

import com.proofsecure.paros.network.HttpBody;
import com.proofsecure.paros.network.HttpHeader;
import com.proofsecure.paros.network.HttpMalformedHeaderException;
import com.proofsecure.paros.network.HttpRequestHeader;
import com.proofsecure.paros.util.Util;


public class ParsedEntity extends javax.swing.tree.DefaultMutableTreeNode {

	public static final int		DEFAULT			= 0;
	public static final int		DO_NOT_SCAN		= 1;

	private static final String	CRLF			= "\r\n";
	private static final String DEFAULT_HEADER =
				"Accept: */*" + CRLF +
				"Accept-Language: en-us" + CRLF +
				"Pragma: no-cache" + CRLF + 
				"User-Agent: Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.0;)" + CRLF;
				
	private static final String a_BASIC_AUTH		= "basicAuth";
	private static final String a_BASIC_AUTH_BASE64	= "basicAuthBase64";
	private static final String HTTP				= "http";
	private static final String HTTPS				= "https";


	private static final String	BASIC				= "Basic ";

	private URI		mUri							= null;
	
	private	boolean	mIsSecure	= false;
	private	String	mHostName	= "";
	private	int		mHostPort	= 80;
	private	HttpRequestHeader	mReqHeader	= new HttpRequestHeader();
	private	HttpBody			mReqBody	= new HttpBody();
	private	String	mCaption	= "";

	protected Hashtable mAttr			= new Hashtable();
	public int	mState		= DEFAULT;
	
	public ParsedEntity() {
	}
	
	public ParsedEntity(String caption) {
		super(caption);
	}

	public ParsedEntity(String caption, HttpRequestHeader req, HttpBody body) throws HttpMalformedHeaderException, URISyntaxException {
		mCaption	= caption;
		setUserObject(mCaption);
		setEntity(req, body);
	}

	public ParsedEntity(String caption, URI uri, boolean isSecure) throws HttpMalformedHeaderException, URISyntaxException {
		mCaption	= caption;
		setUserObject(mCaption);
		setEntity(uri, isSecure);
	}

	public void setEntity(HttpRequestHeader req, HttpBody body) throws HttpMalformedHeaderException, URISyntaxException {	
		String val = "";

		// break Uri
		
		setEntity(new URI(req.getURIHostPath()), req.getSecure());

		// clean up header fields
		getRequestHeader().setMessage(req.toStringAbsoluteUri());
		//getRequestHeader().setURI(Util.getURIPathQuery(mUri));
		getRequestHeader().setHeader(HttpHeader.IF_MODIFIED_SINCE, null);
		getRequestHeader().setHeader(HttpHeader.IF_NONE_MATCH, null);
		getRequestHeader().setVersion(HttpHeader.VERSION_HTTP10);
		getRequestHeader().setHeader(HttpHeader.CONNECTION, null);
		getRequestBody().setBody(body.toString());

		mIsSecure	= req.getSecure();
		mHostName	= req.getHostName();
		mHostPort	= req.getHostPort();

		// process individual attributes or header here.

		// auth code
		val = req.getHeader(HttpHeader.AUTHORIZATION);
		if (val != null) {
			setBasicAuthBase64(val);
		}

	}

	private void setEntity(URI uri, boolean isSecure) throws HttpMalformedHeaderException, URISyntaxException {
		mIsSecure = isSecure;

		if (uri.getScheme().equalsIgnoreCase(HTTPS)) {
			mIsSecure = true;
		}
		mUri		= new URI(Util.getURIHostPath(uri, mIsSecure));
		mHostName = uri.getHost();
		mHostPort = uri.getPort();
		
		String sTmp = mUri.getPath();
		if (sTmp == null || sTmp.equals("")) {
			sTmp = "/";
		}		
		// create dummy header

		getRequestHeader().setMessage("GET " + sTmp + " HTTP/1.0" + CRLF + DEFAULT_HEADER + CRLF);

		getRequestHeader().setSecure(mIsSecure);

	}
	
	/**
	URI of this entity with host and path only like:
	https://www.testingdomain.com/abc
	*/
	public URI getURIHostPath() {
		return mUri;
	}	

	public String getHostName() {
		return mHostName;
	}
	
	public void setHostName(String hostName) {
		mHostName = hostName;
	}

	public int getHostPort()
	{
		return mHostPort;
	}
	
	public void setHostPort(int hostPort) {
		mHostPort = hostPort;
	}

	public boolean getSecure()
	{
		return mIsSecure;
	}
	
	public void setSecure(boolean isSecure) {
		mIsSecure = isSecure;
	}

	/**
	Get Basic Authentication string with Base64 encoding.
	@return	Base64 encoding string.
	*/
	public String getBasicAuthBase64() {
		return (String) getAttr(a_BASIC_AUTH_BASE64);
	}

	/**
	Set basic authentication with base64 in this entity.  This will also change the
	@param	basicAuthBase64 String with base64 encoding.
	*/
	public void setBasicAuthBase64(String basicAuthBase64) {
		if (basicAuthBase64 == null || basicAuthBase64.equals(""))
		{
			mAttr.remove(a_BASIC_AUTH_BASE64);
			mAttr.remove(a_BASIC_AUTH);
			return;
		}

		if (basicAuthBase64.substring(0,BASIC.length()).equalsIgnoreCase(BASIC)) {
			basicAuthBase64 = basicAuthBase64.substring(BASIC.length());
		}

		String basicAuth	= "";
		try  {
			// not implemented yet as encoding from base64 not found
			//basicAuth = new String(staticEncoder.GetChars(Convert.FromBase64String(val)));
			mAttr.put(a_BASIC_AUTH_BASE64, BASIC + basicAuthBase64);
			// also modify the string without base64
			mAttr.put(a_BASIC_AUTH, basicAuth);
		} catch (Exception e) {
		}

	}

	/**
	Get basic authentication string.
	@return	string without base64 encoding.
	*/
	public String getBasicAuth() {
		return (String) getAttr(a_BASIC_AUTH);
	}
	
	public void setBasicAuth(String basicAuth) {
		if (basicAuth == null || basicAuth.equals("")) {
			mAttr.remove(a_BASIC_AUTH);
			mAttr.remove(a_BASIC_AUTH_BASE64);
			return;
		}

		mAttr.put(a_BASIC_AUTH, basicAuth);

		// also modify the string with base64
		String base64 = "";
		//base64 = Convert.ToBase64String(staticEncoder.GetBytes(value)
		mAttr.put(a_BASIC_AUTH_BASE64, BASIC + base64);
	}

	/**
	Get attribute if exist in current object.  Search from parents if not exists.
	@param	attrName	name of attribute
	@return	value of the attribute.  null if not exists.
	*/
	protected Object getAttr(String attrName) {
		ParsedEntity entity = this;
		Object val = null;
		while (entity != null) {
			val = entity.getAttr(attrName);
			if (val != null)	{
				break;
			}

			entity = (ParsedEntity) entity.getParent();
			if (entity == null) {
				break;
			}

		}

		return val;
	}

	public HttpRequestHeader getRequestHeader() {
		return mReqHeader;
	}

	public HttpBody getRequestBody() {
		return mReqBody;
	}
	
	public void setCaption(String caption) {
		mCaption = caption;
		setUserObject(mCaption);
	}
	
	
}
