/*
Paros and its related class files.
Paros is an HTTP/HTTPS proxy for assessing web application security.
Copyright (C) 2003-2004 www.proofsecure.com

This program is free software; you can redistribute it and/or
modify it under the terms of the Clarified Artistic License
as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
Clarified Artistic License for more details.

You should have received a copy of the Clarified Artistic License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
package com.proofsecure.paros.scan;

import java.io.IOException;
import java.util.regex.Pattern;

import com.proofsecure.paros.network.HttpStatusCode;

class TestPerlInjection extends TestAbstractParam {

	private static String PERL_1 = ";ls%20/";
	
	private static String PERL_DELAY_1 = "';wait%2015;#";
	private static String PERL_DELAY_2 = ";wait%2015;#";
	
	private static int TIME_SPREAD = 15000;
	
	private static Pattern patternErrorPERL1 = Pattern.compile("insecure dependency", PATTERN_PARAM);
	private static Pattern patternErrorPERL2 = Pattern.compile("insecure \\$ENV\\{PATH\\}", PATTERN_PARAM);
	
	private String mResBodyNormal 	= "";		// normal response for comparison
	//private String mResBodyError	= "";		// error response for comparison

	TestPerlInjection() {
	}

    public String toString() {
        return "TestPerlInjection";
    }
    
	public String getTestName() {
		return "Perl Injection";
	}	
	
	protected void check(boolean isBody, String paramKey, String paramValue, String query, int insertPos) throws IOException {

		String bingoQuery = null;
		String displayURI = null;
		String newQuery = null;
		String resBody = null;
		
		int pos = 0;
		long defaultTimeUsed = 0;
		long timeUsed = 0;
		long lastTime = 0;
		
		// always try normal query first
		newQuery = insertQuery(query, insertPos, paramKey + "=" + paramValue);
		createMessage(isBody, newQuery);
		lastTime = System.currentTimeMillis();
		sendAndReceive();
		defaultTimeUsed = System.currentTimeMillis() - lastTime;
		if (getResponseHeader().getStatusCode() != HttpStatusCode.OK) {
			return;
		}

		mResBodyNormal = getResponseBody().toString();

		/*
		// 2nd try an always error SQL query
		newQuery = insertQuery(query, insertPos, paramKey + "=" + paramValue + SQL_CHECK_ERR);
		createMessage(isBody, newQuery);
		lastTime = System.currentTimeMillis();
		sendAndReceive();
		defaultTimeUsed = System.currentTimeMillis() - lastTime;
		mResBodyError	= getResponseBody().toString();
		if (checkResult(newQuery)) {
			return;
		}
		*/

		// try PERL query 1
		bingoQuery = insertQuery(query, insertPos, paramKey + "=" + paramValue + PERL_1);
		createMessage(isBody, bingoQuery);
		sendAndReceive();
		displayURI = getRequestHeader().getURIHostPathQuery();

		if (checkResult(bingoQuery)) {
			return;
		}

		if (mResBodyNormal.indexOf("root") > 0 ) {
			// no check needed if "root" exists
			resBody = getResponseBody().toString();
			if (resBody.indexOf("root") >0 ) {
				bingo(20003, AlertItem.RISK_HIGH, AlertItem.WARNING, displayURI, bingoQuery, "");
			}
		}

		// try BLIND PERL using timing 
		newQuery = insertQuery(query, insertPos, paramKey + "=" + paramValue + PERL_DELAY_1);
		createMessage(isBody, newQuery);
		lastTime = System.currentTimeMillis();
		sendAndReceive();
		timeUsed = System.currentTimeMillis() - lastTime;
				
		if (checkTimeResult(newQuery, defaultTimeUsed, timeUsed)) {
			return;
		}

		newQuery = insertQuery(query, insertPos, paramKey + "=" + paramValue + PERL_DELAY_2);
		createMessage(isBody, newQuery);
		lastTime = System.currentTimeMillis();
		sendAndReceive();
		timeUsed = System.currentTimeMillis() - lastTime;
				
		if (checkTimeResult(newQuery, defaultTimeUsed, timeUsed)) {
			return;
		}

	}
	


	private boolean checkResult(String query) {

		StringBuffer sb = new StringBuffer();
		if (getResponseHeader().getStatusCode() != HttpStatusCode.OK
			&& !HttpStatusCode.isServerError(getResponseHeader().getStatusCode())) {
			return false;
		}
		
		if (matchBodyPattern(patternErrorPERL1, sb) 
			|| matchBodyPattern(patternErrorPERL2, sb)) {
			// check for PERL error.  Almost certain.
			bingo(20003, AlertItem.RISK_HIGH, AlertItem.WARNING, "", query, sb.toString());
			return true;
		}
		
		return false;
		
	}

	
	private boolean checkTimeResult(String query, long defaultTimeUsed, long timeUsed) {

		boolean result = checkResult(query);
		if (result) {
			return result;
		}


		if (timeUsed > defaultTimeUsed + TIME_SPREAD - 500) {		
			// allow 500ms discrepancy
			bingo(20003, AlertItem.RISK_HIGH, AlertItem.SUSPICIOUS, "", query, "");
			return true;
		}			
		return false;
	}
	
	protected void scan() throws Exception {
		boolean skip = (getEntity().mState == ParsedEntity.DO_NOT_SCAN);
			
		writeStatus("PERL Injection: " + (skip? "(skipped) " : "") + getRequestHeader().getURIHostPath());
		if (skip) {
			return;
		}
		init();
		checkUrlOrBody(false, myQuery);
		checkUrlOrBody(true, myReqBody.toString());
	}

}
