/*
 Paros and its related class files.
 Paros is an HTTP/HTTPS proxy for assessing web application security.
 Copyright (C) 2003-2004 www.proofsecure.com

 This program is free software; you can redistribute it and/or
 modify it under the terms of the Clarified Artistic License
 as published by the Free Software Foundation.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 Clarified Artistic License for more details.

 You should have received a copy of the Clarified Artistic License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package com.proofsecure.paros.filter;

import javax.swing.JDialog;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.swing.JOptionPane;

import com.proofsecure.paros.Global;
import com.proofsecure.paros.network.HttpBody;
import com.proofsecure.paros.network.HttpRequestHeader;
import com.proofsecure.paros.network.HttpResponseHeader;
//import com.proofsecure.paros.network.HttpMalformedHeaderException;


// Use common patterns
import com.proofsecure.paros.scan.TestCacheAndMisc;

public class DetectUnsafeContent extends AbstractFilter {
	private JDialog dialog = null;
	
	public DetectUnsafeContent(){
		dialog = new PDialog("DetectUnsafeContent");
	}

	public JDialog getDialog(){
		return dialog;
	}

	public void filterRequest(HttpRequestHeader reqHeader, HttpBody reqBody) {
	
	}
	
	/**
	 *   used by FilterManager to run the filter 
	 *   @param    reqHeader	Request Header
	 *   @param    resHeader	Response header to be filtered
	 *   @param	resBody		Response body to be filtered
	 */
	public void filterResponse(HttpRequestHeader reqHeader, HttpResponseHeader resHeader, HttpBody resBody) {
		// no filtering if image 
		if (resHeader.isImage()) {
			return;
		}
		
		try {
			checkAutocomplete(reqHeader, resHeader, resBody);
		} catch (Exception e) {
		}

		try {
			checkMaliciousCode(reqHeader, resHeader, resBody);
		} catch (Exception e) {
		}
	}
	
	public String usage(){
		return "Detect insecure or potentially unsafe response (see output)";
	}    

	private void writeOutput(String hostPath, String msg, String match) {
		Global.parosFrame.logAppend(com.proofsecure.paros.ui.ParosFrame.OUTPUT_LOG, "+ " + hostPath + "\r\n");
		Global.parosFrame.logAppend(com.proofsecure.paros.ui.ParosFrame.OUTPUT_LOG, "- " + msg + "\r\n");
		if (match != null) {
			Global.parosFrame.logAppend(com.proofsecure.paros.ui.ParosFrame.OUTPUT_LOG, "- " + match + "\r\n");
		}
		Global.parosFrame.logAppend(com.proofsecure.paros.ui.ParosFrame.OUTPUT_LOG, "\r\n");
		
	}
	
	private void checkAutocomplete(HttpRequestHeader reqHeader, HttpResponseHeader resHeader, HttpBody body) {

		String txtBody = body.toString();
		String txtForm = null;
		String txtInput = null;
		Matcher matcherForm = TestCacheAndMisc.patternForm.matcher(txtBody);
		Matcher matcherAutocomplete = null;
			
		while (matcherForm.find()) {
			txtForm = matcherForm.group(1);	
			txtInput = matcherForm.group(2);

			//System.out.println(txtForm + txtInput);
			if (txtForm != null && txtInput != null) {
				matcherAutocomplete = TestCacheAndMisc.patternAutocomplete.matcher(txtForm);
				if (matcherAutocomplete.find()) {
					continue;
				}
				matcherAutocomplete = TestCacheAndMisc.patternAutocomplete.matcher(txtInput);
				if (matcherAutocomplete.find()) {
					continue;
				}
				
				writeOutput(reqHeader.getURIHostPath(), "Password field without setting autocomplete to off found.", txtInput);
				
			}
			
		}
		
	}

	private final static int DEFAULT = Pattern.MULTILINE | Pattern.CASE_INSENSITIVE;

	private final static Pattern[] patternBadHeaderList = {
			Pattern.compile("Content-type:\\s+application/hta", DEFAULT),
			Pattern.compile("Content-Disposition: attachment; filename=[^{}]+\\{[\\w\\d\\-]*\\}[^{}]+", DEFAULT),
	};

	private final static String[] patternBadHeaderDesc = {
		"Suspicious content-type header 'application/hta'",
		"MS IE Microsoft Internet Explorer CLSID File Extension Misrepresentation Vulnerability (http://www.securityfocus.com/bid/9510)"
	};
	
	private final static Pattern[] patternBadBodyList = {
		Pattern.compile("^.*file:javascript:eval.*$", DEFAULT),
		Pattern.compile("<[^>]*CLSID:11111111-1111-1111-1111-11111111111[^>]*>", DEFAULT),
		Pattern.compile("^.*Scripting\\.FileSystemObject.*$", DEFAULT),
		Pattern.compile("^.*new\\s+ActiveXObject.*$", DEFAULT),
		Pattern.compile("<OBJECT\\s+[^>]+>", DEFAULT),
		Pattern.compile("https?://[^\\s\"']+?@[^\\s\"']+?", DEFAULT),
		Pattern.compile("^.*Microsoft\\.XMLHTTP.*$", DEFAULT),		
		Pattern.compile("^.*SaveToFile.*$", DEFAULT),
		Pattern.compile("^.*CreateObject(\\s*[\"']+Adodb.Stream[\"']\\s*)$", DEFAULT),
		Pattern.compile("^.*execcommand.*$", DEFAULT),
		Pattern.compile("ms-its:mhtml:file://", DEFAULT),
		Pattern.compile("<iframe[^>]+src=['\"]*shell:[^>]+>", DEFAULT)
	};

	private final static String[] patternBadBodyDesc = {
			"Suspcious use of javascript 'file:javascript:eval'.",
			"Suspicious ActiveX CLSID 11111111-1111-1111-1111-... being used.",
			"Attempt to access Scripting.FileSystemObject.",
			"Inline creation of ActiveX object.",
			"ActiveX object used.",
			"URL with '@' to obscure hyperlink.",
			"Suspicious use of ActiveX XMLHTTP object (http://www.securityfocus.com/bid/8577)",
			"Suspicious scripting attempt to access local file via SafeToFile.  MS IE Self Executing HTML Arbitrary Code Execution Vulnerability.  (http://www.securityfocus.com/bid/8984)",
			"MS IE Self Executing HTML Arbitrary Code Execution Vulnerability.  (http://www.securityfocus.com/bid/8984)",
			"MS IE ExecCommand Cross-Domain Access Violation Vulnerability (http://www.securityfocus.com/bid/9015)",
			"MS IE MT-ITS Protocol Zone Bypass Vulnerability (http://www.securityfocus.com/bid/9658)",
			"MS IE Shell: IFrame Cross-Zone Scripting Vulnerability (http://www.securityfocus.com/bid/9628)"
	};
	
	private void checkMaliciousCode(HttpRequestHeader reqHeader, HttpResponseHeader resHeader, HttpBody body) {
		
		Pattern bad = null;
		Matcher matcher = null;
		String txtHeader = reqHeader.toString();
		String txtBody = body.toString();

		// check malicious header
		
		for (int i=0; i<patternBadHeaderList.length; i++){
			bad = patternBadHeaderList[i];
			matcher = bad.matcher(txtHeader);
			while (matcher.find()) {
				writeOutput(reqHeader.getURIHostPath(), patternBadHeaderDesc[i], matcher.group(0));
			}
		}
		
		// check malicous body 
		for (int i=0; i<patternBadBodyList.length; i++){
			bad = patternBadBodyList[i];
			matcher = bad.matcher(txtBody);
			while (matcher.find()) {
				writeOutput(reqHeader.getURIHostPath(), patternBadBodyDesc[i], matcher.group(0));				
			}
		}
	}

	/**
	 * This method is not used yet
	 * @param code
	 * @return
	 */
	private boolean checkAllow(String code) {
				
		Global.parosFrame.toFront();
		String text = JOptionPane.showInputDialog(Global.parosFrame, "Suspicious code found.  Accept it? (OK=Accept, Cancel=filter it)", code);

		// text == null means cancel pressed.  If cancel, string will be filtered
		if ((text != null && !text.equals(""))){
			return false;
		} else {
			return true;
		}
	}
	
	
}
