/*
Paros and its related class files.
Paros is an HTTP/HTTPS proxy for assessing web application security.
Copyright (C) 2003-2004 www.proofsecure.com

This program is free software; you can redistribute it and/or
modify it under the terms of the Clarified Artistic License
as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
Clarified Artistic License for more details.

You should have received a copy of the Clarified Artistic License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

package com.proofsecure.paros.scan;
import java.io.IOException;
import java.util.Vector;
 
import javax.net.ssl.SSLSocket;

import com.proofsecure.paros.Global;

class TestCipherSuite extends AbstractServerTest {

	private String[] mCipherSuite = null;

	private static final String[] CIPHER_SUITE_NULL = {"SSL_RSA_WITH_NULL_MD5", "SSL_RSA_WITH_NULL_SHA"};
	private static final String[] CIPHER_SUITE_WEAK = {
		"SSL_DH_anon_WITH_RC4_128_MD5",
		"TLS_DH_anon_WITH_AES_128_CBC_SHA",
		"SSL_DH_anon_WITH_3DES_EDE_CBC_SHA",
		"SSL_DH_anon_WITH_DES_CBC_SHA",
		"SSL_DH_anon_EXPORT_WITH_RC4_40_MD5",
		"SSL_DH_anon_EXPORT_WITH_DES40_CBC_SHA",
		"SSL_RSA_EXPORT_WITH_RC4_40_MD5",
		"SSL_RSA_EXPORT_WITH_DES40_CBC_SHA",
		"SSL_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA",
		"SSL_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA"
	};
		
	private static final String[] CIPHER_SUITE_MEDIUM = {
		"SSL_RSA_WITH_DES_CBC_SHA",
		"SSL_DHE_RSA_WITH_DES_CBC_SHA",
		"SSL_DHE_DSS_WITH_DES_CBC_SHA"
	};
	
    public String toString() {
        return "TestCipherSuite";
    }

	public String getTestName() {
		return "SSL Cipher Suite";
	}
	
	public int getAlertID() {
		return 10008;
	}

	/*
	SSL_RSA_WITH_RC4_128_MD5
	SSL_RSA_WITH_RC4_128_SHA
	TLS_RSA_WITH_AES_128_CBC_SHA
	TLS_DHE_RSA_WITH_AES_128_CBC_SHA
	TLS_DHE_DSS_WITH_AES_128_CBC_SHA
	SSL_RSA_WITH_3DES_EDE_CBC_SHA
	SSL_DHE_RSA_WITH_3DES_EDE_CBC_SHA
	SSL_DHE_DSS_WITH_3DES_EDE_CBC_SHA

	medium
	SSL_RSA_WITH_DES_CBC_SHA
	SSL_DHE_RSA_WITH_DES_CBC_SHA
	SSL_DHE_DSS_WITH_DES_CBC_SHA


	weak
	SSL_DH_anon_WITH_RC4_128_MD5
	TLS_DH_anon_WITH_AES_128_CBC_SHA
	SSL_DH_anon_WITH_3DES_EDE_CBC_SHA
	SSL_DH_anon_WITH_DES_CBC_SHA
	SSL_DH_anon_EXPORT_WITH_RC4_40_MD5
	SSL_DH_anon_EXPORT_WITH_DES40_CBC_SHA
	SSL_RSA_EXPORT_WITH_RC4_40_MD5
	SSL_RSA_EXPORT_WITH_DES40_CBC_SHA
	SSL_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA
	SSL_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA	

	null
	SSL_RSA_WITH_NULL_MD5
	SSL_RSA_WITH_NULL_SHA

	*/
	

	/**
	Init is used to add test case.
	*/
	protected void init(ParsedEntity entity) throws Exception {
		super.init(entity);
		TestCipherSuite test = new TestCipherSuite();
		test.initForRun(entity);
		addTestCase(test);
	}
	
	protected void initForRun(ParsedEntity entity) throws Exception {
		super.init(entity);
	}
	
	
	protected void scan() throws Exception {

		// no need to check if not secure
		if (!getRequestHeader().getSecure()) {
			return;
		}

		String host = getRequestHeader().getHostName();
		int port = getRequestHeader().getHostPort();
		
		SSLSocket socket = null;
		Vector supportedList = new Vector();
		
		try {
		
			socket = getSSLSocket(host, port);
			mCipherSuite = socket.getSupportedCipherSuites();
			//for (int i=0; i<mCipherSuite.length; i++) {
			//	System.out.println("Checking:\t" + mCipherSuite[i]);
			//}
		} catch (Exception e) {
			e.printStackTrace();
		}

		for (int i=0; i<mCipherSuite.length; i++) {
			writeStatus("SSLCipherSuite: " + mCipherSuite[i]);
			checkCipher(host, port, mCipherSuite[i], supportedList);
		}
		
		StringBuffer sbAll = new StringBuffer();
		StringBuffer sbWeak = new StringBuffer();
		StringBuffer sbNull = new StringBuffer();
		String cipher = null;
		for (int i=0; i<supportedList.size(); i++) {
			cipher = (String) supportedList.elementAt(i);
			if (checkDefinition(CIPHER_SUITE_NULL, cipher)) {
				sbNull.append(cipher + "\r\n");
			} else if (checkDefinition(CIPHER_SUITE_WEAK, cipher)) {
				sbWeak.append(cipher + "\r\n");
			}
			sbAll.append(cipher + "\r\n");
		}
		
		if (sbNull.length() > 0) {
			bingo(10009, AlertItem.RISK_MEDIUM, AlertItem.WARNING, "", "", sbNull.toString());
		}

		if (sbWeak.length() > 0) {
			bingo(10014, AlertItem.RISK_LOW, AlertItem.WARNING, "", "", sbWeak.toString());
		}
		
		if (sbAll.length() > 0) {
			bingo(10008, AlertItem.RISK_INFO, AlertItem.NONE, "", "", sbAll.toString());
		}			
			
	}
	
	private boolean checkDefinition(String[] def, String cipher) {
		for (int i=0; i<def.length; i++) {
			if (cipher.compareToIgnoreCase(def[i]) == 0) {
				return true;
			}
		}
		return false;
	}
	
	private SSLSocket getSSLSocket(String host, int port) throws IOException {
		SSLSocket socket = null;
		if (!Global.config.isUseProxyChain() || Global.config.isSkipProxyChain(host)) {
			socket = Global.ssl.clientNoHandshake(host, port, false);
		} else {
			socket = Global.ssl.clientViaProxyNoHandshake(host, port, Global.config.getProxyChainName(), Global.config.getProxyChainPort(), false);
		}
		return socket;
	}
	
	private void checkCipher(String host, int port, String suite, Vector supportedList) {
		String suites[] = new String[1];
		suites[0] = suite;
		SSLSocket socket = null;
		
		try {
			socket = getSSLSocket(host, port);
			
			socket.setEnabledCipherSuites(suites);
			socket.startHandshake();
			supportedList.add(suite);
			socket.close();
//			System.out.println(suite);
		} catch (Exception e) {
//			e.printStackTrace();
		}
		
	}
}