/*
Paros and its related class files.
Paros is an HTTP/HTTPS proxy for assessing web application security.
Copyright (C) 2003-2004 www.proofsecure.com

This program is free software; you can redistribute it and/or
modify it under the terms of the Clarified Artistic License
as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
Clarified Artistic License for more details.

You should have received a copy of the Clarified Artistic License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
package com.proofsecure.paros.scan;

import java.io.File;
import java.util.Iterator;
import java.util.TreeSet;

import com.proofsecure.paros.Global;
import com.proofsecure.paros.util.Util;

public abstract class AbstractScanner implements Runnable {

	protected static final String CRLF = "\r\n";
	protected static java.text.DecimalFormat decFormat = new java.text.DecimalFormat("##0.###");
	protected ParsedEntity	mStartEntity	= null;
	private Thread 			myThread		= null;
	protected boolean		mIsStarted		= false;
	protected boolean		mIsSignalStop	= false;
	protected AbstractTest 	mTestModel		= null;
	static Analyser	mAnalyser		= null;
	protected AlertDB		mAlertDB = new AlertDB();
	protected TreeSet			mAlertList = new TreeSet();
	
	public AbstractScanner() throws Exception {
		try {
			mAlertDB.read("alerts.xml");
		} catch (Exception e) {
			System.out.println("Error reading alerts.xml.  Please make sure it is in the program path.");
			e.printStackTrace();
			throw e;
		}

	}

	public void startScan(ParsedEntity entity) {
		if (mIsStarted) {
			return;
		}
		
		Global.currentScanner = this;
		
		removeAllAlert();
					
		// create new thread for scanning to avoid freezing the GUI
		mStartEntity	= entity;
		myThread	= new Thread(this);
		myThread.setDaemon(true);
		myThread.start();
		mIsStarted	= true;
	}
	
	public void stopScan() {
		if (!mIsStarted) {
			return;
		}
		
		mIsSignalStop = true;
		if (mTestModel != null) {
			mTestModel.stop();
		}
		
	}

	public boolean isStarted() {
		return mIsStarted;
	}

	public int getScannerThread() {
		return Global.config.getScannerThread();
	}

	/**
	Run scans from selected node
	*/
	public void run() {
	
		long	startTime	= 0,
				diffTime 	= 0;

		try{					
			Global.proxy.stopServer();
			startTime = System.currentTimeMillis();
	
	
	        writeOutput("Scanning using " + getScannerThread() + " thread(s).");
			
			process();
			
			diffTime = System.currentTimeMillis() - startTime;
			String diffTimeString = decFormat.format((double) (diffTime/1000.0)) + "s";
	
			if (mIsSignalStop) {
				writeOutput("Stopping scan...");
				if (mTestModel != null) {
					mTestModel.getThreadPool().waitAllThreadComplete(15000);
				}
				//Util.sleep(15000);
				writeOutput("Scanning stopped.");
				endOfScan();
				return;
			}
			writeOutput("Scanning completed in " + diffTimeString);
			endOfScan();
			endOfProcess();
	
		} catch(Exception e){
			System.out.println(e.getMessage());
		}
	}	


	abstract protected void process() throws Exception;
	abstract protected void endOfProcess();
		
	protected void endOfScan() {
		Global.currentScanner = null;

		Global.proxy.startServer();
		mIsStarted = false;
		mIsSignalStop = false;
		writeStatus(" ");
	}	



	protected void writeOutput(String text) {
		com.proofsecure.paros.util.Util.writeOutput(text);
	}

	protected void writeStatus(String text) {	
		com.proofsecure.paros.util.Util.writeStatus(text);
	}
	
	static Analyser getAnalyser() {
		return mAnalyser;
	}
	
	public AlertDB getAlertDB() {
		return mAlertDB;
	}
	
	public synchronized void writeAlertToOutput(AlertItem alert) {
		StringBuffer sb = new StringBuffer();
		sb.append(AlertItem.MSG_RISK[alert.risk] + " risk (");
		sb.append(AlertItem.MSG_RELIABILITY[alert.reliability] + ") - " + alert.alert + CRLF);
		//sb.append("Description:\t" + alert.desc + CRLF);
		sb.append("URI:\t" + alert.uri + CRLF);
		sb.append("Parameter:\t" + alert.param + CRLF);
		//sb.append("Solution:\t" + alert.solution + CRLF);
		//sb.append("Reference:\t" + alert.reference + CRLF);
		sb.append("******");
		writeOutput(sb.toString());
	}
	
	public synchronized void writeAlert(AlertItem alert) {

		switch (alert.id) {
			// server header
			case 10010:	if (addAlertUniqueOtherInfo(alert)) {
							writeAlertToOutput(alert);
						}
						break;
			default:	addAlert(alert);
						writeAlertToOutput(alert);
		}		
	}
	
	public void generateAlertReport() {
		
		AlertItem item = null;
		
		if (!Global.isReportEnabled) {
			return;
		}
		
		//File outFile = new File(Global.DIR_PROJECT + File.separator + "rp_alert.xml");
		
	    try{

			/*    
    		if (outFile.exists()) {
      			int answer = JOptionPane.showConfirmDialog(Global.parosFrame, "Report file exists. Overwrite it?","Warning",JOptionPane.YES_NO_OPTION);
      			if (answer != JOptionPane.YES_OPTION) {        
        			// User did not click "yes"
        			return;
      			}
      		}
    
    		BufferedWriter out = new BufferedWriter(new FileWriter(outFile));
		    */
		    
		    StringBuffer sb = new StringBuffer(500);
    		sb.append("<?xml version=\"1.0\"?>");
			sb.append("<report>\r\n");
			sb.append("This report was generated at " + Report.getCurrentDateTimeString() + ".");
    		for (Iterator iterator = mAlertList.iterator(); iterator.hasNext(); ) {
				item = (AlertItem) iterator.next();
				sb.append(item.toXML());
    		}
    		sb.append("</report>");	
    		
    		//out.write(sb.toString());
			//out.close();
		    
		    //File report = Report.fileToHtml(outFile.getAbsolutePath(), Global.DIR_REPORT + File.separator + "rp_alert.xsl", Global.PATH_SCAN_REPORT);
		    File report = Report.stringToHtml(sb.toString(), Global.DIR_REPORT + File.separator + "rp_alert.xsl", Global.PATH_SCAN_REPORT);
		    
		    Util.showMessageDialog("Scanning report generated.  If it does not show up after clicking OK,\r\nplease browse the file at " + report.getAbsolutePath()); 
  
  			Report.openBrowser(report.getAbsolutePath());
  			
    	}catch(Exception ae){
      		Util.showMessageDialog("File creation error."); 
    	}
		
	}


	public synchronized void removeAllAlert() {
		if (mAlertList == null || mAlertList.size() > 0) {
			mAlertList = new TreeSet();
		}
		// clean up the scan log
		if (Global.isDumpScan){
			AbstractTest.resetDump();
		}

	}
	
	/**
	This is an internal routine for listing all alerts.  Not to be directly called.
	*/
	public synchronized void listAllAlert() {
		mAlertList = new TreeSet();
		AlertItem item = null;
		for (Iterator iterator = mAlertDB.getTable().values().iterator(); iterator.hasNext();) {
			item = (AlertItem) iterator.next();
			item = item.newInstance();
			item.setRiskReliability(AlertItem.RISK_HIGH, AlertItem.WARNING);
			writeAlert(item);
		}
		generateAlertReport();

	}
	
	/**
	Add alert irrespective of repetitiveness (ie id, uri and param)
	*/
	synchronized void addAlert(AlertItem alert) {
		mAlertList.add(alert);		
	}
	
	/**
	Add alert with different URL grouped.
	@return	true = no duplicate, added.
			false = duplicate, not added.
	*/
	synchronized boolean addAlertUniqueOtherInfo(AlertItem alert) {
		AlertItem item = null;

   		for (Iterator iterator = mAlertList.iterator(); iterator.hasNext(); ) {
			item = (AlertItem) iterator.next();
			if (alert.id == item.id && alert.otherInfo.equalsIgnoreCase(item.otherInfo)) {
				return false;
			}
		}
		
		// not duplicate
		addAlert(alert);
		return true;
	}
			

}
