/*
Paros and its related class files.
Paros is an HTTP/HTTPS proxy for assessing web application security.
Copyright (C) 2003-2004 www.proofsecure.com

This program is free software; you can redistribute it and/or
modify it under the terms of the Clarified Artistic License
as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
Clarified Artistic License for more details.

You should have received a copy of the Clarified Artistic License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
package com.proofsecure.paros.scan;
 
import java.io.IOException;
import java.net.Socket;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.proofsecure.paros.log.Dump;
import com.proofsecure.paros.network.HttpBody;
import com.proofsecure.paros.network.HttpHeader;
import com.proofsecure.paros.network.HttpInputStream;
import com.proofsecure.paros.network.HttpOutputStream;
import com.proofsecure.paros.network.HttpRequestHeader;
import com.proofsecure.paros.network.HttpResponseHeader;
import com.proofsecure.paros.network.HttpStatusCode;
import com.proofsecure.paros.util.ThreadPool;
import com.proofsecure.paros.util.Util;
import com.proofsecure.paros.Global;

abstract public class AbstractTest implements Runnable {

	protected final static String	CRLF			= "\r\n";
	protected final static int PATTERN_PARAM	= Pattern.CASE_INSENSITIVE | Pattern.MULTILINE | Pattern.DOTALL;
	protected final static int WAIT_TIME_MILLIS	= 30000;

	protected final static String	DEFAULT_HEADER	=
				"Accept: */*" + CRLF +
				"Accept-Language: en-us" + CRLF +
				"Pragma: no-cache" + CRLF + 
				"User-Agent: Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.0;)" + CRLF;

	protected	HttpRequestHeader	mReqHeader	= new HttpRequestHeader();
	protected	HttpBody			mReqBody	= new HttpBody(); 
	protected	HttpResponseHeader	mResHeader	= new HttpResponseHeader();
	protected	HttpBody			mResBody	= new HttpBody();

	protected	ThreadPool			mThreadPool	= null;

	// Note all member variables except the above shall also be initialized in method init().
	private		Socket				mSocket		= null;
	private		ParsedEntity		mCurEntity		= null;
	private		HttpInputStream		mHttpIn		= null;
	private		HttpOutputStream	mHttpOut	= null;
	private		boolean				mAbsoluteUriRequired	= false;
	private		boolean				mIsSignalStop = false;
	private 	static Dump	dumpScan = null;
	/**
	Not to be directly created by others.
	*/
	protected AbstractTest() {
	}

	/**
	 * 
	 * @return the static dumpScan for reset/restart 
	 */
	public static void resetDump(){
		dumpScan = new Dump("scan",false,true);
	}
	/**
	Abstract method to start a test.
	*/
	abstract void startTest(ParsedEntity startEntity);

	/**
	Name of the test.
	*/
	public abstract String getTestName();

	/**
	Execute test(s) in a thread.
	*/
	public void run() {

		try  {
			scan();
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			close();
		}
	}

	/**
	Signal test to stop gracefully.
	*/
	public void stop() {
		mIsSignalStop = true;
	}

	
	/**
	Scan to be performed in one thread's life.
	*/
	abstract protected void scan() throws Exception ;

	protected void writeOutput(String text) {
		Global.scanner.writeOutput(text);
	}
	
	protected void writeAlert(AlertItem alert) {
		Global.scanner.writeAlert(alert);
	}
		
	protected void writeStatus(String text) {
		Global.scanner.writeStatus(text);
	}

	protected ParsedEntity getEntity() {
		return mCurEntity;
	}

	protected HttpInputStream getHttpInputStream() {
		return mHttpIn;
	}
	
	protected HttpOutputStream getHttpOutputStream() {
		return mHttpOut;
	}

	protected HttpRequestHeader getRequestHeader() {
		return mReqHeader;
	}

	protected HttpBody getRequestBody() {
		return mReqBody;
	}

	protected HttpResponseHeader getResponseHeader() {
		return mResHeader;
	}

	protected HttpBody getResponseBody() {
		return mResBody;
	}


	protected void sendAndReceive() throws IOException {
		
		HttpRequestHeader reqHeader = new HttpRequestHeader(getRequestHeader().toString());
		//HttpRequestHeader reqHeader = getRequestHeader();
		HttpResponseHeader tmpResHeader = null;
		HttpBody tmpResBody = null;
		
		reqHeader.setHostName(getRequestHeader().getHostName());
		reqHeader.setHostPort(getRequestHeader().getHostPort());
		reqHeader.setSecure(getRequestHeader().getSecure());
		
		connect();
		
		//System.out.println(getRequestHeader().getStartLine());
		reqHeader.setAbsoluteUriRequired(mAbsoluteUriRequired);
		reqHeader.setContentLength(getRequestBody().length());

		long count = System.currentTimeMillis();
		if (Global.isDumpScan && dumpScan!=null){
			if (getRequestBody() != null && getRequestBody().length()>0)
				dumpScan.log(Dump.REQUEST_FILE, count, reqHeader.toString(), getRequestBody().toString(),(reqHeader.getSecure()?"https":"http"));
			else
				dumpScan.log(Dump.REQUEST_FILE, count, reqHeader.toString(), null, (reqHeader.getSecure()?"https":"http"));
		}
		mHttpOut.write(reqHeader);
		mHttpOut.write(getRequestBody());

		// read one response
		Thread.yield();
		tmpResHeader = (HttpResponseHeader) mHttpIn.readHeader();
		tmpResBody = mHttpIn.readBody();

		close();

//		if (mCurEntity.getUserObject()!=null && mCurEntity.getUserObject().toString().equals("ScannerOneRequest")){			
//		}
//		else{			
			// loop at most 5 times for indirect response
			for (int i=0; i<5
					&& (HttpStatusCode.isRedirection(tmpResHeader.getStatusCode())
					&& tmpResHeader.getStatusCode() != HttpStatusCode.NOT_MODIFIED); i++) {
				String location = tmpResHeader.getHeader(HttpRequestHeader.LOCATION);
	
				reqHeader.setURI(location);
				
				reqHeader.setMethod("GET");
				
				connect(reqHeader.getHostName(), reqHeader.getHostPort(), reqHeader.getSecure());
				
				mHttpOut.write(getRequestHeader());
				mHttpOut.write(getRequestBody());
	
				// read response
				Thread.yield();
				tmpResHeader = (HttpResponseHeader) mHttpIn.readHeader();
				tmpResBody = mHttpIn.readBody();
				close();
			}	
//		}
			

		mResHeader	= tmpResHeader;
		mResBody	= tmpResBody;

		if (Global.isDumpScan && dumpScan!=null){
			if (mResBody != null && mResBody.length()>0)
				dumpScan.log(Dump.RESPONSE_FILE, count, mResHeader.toString(), mResBody.toString(), (reqHeader.getSecure()?"https":"http"));
			else
			dumpScan.log(Dump.RESPONSE_FILE, count, mResHeader.toString(), null, (reqHeader.getSecure()?"https":"http"));
		}

	}

	protected void init(ParsedEntity entity) throws Exception {

		// clean up member variables
		mSocket		= null;
		mCurEntity	= null;
		mHttpIn		= null;
		mHttpOut	= null;

		// set up member variables for this instance
		mCurEntity				= entity;
		mAbsoluteUriRequired	= false;

		mReqHeader.setMessage(entity.getRequestHeader().toStringAbsoluteUri());
		mReqHeader.setHostName(entity.getHostName());
		mReqHeader.setHostPort(entity.getHostPort());
		mReqHeader.setSecure(entity.getSecure());
		mReqHeader.setHeader(HttpHeader.IF_MODIFIED_SINCE, null);
		mReqHeader.setHeader(HttpHeader.IF_NONE_MATCH, null);
		mReqBody.setBody(entity.getRequestBody().toString());
	}

	protected void close() {
		try {
			mHttpIn.close();
			mHttpOut.close();
		} catch (Exception e) {
		}
		Util.closeSocket(mSocket);
	}

	
	protected boolean matchBodyPattern(Pattern pattern, StringBuffer sb) {
		Matcher matcher = pattern.matcher(getResponseBody().toString());
		boolean result = matcher.find();
		if (result) {
			if (sb != null) {
				sb.append(matcher.group());
			}
		}
		return result;
	}

	protected boolean matchHeaderPattern(String header, Pattern pattern) {
		String val = getResponseHeader().getHeader(header);
		if (val == null) {
			return false;
		}
			
		Matcher matcher = pattern.matcher(val);
		return matcher.find();
	}
	
	private void connect() throws IOException {
		connect(mCurEntity.getHostName(), mCurEntity.getHostPort(), mCurEntity.getSecure());
	}
	
	private void connect(String hostName, int hostPort, boolean isSecure) throws IOException {
		mSocket		= Util.connect(mCurEntity.getHostName(), mCurEntity.getHostPort(), mCurEntity.getSecure());
		mSocket.setSoTimeout(WAIT_TIME_MILLIS);
		mHttpIn 	= new HttpInputStream(mSocket.getInputStream());
		mHttpOut	= new HttpOutputStream(mSocket.getOutputStream());

		if (!isSecure && Util.checkAndUseProxy(hostName)) {
			mAbsoluteUriRequired = true;
		} else {
			mAbsoluteUriRequired = false;
		}
	}
		
	
	protected boolean isFileExist(
		ParsedEntity entity, HttpRequestHeader reqHeader, HttpResponseHeader resHeader, HttpBody body) {
		return Scanner.getAnalyser().isFileExist(entity, reqHeader, resHeader, body);
	}
	
	protected boolean isSignalStop() {
		return mIsSignalStop;
	}
	
	protected void bingo(int id, int risk, int reliability, String uri, String param, String otherInfo) {

		AlertItem alert = Global.scanner.getAlertDB().getInstance(id);
		alert.setRiskReliability(risk, reliability);
		if (uri == null || uri.length() == 0) {
			uri = getRequestHeader().getURIHostPathQuery();
		}
		
		if (uri.length() > 200) {
			uri = uri.substring(200);
		}

		alert.setUri(uri);
		alert.setParam(param);
		alert.setOtherInfo(otherInfo);
		writeAlert(alert);
	}
	
	ThreadPool getThreadPool() {
		return mThreadPool;
	}
	

}
