/*
Paros and its related class files.
Paros is an HTTP/HTTPS proxy for assessing web application security.
Copyright (C) 2003-2004 www.proofsecure.com

This program is free software; you can redistribute it and/or
modify it under the terms of the Clarified Artistic License
as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
Clarified Artistic License for more details.

You should have received a copy of the Clarified Artistic License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
package com.proofsecure.paros.scan;

public class AlertItem implements Comparable  {

	public static final int RISK_INFO 	= 0;
	public static final int RISK_LOW 	= 1;
	public static final int RISK_MEDIUM = 2;
	public static final int RISK_HIGH 	= 3;

	public static final int NONE = 0;
	public static final int SUSPICIOUS = 1;
	public static final int WARNING = 2;
	
	public static final String MSG_RISK[] = {"Informational", "Low", "Medium", "High"};
	public static final String MSG_RELIABILITY[] = {"None", "Suspicious", "Warning"};
	
	public int risk = RISK_INFO;
	public int reliability = WARNING;

	public int		id = 0;
	public String 	alert = "";
	public String 	uri = "";
	public String 	desc = "";
	public String 	param = "";
	public String 	otherInfo = "";
	public String 	solution = "";
	public String	reference = "";
	
	public AlertItem(int paramID) {
		id = paramID;
	}
	
	public AlertItem(int paramID, int paramRisk, int paramReliability, String paramAlert) {
		this(paramID);
		setRiskReliability(paramRisk, paramReliability);
		setAlert(paramAlert);
	}

	public void setRiskReliability(int paramRisk, int paramReliability) {
		risk = paramRisk;
		reliability = paramReliability;
	}
	
	public void setAlert(String paramAlert) {
		alert = new String(paramAlert);
	}
	
	

	public void setDetail(String paramDesc, String paramUri, String paramParam, String paramOtherInfo, String paramSolution, String paramReference) {
		setDesc(paramDesc);
		setUri(paramUri);
		setParam(paramParam);
		setOtherInfo(paramOtherInfo);
		setSolution(paramSolution);
		setReference(paramReference);
	}

	public void setUri(String paramUri) {
		uri = new String(paramUri);
	}
	
	
	public void setDesc(String paramDesc) {
		desc = new String(paramDesc);
	}
	
	public void setParam(String paramParam) {
		param = new String(paramParam);
	}
	
	
	public void setOtherInfo(String paramOtherInfo) {
		otherInfo = new String(paramOtherInfo);
	}

	public void setSolution(String paramSolution) {
		solution = new String(paramSolution);
	}

	public void setReference(String paramReference) {
		reference = new String(paramReference);
	}

	public int compareTo(Object o2) throws ClassCastException {
		AlertItem alert2 = (AlertItem) o2;
		int result = 0;
		
		if (risk < alert2.risk) {
			return -1;
		} else if (risk > alert2.risk) {
			return 1;
		}
		
		if (reliability < alert2.reliability) {
			return -1;
		} else if (reliability > alert2.reliability) {
			return 1;
		}
		
		result = alert.compareToIgnoreCase(alert2.alert);
		if (result != 0) {
			return result;
		}
		
		result = alert.compareToIgnoreCase(alert2.uri);
		if (result != 0) {
			return result;
		}
		
		result = alert.compareToIgnoreCase(alert2.param);
		if (result != 0) {
			return result;
		}
		
		return otherInfo.compareToIgnoreCase(alert2.otherInfo);
	} 


	/**
	Override equals.  Alerts are equal if the alert, uri and param is the same.
	*/
	public boolean equals(Object obj) {
		AlertItem item = null;
		if (obj instanceof AlertItem) {
			item = (AlertItem) obj;
			if ((id == item.id) && uri.equalsIgnoreCase(item.uri)
				&& param.equalsIgnoreCase(item.param) && otherInfo.equalsIgnoreCase(item.otherInfo)) {
				return true;
			}
		}
		return false;
	}
	
	/**
	Create a new instance of AlertItem with same members.
	*/
	public AlertItem newInstance() {
		AlertItem item = new AlertItem(this.id);
		item.setRiskReliability(this.risk, this.reliability);
		item.setAlert(this.alert);
		item.setDetail(this.desc, this.uri, this.param, this.otherInfo, this.solution, this.reference);
		return item;
	}
	
	public String toXML() {
		StringBuffer sb = new StringBuffer(150);
		sb.append("<alertitem>\r\n");
		sb.append("<alertid>" + id + "</alertid>");
		sb.append("<alert>" + alert + "</alert>");
		sb.append("<riskcode>" + risk + "</riskcode>");
		sb.append("<reliability>" + reliability + "</reliability>");
		sb.append("<riskdesc>" + replaceEntity(MSG_RISK[risk] + " (" + MSG_RELIABILITY[reliability] + ")") + "</riskdesc>");
		sb.append("<desc>" + paragraph(replaceEntity(desc)) + "</desc>");
		sb.append("<uri>" + paragraph(replaceEntity(uri)) + "</uri>");
		sb.append("<param>" + paragraph(replaceEntity(param)) + "</param>");
		sb.append("<otherinfo>" + paragraph(replaceEntity(otherInfo)) + "</otherinfo>");
		sb.append("<solution>" + paragraph(replaceEntity(solution)) + "</solution>");
		sb.append("<reference>" + paragraph(replaceEntity(reference)) + "</reference>");
		
		sb.append("</alertitem>\r\n");
		return sb.toString();
	}
	
	private String replaceEntity(String text) {
		String result = null;
		if (text != null) {
			result = text;
			
			result = Report.entityEncode(text);
		}
		return result;
	}
	
	private String paragraph(String text) {
		String result = null;
		result = text.replaceAll("\\r\\n","<br/>").replaceAll("\\n","<br/>");
		return result;
	}
		
}	


