/************************************************************************/
/*									*/
/* Sentry - Port Scan Detector						*/
/*									*/
/* Author: Craig H. Rowland <crowland@psionic.com> 			*/
/*			    <crowland@bipolar.net>			*/
/* Created: 10-12-97							*/
/*									*/
/* This software is Copyright(c) 1997-98 Craig H. Rowland		*/
/*									*/
/* This software is free to use provided the enclosed DISCLAIMER is     */
/* read and adhered to.							*/
/*									*/
/* Use of this software in a commercial product offering is prohibited.	*/
/* Code changes are permitted for PERSONAL USE ONLY and may not be 	*/
/* distributed.								*/
/* Send all changes/modifications/bugfixes to the above addresses.	*/
/*									*/
/* This software is provided AS-IS and has no warranty or guarantee of  */
/* any type. Use at your own risk.					*/
/*									*/
/*$Id: sentry.c,v 1.2 1998/03/11 04:24:03 crowland Exp crowland $*/
/************************************************************************/


#include "sentry.h"
#include "sentry_config.h"


/* global variable for state engine */
char scanDetectHost[MAXSTATE][IPMAXBUF];
int scanDetectCount=0;

/* logging level */
int logLevel=0;


/* here we go */
void main(int argc, char *argv[])
{


/* check args */
   if(argc != 2)
   {
      usage();
      abacusExit(ERROR);
   }

   if((geteuid()) && (getuid()) != 0)
   {
      printf("You need to be root to run this.\n");
      abacusExit(ERROR);
   }


/* if the requested TCP port do this */
   if(strcmp(argv[1],"-tcp") == 0)
   {
      abacusStart();
      if(abacusCheckConfig() == FALSE)
      {
	 abacusLog("adminalert: ERROR: Configuration files are missing/corrupted. Shutting down.\n");
	 abacusExit(ERROR);
      }
      if(abacusDaemonSeed() == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not go into daemon mode. Shutting down.\n");
	 abacusExit(ERROR);
      }
      if(sentryModeTCP() == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not go into sentry mode. Shutting down.\n");
	 abacusExit(ERROR);
      }
   }


   /* else do a UDP port bind */
   else if(strcmp(argv[1],"-udp") == 0)
   {
      abacusStart();
      if(abacusCheckConfig() == FALSE)
      {
	 abacusLog("adminalert: ERROR: Configuration files are missing/corrupted. Shutting down.\n");
	 abacusExit(ERROR);
      }
      if(abacusDaemonSeed() == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not go into daemon mode. Shutting down.\n");
	 abacusExit(ERROR);
      }
      if(sentryModeUDP() == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not go into sentry mode. Shutting down.\n");
	 abacusExit(ERROR);
      }
   }
   else
      usage();

   abacusExit(TRUE);
}





/* This is the sentrymode for TCP sockets. It will listen and accept */
/* calls to the the specified port and then nuke the caller IP */
int sentryModeTCP(void)
{

   struct sockaddr_in client, server;
   int length, portCount=0, ports[MAXSOCKS], openSockfd[MAXSOCKS], incomingSockfd, result=TRUE;
   int count=0,scanDetectTrigger=TRUE, showBanner=FALSE;
   int selectResult=0, gotBound=FALSE;
   char logbuffer[MAXBUF], *temp, target[IPMAXBUF], bannerBuffer[MAXBUF], configToken[MAXBUF];
   fd_set selectFds;


   if((abacusConfigTokenRetrieve("TCP_PORTS", configToken)) == FALSE)
   {
      abacusLog("adminalert: ERROR: Could not read TCP_PORTS option from config file");
      abacusExit(ERROR);
   }

   /* break out the ports */
   if((temp=(char *)strtok(configToken, ",")) != NULL)
   {
      ports[0]=atoi(temp);
      for(count=1; count < MAXSOCKS; count++)
      {
	 if((temp=(char *)strtok(NULL,",")) != NULL)
	    ports[count]=atoi(temp);
	 else
	    break;
      }
      portCount=count;
   }
   else
   {
      abacusLog("adminalert: ERROR: No TCP ports supplied in config file. Aborting");
      abacusExit(ERROR);
   }

   /* read in the banner if one is given */
   if((abacusConfigTokenRetrieve("PORT_BANNER", configToken)) == TRUE)
   {
      showBanner=TRUE;
      strncpy(bannerBuffer, configToken, MAXBUF);
      bannerBuffer[MAXBUF-1]='\0';
   }


   /* setup select call */
   FD_ZERO(&selectFds);

   for(count=0; count < portCount; count++ )
   {
      snprintf(logbuffer, MAXBUF, "adminalert: Going into listen mode on TCP port: %d\n",ports[count]);
      abacusLog(logbuffer);
      if((openSockfd[count]=abacusOpenTCPSocket()) == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not open TCP socket. Aborting.\n");
	 abacusExit(ERROR);
      }

      if(abacusBindSocket(openSockfd[count], client, server, ports[count]) == ERROR)
      {
	 snprintf(logbuffer, MAXBUF, "adminalert: ERROR: could not bind TCP socket: %d. Attempting to continue\n",             	  ports[count]);
	 abacusLog(logbuffer);
      }
      else /* well we at least bound to one socket so we'll continue */
	 gotBound=TRUE;
   }

   /* if we didn't bind to anything then abort */
   if(gotBound == FALSE)
   {
      abacusLog("adminalert: ERROR: could not bind ANY TCP sockets. Shutting down.\n");
      abacusExit(ERROR);
   }


   /* length for accept() call */
   length = sizeof(client);

   /* main loop for multiplexing/resetting */
   for( ; ; )
   {
      	 /* set up select call */
	 for(count = 0; count < portCount; count++)
            FD_SET(openSockfd[count], &selectFds);

   	 /* setup the select multiplexing (blocking mode)*/
	 selectResult=select(MAXSOCKS, &selectFds, NULL, NULL, (struct timeval *)NULL);

   	 /* something blew up */
	 if(selectResult < 0)
	 {
	    abacusLog("adminalert: ERROR: select call failed. Shutting down.\n");
	    abacusExit(ERROR);
	 }
      else if(selectResult == 0)
      {
#ifdef DEBUG
	 abacusLog("Select timeout");
#endif
      }

   	 /* select is reporting a waiting socket. Poll them all to find out which */
	 else if(selectResult > 0)
	 {
	    for(count = 0; count < portCount; count++)
	    {
	       if(FD_ISSET(openSockfd[count], &selectFds))
	       {
		  incomingSockfd = accept(openSockfd[count], (struct sockaddr *)&client, &length);
		  if(incomingSockfd < 0)
		  {
		     snprintf(logbuffer, MAXBUF, "attackalert: Possible stealth scan from unknown host to TCP port: %d                               (accept failed)",ports[count]);
		     abacusLog(logbuffer);
		     break;
		  }

        	  /* copy the clients address into our buffer for nuking */
		  strncpy(target, inet_ntoa(client.sin_addr.s_addr), IPMAXBUF);
		  target[IPMAXBUF-1]='\0';
                  /* check if we should ignore this IP */
		  result = abacusNeverBlock(target);

		  if(result == ERROR)
		  {
		     snprintf(logbuffer, MAXBUF, "attackalert: ERROR: cannot open ignore file. Blocking host anyway.\n");
		     abacusLog(logbuffer);
		     result = FALSE;
		  }

		  if(result == FALSE)
		  {
                     /* report the connection */
                     /* check if they've visited before */
  		     scanDetectTrigger=checkStateEngine(target);

                     if(scanDetectTrigger == TRUE)
		     {
                           /* show the banner if one was selected */
			   if(showBanner == TRUE)
                              write(incomingSockfd, bannerBuffer, strlen(bannerBuffer));
                           /* we don't need the bonehead anymore */
			   close(incomingSockfd);
			   snprintf(logbuffer, MAXBUF, "attackalert: Connect from host: %s to TCP port: %d",
                                    target ,ports[count]);
			   abacusLog(logbuffer);
                        /* check if this target is already blocked */
                   	if(abacusIsBlocked(target) == FALSE)
			{
                           /* toast the prick */
			   if(disposeTCP(target) != TRUE)
			   {
			      snprintf(logbuffer, MAXBUF, "attackalert: ERROR: Could not block host %s !!"
                                       ,target);
			      abacusLog(logbuffer);
			   }
			   else
			      abacusWriteBlocked(target, ports[count]);
			}
			else
			{
			   snprintf(logbuffer, MAXBUF, "attackalert: Host: %s is already blocked. Ignoring",
			            target);
			   abacusLog(logbuffer);
			}
		     }
		  }
                  /* never block this person*/
		  else
		     close(incomingSockfd);

                  /* safety */
		  close(incomingSockfd);
	       }/* end if(FD_ISSET) */
	    } /* end for() */
	 } /* end else (selectResult > 0) */
   } /* end main for(; ; ) loop */

/* not reached */
close(incomingSockfd);
}





/* This is the sentrymode for UDP sockets. It will listen and accept */
/* calls to the the specified port and then nuke the caller IP */
int sentryModeUDP(void)
{
   struct sockaddr_in client, server;
      int length, ports[MAXSOCKS], openSockfd[MAXSOCKS], result=TRUE;
      int count=0, portCount=0, selectResult=0, scanDetectTrigger=0, gotBound=FALSE, showBanner=FALSE;
      char logbuffer[MAXBUF], *temp, target[IPMAXBUF], bannerBuffer[MAXBUF],configToken[MAXBUF], buffer[MAXBUF];
      fd_set selectFds;


   if((abacusConfigTokenRetrieve("UDP_PORTS", configToken)) == FALSE)
   {
      abacusLog("adminalert: ERROR: Could not read UDP_PORTS option from config file");
      abacusExit(ERROR);
   }

   /* break out the ports */
   if((temp=(char *)strtok(configToken, ",")) != NULL)
   {
      ports[0]=atoi(temp);
	 for(count=1; count < MAXSOCKS; count++)
         {
         	if((temp=(char *)strtok(NULL,",")) != NULL)
                   ports[count]=atoi(temp);
                else
                   break;
         }
      portCount=count;
   }
   else
   {
      abacusLog("adminalert: ERROR: No UDP ports supplied in config file. Aborting");
      abacusExit(ERROR);
   }

   /* read in the banner if one is given */
   if((abacusConfigTokenRetrieve("PORT_BANNER", configToken)) == TRUE)
   {
      showBanner=TRUE;
      strncpy(bannerBuffer, configToken, MAXBUF);
      bannerBuffer[MAXBUF-1]='\0';
   }

   /* setup select call */
   FD_ZERO(&selectFds);

   for(count=0; count < portCount; count++ )
   {
      snprintf(logbuffer, MAXBUF, "adminalert: Going into listen mode on UDP port: %d\n",ports[count]);
	       abacusLog(logbuffer);
      if((openSockfd[count]=abacusOpenUDPSocket()) == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not open UDP socket. Aborting\n");
	 abacusExit(ERROR);
      }
      if(abacusBindSocket(openSockfd[count], client, server, ports[count]) == ERROR)
      {
	 snprintf(logbuffer, MAXBUF, "adminalert: ERROR: could not bind UDP socket: %d. Attempting to continue\n",
                  ports[count]);         
         abacusLog(logbuffer);
      }
      else /* well we at least bound to one socket so we'll continue */
	 gotBound=TRUE;
   }

/* if we didn't bind to anything then abort */
   if(gotBound == FALSE)
   {
      abacusLog("adminalert: ERROR: could not bind ANY UDP sockets. Shutting down.\n");
	 abacusExit(ERROR);
   }


/* length for accept() call */
length = sizeof(client);

/* main loop for multiplexing/resetting */
for( ; ; )
{
   	/* set up select call */
	 for(count = 0; count < portCount; count++)
            FD_SET(openSockfd[count], &selectFds);

   	/* setup the select multiplexing (blocking mode) */
	selectResult=select(MAXSOCKS, &selectFds, NULL, NULL, (struct timeval *)NULL);

      	if(selectResult < 0)
        {
           abacusLog("adminalert: ERROR: select call failed. Shutting down.\n");
	   abacusExit(ERROR);
        }
        else if(selectResult == 0)
        {
#ifdef DEBUG
	 abacusLog("Select timeout");
#endif
        }

   	/* select is reporting a waiting socket. Poll them all to find out which */
   	else if(selectResult > 0)
        {
	 for(count = 0; count < portCount; count++)
	 {
	    if(FD_ISSET(openSockfd[count], &selectFds))
	    {
            /* here just read in one byte from the UDP socket, that's all we need to */
            /* know that this person is a jerk */
	    if(recvfrom(openSockfd[count], buffer, 1, 0, (struct sockaddr *)&client, &length) < 0)
	    {
		snprintf(logbuffer, MAXBUF, "adminalert: ERROR: could not accept incoming socket for UDP port: %d\n"
                   	 , ports[count]);
		abacusLog(logbuffer);
		break;
	    }

            /* copy the clients address into our buffer for nuking */
	    strncpy(target,inet_ntoa(client.sin_addr.s_addr), IPMAXBUF);
	    target[IPMAXBUF-1]='\0';
#ifdef DEBUG
            snprintf(logbuffer, MAXBUF, "debug: sentryModeUDP: accepted UDP connection from: %s\n", target);
		     abacusLog(logbuffer);
#endif
            /* check if we should ignore this IP */
  	    result = abacusNeverBlock(target);
	    if(result == ERROR)
	    {
		snprintf(logbuffer, MAXBUF, "attackalert: ERROR: cannot open ignore file. Blocking host anyway.\n");
		         abacusLog(logbuffer);
  	        result = FALSE;
	    }
	    if(result == FALSE)
	    {
               /* check if they've visited before */
               scanDetectTrigger=checkStateEngine(target);
               if(scanDetectTrigger == TRUE)
               {
                  /* show the banner if one was selected */
                  if(showBanner == TRUE)
                     sendto(openSockfd[count], bannerBuffer, strlen(bannerBuffer),0,
			    (struct sockaddr *)&client, length);
                  /* report the connection */
                  snprintf(logbuffer, MAXBUF, "attackalert: Connect from host: %s to UDP port: %d",
			   target ,ports[count]);
                  abacusLog(logbuffer);
                  /* check if this target is already blocked */
                  if(abacusIsBlocked(target)== FALSE)
                  {
                  /* toast the prick */
                     if(disposeUDP(target) != TRUE)
                     {
                  	snprintf(logbuffer, MAXBUF, "attackalert: ERROR: Could not block host %s !!"
			         ,target);
                        abacusLog(logbuffer);
                     }
                     else
                        abacusWriteBlocked(target, ports[count]);
                  }
                  else
		  {
                     snprintf(logbuffer, MAXBUF, "attackalert: Host: %s is already blocked. Ignoring",
			      target);
                     abacusLog(logbuffer);
                  }
               }
            }
         }/* end if(FD_ISSET) */
        } /* end for() */
   } /* end else (selectResult > 0) */
} /* end main for(; ; ) loop */

} /* end UDP sentry */




/* kill the TCP connection depending on config option */
int disposeTCP(char *target)
{
   int status=TRUE;

/* Should we ignore TCP from active response? */
   if(abacusDoBlockTCP() == TRUE)
   {
/* run external command first, hosts.deny second, dead route last */
   	if(abacusKillRunCmd(target) != TRUE)
           status=FALSE;
        else if(abacusKillHostsDeny(target) != TRUE)
           status=FALSE;
        else if(abacusKillRoute(target) != TRUE)
           status=FALSE;
   }
   else
      abacusLog("attackalert: Ignoring TCP response per configuration file setting.");
           
   return(status);
}


/* kill the UDP connection depending on config option */
int disposeUDP(char *target)
{
   int status=TRUE;


/* Should we ignore UDP from active response? */
   if(abacusDoBlockUDP() == TRUE)
   {
/* run external command first, hosts.deny second, dead route last */
	 if(abacusKillRunCmd(target) != TRUE)
            status=FALSE;
	 else if(abacusKillHostsDeny(target) != TRUE)
            status=FALSE;
	 else if(abacusKillRoute(target) != TRUE)
            status=FALSE;
   }
   else
      abacusLog("attackalert: Ignoring UDP response per configuration file setting.");

   return(status);
}


/* duh */
void usage(void)
{
   printf("Sentry - Port Scan Detector.\n");
      printf("Author: Craig H. Rowland <crowland@psionic.com>\n");
      printf("Version: %s\n\n", VERSION);
      printf("usage: sentry [-tcp -udp]\n\n");
}



/* our cheesy state engine to monitor who has connected here before */
int checkStateEngine(char *target)
{
   int count=0, scanDetectTrigger=TRUE;
      int gotOne=0, configTriggerCount=0;
      char logbuffer[MAXBUF], configToken[MAXBUF];


/* get SCAN_TRIGGER option from config */
   if((abacusConfigTokenRetrieve("SCAN_TRIGGER", configToken)) == FALSE)
   {
      abacusLog("adminalert: ERROR: Could not read SCAN_TRIGGER option from config file. Disabling SCAN DETECTION");
	 configTriggerCount=0;
   }
   else
   {
#ifdef DEBUG
      snprintf(logbuffer, MAXBUF, "debug: sentryModeTCP: retrieved SCAN_TRIGGER option: %s \n", configToken);
	 abacusLog(logbuffer);
#endif
      configTriggerCount=atoi(configToken);
   }


/* This is the rather dumb scan state engine. It maintains     */
/* an array of past hosts who triggered a connection on a port */
/* when a new host arrives it is compared against the array */
/* if it is found in the array it increments a state counter by */
/* one and checks the remainder of the array. It does this until */
/* the end is reached or the trigger value has been exceeded */
/* This would probably be better as a linked list, but for the number */
/* of hosts we are tracking this is just as good. */

   gotOne=1; /* our flag counter if we get a match */
      scanDetectTrigger = TRUE; /* set to TRUE until set otherwise */

   if(configTriggerCount > 0 ) /* only do this if they want the option */
   { /* crawl through the array looking for a previous host connect */
      for(count = 0; count < MAXSTATE; count++)
      { /* if the array has the IP address then increment the gotOne counter and */
/* check the trigger value. If it is exceeded break out of the loop and */
/* set the detecttrigger to TRUE */
	    if(strstr(scanDetectHost[count], target) != (char)NULL)
	 { /* compare the number of matches to the configured trigger value */
/* if we've exceeded we can stop this noise */
	       if(++gotOne >= configTriggerCount )
	    {
	       scanDetectTrigger = TRUE; /* got one */
#ifdef DEBUG
	       snprintf(logbuffer, MAXBUF, "debug: checkStateEngine: host: %s has exceeded trigger value: %d\n", 		                    	scanDetectHost[count], configTriggerCount);
		abacusLog(logbuffer);
#endif
	       break;
	    }
	 }
/* Ok not a match, set the trigger to false and continue on */
	    else
	    scanDetectTrigger = FALSE;
      }

/* now add the fresh meat into the state engine */
/* if our array is still less than MAXSTATE large add it to the end */
      if(scanDetectCount < MAXSTATE)
      {
	 strncpy(scanDetectHost[scanDetectCount], target, IPMAXBUF);
	    scanDetectHost[scanDetectCount][IPMAXBUF-1]='\0';
	    scanDetectCount++;
      }
      else /* otherwise tack it to the beginning and start overwriting older ones */
      {
	 scanDetectCount=0;
	    strncpy(scanDetectHost[scanDetectCount], target, IPMAXBUF);
	    scanDetectHost[scanDetectCount][IPMAXBUF-1]='\0';
      }

#ifdef DEBUG
      for(count = 0; count < MAXSTATE; count++)
      {
	 snprintf(logbuffer, MAXBUF, "debug: checkStateEngine: state engine host: %s -> position: %d Detected: %d\n",	    scanDetectHost[count], count, scanDetectTrigger);
	    abacusLog(logbuffer);
      }
#endif
/* end catch to set state if configTriggerCount == 0 */
      if(gotOne >= configTriggerCount )
	 scanDetectTrigger = TRUE;
   } /* end if(scanDetectTrigger > 0)  */

   return(scanDetectTrigger);
}


