/****************************************************************************************/
/*											*/
/* Sentry - Port Scan Detector IO Library						*/
/*											*/
/* Author: Craig H. Rowland <crowland@psionic.com> <crowland@bipolar.net>		*/
/* Created: 10-12-97									*/
/*											*/
/* This software is Copyright(c) 1997-98 Craig H. Rowland				*/
/*											*/
/* This software is free to use provided the enclosed DISCLAIMER is read and adhered	*/
/* to. Use of this software in a commercial product offering is prohibited.		*/
/* Code changes are permitted for PERSONAL USE ONLY and may not be distributed.		*/
/* Send all changes/modifications/bugfixes to the above addresses.			*/
/*											*/
/* This software is provided AS-IS and has no warranty or guarantee of any type.	*/
/*											*/
/*											*/
/* Id: $Id: sentry_io.c,v 1.2 1998/03/11 04:24:13 crowland Exp crowland $	*/
/****************************************************************************************/

#include "sentry.h"
#include "sentry_io.h"
#include "sentry_config.h"

/* Main logging function to surrogate syslog */
void abacusLog(char *logentry)
{
   openlog("abacus_sentry",LOG_PID, ABACUS_SYSLOG_FACILITY);
   syslog(ABACUS_SYSLOG_LEVEL,"%s", logentry);
   closelog();
}


void abacusExit(int status)
{
   abacusLog("adminalert: abacus is shutting down\n");
   exit(status);
}


void abacusStart(void)
{
   char logbuffer[MAXBUF];

   snprintf(logbuffer, MAXBUF, "adminalert: Sentry %s is starting.\n", VERSION);
   abacusLog(logbuffer);
}


/* The daemonizing code copied from Advanced Programming */
/* in the UNIX Environment by W. Richard Stevens with minor changes */
int abacusDaemonSeed(void)
{
   int childpid;


/* handle signals */
   signal(SIGALRM, SIG_IGN);
   signal(SIGHUP, SIG_IGN);
   signal(SIGPIPE, SIG_IGN);
   signal(SIGTERM, abacusExit);
   signal(SIGABRT,abacusExit);
   signal(SIGURG, abacusExit);

   if((childpid = fork()) < 0)
      return(ERROR);
   else if(childpid > 0)
      exit(0);

   setsid();
   chdir("/");

   /* default umask */
   umask(077);

   return(TRUE);
}




/* check hosts that should never be blocked */
int abacusNeverBlock(char *target)
{
   FILE *input;
   char buffer[MAXBUF], logbuffer[MAXBUF], configToken[MAXBUF];


   /* get ignore file name from config */
   if((abacusConfigTokenRetrieve("IGNORE_FILE", configToken)) == TRUE)
   {
#ifdef DEBUG
      snprintf(logbuffer, MAXBUF, "debug: abacusNeverBlock: Opening ignore file: %s \n", configToken);
      abacusLog(logbuffer);
#endif

      if((input=fopen(configToken,"r")) == NULL)
	 return(ERROR);
   }
   else
      return(ERROR);


#ifdef DEBUG
   snprintf(logbuffer, MAXBUF, "debug: abacusNeverBlock: Doing lookup for host: %s \n", target);
   abacusLog(logbuffer);
#endif

   while(fgets(buffer, MAXBUF, input) != NULL)
   {
      if((strstr(buffer, target)) != (char)NULL)
      {
#ifdef DEBUG
	 snprintf(logbuffer, MAXBUF, "debug: abacusNeverBlock: Host: %s found in ignore file\n", target);
	 abacusLog(logbuffer);
#endif
	 fclose(input);
	 return(TRUE);
      }
   }
#ifdef DEBUG
   snprintf(logbuffer, MAXBUF, "debug: abacusNeverBlock: Host: %s NOT found in ignore file\n", target);
   abacusLog(logbuffer);
#endif

   fclose(input);
   return(FALSE);
}


/* Make sure the config file is available */
int abacusCheckConfig(void)
{
   FILE *input;
   int status=TRUE;
   char logbuffer[MAXBUF], configToken[MAXBUF];

   /* open the config file */
   if((input=fopen(CONFIG_FILE, "r")) == NULL)
   {
      snprintf(logbuffer, MAXBUF, "adminalert: Cannot open config file: %s. Exiting\n", CONFIG_FILE);
      abacusLog(logbuffer);
      status=FALSE;
   }
   else
      fclose(input);


   if((abacusConfigTokenRetrieve("BLOCKED_FILE", configToken)) == TRUE)
   {
#ifdef DEBUG
      snprintf(logbuffer, MAXBUF, "debug: abacusCheckConfig: Removing old block file: %s \n", configToken);
      abacusLog(logbuffer);
#endif

      if((input=fopen(configToken, "w")) == NULL)
      {
	 snprintf(logbuffer, MAXBUF, "adminalert: ERROR: Cannot delete blocked file on startup: %s.\n", configToken);
	 abacusLog(logbuffer);
	 status=FALSE;
      }
      else
         fclose(input);
   }
   else
   {
      snprintf(logbuffer, MAXBUF, "adminalert: ERROR: Cannot find blocked filename in config file\n");
      abacusLog(logbuffer);
      status=FALSE;
   }

   return(status);
}


/* This writes out blocked hosts to the blocked file. It adds the hostname */
/* time stamp, and port connection that was acted on */
int abacusWriteBlocked(char *target, int port)
{
   FILE *output;
   int blockedStatus=TRUE, historyStatus=TRUE;
   char logbuffer[MAXBUF], configToken[MAXBUF];


   struct tm *tmptr;

   time_t current_time;
   current_time=time(0);
   tmptr=localtime(&current_time);


   if((abacusConfigTokenRetrieve("BLOCKED_FILE", configToken)) == TRUE)
   {
#ifdef DEBUG
      snprintf(logbuffer, MAXBUF, "debug: abacusWriteBlocked: Opening block file: %s \n", configToken);
      abacusLog(logbuffer);
#endif


      if((output=fopen(configToken, "a")) == NULL)
      {
	 snprintf(logbuffer, MAXBUF, "adminalert: ERROR: Cannot open blocked file: %s.\n", configToken);
	 abacusLog(logbuffer);
	 blockedStatus=FALSE;
      }
      else
      {

	 fprintf(output,"%ld - %d/%d/%d %d:%d:%d Host: %s Port: %d Blocked\n",
	    current_time, tmptr->tm_mon+1, tmptr->tm_mday, tmptr->tm_year,
	    tmptr->tm_hour, tmptr->tm_min, tmptr->tm_sec, target, port);
	 fclose(output);
	 blockedStatus=TRUE;
      }
   }
   else
   {
      snprintf(logbuffer, MAXBUF, "adminalert: ERROR: Cannot find blocked filename in config file\n");
      abacusLog(logbuffer);
      blockedStatus=FALSE;
   }


   /* write out to history file */
   if((abacusConfigTokenRetrieve("HISTORY_FILE", configToken)) == TRUE)
   {
#ifdef DEBUG
      snprintf(logbuffer, MAXBUF, "debug: abacusWriteBlocked: Opening history file: %s \n", configToken);
      abacusLog(logbuffer);
#endif

      if((output=fopen(configToken, "a")) == NULL)
      {
	 snprintf(logbuffer, MAXBUF, "adminalert: ERROR: Cannot open history file: %s.\n", configToken);
	 abacusLog(logbuffer);
	 historyStatus=FALSE;
      }
      else
      {

	 fprintf(output,"%ld - %d/%d/%d %d:%d:%d Host: %s Port: %d\n",
	    current_time, tmptr->tm_mon+1, tmptr->tm_mday, tmptr->tm_year,
	    tmptr->tm_hour, tmptr->tm_min, tmptr->tm_sec, target, port);
	 fclose(output);
	 historyStatus=TRUE;
      }
   }
   else
   {
      snprintf(logbuffer, MAXBUF, "adminalert: ERROR: Cannot find history filename in config file\n");
      abacusLog(logbuffer);
      historyStatus=FALSE;
   }

   if(historyStatus || blockedStatus == FALSE)
      return(FALSE);
   else
      return(TRUE);
}




/* This reads a token from the config file up to the "=" and returns the string */
/* up to the first space or NULL */
int abacusConfigTokenRetrieve(char *token, char *configToken)
{
   FILE *config;
   char logbuffer[MAXBUF], buffer[MAXBUF];
   int count=0;


   /* clear out the buffer */
   bzero(buffer, MAXBUF);

   if((config=fopen(CONFIG_FILE,"r")) == NULL)
   {
      snprintf(logbuffer, MAXBUF, "adminalert: ERROR: Cannot open config file: %s.\n", CONFIG_FILE);
      abacusLog(logbuffer);
      return(ERROR);
   }
   else
   {
#ifdef DEBUG
      snprintf(logbuffer, MAXBUF, "debug: abacusConfigTokenRetrieve: checking for token %s",token);
      abacusLog(logbuffer);
#endif
      while((fgets(buffer, MAXBUF, config)) != NULL)
      {
            /* this skips comments */
	    if(buffer[0] != '#')
	    {
#ifdef DEBUG
	       snprintf(logbuffer, MAXBUF, "debug: abacusConfigTokenRetrieve: data: %s",buffer);
	       abacusLog(logbuffer);
#endif
                  /* search for the token */
		  if(strstr(buffer, token) != (char)NULL)
		  { /* cut off the '=' and send it back */
		     strncpy(buffer ,strstr(buffer,"\"")+1, MAXBUF);
		     buffer[MAXBUF-1]='\0';
                     /* strip off unprintables/linefeeds (if any) */
                     count=0;
		     while(count < MAXBUF)
		     {
			if((isprint(buffer[count])) && buffer[count] != '"')
			   configToken[count] = buffer[count];
			else /* nullify it */
			{
			   configToken[count]='\0';
			   break;
			}
			count++;
		     }
#ifdef DEBUG
		     snprintf(logbuffer, MAXBUF, "debug: abacusConfigTokenRetrieved token: %s\n", configToken);
		     abacusLog(logbuffer);
#endif
                     configToken[MAXBUF-1]='\0';
		     fclose(config);
		     return(TRUE);
		  }
	    }
      }
      fclose(config);
      return(FALSE);
   }

}




/* This will bind a socket to a port. It works for UDP/TCP */
int abacusBindSocket(int sockfd, struct sockaddr_in client,
   struct sockaddr_in server, int port)
{
   char status[MAXBUF];
#ifdef DEBUG
   snprintf(status, MAXBUF, "debug: abacusBindSocket: Binding to port: %d\n", port);
   abacusLog(status);
#endif

   bzero((char *) &server, sizeof(server));
   server.sin_family = AF_INET;
   server.sin_addr.s_addr = htonl(INADDR_ANY);
   server.sin_port = htons(port);

   if(bind(sockfd, (struct sockaddr *)&server, sizeof(server)) < 0)
   {
#ifdef DEBUG
      abacusLog("debug: abacusBindSocket: Binding failed\n");
#endif
      return(ERROR);
   }
   else /* the bind worked so now we call listen() and party on */
   {
#ifdef DEBUG
      abacusLog("debug: abacusBindSocket: Binding successful. Doing listen\n");
#endif
      listen(sockfd, 5);
      return(TRUE);
   }
}




/* Open a TCP Socket */
int abacusOpenTCPSocket(void)
{
   int sockfd;

#ifdef DEBUG
   abacusLog("debug: abacusOpenTCPSocket: opening TCP socket\n");
#endif

   if((sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
      return(ERROR);
   else
      return(sockfd);
}




/* Open a UDP Socket */
int abacusOpenUDPSocket(void)
{
   int sockfd;

#ifdef DEBUG
   abacusLog("debug: openUDPSocket opening UDP socket\n");
#endif

   if((sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
      return(ERROR);
   else
      return(sockfd);
}


/* This will use a system() call to change the route of the target host to */
/* a dead IP address on your LOCAL SUBNET. */
int abacusKillRoute(char *target)
{
   char killString[MAXBUF], logbuffer[MAXBUF], cleanAddr[MAXBUF];
   int killStatus=ERROR, count=0;

/* get the route command to run */
   if((abacusConfigTokenRetrieve("KILL_ROUTE", killString)) == TRUE)
   {
#ifdef DEBUG
      snprintf(logbuffer, MAXBUF, "debug: abacusKillRoute: found route command: %s\n" ,killString );
      abacusLog(logbuffer);
#endif
   }
   /* option not found, (disabled), return true */
   else
   {
#ifdef DEBUG
      abacusLog("debug: KILL_ROUTE option not found. Skipping");
#endif
      return(TRUE);
   }

   /* do quick check to see if the string delimiter is present */
   if(strstr(killString, "$TARGET$") == (char)NULL)
   {
      abacusLog("adminalert: No target variable specified in KILL_ROUTE option. Skipping.\n");
      return(ERROR);
   }

   /* This should never happen as we are only dealing with an IP address as reported */
   /* by the inet_ntoa() call, but we are going to  sanitize the string anyway */
   /* numbers and '.' only please.... */
   for(count=0; count < IPMAXBUF; count++)
   {
      if(isdigit(target[count]))
	 cleanAddr[count]=target[count];
      else if(target[count] == '.')
	 cleanAddr[count]=target[count];
      else
	 cleanAddr[count]='\0';
   }


   /* replace target variable with the attacking host IP */
   if(abacusSubstString(cleanAddr, "$TARGET$", killString) == 0)
   {
      abacusLog("adminalert: No target variable specified in KILL_ROUTE option. Skipping.\n");
      return(ERROR);
   }

#ifdef DEBUG
   snprintf(logbuffer, MAXBUF, "debug: abacusKillRoute: running route command: %s\n" , killString );
   abacusLog(logbuffer);
#endif

   /* Kill the bastard and report a status */
   killStatus=system(killString);

   /* if the kill status is 127 the exec call failed in the shell */
   if(killStatus == 127)
   {
      snprintf(logbuffer, MAXBUF, "adminalert: ERROR: There was an error trying to block host (exec fail) %s"
	 ,target);
      abacusLog(logbuffer);
      return(ERROR);
   }
   /* otherwise it is something else */
   else if (killStatus < 0)
   {
      snprintf(logbuffer, MAXBUF, "adminalert: ERROR: There was an error trying to block host (system fail) %s"
	 ,target);
      abacusLog(logbuffer);
      return(ERROR);
   }
   else
   {
      /* report success */
       snprintf(logbuffer, MAXBUF, "attackalert: Host %s has been blocked via dropped route."
	        ,target);
       abacusLog(logbuffer);
       return(TRUE);
   }
}



/* This will run a specified command with TARGET as the option if one is given. */
/* use sparingly */
int abacusKillRunCmd(char *target)
{
   char killString[MAXBUF], logbuffer[MAXBUF], cleanAddr[MAXBUF];
   int killStatus=ERROR, count=0;

   /* get the command to run */
   if((abacusConfigTokenRetrieve("KILL_RUN_CMD", killString)) == TRUE)
   {
#ifdef DEBUG
      snprintf(logbuffer, MAXBUF, "debug: abacusKillRunCmd: found third party command.\n");
      abacusLog(logbuffer);
#endif
   }
   /* option not found, (disabled), return true */
   else
   {
#ifdef DEBUG
      abacusLog("debug: KILL_RUN_CMD option not found. Skipping");
#endif
      return(TRUE);
   }


   /* do quick check to see if the string delimiter is present */
   if(strstr(killString, "$TARGET$") == (char)NULL)
   {
      abacusLog("adminalert: No target variable specified in KILL_RUN_CMD option. Skipping.\n");
      return(ERROR);
   }

   /* This should never happen as we are only dealing with an IP address as reported */
   /* by the inet_ntoa() call, but we are going to  sanitize the string anyway */
   /* numbers and '.' only please.... */
   for(count=0; count < IPMAXBUF; count++)
   {
      if(isdigit(target[count]))
	 cleanAddr[count]=target[count];
      else if(target[count] == '.')
	 cleanAddr[count]=target[count];
      else
	 cleanAddr[count]='\0';
   }


   /* replace target variable with the attacking host IP */
   /* for the third party command a target is not required, but I don't */
   /* know why one would do this...*/
   abacusSubstString(cleanAddr, "$TARGET$", killString);


   /* Kill the bastard and report a status */
   killStatus=system(killString);

   /* if the kill status is 127 the exec call failed in the shell */
   if(killStatus == 127)
   {
      snprintf(logbuffer, MAXBUF, "adminalert: ERROR: There was an error trying to run command (exec fail) %s"
	 ,target);
      abacusLog(logbuffer);
      return(ERROR);
   }
   /* otherwise it is something else */
   else if (killStatus < 0)
   {
      snprintf(logbuffer, MAXBUF, "adminalert: ERROR: There was an error trying to run command (system fail) %s"
	 ,target);
      abacusLog(logbuffer);
      return(ERROR);
   }
   else
   {
      /* report success */
      snprintf(logbuffer, MAXBUF, "attackalert: External command run for host  %s."
                  ,target);
      abacusLog(logbuffer);
      return(TRUE);
   }
}


/* this function will drop the host into the TCP wrappers hosts.deny file to deny */
/* all access. The drop route metod is preferred as this stops UDP attacks as well*/
/* as TCP. You may find though that host.deny will be a more permanent home..*/
int abacusKillHostsDeny(char *target)
{

   FILE *output;
   char killString[MAXBUF], logbuffer[MAXBUF], cleanAddr[MAXBUF];
   int count=0;


   /* get wrapper form to drop in */
   if((abacusConfigTokenRetrieve("KILL_HOSTS_DENY", killString)) == TRUE)
   {
#ifdef DEBUG
      snprintf(logbuffer, MAXBUF, "debug: abacusKillHostsDeny: found hosts.deny string.\n");
      abacusLog(logbuffer);
#endif
   }
   /* option not found, (disabled), return true */
   else
   {
#ifdef DEBUG
      abacusLog("debug: KILL_HOSTS_DENY option not found. Skipping");
#endif
      return(TRUE);
   }

   /* do quick check to see if the string delimiter is present */
   if(strstr(killString, "$TARGET$") == (char)NULL)
   {
      abacusLog("adminalert: No target variable specified in KILL_HOSTS_DENY option. Skipping.\n");
      return(ERROR);
   }

   /* This should never happen as we are only dealing with an IP address as reported */
   /* by the inet_ntoa() call, but we are going to  sanitize the string anyway */
   /* numbers and '.' only please.... */
   for(count=0; count < IPMAXBUF; count++)
   {
      if(isdigit(target[count]))
	 cleanAddr[count]=target[count];
      else if(target[count] == '.')
	 cleanAddr[count]=target[count];
      else
	 cleanAddr[count]='\0';
   }


   /* replace target variable with the attacking host IP */
   if(abacusSubstString(cleanAddr, "$TARGET$", killString) == 0)
   {
      abacusLog("adminalert: No target variable specified in KILL_ROUTE option. Skipping.\n");
      return(ERROR);
   }

   /* open the ignore file */
   if((output=fopen(WRAPPER_HOSTS_DENY, "a")) == NULL)
   {
      snprintf(logbuffer, MAXBUF, "adminalert: cannot open hosts.deny file: %s for blocking."
	 ,WRAPPER_HOSTS_DENY);
      abacusLog(logbuffer);
      snprintf(logbuffer, MAXBUF, "securityalert: ERROR: There was an error trying to block host %s"
	 ,target);
      abacusLog(logbuffer);
      return(FALSE);
   }
   else
   {
      fprintf(output,"%s\n",killString);
      fclose(output);
      /* report success */
      snprintf(logbuffer, MAXBUF, "attackalert: Host %s has been blocked via wrappers."
	       ,target);
      abacusLog(logbuffer);
      return(TRUE);
   }
}


/* check if the host is already blocked */
int abacusIsBlocked(char *target)
{
   FILE *input;
   char buffer[MAXBUF],logbuffer[MAXBUF], configToken[MAXBUF];


   if((abacusConfigTokenRetrieve("BLOCKED_FILE", configToken)) == TRUE)
   {
#ifdef DEBUG
      snprintf(logbuffer, MAXBUF, "debug: abacusIsBlocked: Opening block file: %s \n", configToken);
      abacusLog(logbuffer);
#endif


      if((input=fopen(configToken, "r")) == NULL)
      {
	 snprintf(logbuffer, MAXBUF, "adminalert: ERROR: Cannot open blocked file: %s for reading. Will create.\n",	    configToken);
	 abacusLog(logbuffer);
	 return(FALSE);
      }
      else
      {
	 while(fgets(buffer, MAXBUF, input) != NULL)
	 {
	    if(strstr(buffer, target) != (char)NULL)
	    {
#ifdef DEBUG
	       snprintf(logbuffer, MAXBUF, "debug: abacusIsBlocked: Host: %s found in blocked file\n", target);
	       abacusLog(logbuffer);
#endif
	       fclose(input);
	       return(TRUE);
	    }
	 }
      }
#ifdef DEBUG
      snprintf(logbuffer, MAXBUF, "debug: abacusIsBlocked: Host: %s NOT found in blocked file\n", target);
      abacusLog(logbuffer);
#endif
   }
   fclose(input);
   return(FALSE);
}


/* This will search a string for target and replace it with victim and returns the number
   of substitutions */
int abacusSubstString(char *target, char *victim, char *string)
{
   int numberOfSubst=0, stringLength=0;
   char *tempPtr, logbuffer[MAXBUF], tempString[MAXBUF];

#ifdef DEBUG
   snprintf(logbuffer, MAXBUF, "debug: abacusSubstString: Processing string: %s", string);
   abacusLog(logbuffer);
   snprintf(logbuffer, MAXBUF, "debug: abacusSubstString: Processing search text: %s", victim);
   abacusLog(logbuffer);
   snprintf(logbuffer, MAXBUF, "debug: abacusSubstString: Processing replace text: %s", target);
   abacusLog(logbuffer);
#endif

/* string not found in target */
   if(strstr(string, victim) == (char)NULL)
      return(0);

/* String too long */
   if(strlen(string) > MAXBUF)
      return(0);

   if((tempPtr=(char *)strtok(string, " ")) == (char)NULL)
      return(0);
   else
   {
      numberOfSubst++;
      if(strstr(tempPtr,victim) != (char)NULL)
      {
	 strncpy(tempString, target, MAXBUF);
	 tempString[MAXBUF-1]='\0';
	 stringLength=strlen(target);
      }
      else
      {
	 strncpy(tempString, tempPtr, MAXBUF);
	 tempString[MAXBUF-1]='\0';
	 stringLength=strlen(tempPtr);
      }

      /* replace the space we nuked */
      strncat(tempString, " ");

      while((tempPtr=(char *)strtok(NULL," ")) != (char)NULL)
      {
	 if(stringLength > MAXBUF)
	    break;

	 strncat(tempString," ");
	 numberOfSubst++;

            /* if the token has our victim string in it, then replace, otherwise copy */
	    if(strstr(tempPtr,victim) != (char)NULL)
	    {
	       stringLength+=strlen(target);
	       strncat(tempString, target, MAXBUF);
	    }
	 else
	 {
	    stringLength+=strlen(tempPtr);
	    strncat(tempString, tempPtr, MAXBUF);
	 }
      }
      strncpy(string, tempString, MAXBUF);
      tempString[MAXBUF-1]='\0';
   }
#ifdef DEBUG
   snprintf(logbuffer, MAXBUF, "debug: abacusSubstString: Result subst string: %s", string);
   abacusLog(logbuffer);
#endif
   return(numberOfSubst);
}


/* This function will check to see if we should ignore UDP scans and not block */
/* them. This prevents DOS attacks */
int abacusDoBlockUDP(void)
{
   char configToken[MAXBUF], logbuffer[MAXBUF];

   if((abacusConfigTokenRetrieve("BLOCK_UDP", configToken)) == TRUE)
   {
#ifdef DEBUG
      snprintf(logbuffer, MAXBUF, "debug: abacusDoBlockUDP: found BLOCK_UDP string.\n");
      abacusLog(logbuffer);
#endif
   }
/* option not found, (disabled), return false */
   else
   {
#ifdef DEBUG
      abacusLog("debug: abacusDoBlockUDP: BLOCK_UDP option not found. Assuming TRUE.");
#endif
      return(FALSE);
   }

   if(configToken[0] == '0')
      return(FALSE);
   else
      return(TRUE);
}

/* This function will check to see if we should ignore TCP */
int abacusDoBlockTCP(void)
{
   char configToken[MAXBUF], logbuffer[MAXBUF];

   if((abacusConfigTokenRetrieve("BLOCK_TCP", configToken)) == TRUE)
   {
#ifdef DEBUG
      snprintf(logbuffer, MAXBUF, "debug: abacusDoBlockTCP: found BLOCK_TCP string.\n");
      abacusLog(logbuffer);
#endif
   }
/* option not found, (disabled), return false */
   else
   {
#ifdef DEBUG
      abacusLog("debug: abacusDoBlockTCP: BLOCK_TCP option not found. Assuming TRUE.");
#endif
      return(FALSE);
   }

   if(configToken[0] == '0')
      return(FALSE);
   else
      return(TRUE);
}



