//sipb_bnftools.cpp
//Copyright (C) 2003 Metalink LTD
//Author: Rodionov Sergey (seger@metalinkltd.com)
//This program is distributed under terms of GPL (see LICENSE)

#include "sipb_bnftools.h"
#include "sipb_md5.h"
#include "sipb_addfun.h"
#include <sstream>
#include <iomanip>
#include <stdexcept>
#include <iostream>
using namespace std;

//------BASE PART--------
string sipb_bnftools::hstr(string in)
{
   if (in.size()==0)
     return "";
   ostringstream out;
   out<<hex<<"%x";
   for (unsigned int i=0;i<in.size();i++)
     {
	out<<hex<<setw(2)<<setfill('0')<<(unsigned int)((unsigned char)in[i]);
	if (i<in.size()-1)
	  out<<'.';
     }
   return out.str();
}
//                                                                           
string sipb_bnftools::str(string in)
{
   return "\""+ in+ "\""; 
}
//                                                                           
string sipb_bnftools::str(int in)
{
   ostringstream out;
   out<<in;
   return "\""+out.str()+"\"";
}
//                                                                           
string sipb_bnftools::rand_nalpha(int l)
{
   if (l<=0)
     return string("");
   int size = 1+random() % l;
   return nalpha(size);
}
//                                                                           
string sipb_bnftools::nalpha(int size)
{
   if (size==0)
     return "";
   string rez;
   char c;
   for (int i=0;i<size;i++)
     {
	c=0;
	while (!isalpha(c))
	  c=random()%256;
	rez.push_back(c);
     }
   return rez;
}
//                                                                           
string sipb_bnftools::nprint(int size)
{
   if (size==0)
     return "";
   string rez;
   char c;
   for (int i=0;i<size;i++)
     {
	c=0;
	while (!isprint(c))
	  c=random()%256;
	rez.push_back(c);
     }
   return rez;
}
//                                                                            
void sipb_bnftools::create_topacket(bnf_parser&p,string rule,sipb_stpacket*pk,
				    int rand_param)
{
   p.reparse(rule, pk->pack, rand_param);
}
//                                                                           
bool sipb_bnftools::create_badrequest(string& rez,bnf_parser&pcr,bnf_parser&ppc,
				      int rand_param,int max_clen,int c_quan,
				      int ntry)
{
   for (int i=0;i<ntry;i++)
     {
	pcr.reparse("Request",rez,rand_param);
	make_change(rez,max_clen,c_quan);
	bnf_parsval tmp;
	if (!ppc.try_parse(tmp,"Request",rez))  //Ok bad packet
	  return true;
     }
   return false;
}
//                                                                           
void sipb_bnftools::make_change(string& rez,int max_clen,int c_quan)
{
   if (max_clen<=0 || c_quan<=0)
     throw logic_error("Error in sipb_bnftools::make_change");
   if (rez.size()==0)
     throw logic_error("Error in sipb_bnftools::make_change rez.size()==0");
   for (int i=0;i<c_quan;i++)
     {
	int pos =random() % rez.size();
	int clen=random() % (max_clen + 1);
	if (random() % 2) //instart garbage
	  {
	     for (int j=0;j<clen;j++)
	       rez.insert(pos,1,random() % 255);
	  }
	else
	  {
	     for (int j=0;j<clen && ((j+pos) < (int)rez.size());j++)
	       rez.at( j + pos )=random()%255;
	  }
     }
}
//                                                                           
//-----READ PART-------
//                                                                           
string sipb_bnftools::read_qstr(bnf_parsval* val)
{
   vector<bnf_parsval*> v;
   val->get("quoted-string-data",v);
   if (v.size()!=1)
     logic_error("Error in sipb_bnftools::read_qstr");
   return v[0]->get_data();
}
//                                                                           
//-------FULLSET PART-------
//                                                                           
void sipb_bnftools
  ::fullset_optval(bnf_parser&pcr, string servhost_name,int sport,
		   string username,string userhost_name,
		   int max_contlen,bool rand_set_cont)
{
   sipb_bnftools::set_rulebydef(pcr);

   sipb_bnftools::set_request(pcr,random() % (max_contlen + 1),"",
			      "Contact Expires",rand_set_cont);
   if (random()%2)  //may be to host and maxforwards>0
     {
	sipb_bnftools::set_reqline(pcr, "OPTIONS", servhost_name,"",
				   sport);
	sipb_bnftools::set_maxfor(pcr,70);
     }
   else             //may be to any and maxforwards=0
     {
	sipb_bnftools::set_randhost_reqline(pcr,"OPTIONS");
	sipb_bnftools::set_maxfor(pcr,0);
     }
   sipb_bnftools::set_to_nottag(pcr, servhost_name,"",sport);
   sipb_bnftools::set_cseq(pcr,"OPTIONS");
   sipb_bnftools::set_from(pcr,userhost_name,username,1 + random()%40000);
}
//                                                                           
//-----SET PART----------
//                                                                           
void sipb_bnftools::set_resetrules(bnf_parser&p, string rulesl,char sep)
{
   vector<string> rl;
   vec_add(rulesl,rl,sep);
   for (unsigned int i=0;i<rl.size();i++)
     p.reset_rule(rl[i]);
}
//                                                                           
void sipb_bnftools::set_rulebydef(bnf_parser&p)
{
   //TODO: make it more GENERAL
   //Accept
   p.reset_rule("Accept","\"Accept\" HCOLON " + hstr("application/sdp") +
		" [ SEMI "+hstr("level = 1")+"]" );
   
   p.reset_rule("Accept-Encoding","\"Accept-Encoding\" HCOLON "+hstr("gzip"));
   p.reset_rule("Accept-Language","\"Accept-Language\" HCOLON "+hstr("en"));
   p.reset_rule("Alert-Info",string("\"Alert-Info\" HCOLON ") + 
		hstr("<http://www.example.com/sound/moo.wav>"));
   p.reset_rule("Authentication-Info", string("\"Authentication-Info\"")+
		"HCOLON nextnonce [COMMA nonce-count]");
   p.reset_rule("Call-Info",string("\"Call-Info\" HCOLON ")+ 
		hstr("<http://www.example.com/alice/photo.jpg> ;purpose=icon,")+
		hstr("<http://www.example.com/alice/> ;purpose=info"));
   p.reset_rule("Allow",string("\"Allow\" HCOLON INVITEm COMMA ") + 
		" ACKm COMMA OPTIONSm COMMA BYEm COMMA CANCELm COMMA "+
		" REGISTERm [ COMMA " + hstr("SOME")+" ]" );
   //TODO: set random contact param
   
   p.reset_rule("Contact",string("(\"Contact\" / \"m\" ) HCOLON ")+
		hstr("<sip:seger@127.0.0.1>;q=60"));
   p.reset_rule("Error-Info","\"Error-Info\" HCOLON " +
		hstr("<sip:oblom@oblom.ru>"));
   p.reset_rule("Content-Encoding",string("( \"Content-Encoding \" / \"e\" )")+
		" HCOLON "+ hstr("gzip"));
   p.reset_rule("Content-Language","\"Content-Language\" HCOLON " +
		hstr("ru"));
   p.reset_rule("Content-Length","( \"Content-Length\" / \"l\" ) HCOLON \"0\"");
   p.reset_rule("Content-Type","( \"Content-Type\" / \"c\" ) HCOLON "+
		hstr("application/sdp"));
   //List of good headers (not need to change look to sipb_bnfrules_forcreate)
   //Good - Expires Min-Expires MIME-Version Organization
   //Good - Priority Server Subject Timestamp
   //Good - Unsupported
   p.reset_rule("Retry-After",string("\"Retry-After\" ")+ 
		" HCOLON delta-seconds [ comment ] [ SEMI retry-param ]");
   p.reset_rule("retry-param","\"duration\" EQUAL delta-seconds");
   p.reset_rule("comment","LPAREN *( ctext / quoted-pair ) RPAREN"); //set light comment
   p.reset_rule("Warning",string("\"Warning\" ")+
		"HCOLON warning-value [COMMA warning-value]");
   p.reset_rule("warn-agent"," ( hostname / IPv4address ) [ \":\" port ] / pseudonym ");
   //for extension-header
   p.reset_rule("header-name",hstr("SOMEONE"));
}
//                                                                           
void sipb_bnftools::set_message(bnf_parser&p ,string rule_name,
				string first_rule,vector<string>& nec,
				vector<string>&opt,int c_line)
{
   //Remove all nec from opt
   vector<string>::iterator it;
   for (unsigned int i=0;i<nec.size();i++)
     {
	it=find(opt.begin(),opt.end(),nec[i]);
	if (it!=opt.end())
	  opt.erase(it);
     }
   //Remove duplicate nec-nec and opt-opt
   vector<string> tmp=nec;
   sort(tmp.begin(),tmp.end());
   nec.resize(0);
   unique_copy(tmp.begin(),tmp.end(),back_inserter(nec));
   tmp=opt;
   sort(tmp.begin(),tmp.end());
   opt.resize(0);
   unique_copy(tmp.begin(),tmp.end(),back_inserter(opt));
   //Check Duplicat 
   //in nec (nec-nec nec-opt)
   //TODO: remove this part (we remove all duplicat before).
   for (unsigned int i=0;i<nec.size();i++)
     {
	it=find(nec.begin() + i + 1,nec.end(),nec[i]);
	if (it!=nec.end())
	  throw  logic_error("Error in sipb_bnftools::set_message duplicat nec-nec");
	it=find(opt.begin(),opt.end(),nec[i]);
	if (it!=opt.end())
	  throw logic_error("Error in sipb_bnftools::set_message duplicat nec-opt");
     }
   //in opt (opt-opt)
   for (unsigned int i=0;i<opt.size();i++)
     {
	it=find(opt.begin() + i + 1,opt.end(),opt[i]);
	if (it!=opt.end())
	  throw  logic_error("Error in sipb_bnftools::set_message duplicat opt-opt");
     }
   //general part
   string rul;
   int n;
   rul+=first_rule+" ";
   while (nec.size() > 0 || opt.size() > 0)
     {
	if (random() % 2)  
	  {
	     if (nec.size()>0)
	       {
		  n=random() % nec.size();
		  rul+=nec.at(n) + " CRLF ";
		  nec.erase(nec.begin() + n);
	       }
	  }
	else
	  {
	     if (opt.size()>0)
	       {
		  n=random() % opt.size();
		  rul+="[ "+opt.at(n) + "  CRLF ] ";
		  opt.erase(opt.begin() + n);
	       }
	  }
     }
   rul+=" CRLF ";
   rul+=hstr(nalpha(c_line));
   p.reset_rule(rule_name,rul);
}
//                                                                           
void sipb_bnftools::set_message(bnf_parser&p,string rule_name,
				string first_rule,int c_len,string nec_add,
				string opt_add,bool rand_cont)
{
   vector<string> nec;
   vector<string> opt;
   vec_add(nec_add,nec); 
   vec_add(opt_add,opt);
   vec_add("Via To From CSeq Call-ID Max-Forwards",nec);
   vec_add("Accept Alert-Info Allow Authentication-Info Call-Info",opt);   
   vec_add("Date Error-Info Min-Expires MIME-Version",opt);
   vec_add("Organization Priority Retry-After Server Subject Timestamp",opt);
   vec_add("Unsupported User-Agent Warning extension-header",opt);

   //content
   vec_add("Content-Disposition Content-Encoding",opt);
   vec_add("Content-Language Content-Type",opt);
   if ( random() % 2  || !rand_cont) //may add content/
     {
	vec_add("Content-Length",nec);
	set_message(p,rule_name,first_rule,nec,opt,c_len);
     }
   else
     {
	c_len=0;
	vec_add("Content-Length",opt);
	set_message(p,rule_name,first_rule,nec,opt,0);
     }
   p.reset_rule("Content-Length","( \"Content-Length \" / \"l\" )" +
		string(" HCOLON ") + str(c_len));
}
void sipb_bnftools::set_request(bnf_parser&p,int c_len,string nec_add,
				string opt_add,bool rand_cont)
{
   set_message(p,"Request","Request-Line",c_len,nec_add,opt_add,rand_cont);
}
//                                                                            
void sipb_bnftools::set_reqline(bnf_parser& p,string method,string host,
				string username, int to_port)
{
   p.reset_rule("Request-Line",hstr(method)+" SP Request-URI SP" +
		"\"SIP/2.0\"" + " CRLF ");
   
   //Create rule SIP-URI-Request
   set_sip_pr(p,"Request",host,username,to_port);
   
   p.reset_rule("Request-URI"," SIP-URI-Request ");
}
//                                                                            
void sipb_bnftools::set_randhost_reqline(bnf_parser&p,string method)
{
   p.reset_rule("Request-Line",hstr(method)+" SP Request-URI SP " +
		"\"SIP/2.0\"" + " CRLF ");
   p.add_rule("SIP-URI-Request",str("sip:")+
	      "[user \"@\"] (hostname / IPv4address) [ \":\" port ]");
   p.reset_rule("Request-URI","SIP-URI-Request");
}
//                                                                            
void sipb_bnftools::set_via(bnf_parser& p ,string host, string tr,int port)
{
   p.reset_rule("Via","( \"Via\" / \"v\" ) HCOLON via-parm");
   p.reset_rule("via-parm",string("sent-protocol")+" LWS sent-by " +
		" [ SEMI via-ttl ] [ SEMI via-branch ]");
   p.reset_rule("sent-protocol",str("SIP") + " SLASH \"2.0\" SLASH " + 
		str(tr));
   p.reset_rule("sent-by",hstr(host) + " COLON " + str(port));
}
//                                                                           
void sipb_bnftools::set_to_nottag(bnf_parser&p, string host,string username,
				  int to_port)
{
   //create new rules
   p.add_rule("display-name-To"," 1*(token LWS) / quoted-string ");
   set_sip_pr(p, "To", host, username, to_port);
   p.add_rule("addr-spec-To","SIP-URI-To");
   p.add_rule("name-addr-To","[ display-name-To ] LAQUOT addr-spec-To RAQUOT");
   
   //create general rule
   p.reset_rule("To", string("( \"To\" / \"t\" )") +" HCOLON "+ 
		" ( name-addr-To/ addr-spec-To ) ");
}
//                                                                           
void sipb_bnftools::set_to_withtag(bnf_parser&p, string host,string username,
				   int to_port)
{
   //create new rules
   p.add_rule("display-name-To"," 1*(token LWS) / quoted-string ");
   set_sip_pr(p, "To", host, username, to_port);
   p.add_rule("addr-spec-To","SIP-URI-To");
   p.add_rule("name-addr-To","[ display-name-To ] LAQUOT addr-spec-To RAQUOT");
   
   //create general rule
   p.reset_rule("To", "( \"To\" / \"t\" )  HCOLON "
		"( name-addr-To/ addr-spec-To ) SEMI \"tag\" EQUAL token");
}
//                                                                           
void sipb_bnftools::set_from(bnf_parser&p, string host,string username,
			     int to_port)
{
   //create new rules
   p.add_rule("display-name-From","1*(token LWS)/ quoted-string");
   set_sip_pr(p,"From",host,username,to_port);
   p.add_rule("addr-spec-From","SIP-URI-From");
   p.add_rule("name-addr-From",string("[ display-name-From ]")+ 
	      " LAQUOT addr-spec-From RAQUOT ");
   p.add_rule("from-param-tag"," \"tag\" EQUAL token ");
   
   p.reset_rule("From","( \"From\" / \"f\" ) HCOLON from-spec");
   p.reset_rule("from-spec",
		"( name-addr-From / addr-spec-From ) [ SEMI from-param-tag ]");
}
//                                                                           
void sipb_bnftools::set_contact(bnf_parser& p,string host,string username,
				int cont_port,bool rand_setport,
				string rule_name,int expir,
				bool rand_setexpir)
{
   string display_name="display-name-"+rule_name;
   string addr_spec="addr-spec-"+rule_name;
   string name_addr="name-addr-"+rule_name;
   string param_q="param-q-"+rule_name;
   string param_exp="param-expires-"+rule_name;
   string sip_uri="SIP-URI-"+rule_name; //  created by set_sip_pr
   
   p.add_rule(display_name,"1*(token LWS)/ quoted-string");
   set_sip_pr(p,rule_name,host,username,cont_port,rand_setport);
   p.add_rule(addr_spec,sip_uri);
   p.add_rule(name_addr,"[ "+display_name+" ]"+ 
	      " LAQUOT "+addr_spec+" RAQUOT ");
   p.add_rule(param_q,"\"q\" EQUAL qvalue"); 
   
   p.add_rule(param_exp,"\"expires\" EQUAL "+str(expir));
   if (expir<0)  //not send expires
     {
	p.add_rule(rule_name,string("(\"Contact\" / \"m\" ) HCOLON ")+
		   "( " + name_addr + " /  " + addr_spec + " ) "+
		   "[SEMI "+ param_q + " ] ");
	return;
     }
   if (rand_setexpir)
     p.add_rule(rule_name,string("(\"Contact\" / \"m\" ) HCOLON ")+
		"( " + name_addr + " /  " + addr_spec + " ) "+
		"[SEMI "+ param_q + " ] [SEMI "+param_exp +" ]");   
   else
     p.add_rule(rule_name,string("(\"Contact\" / \"m\" ) HCOLON ")+
		"( " + name_addr + " /  " + addr_spec + " ) "+
		"[SEMI "+ param_q + " ] SEMI "+param_exp);
}
//                                                                           
void sipb_bnftools::set_maxfor(bnf_parser&p,int mf)
{
   p.reset_rule("Max-Forwards",str("Max-Forwards") + "HCOLON" + str(mf));
}
//                                                                           
void sipb_bnftools::set_cseq(bnf_parser&p,string name,int val)
{
   p.add_rule("Method-CSeq",hstr(name));
   if (val<0)
     p.reset_rule("CSeq","\"CSeq\" HCOLON 1*9DIGIT LWS Method-CSeq");
   else
     p.reset_rule("CSeq","\"CSeq\" HCOLON "+str(val)+" LWS Method-CSeq");
}
//                                                                           
void sipb_bnftools::set_require(bnf_parser&p,string req)
{
   p.reset_rule("Require","\"Require\" HCOLON " + hstr(req));
}
//                                                                           
void sipb_bnftools::set_accept(bnf_parser&p,string req)
{
   p.reset_rule("Accept","\"Accept\" HCOLON " + hstr(req));
}
//                                                                           
void sipb_bnftools::set_subject(bnf_parser&p,string s)
{
   p.reset_rule("Subject","(\"Subject\" / \"s\" ) HCOLON " + hstr(s));
}
//                                                                           
void sipb_bnftools::set_conttype(bnf_parser& p, string s)
{
   p.reset_rule("Content-Type","( \"Content-Type\" / \"c\" ) HCOLON "+hstr(s));
}
//                                                                           
void sipb_bnftools::set_callid(bnf_parser&p,string val)
{
   if (val.size()>0)
     p.reset_rule("Call-ID","( \"Call-ID\" / \"i\" ) HCOLON "+hstr(val));
   else  //reset by default
     p.reset_rule("Call-ID","( \"Call-ID\" / \"i\" ) HCOLON callid");
}
//                                                                           
void sipb_bnftools::set_expires(bnf_parser&p,int expir)
{
   p.reset_rule("Expires","\"Expires\" HCOLON " + str(expir));
}
//                                                                           
void sipb_bnftools::set_contact_star(bnf_parser& p,string rule_name)
{
   p.add_rule(rule_name,"(\"Contact\" / \"m\" ) HCOLON STAR");
}
//                                                                           
void sipb_bnftools::set_sip_pr(bnf_parser&p,string prefix, string host,
			       string user_name,int port,bool rand_setport)
{
   //create rulle hostport-prefix
   
   if (port>0)
     {
	if (rand_setport)
	  p.add_rule("hostport-"+prefix , hstr(host)+" [ \":\" " +
		     str(port) + " ] ");
	else
	  p.add_rule("hostport-"+prefix , hstr(host)+"  \":\" " +
		     str(port) + "  ");
	  
     }
   else
     p.add_rule("hostport-"+prefix, hstr(host));

   //create rule SIP-UTI-prefix
   if (user_name!="")
     p.add_rule("SIP-URI-"+prefix, str("sip:") +" " + hstr(user_name) + " "+
		hstr("@")+" hostport-"+prefix);
   else
     p.add_rule("SIP-URI-"+prefix,str("sip:") +
		" hostport-"+prefix);
}
//                                                                            
bool sipb_bnftools::set_auth(bnf_parser&pcr,bnf_parsval* challenge,
			     string uri,string username,string password,
			     string method,sipb_errwarn& ew)
{
   bnf_parsval* w,*ch;
   if (!challenge->get_one("WWW-Authenticate",w))
     {	
	ew.add_error("Cant find WWW-Authenticate in 401-responce");
	return false;
     }
   if (!w->get_one("challenge",ch))
     {
	ew.add_error("Cant find challenge in WWW-Authenticate, or multi challenge");
	return false;
     }
   string realm,nonce;
   bnf_parsval* tmp;
   if (!ch->get_one("realm",tmp))
     {
	ew.add_error("Cant find real in WWW-Authenticate");
	return false;
     }
   realm=read_qstr(tmp);
   if (!ch->get_one("nonce",tmp))
     {
	ew.add_error("Cant find nonce in WWW-Authenticate");
	return false;
     }
   nonce=read_qstr(tmp);
   if (!ch->is_present("md5-str"))
     {
	ew.add_warning("Cant find MD5 - string in WWW-Authenticate, we work only with MD5");
	return false;
     }
   //TODO: change from hstr
   pcr.reset_rule("credentials",str("Digest")+" LWS digest-response ");
   pcr.add_rule("username-dr",str("username")+" EQUAL LDQUOT "+
		hstr(username)+" RDQUOT");
   pcr.add_rule("realm-dr",str("realm")+" EQUAL LDQUOT "+hstr(realm)+
		" RDQUOT");
   pcr.add_rule("nonce-dr",str("nonce")+" EQUAL LDQUOT "+hstr(nonce)+
		" RDQUOT");
   pcr.add_rule("digest-uri-dr",str("uri")+" EQUAL LDQUOT "+hstr(uri)+
		" RDQUOT");
   string resp;
   resp=sipb_md5::create_dresp(username,realm,password,method,uri,nonce);
   pcr.add_rule("dresponce-dr",str("response")+" EQUAL LDQUOT "+ hstr(resp)+
		" RDQUOT");
   pcr.reset_rule("digest-response",string("username-dr COMMA realm-dr ")+
		  "COMMA nonce-dr COMMA digest-uri-dr COMMA dresponce-dr");   
   return true;
}
//                                                                            
bool sipb_bnftools::set_resp_fromreq(bnf_parser&p,bnf_parsval* request,
				     int status_code,string status_str,
				     int c_len,bool rand_cont,
				     sipb_errwarn&ew)
{
   //set from
   bnf_parsval* pv;
   if (!request->get_one("From",pv))
     {
	ew.add_error("Can't find From field (or multi From)");
	return false;
     }
   p.add_rule("From",hstr(pv->get_data()));
   //set To (and add to-tag)
   if (!request->get_one("To",pv))
     {
	ew.add_error("Can't find To field (or multi To)");
	return false;
     }
   if (!pv->is_present("tag-param"))  //we MUST add to-tag
     p.add_rule("To",hstr(pv->get_data())+" SEMI \"tag\" EQUAL token");
   else 
     p.add_rule("To",hstr(pv->get_data()));
   //set Call-ID
   if (!request->get_one("Call-ID",pv))
     {
	ew.add_error("Can't find Call-ID (or multi Call-ID)");
	return false;
     }
   p.add_rule("Call-ID",hstr(pv->get_data()));
   //set CSeq
   if (!request->get_one("CSeq",pv))
     {
	ew.add_error("Can't find CSeq (or multi CSeq)");
	return false;
     }
   p.add_rule("CSeq",hstr(pv->get_data()));
   //set Via (may be multi)
   vector<bnf_parsval*> pvs;
   request->get("Via",pvs);
   string string_via;
   for (unsigned int i=0;i<pvs.size();i++)
     {
	string_via+=hstr(pvs[i]->get_data())+" ";
	if (i != pvs.size() - 1)
	  string_via+="CRLF  ";
     }
   p.add_rule("Via",string_via);
   
   p.add_rule("Status-Line","\"SIP/2.0\" SP "+
	      hstr(int_to_str(status_code))+" SP "+
	      hstr(status_str)+" CRLF");
   //now we set Response rule
   set_message(p,"Response","Status-Line",c_len,"","",rand_cont);
   return true;
}
//                                                                            
//---------CHECK PART------------
//                                                                            
bool sipb_bnftools::cmp_uri(bnf_parsval* v1,bnf_parsval * v2)
{
   //TODO: make remove escaped!
   //TODO: SIPS-URI
   //TODO: header can swap
   if ( ( v1->name() != "SIP-URI" ) || ( v2->name() != "SIP-URI" ))
     throw logic_error("Error in sipb_bnftools::cmp_uri unsupported uri sheme");
   
   //1. compare userinfo (case sensetivity)
   string s1,s2;
   if (v1->get_one("userinfo",s1)) //if userinfo is present
     {
	if (!v2->get_one("userinfo",s2)  || (s1 != s2) )
	  return false;
     }
   else 
     if (v2->get_one("userinfo",s2))
       return false;
   
   //2. compare hostport (case-insensitivity)
   if (!v1->get_one("hostport",s1) || !v2->get_one("hostport",s2))
     throw logic_error("in sipb_bnftools::cmp_uri not found hostport");
   if (! casecmp( s1, s2 ))
     return false;
   
   //3. compare uri parameters
   bnf_parsval* p1,*p2;
   if (!v1->get_one("uri-parameters",p1) || !v2->get_one("uri-parameters",p2))
     throw logic_error("in sipb_bnftools::cmp_uri not found uri-parameters");
   
   if (!casecmp_values(p1,p2,"transport-param-trans")|| 
       !casecmp_values(p1,p2,"user-param-param") ||
       !cmp_values(p1,p2,"Method") ||
       !casecmp_values(p1,p2,"lr-param") ||
       !casecmp_values(p1,p2,"maddr-param-host")||
       !casecmp_values(p1,p2,"ttl"))
     return false;
   //4. compare headers
   if (!casecmp_values(v1,v2,"headers"))
     return false;
   return true;
}
//                                                                            
bool sipb_bnftools::cmp_viaparm_notrecived(bnf_parsval *v1,bnf_parsval*v2)
{
   if (v1->name()!="via-parm" || v2->name()!="via-parm")
     throw logic_error("Error in sipb_bnftools::cmp_via_notrecived not via-parm");
   //1. cmp sent-protocol
   bnf_parsval*p1,*p2;
   if (!v1->get_one("sent-protocol",p1) || !v2->get_one("sent-protocol",p2))
     throw logic_error("Error in cmp_viaparm_notrecived not found sent-protocol value");
   if (!casecmp_values(p1,p2,"protocol-name")  ||
       !casecmp_values(p1,p2,"protocol-version") ||
       !casecmp_values(p1,p2,"transport"))
     return false;
   //2. cmp sent_by 
   string s1,s2;
   if (!v1->get_one("sent-by",p1) || !v2->get_one("sent-by",p2))
     throw logic_error("Error in cmp_via_notrecived not find sent-by in via-parm");
   //2.1 cmp host in sent-by
   if (!p1->get_one("host",s1) || !p2->get_one("host",s2))
     throw logic_error("Error in cmp_via_notrecived not find host in sent-by");
   if ( !casecmp( s1 , s2 ) )
     return false;
   //2.2 cmp port in sent-by
   if (p1->get_one("port",s1))
     {
	if ( !p2->get_one("port",s2) || s1 != s2)
	  return false;
     }
   else
     if (p2->get_one("port",s2))
       return false;
   //3. via-params (without recived)
   if (!cmp_values(v1,v2,"ttl") ||
       !casecmp_values(v1,v2,"via-maddr-param") ||
       !cmp_values(v1,v2,"via-branch-param") ||
       !casecmp_values(v1,v2,"via-extension"))
     return false;
   return true;
}
//                                                                            
bool sipb_bnftools::cmp_from(bnf_parsval*v1,bnf_parsval*v2)
{
   if (v1->name()!="From" || v2->name()!="From")
     throw logic_error("Error in sipb_bnftools::cmp_from not From values");
   
   if ( !cmp_addrspec(v1,v2) || 
	!cmp_values(v1,v2,"tag-param-param") ||
	!cmp_values(v1,v2,"generic-param")) 
     return false;
   return true;
}
//                                                                            
bool sipb_bnftools::cmp_to_nottag(bnf_parsval*v1,bnf_parsval*v2)
{
   if (v1->name()!="To" || v2->name()!="To")
     throw logic_error("Error in sipb_bnftools::cmp_to_notag not To values");
  
   if (!cmp_addrspec(v1,v2) || 
       !cmp_values(v1,v2,"generic-param"))
     return false;
   return true;
}
//                                                                            
bool sipb_bnftools::cmp_cseq(bnf_parsval* v1,bnf_parsval*v2)
{
   if (v1->name()!="CSeq" || v2->name()!="CSeq")
     throw logic_error("Error in sipb_bnftools::cmp_cseq no CSeq values");
   string s1,s2;
   if (!v1->get_one("CSeq-value",s1) || !v2->get_one("CSeq-value",s2))
     throw logic_error("Error in sipb_bnftools::cmp_cseq no CSeq-value values");
   if (atoi(s1.c_str()) != atoi(s2.c_str()))
     return false;
   if (!cmp_values(v1,v2,"CSeq-Method"))
     return false;
   return true;
}
//                                                                            
void sipb_bnftools::check_statuscode(bnf_parsval*pv_recv,string expect_sc,
				     sipb_errwarn& ew)
{
   string stc;
   if (!pv_recv->get_one("Status-Code",stc))
     ew.add_error("Can't find Status-Code in responce");
   if (stc!=expect_sc)
     ew.add_error("We wait "+expect_sc+" Status-Code but recive "+stc);
}
//                                                                            
bool sipb_bnftools::check_statuscode(bnf_parsval*pv_recv,string expect_sc)
{
   string stc;
   if (!pv_recv->get_one("Status-Code",stc))
     return false;
   if (stc!=expect_sc)
     return false;
   return true;
}
//                                                                            
void sipb_bnftools::check_base_req(bnf_parsval*pv_recv, bnf_parsval*pv_send,
				   sipb_errwarn&ew,sipb_errwarn&ew_gen)
{
   bnf_parsval *ps,*pr;
   //0. Read Status-Code (we need read status, for To-tag check)
   int status_code=0;
   string status_code_str;
   if (!pv_recv->get_one("Status-Code",status_code_str))
     ew.add_error("Can't find Status-Code");
   status_code=atoi(status_code_str.c_str());
   if (status_code==0)  //bad status code
     ew.add_error("Bad Status-Code="+status_code_str);
   //1. Check SIP-Version
   if ( !pv_recv->get_one("SIP-Version",pr) )
     ew.add_error("Can't find SIP-Version in Response");
   else
     if ( !casecmp(pr->get_data(),"sip/2.0") )
       ew.add_error("SIP-Version not sip/2.0");
   //2. Check From
   if ( !pv_send->get_one("From",ps) )
     {
	ew_gen.add_error("Can't find From in send packet, or multi From ");
	if (pv_send->is_present("From"))
	  ew_gen.add_error("BUT PRESENT");
     }
   else if ( !pv_recv->get_one("From",pr) )
     ew.add_error("Can't find From in response packet, or multi From (probably bad grammar)");
   else if ( !cmp_from(ps,pr) )
     ew.add_error("From in response not equivalent From in request but MUST be (rfc3261 8.2.6.2)");
   //3. Check Call-ID
   if ( !pv_send->get_one("Call-ID",ps) )
     ew_gen.add_error("Can't find Call-ID in send packet, or multi Call-ID");
   else if ( !pv_recv->get_one("Call-ID",pr) )
     ew.add_error("Can't find Call-ID in response packet, or multi Call-ID (probably bad grammar)");
   else if ( !cmp_values(ps,pr,"callid"))	
     ew.add_error("Call-ID in response not equivalent Call-ID in request but MUST be (rfc3261 8.2.6.2)");
   //4. Check CSeq
   if ( !pv_send->get_one("CSeq",ps) )
     ew_gen.add_error("Can't find CSeq in send packet, or multi CSeq");
   else if ( !pv_recv->get_one("CSeq",pr) )
     ew.add_error("Can't find CSeq in response packet, or multi CSeq (probably bad grammar)");
   else if ( !cmp_cseq(ps,pr) )	
     ew.add_error("CSeq in response not equivalent CSeq in request but MUST be (rfc3261 8.2.6.2)");
   //5. Check Via
   if ( !pv_send->get_one("via-parm",ps) )
     ew_gen.add_error("Can't find Via in send packet, or multi Via"); 
     //we check response! in responce must be single via
   else if ( !pv_recv->get_one("via-parm",pr) )
     ew.add_error("Can't find Via in response packet, or multi Via");
   else if ( !cmp_viaparm_notrecived(ps,pr) )	
     ew.add_error(string("Via in response not equivalent Via in request")+
		  " but MUST be (rfc3261 8.2.6.2), except recived) ");
   //6.   Check To
   if ( !pv_send->get_one("To",ps) )
     {
	ew_gen.add_error("Can't find To in send packet, or multi To");
	if (pv_send->is_present("To"))
	  ew_gen.add_error("BUT PRESENT");
     }
   else if ( !pv_recv->get_one("To",pr) )
     ew.add_error("Can't find To in response packet, or multi To (probably bad grammar)");
   else if ( !cmp_to_nottag(ps,pr) )	
     ew.add_error("To in response not equivalent To in request but MUST be (rfc3261 8.2.6.2)");
   //6.1 check add to-tag
   if ( !ps->is_present("tag-param") ) //not find To-tag in send(request)
     if (!pr->is_present("tag-param"))  //not find To-tag in recv(responce)
       if ( status_code != 100 )  //for 100 Trying we only may add to-tag
	 ew.add_error("In To-field tag-parameter not found. But UAS MUST "
		    "add a tag parameter in request (rfc3261 8.2.6.2)");
   //7 Check Content-length
   bnf_parsval* tmp_pv;
   int real_clen;
   if (!pr->get_one("message-body",tmp_pv))
     real_clen=0;
   else 
     real_clen=tmp_pv->get_data().size();
   if (!pr->get_one("Content-Length",tmp_pv))
     {
	if ( real_clen > 0 )
	  ew.add_error("Content is present but Content-length not present");
     }
   else
     {
	int clen=atoi(tmp_pv->get_data().c_str());
	if (clen!=real_clen)
	  ew.add_error("Content-Length!=real content length");
     }
}
//                                                                            
void sipb_bnftools::check_grammar(bnf_parser&fc,bnf_parser&fp,int n_rand,int n)
{
	     
   string rez;
   int len=0;
   vector<string> bad;
   map<string,bnf_rule*>::iterator it=fc.rules.begin();
   for (;it!=fc.rules.end();it++)
     {
	if (fp.is_rule_exist(it->first))
	  {
	     cout<<it->first<<endl;
	     for (int j=0;j<n;j++)
	       {
		  fc.reparse(it->first,rez,n_rand);
		  bnf_parsval val;
		  if (!fp.try_parse(val,it->first,rez))
		    {
		       cout<<it->first<<endl;
		       cout<<"------"<<endl;
		       cout<<rez<<endl;
		       cout<<"------"<<endl;
		       j=n;
		       bad.push_back(it->first);
		    }
	       }	     
	  }
	
     }
   cout<<endl<<"Bad!!!!!!"<<endl;
   for (unsigned int i=0;i<bad.size();i++)
     cout<<bad[i]<<endl;
}
//                                                                            
void sipb_bnftools::check_regcontact(bnf_parsval*pv_recv,bnf_parsval*pv_send,
				      sipb_errwarn&ew)
{
   vector<bnf_parsval*> pv_s;
   vector<bnf_parsval*> pv_r;
   pv_send->get("contact-param",pv_s);
   pv_recv->get("contact-param",pv_r);
   if (pv_s.size()==0)
     throw logic_error("Error in sipb_bnftools:::check_regcontact not find Contact in send packet");
   //check expiration
   for (unsigned int i=0;i<pv_r.size();i++)
     if (pv_r[i]->get_quan("c-p-expires")!=1)
       ew.add_error("Can't find expires parameter in Contact, "
		    "or multi expires, but registrar MUST add it "
		    "(rfc3261 10.3 item 8)");
   //get general Expires field
   string epar_gen;
   if (!pv_recv->get_one("expires-delta-seconds",epar_gen))
     epar_gen="60"; //set by default
   for (unsigned int i=0;i<pv_s.size();i++)
     {
	//read expires parameter
	string epar;
	if (!pv_s[i]->get_one("c-p-expires-ds",epar))
	  epar=epar_gen;
	//find same Contact in responce
	bnf_parsval* tmp_s;
	bnf_parsval* tmp_r=0;	
	if (!pv_s[i]->get_one("addr-spec",tmp_s))
	  throw logic_error("Error in sipb_bnftools:::check_regcontact");

	for (unsigned int j=0;j<pv_r.size() && tmp_r==0 ;j++)
	  {
	     if (!pv_r[j]->get_one("addr-spec",tmp_r))
	       throw logic_error("Error in sipb_bnftools:::check_regcontact");
	     if (!cmp_addrspec(tmp_s,tmp_r))
	       tmp_r=0;
	  }
	if (tmp_r==0)  //not find 
	  {
	     if (atoi(epar.c_str()) != 0)
	       ew.add_error("In request, in Contact header \""+
			    pv_s[i]->get_data()+"\" expires parameter != 0 but in "+
			    " response we can't find this Contact header (rfc3261 10.3)");
	  }
	else
	  {
	     if (atoi(epar.c_str()) == 0)
	       ew.add_error("In request , in Contact header \""+
			    pv_s[i]->get_data()+"\" expires parameter == 0 "+
			    "but in responce we find this Contact header (rfc3261 10.3)");
	  }
     }
}
//                                                                            
bool sipb_bnftools::cmp_addrspec_uh_port(bnf_parsval*as1,bnf_parsval*as2)
{
   if (as1->name() != "addr-spec" || as2->name() != "addr-spec")
     throw logic_error("Error in sipb_bnftools::cmp_addrspec_cononic");
   bnf_parsval *p1,*p2;
   if ( ( as1->get_one("SIP-URI",p1) && as2->get_one("SIP-URI",p2) ) ||
	( as1->get_one("SIPS-URI",p1) && as2->get_one("SIPS-URI",p2) ))
     {
	if (!cmp_values(p1,p2,"userinfo"))
	  return false;
	if (!casecmp_values(p1,p2,"hostport")) //my be port by default
	  {
	     bnf_parsval *pp1,*pp2;
	     if (!p1->get_one("hostport",pp1) || !p2->get_one("hostport",pp2))
	       throw logic_error("Error in sipb_bnftools::cmp_addrspec_cononic");
	     if (!casecmp_values(pp1,pp2,"host"))
	       return false;
	     string port;
	     if ((!pp1->get_one("port",port)) && (!pp2->get_one("port",port)))
	       throw logic_error("Error in sipb_bnftools::cmp_addrspec_cononic");
	     if (pp1->get_one("port",port))
	       if ( (pp2->get_quan("port") != 0 ) || port!="5060")
		 return false;
	     if (pp2->get_one("port",port))
	       if ((pp1->get_quan("port")!=0) || (port!="5060"))
		 return false;
	  }
	return true;
     }
   else
     throw logic_error("Error in sipb_bnftools::cmp_addrspec_cononic work only wirh SIP-URI or SIPS-URI");
}
//                                                                            
bool sipb_bnftools::find_contact_port(bnf_parsval*pv_recv, int port)
{
   vector<bnf_parsval*> v;
   pv_recv->get("contact-param",v);
   for (unsigned int i=0;i<v.size();i++)
     {
	bnf_parsval*hp;
	if (!v[i]->get_one("hostport",hp))
	  throw logic_error("Error in sipb_bnftools::find_contact_port");
	string p;
	if (hp->get_one("port",p)  && (port == atoi(p.c_str())) )
	  return true;
     }
   return false;
}
//                                                                            
void sipb_bnftools::vec_add(string s,vector<string>& v,char sep)
{
   string tmp;
   string::const_iterator it=s.begin();
   while (it!=s.end() && *it==sep) it++;
   while (it!=s.end())
     {
	if (*it==sep)
	  {
	     v.push_back(tmp);
	     tmp="";
	     while (it!=s.end() && *it==sep) it++;
	  }
	else
	  tmp.push_back(*(it++));
     }
   if (tmp!="")
     v.push_back(tmp);
}
//                                                                            
bool sipb_bnftools::casecmp(const string& s1,const string&s2)
{
   string::const_iterator it1=s1.begin();
   string::const_iterator it2=s2.begin();
   while ( it1 != s1.end() && it2 != s2.end() )
     {
	if (toupper( *it1 ) != toupper( *it2 ) )
	  return false;
	it1++;
	it2++;
     }
   return ( it1 == s1.end() ) && ( it2 == s2.end() );
}
//                                                                            
bool sipb_bnftools::cmp_values(bnf_parsval*v1,bnf_parsval*v2,string rule,
			       bool case_sence)
{
   vector<bnf_parsval*> pv1,pv2;
   v1->get(rule,pv1);
   v2->get(rule,pv2);
   if (pv1.size()!=pv2.size())
     return false;
   for (unsigned int i=0;i<pv1.size();i++)
     {
	if (case_sence)
	  {
	     if (pv1[i]->get_data() != pv2[i]->get_data())
	       return false;
	  }
	else
	  {
	     if ( !casecmp(pv1[i]->get_data(),pv2[i]->get_data()) )
	       return false;
	  }
     }
   return true;
}
//                                                                            
bool sipb_bnftools::cmp_addrspec(bnf_parsval*v1,bnf_parsval*v2)
{
   //v1 and v2 may be addr-spec or name-addr
   //1. compare display-name if present in name-addr
   //We must'nt compare display-name (20.20)
   //2. compare URI
   bnf_parsval *p1,*p2;
   if ( (v1->get_one("SIP-URI",p1) || v1->get_one("SIPS-URI",p1)) &&
	(v2->get_one("SIP-URI",p2) || v2->get_one("SIPS-URI",p2)))
     {
	if (!cmp_uri(p1,p2))
	  return false;
     }
   else
     throw logic_error("Error in cmp_addrspec Not find SIP or SIPS URS in values");
   return true;
}
//                                                                            
//                                                                            
bool sipb_bt_challenge::read(bnf_parsval* v,string& err)
{
   vector<bnf_parsval*> cl;
   v->get("challenge",cl);
   if (cl.size()!=1)
     {
	err="Not one challenge";
	return false;
     }
   if (!cl[0]->is_present("digest-cln"))
     {
	err="Not present any digest-cln";
	return false;
     }
   if (!cl[0]->is_present("realm"))
     {
	err="Not present realm";
	return false;
     }
   if (!cl[0]->is_present("nonce"))
     {
	err="Not present nonce";
	return false;
     }
   if (!cl[0]->is_present("algorithm"))
     algo=MD5; //by default
   vector<bnf_parsval*> d;
   cl[0]->get("digest-cln",d);
   for (unsigned int i=0;i<d.size();i++)  //over all digest-cln
     {
	if (d[i]->is_present("realm"))
	  _realm=sipb_bnftools::read_qstr(d[i]);
	else if (d[i]->is_present("nonce"))
	  _nonce=sipb_bnftools::read_qstr(d[i]);
	else if (d[i]->is_present("algorithm"))
	  {
	     if (d[i]->is_present("md5_str"))
	       algo=MD5;
	     else if (d[i]->is_present("md5_sess_str"))
	       algo=MD5_session;
	     else 
	       algo=other_algo;
	  }
     }
   return true;
}
