#!/usr/bin/perl
#
# $Id: snort_stat.pl,v 1.6.6.1 2000/03/08 04:29:39 yenming Exp $
# $Revision: 1.6.6.1 $
#
# snort_stat.pl is a perl script trying to generate statistical data from every
# day snort log file.
#
# USAGE: cat <snort_log> | snort_stat.pl -r -a -h
#        -r: resolve IP address to domain name
#        -a: snort alert format
#        -h: produce html output
#
# or put it in the root's crontab file:
#59      10      *       *       *       root    cat /var/log/authlog | /etc/snort_stat.pl | sendmail root
#
# $Author: yenming $
# Yen-Ming Chen, <chenym+@CMU.EDU>
# $Date: 2000/03/08 04:29:39 $
# 
# Angelos Karageorgiou, <angelos@stocktrade.gr>
# contributed the DNS resolve and cache
#
# 2000.03.06 - modifications to read snort alert file
#            - added html output option
#              Andrew R. Baker <andrewb@uab.edu>
#

use Getopt::Std; # use Getopt for options
use Socket;      # use socket for resolving domain name from IP
%HOSTS = {};     # Hash for IP <-> domain name mapping  

getopts('arh');    # get options in command line

# process whatever comes in
while (<>) {
  if ($opt_a) {
    # process data from a snort alert file
    chomp();
    # if the line is blank, go to the next one
    if ( $_ eq "" )  { next }
    # is this line an alert message
    unless ( $_ =~ /^\[\*\*\]/ ) { 
      print STDERR "Warning, file may be corrupt.\n";
      next; 
    }
    $a = <>;
    chomp($a);
    unless ( $a eq "" ) {
      # strip off the [**] from either end.
      s/(\s)*\[\*\*\](\s)*//g;
    } else { 
      print STDERR "Warning, file may be incomplete\n";
      next;
    }
    $sig = $_;
    $a =~ m/^(\d+)\/(\d+)\-(\d+)\:(\d+)\:(\d+)\.(\d+)\s([\d\.]+)[\:]*([\d]*)\s
            [\-\>]+\s([\d\.]+)[\:]*([\d]*)/ox;
         
    $month  = $1; $day   = $2;  $hour  = $3; $minute = $4;
    $second = $5; $saddr  = $7; $host = "localhost";
    $sport  = $9; $daddr = $9; $dport = $10;
 } else {
  # If this is a snort log
  if ($_ =~ m/^(\w{3})\s+(\d+)\s(\d+)\:(\d+)\:(\d+)\s([\w+\.]*)\s[\w+\/\[\d+\]]*:\s([^:|.]+):\s([\d\.]+)[\:]?([\d]*)\s[\-\>]+\s([\d\.]+)[\:]?([\d]*)/ox)
    {
      $month  = $1; $day   = $2;  $hour  = $3; $minute = $4;
      $second = $5; $host  = $6;  $sig   = $7; $saddr  = $8;
      $sport  = $9; $daddr = $10; $dport = $11;
    } else {
      next;
    }
  }
  # if the resolve switch is on
  if ($opt_r) {
    $saddr = resolve($saddr);
    $daddr = resolve($daddr);
  }

  # put those data into a big matrix
  push @result ,[$month,$day,$hour,$minute,$second,$host,$sig,$saddr,$sport,$daddr,$dport];
  $total++;
  next;                    
} # end of snort log

# begin statistics

for $i ( 0 .. $#result ) {
  # for the same pair of attacker and victim with same sig
  # to see the attack pattern
  # used in same_attack()
  $s0{"$result[$i]->[9]:$result[$i]->[7]:$result[$i]->[6]"}++;
  # for the same pair of attacker and victim 
  # to see how many ways are being tried
  # used in same_host_dest()
  $s1{"$result[$i]->[7]:$result[$i]->[9]"}++;
  # from same host use same method to attack 
  # to see how many attacks launched from one host
  # used in same_host_sig()
  $s2{"$result[$i]->[6]:$result[$i]->[7]"}++;
  # to same victim with same method
  # to see how many attacks received by one host
  # used in same_dest_sig_stat()
  $s3{"$result[$i]->[6]:$result[$i]->[9]"}++;
  # same signature
  # to see the popularity of one attack method
  # used in attack_distribution()
  $s4{"$result[$i]->[6]"}++;
}

# begin report

print_head();
print_date();
same_attack();
same_host_dest();
same_host_sig();
same_dest_sig_stat();
attack_distribution();
print_footer();

# print the header (e.g. for mail)
sub print_head {
  if($opt_h) {
    print "<html>\n<head>\n";
    print "<title>Snort Statistics</title>";
    print "</head>\n<body>\n";
    print "<h1>Snort Statistics</h1>\n";
  } else { 
    print "Subject: snort daily report\n\n";
  }
}

# print the time of begin and end of the log
sub print_date {
  if($opt_h) {
    print "<table>\n";
    print "<tr><td>The log begins at:</td>\n";
    print "<td>$result[0]->[0] $result[0]->[1] $result[0]->[2]:$result[0]->[3]:$result[0]->[4]</td></tr>\n";
    print "<tr><td>The log ends at:</td>\n";
    print "<td>$result[$#result]->[0] $result[$#result]->[1] $result[$#result]->[2]:$result[$#result]->[3]:$result[$#result]->[4]</td></tr>\n";
    print "</table>\n";
    print "<hr>\n";
  } else {
    print "The log begins from: $result[0]->[0] $result[0]->[1] $result[0]->[2]:$result[0]->[3]:$result[0]->[4]\n";
    print "The log ends     at: $result[$#result]->[0] $result[$#result]->[1] $result[$#result]->[2]:$result[$#result]->[3]:$result[$#result]->[4]\n";
  }
}

# to see the frequency of the attack from a certain pair of 
# host and destination
sub same_attack {
  if($opt_h) {
    print "<h3>The number of attack from same host to same destination using same method</h3>\n";
    print "<table>\n";
    print "<tr><th># of attacks</th><th>from</th><th>to</th><th>with</th</tr>";
    foreach $k (sort { $s0{$b} <=> $s0{$a} } keys %s0) { 
      @_ = split ":",$k;
      print "<tr><td>$s0{$k}</td><td>$_[1]</td><td>$_[0]</td>
             <td>$_[2]</td></tr>\n" if $s0{$k} >1;
    }
    print "</table><hr>\n";
  } else {
    format SAME_ATTACK_TOP =
    
    
The number of attack from same host to same destination using same method
=========================================================================
  #  of 
 attacks        from                  to                    with
=========================================================================
.
    $~=SAME_ATTACK_TOP;
    write;
  
    foreach $k (sort { $s0{$b} <=> $s0{$a} } keys %s0) { 
      @_ = split ":",$k;
      printf("   %-2d      %-20s   %-20s %-20s\n",
              $s0{$k},$_[1],$_[0],$_[2]) if $s0{$k} >1;
    }
  }
}

# to see the percentage and number of attacks from a host to a destination
sub same_host_dest {
  if($opt_h) {
    print "<h3>Percentage and number of attacks from a host to a destination</h3>\n";
    print "<table>\n";
    print "<tr><th>%</th><th># of attacks</th><th>from</th><th>to</th></tr>\n";
    foreach $k (sort { $s1{$b} <=> $s1{$a} } keys %s1) {
      @_ = split ":",$k;
      printf("<tr><td>%-2.2f</td><td>%-2d</td><td>%-20s</td><td>%-20s</td>
              <td>\n",$s1{$k}/$total*100,$s1{$k},$_[0],$_[1]) if $s1{$k} > 1;
    }
    print "</table><hr>\n";
  } else {
    format SAME_HOST_DEST_TOP =
    
    
Percentage and number of attacks from a host to a destination
=============================================================
       #  of 
  %   attacks          from                      to             
=============================================================
.
    $~ = SAME_HOST_DEST_TOP;
    write;
  
    foreach $k (sort { $s1{$b} <=> $s1{$a} } keys %s1) {
      @_ = split ":",$k;
      printf("%-2.2f   %-2d      %-20s   %-20s\n",$s1{$k}/$total*100,$s1{$k},$_[0],$_[1]) if $s1{$k} > 1;
    }
  }
}

# to see how many attacks launched from one host
sub same_host_sig {
  if ($opt_h) {
    print "<h3>Percentage and number of attacks from one host to any with same method</h3>\n";
    print "<table>\n";
    print "<tr><th>%</th><th># of attacks</th><th>from</th><th>type</th></tr>\n";
    foreach $k (sort { $s2{$b} <=> $s2{$a} } keys %s2) {
      @_ = split ":",$k;
      printf("<tr><td>%-2.2f</td><td>%-4d</td><td>%-20s</td><td>%-28s</td>
              </tr>\n",$s2{$k}/$total*100,$s2{$k},$_[1],$_[0]) if $s2{$k} > 1;
    }
    print "</table><hr>\n";
  } else { 
    format SAME_HOST_SIG_TOP =
    
    
Percentage and number of attacks from one host to any with same method
===================================================================
       #  of 
  %   attacks           from                    type             
===================================================================
.
    $~ = SAME_HOST_SIG_TOP;
    write;
  
    foreach $k (sort { $s2{$b} <=> $s2{$a} } keys %s2) {
      @_ = split ":",$k;
      printf("%-2.2f   %-4d   %-20s    %-28s\n",$s2{$k}/$total*100,$s2{$k},$_[1],$_[0]) if $s2{$k} > 1;
    }
  }
}

# to see how many attacks received by one host
sub same_dest_sig_stat {
  if ($opt_h) {
    print "<h3>Percentage and number of attacks to one certain host</h3>\n";
    print "<table>\n";
    print "<tr><th>%</th><th># of attacks</th><th>to</th><th>type</th></tr>\n";
    foreach $k (sort { $s3{$b} <=> $s3{$a} } keys %s3) {
      @_ = split ":",$k;
      printf("<tr><td>%-2.2f</td><td>%-4d</td><td>%-25s</td><td>%-28s</td><td>\n",$s3{$k}/$total*100,$s3{$k},$_[1],$_[0]) if $s3{$k} > 1;
    }
    print "</table><hr>\n";
  } else {
    format SAME_DEST_SIG_TOP =
    
    
The percentage and number of attacks to one certain host 
===================================================================
       #  of 
  %   attacks           to                      type             
===================================================================
.
    $~ = SAME_DEST_SIG_TOP;
    write;
  
    foreach $k (sort { $s3{$b} <=> $s3{$a} } keys %s3) {
      @_ = split ":",$k;
      printf("%-2.2f   %-4d    %-25s   %-28s\n",$s3{$k}/$total*100 ,
           $s3{$k},$_[1],$_[0]) if $s3{$k} > 1;
    }
  }
}

# to see the popularity of one attack method
sub attack_distribution {
  if($opt_h) {
    print "<h3>The distribution of attack methods</h3>\n";
    print "<table>\n";
    print "<tr><th>%</th><th># of attacks</th><th>methods</th></tr>\n";
    foreach $k (sort { $s4{$b} <=> $s4{$a} } keys %s4) {
      @_ = split ":",$k;
      printf("<tr><td>%-2.2f</td><td>%-4d</td><td>%-32s</td></tr>\n",
              $s4{$k}/$total*100,$s4{$k},$_[0]);
    }
    print "</table><hr>\n";
  } else {
    format ATTACK_DISTRIBUTION_TOP =
    
    
The distribution of attack methods
===================================================================
        #  of 
  %    attacks              methods       
===================================================================
.
    $~ = ATTACK_DISTRIBUTION_TOP;
    write;
  
    foreach $k (sort { $s4{$b} <=> $s4{$a} } keys %s4) {
      @_ = split ":",$k;
      printf("%2.2f    %-4d        %-32s\n",$s4{$k}/$total*100,$s4{$k},$_[0]);
    }
  }
}

# print the footer (needed for html)
sub print_footer {
  if($opt_h) { 
    print "</body>\n</html>\n";
  } else { }
}

#
# resolve host name and cache it
# contributed by: Angelos Karageorgiou, <angelos@stocktrade.gr>
# edited by: $Author: yenming $
#
sub resolve {
  local $mname, $miaddr, $mhost = shift;
  $miaddr = inet_aton($mhost);
  # print "$mhost\n";
  if (!$HOSTS{$mhost}) {
    $mname = gethostbyaddr($miaddr, AF_INET);
    if ($mname =~ /^$/) {
      $mname = $mhost;
    }
    $HOSTS{$mhost} = $mname;
  }
  return $HOSTS{$mhost};
}
