/*
 * RAZOR imap/pop scan
 * $Id: imap.c,v 1.1 2000/11/06 15:32:06 loveless Exp $
 *
 * $Log: imap.c,v $
 * Revision 1.1  2000/11/06 15:32:06  loveless
 * Moved to sectools section of CVS
 *
 * Revision 1.12  2000/08/03 21:00:48  loveless
 * Added patrice salnot <psalnot@gpsc.fr> patch for OpenBSD
 *
 * Revision 1.11  2000/07/24 17:14:43  loveless
 * Updated Docs references to reflect changes to filenames
 *
 * Revision 1.10  2000/07/24 14:37:20  loveless
 * Fixed some warnings
 *
 * Revision 1.9  2000/07/24 11:47:27  paul
 * Added getopt.h. Compiled with -Wall and added missing close comment
 *
 * Revision 1.8  2000/07/20 19:00:17  loveless
 * Fairly major rewrite. Was failing to connection properly, in uncovering
 * the bug found some logic problems that gave false positives/negatives.
 * So rewrote most of Matt's code, tested against known vul and non-vul
 * hosts and everything is peachy.
 *
 * Revision 1.7  2000/07/18 21:06:36  loveless
 * Made sure all output went to stdout. Adjusted verbosity in a couple spots.
 *
 * Revision 1.6  2000/07/14 14:36:23  loveless
 * Made minor adjustments to output to match Docs info.
 *
 * Revision 1.5  2000/06/29 13:52:29  loveless
 * Fixed very minor display bug in a couple of printf's.
 *
 * Revision 1.4  2000/06/29 00:46:58  mconover
 * Prepended newlines to error messsages
 *
 * Revision 1.3  2000/06/29 00:44:40  mconover
 * Fixed bug with sockets (apparently need to close and reopen a socket
 * each time you want to connect to a new port)
 *
 * Revision 1.2  2000/06/28 21:13:22  loveless
 * Added -s and related stuff for integration into VLAD. Changed some stderr
 * stuff to stdout, mainly to make things easier in the vlad.pl script.
 *
 * Revision 1.1  2000/06/28 19:21:41  loveless
 * Initial rev
 *
 */

#include <stdio.h>
#include <stdlib.h>
#ifndef __OpenBSD__
#include <getopt.h>
#endif
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <ctype.h>
#include <errno.h>
#include <netdb.h>

#define ERROR -1
#define VERSION "1.1"
#define TIMEOUT 10

#define IMAP_PORT 143
#define POP2_PORT 109
#define POP3_PORT 110

int timeout = TIMEOUT;

char *host = NULL;
char quiet = 0, verbostic = 0;
char banner[64], *ptr, *version, *tmpptr;
int found_one = 0;

/* fix implicit declarations */
unsigned long int inet_addr(const char *cp);
char *inet_ntoa(struct in_addr in);

void usage(char *progname)
{
   fprintf(stdout, "Usage: %s [options] host\n\n", progname);
   fprintf(stdout, "Options are:\n"
           "-h: display this menu\n"
           "-q: suppress normal output\n"
           "-v: verbostic (detailed) output\n"
           "-t timeout: specify timeout in seconds (default = %d)\n\n",
           TIMEOUT);

   exit(0);
}

void handle_timeout(int signum)
{
   fprintf(stdout,"\nTimed out occured while waiting for a response\n");
   exit(ERROR);
}

void check_imap(void)
{
  if (verbostic)
  {
    fprintf(stdout,"banner:%c%s\n%c", ((strlen(banner) > 64) ? '\n' : ' '),
          banner, ((banner[strlen(banner)-1] == '\n') ? '\0' : '\n'));      
  }

  if ((ptr = strstr(banner, "QPOP (version ")))
  {
    ptr += 14, version = ptr; /* skip "QPOP (version " */
    ptr = strchr(ptr, '/');
    if (ptr) *ptr = '\0'; 

    if (version[0] < '2' || (version[0] == '2' && version[3] < '5'))
    {
      found_one = 1;
      if (verbostic)
      {
        fprintf(stdout,"qpopper\n-------\n\n");
        fprintf(stdout,"Description:\nA buffer overflow in PASS parsing allows remote access in qpopper 2.4 and earlier\n\n");
        fprintf(stdout,"Fix:\nUpgrade to the latest version of qpopper (http://www.eudora.com/qpopper/#CURRENT)\n\n");
        fprintf(stdout,"Related URLs:\nhttp://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-1999-0006\nhttp://www.cert.org/advisories/CA-98.08.qpopper_vul.html\n\n");
      }
      else
      {
        fprintf(stdout,"%s is potentially vulnerable to a known qpopper bug "
               "(CVE-1999-0006)\n", host);
      }
    }
  }
  else if ((ptr = strstr(banner, "IMAP4rev1")))
  {
    if ((tmpptr = strstr(banner, "IMAP4rev1 Service "))) ptr += 18;
    else ptr += 11;
    version = ptr; /* skip "IMAP4rev1 v" */

    ptr = strchr(ptr, ' ');
    if (ptr) *ptr = '\0';

    if (((isdigit(version[0])) && (!isdigit(version[1]))) || 
        (!strncmp(version, "10.", 3) && (version[3] <= '2' ||
        (version[3] == '2' && (version[4] < '3' || (version[4] == '3' &&
         version[5] < '4'))))))
    {
      if (verbostic)
      {
        fprintf(stdout,"imap\n----\n\n");
        fprintf(stdout,"Description:\nA buffer overflow in authentication allows execution of arbitrary\ncommands in IMAP4rev1 10.234 (and earlier) and imap-4.1.BETA\n\n");
        fprintf(stdout,"Fix:\nUpgrade to the latest version of imapd (ftp://ftp.cac.washington.edu/mail/imap.tar.Z)\n\n");
        fprintf(stdout,"Related URLs:\nhttp://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-1999-0005\nhttp://www.cert.org/advisories/CA-98.09.imapd.html\n\n");
      }
      else
      {
        fprintf(stdout,"%s is potentially vulnerable to a known imap bug "
                "(CVE-1999-0005)\n", host);
      }
    }
  }
}

void check_pop3(void)
{
  if (verbostic)
  {
    fprintf(stdout,"banner:%c%s\n%c", ((strlen(banner) > 64) ? '\n' : ' '),
           banner, ((banner[strlen(banner)-1] == '\n') ? '\0' : '\n'));      
  }

  if ((ptr = strstr(banner, "POP3 ")))
  {
    if((tmpptr = strstr(banner, "POP3 Server (Version "))) ptr += 21;
    else ptr += 5;  /* skip "POP3 " */

    version = ptr;
    ptr = strchr(ptr, ' ');
    if (ptr) *ptr = '\0';

    if ((version[0] == '1') || (version[0] == '2') || ((version[0] == '3') && (version[0] <= '2')))
    {
      found_one = 1;
      if (verbostic)
      {
        fprintf(stdout,"POP3\n----\n\n");
        fprintf(stdout,"Description:\nThere is a buffer overflow in the Univ. of Washington's POP/IMAP servers\n\n");
        fprintf(stdout,"Fix:\nUpgrade to the latest version of pop/imap (ftp://ftp.cac.washington.edu/mail/imap.tar.Z)\n\n");
        fprintf(stdout,"Related URLs:\nhttp://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-1999-0042\nhttp://www.cert.org/advisories/CA-97.09.imap_pop.html\n\n");
      }
      else
      {
        fprintf(stdout,"%s is potentially vulnerable to a known pop/imap bug "
                "(CVE-1999-0042)\n", host);
      }
    }
  }
  else if ((ptr = strstr(banner, "vpop ")))
  {
    ptr += 5, version = ptr; /* skip "vpop " */

    ptr = strchr(ptr, ' ');
    if (ptr) *ptr = '\0';
    if (version[0] < '3' || (version[0] == '3' && (version[2] < '4' ||
       (version[2] == '4' && (((version[4] == '1') &&
        (version[5] == '0')) || ((version[4] <= '9') && (version[5] > '9')))))))
    {
      found_one = 1;
      if (verbostic)
      {
        fprintf(stdout,"vpopmail\n--------\n\n");
        fprintf(stdout,"Description:\nA buffer overflow occurs with a long username or password (from vchkpw) in\nvpopmail 3.4.10 and earlier\n\n");
        fprintf(stdout,"Fix:\nUpgrade to the latest version of vpopmail (http://www.inter7.com/vpopmail)\n\n");
        fprintf(stdout,"Related URLs:\nhttp://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2001-0091\n\n");
      }
      else
      {
        fprintf(stdout,"%s is potentially vulnerable to a known vpopmail bug "
               "(CVE-2000-0091)\n", host);
      }
    }
  }
}

void check_pop2(void)
{
  if (verbostic)
  {
    fprintf(stdout,"banner:%c%s\n%c", ((strlen(banner) > 64) ? '\n' : ' '),
            banner, ((banner[strlen(banner)-1] == '\n') ? '\0' : '\n'));
  }

  if ((ptr = strstr(banner, "POP2 ")))
  {
    if ((tmpptr = strstr(banner, "POP2 Server (Version "))) ptr += 21;
    else ptr += 5; 
    version = ptr; /* skip "POP2 Server (Version " */

    ptr = strchr(ptr, ')');
    if (ptr) *ptr = '\0';

    if ((version[0] < '4') || ((version[0] == '4') && (version[2] <= '4')))
    {
      found_one = 1;
      if (verbostic)
      {
        fprintf(stdout,"POP2\n----\n\n");
        fprintf(stdout,"Description:\nA buffer overflow in the FOLD command allows remote access in IMAP's pop2d\n\n");
        fprintf(stdout,"Fix:\nUpgrade to the latest version of pop2d (ftp://ftp.washington.edu/mail/imap.tar.Z)\n\n");
        fprintf(stdout,"Related URLs:\nhttp://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-1999-0920\n\n");
      }
      else
      {
        fprintf(stdout,"%s is potentially vulnerable to a known pop2d bug "
               "(CVE-1999-0920)\n", host);
      }
    }
  }
}

int main(int argc, char **argv)
{
   int sockfd, sansmode = 0;

   char opt;

   struct sockaddr_in daddr;
   struct hostent *hent = NULL;

   signal(SIGALRM, handle_timeout);
   signal(SIGSEGV, SIG_IGN), signal(SIGPIPE, SIG_IGN);

   if (argc < 2) usage(argv[0]);

   while ((opt = getopt(argc, argv, "svqht")) != EOF)
      switch (opt)
      {
         case 's':
            sansmode = 1;
            break;
         case 'h': usage(argv[0]);
         case 'q': 
            quiet = 1;
            break;
         case 'v':
            verbostic = 1;
            break;

         case 't':
            if (argc < 3 || optind < 2) usage(argv[0]);

            timeout = atoi(argv[optind++]);
            break;
      }

   if (optind >= argc) usage(argv[0]);

   else if (optind && argv[optind] && argv[optind][0] != '-') 
      host = strdup(argv[optind]);

   if (!host) usage(argv[0]);

   /* conflicting options */
   if (verbostic && quiet)
   {
      fprintf(stdout,"Quiet mode and verbostic mode conflict!\n");
      free(host), exit(0);
   }

   if (sansmode)
   {
      verbostic = 0;
      quiet = 1;
   }

   if (!quiet)
   {
      fprintf(stdout,"VLAD the Scanner -- IMAP/POP checker Version %s\n"
             "http://razor.bindview.com/tools/vlad/\n\n", 
             VERSION);
   }

   if (sansmode) fprintf(stdout,"#9 - IMAP/POP vulnerabilities\n\n");
   
   if (verbostic) fprintf(stdout,"Attempting to resolve %s...\n", host);

   alarm(timeout);

   hent = gethostbyname(host);
   if (!hent)
   {
      herror("gethostbyname");
      free(host), exit(ERROR);
   }

   memcpy(&daddr.sin_addr.s_addr, hent->h_addr, sizeof(u_long));

   if (verbostic) 
      fprintf(stdout,"Resolved %s to %s\n", host, inet_ntoa(daddr.sin_addr));

   alarm(0);

   daddr.sin_family = AF_INET;

   /*
    * IMAP check
    */
   daddr.sin_port = htons(IMAP_PORT);

   sockfd = socket(AF_INET, SOCK_STREAM, 0);

   if (verbostic) 
     fprintf(stdout,"Connecting to %s (port %d) with a %d second timeout...",
                host, IMAP_PORT, timeout);

   alarm(timeout);

   if (connect(sockfd, (struct sockaddr *)&daddr, sizeof(daddr)) == ERROR)
   {
      if (verbostic) fprintf(stdout, "\nError connecting to %s (port %d): %s\n\n", host,
              IMAP_PORT, strerror(errno));

      close(sockfd);
   }
   else /* connected to IMAP_PORT */
   {
     alarm(0);
     if (verbostic) printf(" connected\n");

     if (recv(sockfd, banner, sizeof(banner)-1, 0) == ERROR)
     {
       fprintf(stderr, "error with recv(): %s\n", strerror(errno));
       close(sockfd);
     }
     else
     {
       check_imap();
       close(sockfd);
     }
   } /* end of IMAP check */

   /*
    * POP3 check
    */
   daddr.sin_port = htons(POP3_PORT);

   sockfd = socket(AF_INET, SOCK_STREAM, 0);

   if (verbostic) 
     fprintf(stdout,"Connecting to %s (port %d) with a %d second timeout...",
            host, POP3_PORT, timeout);

   alarm(timeout);

   if (connect(sockfd, (struct sockaddr *)&daddr, sizeof(daddr)) == ERROR)
   {
      if (verbostic) fprintf(stdout, "\nError connecting to %s (port %d): %s\n\n", host,
              POP3_PORT, strerror(errno));

      close(sockfd);
   }
   else /* connected to POP3_PORT */
   {
     alarm(0);

     if (verbostic) fprintf(stdout," connected\n");

     if (recv(sockfd, banner, sizeof(banner)-1, 0) == ERROR)
     {
       fprintf(stderr, "error with recv(): %s\n", strerror(errno));
       close(sockfd);
     }
     else
     {
       check_pop3();
       close(sockfd);
     }
   } /* end of POP3 check */

   /*
    * POP2 check
    */

   daddr.sin_port = htons(POP2_PORT);

   sockfd = socket(AF_INET, SOCK_STREAM, 0);

   if (verbostic)
     fprintf(stdout,"Connecting to %s (port %d) with a %d second timeout...",
             host, POP2_PORT, TIMEOUT);

   alarm(timeout);
   if (connect(sockfd, (struct sockaddr *)&daddr, sizeof(daddr)) == ERROR)
   {
      if (verbostic) fprintf(stderr, "\nError connecting to %s (port %d): %s\n", host,
              POP2_PORT, strerror(errno));
      close(sockfd);
   }
   else
   {
     alarm(0);

     if (verbostic) fprintf(stdout," connected\n");

     if (recv(sockfd, banner, sizeof(banner)-1, 0) == ERROR)
     {
       fprintf(stderr, "Error with recv(): %s\n", strerror(errno));
       close(sockfd);
     }
     else
     {
       check_pop2();
       close(sockfd);
     }
   } /* end of POP2 check */

   if (sansmode)
   {
      if (found_one) fprintf(stdout,"  Possible vulnerable IMAP/POP version found. Refer to Docs/imapandpop.html\n\n");
      else fprintf(stdout,"  No problems related to #9\n\n");
   }

   free(host);
   exit(0);
}
