#!/usr/bin/perl
use strict;

# Required CPAN modules
use Getopt::Long;
use Net::SNMP;
use Net::Telnet;
use Expect;
use File::Spec;
use Time::HiRes qw(usleep ualarm gettimeofday tv_interval);

my $version = '$Id: pwscan.pl,v 1.1 2000/11/06 15:32:06 loveless Exp $';

my $communityf = "community.db";
my $accountf = "account.db";
my $passf = "password.db";

my @communities;
my @passwords;
my %passwds;

# defaults
my $dbg = 0; # not output in usage()

my $verbosity = 0;
my $timeout = 5;
my $quiet = 0;
my $help = 0;
my $sansmode = 0;
my $fmode = 0;
my $fork = 0;
my $protofirst = 0;

my $found_one = 0;

my %check;

my %totals;
my %calls;

# parse command line
Getopt::Long::Configure('no_ignore_Case');
GetOptions(
	"v+" => \$verbosity,
	"d+" => \$dbg,
	"t=i" => \$timeout,
	"M" => \$check{'snmp'},
	"T" => \$check{'telnet'},
	"R" => \$check{'rlogin'},
	"P" => \$check{'pop3'},
	"I" => \$check{'imap'},
	"F" => \$check{'ftp'},
	"S" => \$check{'ssh'},
	"B" => \$check{'smb'},
	"q"   => \$quiet,
	"s"   => \$sansmode,
	"p"   => \$protofirst,
	"f"   => \$fmode,
	"k"   => \$fork,
	"h"   => \$help);

if ($help) {
	usage();
}

if ($sansmode) {
	$quiet = 1;
	$verbosity = 0;
        print "#8 and #10 - Default SNMP communities, default/simple account/passwords\n";
}

if (!$quiet) {
	print "RAZOR password scanner  - version: $version\n";
}

if (!defined($check{'snmp'}) && !defined($check{'telnet'}) &&
	!defined($check{'rlogin'}) && !defined($check{'pop3'}) &&
	!defined($check{'imap'}) && !defined($check{'ssh'}) &&
	!defined($check{'smb'}) && !defined($check{'ftp'})) {
		$check{'snmp'} = $check{'telnet'} = $check{'rlogin'} =
		$check{'pop3'} = $check{'imap'} = $check{'ssh'} =
		$check{'smb'} = $check{'ftp'} = 1;
}

# Check for dependent  executables
# If executable required and this option is selected, then unselect and report
if (defined($check{'smb'}) && !defined(which('smbclient'))) {
	print "Cannot locate smbclient. No SMB checks\n";
	undef $check{'smb'};
}
if (defined($check{'rlogin'}) && !defined(which('rlogin'))) {
	print "Cannot locate rlogin. No Rlogin checks\n";
	undef $check{'rlogin'};
}
if (defined($check{'ssh'}) && !defined(which('ssh'))) {
	print "Cannot locate ssh. No SSH checks\n";
	undef $check{'ssh'};
}
if (defined($check{'telnet'}) && !defined(which('telnet'))) {
	print "Cannot locate telnet. No Telnet checks\n";
	undef $check{'telnet'};
}

if ($verbosity > 2) {print "Timeout: $timeout\n";}

# read community string database
open(DB, $communityf) or die
	$communityf,": $!";
while (<DB>) {
	chomp;
	$_ =~ s/\s?(#.*)?$//; # remove comments and terminating white space
	next if /^$/;
	push @communities,$_;
}
close DB;

# read account name database
# format: name[:password]
# if no password then all default passwords will be attempted
open(DB, $accountf) or die
	$accountf,": $!";
while (<DB>) {
	my $account;
	my $password;

	chomp;
	$_ =~ s/\s?(#.*)?$//; # remove comments and terminating white space
	next if /^$/;
	($account, $password) = split /:/;
	$passwds{$account} = $password;
}
close DB;

# read common password database
open(DB, $passf) or die
	$passf,": $!";
while (<DB>) {
	chomp;
	$_ =~ s/\s?(#.*)?$//; # remove comments and terminating white space
	next if /^$/;
	push @passwords,$_;
}
close DB;

my $target = shift or usage();

print "Checking $target\n" unless $sansmode;

if (defined($check{'snmp'})) {
	if ($verbosity > 0) {print "SNMP check\n";}
	foreach my $community (@communities) {
		my $xtime = trysnmp($target, $community);
		$totals{'snmp'} += $xtime;
		$calls{'snmp'} ++;
		if ($verbosity > 1) {print "Time(snmp): $xtime\n";}
	}
}

if ($protofirst) {
	foreach my $proto ('ftp', 'rlogin', 'ssh', 'pop3', 'imap', 'telnet', 'smb') {
		if (defined($check{$proto})) {
			if (!$fork || !fork()) {
				foreach my $account (keys(%passwds)) {
					my $xtime;
					if (!defined($passwds{$account})) {
						my $pass;

						foreach $pass (@passwords) {
							# '%' represents same as username
							# '%%' represents null password
							if ($pass eq '%') { 
								$pass = $account;
							} elsif ($pass eq '%%') {
								$pass = '';
							}
							$xtime = try($proto, $target,
								$account, $pass);
							$totals{$proto} += $xtime;
							$calls{$proto} ++;
							if ($verbosity > 1) {print "Time($proto): $xtime\n";}
						}
					} else {
						$xtime = try($proto, $target, $account,
							$passwds{$account});
						$totals{$proto} += $xtime;
						$calls{$proto} ++;
						if ($verbosity > 1) {print "Time($proto): $xtime\n";}
					}
				}
			}
		}
	}
} else { # account first
	foreach my $account (keys(%passwds)) {
		my $proto;

		if (!$fork || !fork()) {
			my $xtime;
			if (!defined($passwds{$account})) {
				my $pass;

				foreach $pass (@passwords) {
					# '%' represents same as username
					# '%%' represents null password
					if ($pass eq '%') { 
						$pass = $account;
					} elsif ($pass eq '%%') {
						$pass = '';
					}
					foreach $proto ('ftp', 'rlogin', 'ssh', 'pop3', 'imap', 'telnet', 'smb') {
						if (defined($check{$proto})) {
							$xtime = try($proto, $target,
								$account, $pass);
							$totals{$proto} += $xtime;
							$calls{$proto} ++;
							if ($verbosity > 1) {print "Time($proto): $xtime\n";}
						}
					}
				}
			} else {
				foreach $proto ('ftp', 'rlogin', 'ssh', 'pop3', 'imap', 'telnet', 'smb') {
					if (defined($check{$proto})) {
						$xtime = try($proto, $target, $account,
							$passwds{$account});
						$totals{$proto} += $xtime;
						$calls{$proto} ++;
						if ($verbosity > 1) {print "Time($proto): $xtime\n";}
					}
				}
			}
		}
	}
}

if ($sansmode) {
	if ($found_one) {
		print "\n  Default account/password or common password vulnerability found\n";
		print "\n  Refer to documents listed above, as well as Docs/passwords.html or\n  Docs/SNMP.html\n\n";
	} else {
		print "\n  No problems related to #8 and #10\n\n";
	}
}

# Print totals/average times, if verbose enough
if ($verbosity > 1) {
	while ((my $key, my $value) = each %totals) {
		printf "Service = %s Time = %.2f Calls = %d Average = %.2f\n",
			$key, $value, $calls{$key}, $value/$calls{$key};
	}
}

sub trysnmp() {
	my $target = shift;
	my $community = shift;
	my $mechanism = 'snmp';

	my $t0;
	my $t1;

	$t0 = [gettimeofday];

	if ($verbosity > 0) {print "$mechanism check. Community:$community\n";}

	(my $session, my $error) = Net::SNMP->session(
		-hostname => $target,
		-timeout => $timeout,
		-community => $community);
	if (!defined($session)) {
		printf("Error: %s.\n", $error);
		exit 1;
	}

	my $sysDescr = '1.3.6.1.2.1.1.1.0';
	if (defined(my $response = $session->get_request($sysDescr))) {
		$t1 = [gettimeofday];
		report_vuln($mechanism, $target, $community,
			$response->{$sysDescr});
	}

	$session->close;
	return tv_interval($t0, $t1);
}

sub try() {
	my ($proto, $target, $account, $pass) = @_;
	for ($proto) {
		if    (/telnet/) {return trytelnet($target, $account, $pass)}
		elsif (/rlogin/) {return tryrlogin($target, $account, $pass)}
		elsif (/pop3/)   {return trypop3($target, $account, $pass)}
		elsif (/imap/)   {return tryimap($target, $account, $pass)}
		elsif (/ftp/)   {return tryftp($target, $account, $pass)}
		elsif (/smb/)   {return trysmb($target, $account, $pass)}
		elsif (/ssh/)   {return tryssh($target, $account, $pass)}
	} 
}
		
sub trytelnet() {
	my $target = shift;
	my $user = shift;
	my $pass = shift;

	my $s;
	my $line;
	my $cmd;
	my @args;

	my $t0;
	my $t1;

	my $mechanism = 'telnet';

	$t0 = [gettimeofday];

	if ($verbosity > 0) {print "$mechanism check. User:$user, pass:$pass\n";}

	$cmd = "telnet";
	@args = ($target);
	$s = Expect->spawn($cmd, @args);
	if ($dbg == 1) {$s->exp_internal(1);}
	if ($dbg > 1) {$s->debug($dbg - 1);}
	if ($verbosity < 2) {
		$s->log_stdout(0);
	}
	if ($s->expect($timeout, "ogin:")) {
		# This is really ugly but telnetd must be flushing after prompts
		# Therefore, sending name or password should be delayed a little
		# after receiving prompt
		sleep(1);
		print $s $user, "\r";
		# If no password is required, then we could get the logged in prompt next
		# Note, we're only catching a typical unix-like shell prompt here.
		if (my $match = $s->expect($timeout, ('word:', -re, '[\$#>] '))) {
			if ($match == 2) { # got a prompt
				$t1 = [gettimeofday];
				report_vuln($mechanism, $target, $user,
					$pass);
			} else {
				sleep(1);
				print $s $pass, "\r";
				# If we get "incorrect" or "word:" we skip the timeout, presuming
				# the password has been rejected
				if ($s->expect($timeout, ('word:', 'incorrect', '-re', '[\$#>] ')) > 2) {
					$t1 = [gettimeofday];
					report_vuln($mechanism, $target, $user,
						$pass);
				}
			}
		}
	}
	$s->hard_close();
	return tv_interval($t0, $t1);
}

sub tryrlogin() {
	my $target = shift;
	my $user = shift;
	my $pass = shift;

	my $s;
	my $line;
	my $cmd;
	my @args;

	my $t0;
	my $t1;

	my $mechanism = 'rlogin';

	$t0 = [gettimeofday];

	if ($verbosity > 0) {print "$mechanism check. User:$user, pass:$pass\n";}

	$cmd = "rlogin";
	@args = ($target, "-l", $user);
	$s = Expect->spawn($cmd, @args);
	if ($dbg == 1) {$s->exp_internal(1);}
	if ($dbg > 1) {$s->debug($dbg - 1);}
	if ($verbosity < 2) {
		$s->log_stdout(0);
	}
	# If no password is required, then we could get the logged in prompt next
	# Note, we're only catching a typical unix-like shell prompt here.
	if (my $match = $s->expect($timeout, ('word:', -re, '[\$#>] '))) {
		if ($match == 2) { # got a prompt
			$t1 = [gettimeofday];
			report_vuln($mechanism, $target, $user,
				$pass);
		} else {
			print $s $pass, "\r";
			# If we get "incorrect" or "word:" we skip the timeout, presuming
			# the password has been rejected
			if ($s->expect($timeout, ('word:', 'incorrect', '-re', '[\$#>] ')) > 2) {
				$t1 = [gettimeofday];
				report_vuln($mechanism, $target, $user,
					$pass);
			}
		}
	}
	$s->hard_close();
	return tv_interval($t0, $t1);
}

sub trypop3() {
	my $target = shift;
	my $user = shift;
	my $pass = shift;

	my $t;
	my $line;

	my $mechanism = 'pop3';

	my $t0;
	my $t1;

	$t0 = [gettimeofday];

	if ($verbosity > 0) {print "$mechanism check. User:$user, pass:$pass\n";}

	$t = Net::Telnet->new(
		Telnetmode => 0,
		Timeout => $timeout,
		Errmode => 'return');

	if ($dbg > 1) {$t->dump_log(*STDOUT)};
	if ($dbg == 1) {$t->input_log(*STDOUT)};
	$t->open(Host => $target, Port => 110) || return;
	$line = $t->getline;
	if ($line =~ /\+OK/ && !$t->errmsg) {
		$t->print("user $user");
		$line = $t->getline;
		if ($line =~ /\+OK/) {
			$t->print("pass $pass");
			$line = $t->getline;
			$t1 = [gettimeofday];
			if ($line =~ /\+OK/ && !$t->errmsg) {
				report_vuln($mechanism, $target, $user,
					$pass);
			}
			$t->print("quit");
		}
	}
	$t->close;
	return tv_interval($t0, $t1);
}

sub tryimap() {
	my $target = shift;
	my $user = shift;
	my $pass = shift;

	my $t;
	my $line;

	my $t0;
	my $t1;

	my $mechanism = 'imap';

	$t0 = [gettimeofday];

	if ($verbosity > 0) {print "$mechanism check. User:$user, pass:$pass\n";}

	$t = Net::Telnet->new(
		Telnetmode => 0,
		Timeout => $timeout,
		Errmode => 'return');

	if ($dbg > 1) {$t->dump_log(*STDOUT)};
	if ($dbg == 1) {$t->input_log(*STDOUT)};
	$t->open(Host => $target, Port => 143) || return;
	$line = $t->getline;
	if ($line =~ / OK/ && !$t->errmsg) {
		$t->print(". login $user $pass");
		$line = $t->getline;
		$t1 = [gettimeofday];
		if ($line =~ / OK/) {
			report_vuln($mechanism, $target, $user,
				$pass);
		}
		$t->print(". logout");
	}
	$t->close;
	return tv_interval($t0, $t1);
}

sub tryftp() {
	my $target = shift;
	my $user = shift;
	my $pass = shift;

	my $t;
	my $line;

	my $t0;
	my $t1;

	my $mechanism = 'ftp';

	$t0 = [gettimeofday];

	if ($verbosity > 0) {print "$mechanism check. User:$user, pass:$pass\n";}

	$t = Net::Telnet->new(
		Telnetmode => 0,
		Timeout => $timeout,
		Errmode => 'return');

	if ($dbg > 1) {$t->dump_log(*STDOUT)};
	if ($dbg == 1) {$t->input_log(*STDOUT)};
	$t->open(Host => $target, Port => 21) || return;
	$line = $t->getline;
	if ($line =~ /^220/ && !$t->errmsg) {
		$t->print("user $user");
		$line = $t->getline;
		if ($line =~ /^331/) {
			$t->print("pass $pass");
			$line = $t->getline;
			$t1 = [gettimeofday];
			if ($line =~ /^230/ && !$t->errmsg) {
				report_vuln($mechanism, $target, $user,
					$pass);
			}
			$t->print("quit");
		}
	}
	$t->close;
	return tv_interval($t0, $t1);
}

sub tryssh() {
	my $target = shift;
	my $user = shift;
	my $pass = shift;

	my $s;
	my $line;
	my $cmd;
	my @args;

	my $t0;
	my $t1;

	my $mechanism = 'ssh';

	$t0 = [gettimeofday];

	if ($verbosity > 0) {print "$mechanism check. User:$user, pass:$pass\n";}

# Only on some ssh clients: "-o", "'checkhostip no'",
# TODO: Check version of ssh before setting arguments
	$cmd = "ssh";
	@args = (
		"-a", "-n", "-x", "-q",
		"-o", "'userknownhostsfile /dev/null'",
		"-o", "'globalknownhostsfile /dev/null'",
		"-o", "'stricthostkeychecking no'",
		"-i", "/dev/null",
		"-l", $user,
		$target);
	$s = Expect->spawn($cmd, @args);
	if ($dbg == 1) {$s->exp_internal(1);}
	if ($dbg > 1) {$s->debug($dbg - 1);}
	if ($verbosity < 2) {
		$s->log_stdout(0);
	}
	# If no password is required, then we could get the logged in prompt next
	# Note, we're only catching a typical unix-like shell prompt here.
	if (my $match = $s->expect($timeout, ('word:', -re, '[\$#>] '))) {
		if ($match == 2) { # got a prompt
			$t1 = [gettimeofday];
			report_vuln($mechanism, $target, $user,
				$pass);
		} else {
			print $s $pass, "\r";
			# If we get "incorrect" or "word:" we skip the timeout, presuming
			# the password has been rejected
			if ($s->expect($timeout, ('word:', 'incorrect', '-re', '[\$#>] ')) > 2) {
				$t1 = [gettimeofday];
				report_vuln($mechanism, $target, $user,
					$pass);
			}
		}
	}
	$s->hard_close();
	return tv_interval($t0, $t1);
}

sub trysmb() {
	my $target = shift;
	my $user = shift;
	my $pass = shift;

	my $s;
	my $line;
	my $cmd;
	my @args;
	my $service;
	my $loc;

	my $t0;
	my $t1;

	my $mechanism = 'smb';

	$t0 = [gettimeofday];

	if ($verbosity > 0) {print "$mechanism check. User:$user, pass:$pass\n";}

	$cmd = "smbclient";
	$service = "//".$target."/ipc\$";
	@args = ($service, "'".$pass."'",
		"-U", $user);
	$s = Expect->spawn($cmd, @args);
	if ($dbg == 1) {$s->exp_internal(1);}
	if ($dbg > 1) {$s->debug($dbg - 1);}
	if ($verbosity < 2) {
		$s->log_stdout(0);
	}
	if ($s->expect($timeout, "smb: ")) {
		$t1 = [gettimeofday];
		report_vuln($mechanism, $target, $user,
			$pass);
	}
	$s->hard_close();
	return tv_interval($t0, $t1);
}

sub report_vuln() {
	my ($mechanism, $target, $user, $details) = @_;
	my $description = "Desc/".$mechanism;

	# Apologies for the ugliness of this:
	# Unless in force mode, delete this account so it doesn't get scanned again
	unless ($fmode) {
		if ($verbosity > 0) {
			print "Deleting account: $user\n";
		}
		delete $passwds{$user};
	}
	$found_one = 1;
	if ($sansmode) {
		print "  Default passwords: $mechanism\n";
		print "  (see $description for details)\n";
		return;
	}
	printf "$target($mechanism):$user:$details\n";
	unless (open (DESC, "< " . $description)) {
		printf "no description found\n";
		return;
	}
	while (<DESC>) {
		print;
	}
	close (DESC);

}

# locate program in path, ala `which prog`
sub which {
	my $prog=shift;

	foreach (File::Spec->path) {
		my $test = File::Spec->catfile($_, $prog);
		return $test if -x $test;
	}
	print "not found $prog\n";
	return undef;
}

sub usage {
	print "Usage: ./pwscan.pl [options] hostname\n";
	print "options are -v verbose  (use more for more verbosity\n";
	print "            -t timeout  (timeout (in secs) for connections, defaults to 5)\n";
	print "            -M          (SNMP)\n";
	print "            -T          (Telnet)\n";
	print "            -R          (Rlogin)\n";
	print "            -P          (POP3)\n";
	print "            -I          (IMAP)\n";
	print "            -F          (FTP)\n";
	print "            -S          (SSH)\n";
	print "            -B          (SMB)\n";
	print "            -p          (protofirst, for each protocol, try each account, instead of vice versa)\n";
	print "            -f          (force, try all protocols, even if previous failure)\n";
	print "            -k          (fork, try all protocols simultaneously)\n";
	print "            -h          (help)\n";
	print "            -q          (quiet, suppress version info)\n";
	exit(1);
}

# $Log: pwscan.pl,v $
# Revision 1.1  2000/11/06 15:32:06  loveless
# Moved to sectools section of CVS
#
# Revision 1.17  2000/07/24 17:14:43  loveless
# Updated Docs references to reflect changes to filenames
#
# Revision 1.16  2000/07/21 22:12:23  loveless
# Added second Docs reference
#
# Revision 1.15  2000/07/21 22:10:30  loveless
# More cleanup, added proper references to Docs files
#
# Revision 1.14  2000/07/19 16:40:05  paul
# Fix for null password in smbclient
#
# Revision 1.13  2000/07/18 16:47:35  paul
# -F fork conflict, changed to -k
#
# Revision 1.12  2000/07/18 14:53:30  paul
# Corrected timing code
#
# Revision 1.11  2000/07/18 14:31:48  paul
# Added timing mode, force mode, fork mode and protocol first options.
#
# Revision 1.10  2000/07/13 17:25:21  paul
# Added null password support (!). Avoid timeout on wrong password.
#
# Revision 1.9  2000/07/11 21:26:29  loveless
# Minor typo in usage() corrected
#
# Revision 1.8  2000/07/10 17:59:13  loveless
# Adjusted output to match other checks
#
# Revision 1.7  2000/07/10 17:05:22  paul
# Missing print. Tested/updated for other ssh clients
#
# Revision 1.6  2000/07/06 17:45:30  paul
# Minor telnet mods
#
# Revision 1.5  2000/07/06 13:05:20  paul
# Mislaid v1.3 log
#
# Revision 1.4  2000/07/06 12:59:00  paul
# Converted telnet -> expect due to timing problems. Added debugging. Misc cleanups.
# Added '%' password == username substitution
#
# Revision 1.3  2000/07/05 17:27:39  paul
# Updated option processing. Check for external command existence
#
# Revision 1.2  2000/06/30 15:28:36  paul
# log -> Log
