/*
 * RAZOR VLAD RPC Scan
 * $Id: rpc.c,v 1.1 2000/11/06 15:32:06 loveless Exp $
 *
 * $Log: rpc.c,v $
 * Revision 1.1  2000/11/06 15:32:06  loveless
 * Moved to sectools section of CVS
 *
 * Revision 1.9  2000/08/03 21:02:15  loveless
 * Added patrice salnot <psalnot@gpsc.fr> patch for OpenBSD.
 *
 * Revision 1.8  2000/07/24 17:14:43  loveless
 * Updated Docs references to reflect changes to filenames
 *
 * Revision 1.7  2000/07/24 11:52:47  paul
 * Added arpa/inet.h for inet_ntoa(), getopt.h for NT, if braces and rm help for -Wall
 *
 * Revision 1.6  2000/07/18 15:25:13  bkeyes
 * Corrected capitalization of OpenBSD in ifdef
 *
 * Revision 1.5  2000/07/18 14:24:31  loveless
 * Added ifdef for OpenBSD
 *
 * Revision 1.4  2000/07/14 14:36:23  loveless
 * Made minor adjustments to output to match Docs info.
 *
 * Revision 1.3  2000/07/11 14:07:47  loveless
 * Fixed minor bug in handling -t param that segfaulted.
 *
 * Revision 1.2  2000/06/28 21:14:28  loveless
 * Changed some stderr to stdout to integrate better with vlad.pl.
 *
 * Revision 1.1  2000/06/28 19:17:18  loveless
 * Initial rev
 *
 */
#include <stdio.h>
#include <unistd.h>
#ifndef __OpenBSD__
#include <getopt.h>
#endif

#include <sys/types.h>
#include <netinet/in.h>
#include <netdb.h>
#include <signal.h>
#include <rpc/rpc.h>
#include <rpc/pmap_clnt.h>
#include <rpc/pmap_prot.h>
#include <arpa/inet.h>
#ifdef __OpenBSD__
#include <sys/socket.h>
#include <stdlib.h>
#include <limits.h>
#endif

#define PORTMAP_PORT 111
#define MOUNT_PROG ((unsigned long) 100005)
#define CALENDAR_PROG ((unsigned long) 100068)
#define TOOLTALK_PROG ((unsigned long) 100083)
#define STATD_PROG ((unsigned long) 100001)
#define SADMIND_PROG ((unsigned long) 100232)

void caughtsig(int sig)
{
  fprintf(stdout,"Unable to talk to host\n\n");  
  exit(sig);
}

void mountd_info(void)
{
  fprintf(stdout,"\nrpc.mountd\n");
  fprintf(stdout,"----------\n\n");
  fprintf(stdout,"CVE Number:\nCVE-1999-0002\n\n");
  fprintf(stdout,"Details:\n");
  fprintf(stdout,"The rpc.mountd was found on this system. On older versions of Linux and Irix\n");
  fprintf(stdout,"systems, this daemon has a remote root buffer overflow vulnerability.\n\n");
  fprintf(stdout,"Fix:\n");
  fprintf(stdout,"Disable the daemon if not needed, or apply the appropriate vendor patch.\n\n");
  fprintf(stdout,"Related URLs:\n");
  fprintf(stdout,"http://www.cert.org/advisories/CA-98.12.mountd.html\n\n");
}

void cmsd_info(void)
{
  fprintf(stdout,"rpc.cmsd\n");
  fprintf(stdout,"--------\n\n");
  fprintf(stdout,"CVE Number:\nCVE-1999-0696\n\n");
  fprintf(stdout,"Details:\n");
  fprintf(stdout,"The rpc.cmsd was found on this system. Vulnerabilities have been found in\n");
  fprintf(stdout,"the SCO, SunOS, Solaris, and HP versions of this daemon which allow for\n");
  fprintf(stdout,"remote root compromise.\n\n");
  fprintf(stdout,"Fix:\n");
  fprintf(stdout,"Disable the daemon if not needed, or apply the appropriate vendor patch.\n\n");
  fprintf(stdout,"Related URLs:\n");
  fprintf(stdout,"http://www.cert.org/advisories/CA-99-08-cmsd.html\n\n");
}

void ttdbd_info(void)
{
  fprintf(stdout,"rpc.ttdbserverd\n");
  fprintf(stdout,"---------------\n\n");
  fprintf(stdout,"CVE Number:\nCVE-1999-0003, CVE-1999-0687\n\n");
  fprintf(stdout,"Details:\n");
  fprintf(stdout,"The rpc.ttdbserverd (ToolTalk) was found on this system. This daemon is\n");
  fprintf(stdout,"typically found on Unix platforms running the Common Desktop Environment (CDE)\n");
  fprintf(stdout,"and X Windows, and is vulnerable to remote root compromise. AIX, HP, Irix,\n");
  fprintf(stdout,"and Sun systems have been known to be vulnerable.\n\n");
  fprintf(stdout,"Fix:\n");
  fprintf(stdout,"Disable the daemon if not needed, or apply the appropriate vendor patch.\n\n");
  fprintf(stdout,"Related URLs:\n");
  fprintf(stdout,"http://www.cert.org/advisories/CA-98.11.tooltalk.html\n\n");
}

void sadmind_info(void)
{
  fprintf(stdout,"sadmind\n");
  fprintf(stdout,"-------\n\n");
  fprintf(stdout,"CVE Number:\nCVE-1999-0977\n\n");
  fprintf(stdout,"Details:\n");
  fprintf(stdout,"The sadmind was found on this system. Older Irix, Sun, AIX, and DGUX systems\n");
  fprintf(stdout,"may have a vulnerable version of this daemon that can lead to remote root\n\n");
  fprintf(stdout,"compromise.\n\n");
  fprintf(stdout,"Fix:\n");
  fprintf(stdout,"Disable the daemon if not needed, or apply the appropriate vendor patch.\n\n");
  fprintf(stdout,"Related URLs:\n");
  fprintf(stdout,"http://www.cert.org/advisories/CA-99-16-sadmind.html\n\n");
}

void statd_info(void)
{
  fprintf(stdout,"rpc.statd\n");
  fprintf(stdout,"---------\n\n");
  fprintf(stdout,"CVE Number:\nCVE-1999-0018, CVE-1999-0019\n\n");
  fprintf(stdout,"Details:\n");
  fprintf(stdout,"The rpc.statd was found on this system. Older Irix, Sun, and HP systems\n");
  fprintf(stdout,"may have a vulnerable version of this daemon that can lead to remote root\n\n");
  fprintf(stdout,"compromise.\n\n");
  fprintf(stdout,"Fix:\n");
  fprintf(stdout,"Disable the daemon if not needed, or apply the appropriate vendor patch.\n\n");
  fprintf(stdout,"Related URLs:\n");
  fprintf(stdout,"http://www.cert.org/advisories/CA-99-05-statd-automountd.html\n\n");
}

void vuln(void)
{
  fprintf(stdout,"Possible High Severity vulnerability\n");
}

void usage(char *prog)
{
  fprintf(stdout,"USAGE: %s [options] [-t timeout_in_seconds, default 10] hostname\n",prog);
  fprintf(stdout,"options are -v verbose (use twice for greater effect)\n");
  fprintf(stdout,"            -h help or usage screen\n");
  fprintf(stdout,"            -q quiet mode (suppress version info)\n");
  fprintf(stdout,"\n");
}

int main(int argc, char **argv)
{
  struct sockaddr_in saddr;
  struct hostent *hp;
  register struct pmaplist *list;
  register struct rpcent *rpc;
  int mountd_found = 0, statd_found = 0, cmsd_found = 0, ttdbd_found = 0;
  int sadmind_found = 0, verbose = 0, quiet = 0;
  int timeout = 10, sansoutput = 0, vuln_found = 0;
  char ch, *prog;

  prog = argv[0];

  if (argc <= 1)
  {
    usage(prog);
    exit(-1);      
  }

  while ((ch = getopt(argc, argv, "st:hvq")) != EOF)
    switch(ch)
    {
      case 's':
        sansoutput++;
        break;
      case 't':
        timeout = (int) strtol(optarg, NULL, 10);
        break;
      case 'h':
        usage(prog);
        exit(-1);
      case 'v':
        verbose++;
        break;
      case 'q':
        quiet++;
        break;
      default:
        usage(prog);
        exit(-1);
    }

  argc -= optind;
  argv += optind;  

  if (sansoutput)
  {
    quiet = 1;
    verbose = 0;
    fprintf(stdout,"#3 and #6 - ToolTalk, Calendar Manager, statd, sadmind, mountd check\n\n");
  }

  if (!quiet) fprintf(stdout,"VLAD the Scanner -- RPC Scan v1.0\nhttp://razor.bindview.com/tools/vlad/\n");
  
  memset(&saddr, 0, sizeof(saddr));
  memset(&hp, 0, sizeof(hp));

  signal(SIGALRM, caughtsig);
  alarm(timeout);

  hp = gethostbyname(*argv);
  if (hp == NULL)
  {
    fprintf(stdout,"ERROR: Unable to resolve %s, unknown host\n", argv[1]);
    exit(-1);
  }

  saddr.sin_addr = *(struct in_addr *)*hp->h_addr_list; 
  saddr.sin_family = PF_INET;
  saddr.sin_port = htons(PORTMAP_PORT);

  if (!quiet) fprintf(stdout,"Checking %s (%s)...\n",hp->h_name,inet_ntoa(saddr.sin_addr));

  for (list = pmap_getmaps(&saddr); list; list = list->pml_next)
  {
    rpc = getrpcbynumber((int) list->pml_map.pm_prog);
    if ((list->pml_map.pm_prog == MOUNT_PROG) && (!mountd_found))
    {
      mountd_found++;
      vuln_found++;
      if (sansoutput)
        fprintf(stdout,"  rpc.mountd (Mountd found, see Docs/sadmindandmountd.html)\n");
      else
      {
        vuln();
        fprintf(stdout,"rpc.mountd\n");
      }
      if(verbose>0) mountd_info();
    }
    if ((list->pml_map.pm_prog == CALENDAR_PROG) && (!cmsd_found))
    {
      cmsd_found++;
      vuln_found++;
      if (sansoutput)
        fprintf(stdout,"  rpc.cmsd (Calendar Manager found, see Docs/RPC.html)\n");
      else
      {
        vuln();
        fprintf(stdout,"rpc.cmsd\n");
      }
      if(verbose>0) cmsd_info();
    }
    if ((list->pml_map.pm_prog == TOOLTALK_PROG) && (!ttdbd_found))
    {
      ttdbd_found++;
      vuln_found++;
      if (sansoutput)
        fprintf(stdout,"  rpc.ttdbd (ToolTalk found, see Docs/RPC.html)\n");
      else
      {
        vuln();
        fprintf(stdout,"rpc.ttdbd\n");
      }
      if(verbose>0) ttdbd_info();
    }
    if ((list->pml_map.pm_prog == STATD_PROG) && (!statd_found))
    {
      statd_found++;
      vuln_found++;
      if (sansoutput)
        fprintf(stdout,"  rpc.statd (Statd found, see Docs/RPC.html)\n");
      else
      {
        vuln();
        fprintf(stdout,"rpc.statd\n");
      }
      if(verbose>0) statd_info();
    }
    if ((list->pml_map.pm_prog == SADMIND_PROG) && (!sadmind_found))
    {
      sadmind_found++;
      vuln_found++;
      if (sansoutput)
        fprintf(stdout,"  rpc.sadmind (Sadmind found, see Docs/sadmindandmountd.html)\n");
      else
      {
        vuln();
        fprintf(stdout,"rpc.sadmind");
      if(verbose>0) sadmind_info();
    }
  }
  }
  if (sansoutput)
  {
    if (vuln_found>0)
    {
      fprintf(stdout,"\n  Possible vulnerable RPC programs found. Refer to the documents\n");
      fprintf(stdout,"  listed above.\n\n");
    }
    else fprintf(stdout,"  No problems related to #3 or #6\n\n");
  }
 
  exit(0);
}

