/*
 * RAZOR Sendmail scan
 * $Id: sendmail.c,v 1.1 2000/11/06 15:32:06 loveless Exp $
 * $Log: sendmail.c,v $
 * Revision 1.1  2000/11/06 15:32:06  loveless
 * Moved to sectools section of CVS
 *
 * Revision 1.9  2000/10/12 16:52:34  loveless
 * Fixed false positive for Sendmail 8.8.*
 *
 * Revision 1.8  2000/08/03 21:01:23  loveless
 * Added patrice salnot <psalnot@gpsc.fr> patch for OpenBSD
 *
 * Revision 1.7  2000/07/24 17:14:43  loveless
 * Updated Docs references to reflect changes to filenames
 *
 * Revision 1.6  2000/07/24 11:53:42  paul
 * getopt.h for NT
 *
 * Revision 1.5  2000/07/20 20:37:42  loveless
 * Made output more consistent.
 *
 * Revision 1.4  2000/07/14 15:20:00  loveless
 * Fixed false positives in versions 8.10 and higher.
 *
 * Revision 1.3  2000/07/14 14:36:24  loveless
 * Made minor adjustments to output to match Docs info.
 *
 * Revision 1.2  2000/06/28 21:15:15  loveless
 * Changed some stderr to stdout for better vlad.pl integration.
 *
 * Revision 1.1  2000/06/28 19:18:39  loveless
 * Initial rev
 *
 */

#include <stdio.h>
#include <stdlib.h>
#ifndef __OpenBSD__
#include <getopt.h>
#endif
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <ctype.h>
#include <errno.h>
#include <netdb.h>

#define ERROR -1
#define VERSION "1.1"
#define TIMEOUT 10

#define SMTP_PORT 25

int timeout = TIMEOUT;

char *host = NULL;
char quiet = 0, verbostic = 0;

/* fix implicit declarations */
unsigned long int inet_addr(const char *cp);
char *inet_ntoa(struct in_addr in);

void usage(char *progname)
{
   fprintf(stdout, "Usage: %s [options] host\n\n", progname);
   fprintf(stdout, "Options are:\n"
           "-h: display this menu\n"
           "-q: suppress normal output\n"
           "-v: verbostic (detailed) output\n"
           "-t timeout: specify timeout in seconds (default = %d)\n\n",
           TIMEOUT);

   exit(0);
}

void handle_timeout(int signum)
{
   printf("\nTimed out occured while waiting for a response\n\n");
   exit(ERROR);
}

void sendmail_info(char *cvenum)
{
   /*
    * CVE-1999-0047
    * CVE-1999-0130 (local only)
    * CVE-1999-0131 (local only)
    * CVE-1999-0203
    * CVE-1999-0204
    * CVE-1999-0206
    */

   printf("sendmail\n--------\n\n");
   printf("CVE Number:\n%s\n\n", cvenum);

   cvenum += 4; /* skip "CVE-" */

   if (!strcmp(cvenum, "1999-0047"))
   {
      printf("Description:\n"
             "A MIME conversion buffer overflow exists in "
             "sendmail 8.8.3-4\n\n");
   }

   else if (!strcmp(cvenum, "1999-0130"))
   {
      printf("Description:\n"
             "Local users can start sendmail in daemon mode and gain root "
             "privileges\nin sendmail 8.7-8.8.2\n\n");
   }

   else if (!strcmp(cvenum, "1999-0131"))
   {
      printf("Description:\n"
             "A buffer overflow and denial via the GECOS field (/etc/passwd) "
             "can give\nroot access to local users in sendmail 8.7.5 and "
             "earlier\n\n");
   }

   else if (!strcmp(cvenum, "1999-0203"))
   {
      printf("Description:\n"
             "Root privileges by specifying an improper \"mail from\" and "
             "invalid\n\"rtcp to\" addres that would cause the mail to "
             "bounce in sendmail 5\n\n");
   }

   else if (!strcmp(cvenum, "1999-0204"))
   {
      printf("Description:\n"
             "An attacker can remotely execute commands with root privileges "
             "via ident\nin sendmail 8.6.9\n\n");
   }

   else if (!strcmp(cvenum, "1999-0206"))
   {
      printf("Description:\n"
             "A MIME buffer overflow can give root access in sendmail "
             "8.8.0-1\n\n");
   }

   /* Always display the same fix */
   printf("Fix:\n"
          "Upgrade to the latest version of sendmail "
          "(http://www.sendmail.org/current-release.html)\n\n");

   if (!strcmp(cvenum, "1999-0047"))
   {
      printf("Related URLs:\n"
             "http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-1999-0047\n"
             "http://www.cert.org/advisories/CA-97.05.sendmail.html\n\n");
   }

   else if (!strcmp(cvenum, "1999-0130"))
   {
      printf("Related URLs:\n"
             "http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-1999-0130\n"
             "http://www.cert.org/advisories/CA-96.24.sendmail.daemon.mode.html\n\n");
   }

   else if (!strcmp(cvenum, "1999-0131"))
   {
      printf("Related URLs:\n"
             "http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-1999-0131\n"
             "http://www.cert.org/advisories/CA-96.20.sendmail_vul.html\n\n");
   }

   else if (!strcmp(cvenum, "1999-0203"))
   {
      printf("Related URLs:\n"
             "http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-1999-0203\n"
             "http://www.cert.org/advisories/CA-95.08.sendmail.v.5.vulnerability.html"
             "\n\n");
   }

   else if (!strcmp(cvenum, "1999-0204"))
   {
      printf("Related URLs:\n"
             "http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-1999-0204\n\n");
   }

   else if (!strcmp(cvenum, "1999-0206"))
   {
      printf("Related URLs:\n"
             "http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-1999-0206\n\n");
   }
}

int main(int argc, char **argv)
{
   int sockfd, sansmode = 0, found_one = 0;

   char opt;
   char banner[64], *ptr, *version, *tmpptr;

   struct sockaddr_in daddr;
   struct hostent *hent = NULL;

   signal(SIGALRM, handle_timeout);
   signal(SIGSEGV, SIG_IGN), signal(SIGPIPE, SIG_IGN);

   if (argc < 2) usage(argv[0]);

   while ((opt = getopt(argc, argv, "svqht")) != EOF)
      switch (opt)
      {
         case 's':
            sansmode = 1;
            break;
         case 'h': usage(argv[0]);
         case 'q': 
            quiet = 1;
            break;
         case 'v':
            verbostic = 1;
            break;

         case 't':
            if (argc < 3 || optind < 2) usage(argv[0]);

            timeout = atoi(argv[optind++]);
            break;
      }

   if (optind >= argc) usage(argv[0]);

   else if (optind && argv[optind] && argv[optind][0] != '-') 
      host = strdup(argv[optind]);

   if (!host) usage(argv[0]);

   /* conflicting options */
   if (verbostic && quiet)
   {
      printf("Quiet mode and verbostic mode conflict!\n");
      close(sockfd), free(host), exit(0);
   }

   if (sansmode)
   {
      verbostic = 0;
      quiet = 1;
   }

   if (!quiet)
   {
      printf("VLAD the Scanner -- Sendmail checker Version %s\n"
             "http://razor.bindview.com/tools/vlad/\n\n", 
             VERSION);
   }

   if (sansmode) printf("#5 - Sendmail vulnerabilities\n\n");
   
   if (verbostic) printf("Attempting to resolve %s...\n", host);

   alarm(timeout);

   hent = gethostbyname(host);
   if (!hent)
   {
      herror("gethostbyname");
      free(host), exit(ERROR);
   }

   memcpy(&daddr.sin_addr.s_addr, hent->h_addr, sizeof(u_long));

   if (verbostic) 
      printf("Resolved %s to %s\n", host, inet_ntoa(daddr.sin_addr));

   alarm(0);

   daddr.sin_family = AF_INET;
   daddr.sin_port = htons(SMTP_PORT);

   sockfd = socket(AF_INET, SOCK_STREAM, 0);

   if (verbostic) 
   {
      printf("Connecting to %s (port %d) with a %d second timeout...",
             host, SMTP_PORT, timeout);
      fflush(stdout);
   }

   alarm(timeout);

   if (connect(sockfd, (struct sockaddr *)&daddr, sizeof(daddr)) == ERROR)
   {
      if (verbostic) fprintf(stdout, "Error connecting to %s (port %d): %s\n\n", host,
              SMTP_PORT, strerror(errno));

      close(sockfd);
   }
   else
   {
     alarm(0);
     if (verbostic) printf(" connected\n");

     if (recv(sockfd, banner, sizeof(banner)-1, 0) == ERROR)
     {
        fprintf(stderr, "error with recv(): %s\n", strerror(errno));
        close(sockfd), free(host), exit(ERROR);
     }

     if (verbostic)
     {
        printf("banner:%c%s\n%c", ((strlen(banner) > 64) ? '\n' : ' '),
               banner, ((banner[strlen(banner)-1] == '\n') ? '\0' : '\n'));      
     }

     ptr = strstr(banner, "Sendmail ");

     if (ptr)
     {
       ptr += 9, version = ptr; /* skip "Sendmail " */
       tmpptr = version;
       tmpptr += 4; /* hack to check for 8. versions greater than 9 */

       ptr = strchr(ptr, '/');
       if (ptr) *ptr = '\0'; 

       if (!strncmp(tmpptr, ".", 1))
         found_one = 0; /* in other words, NOOP */
       else if (!strncmp(version, "8.8.3", 5) || !strncmp(version, "8.8.4", 5))
       {
         if (verbostic) sendmail_info("CVE-1999-0047");
         else
         {
           found_one = 1;
           printf("%s is potentially vulnerable to a known sendmail bug "
                  "(CVE-1999-0047)\n", host);
         }
       }

       /* 8.7 through 8.8.2 */
       else if (!strncmp(version, "8.7", 3) || 
               (!strncmp(version, "8.8.", 4) && version[4] <= '2'))
       {
         if (verbostic) sendmail_info("CVE-1999-0130");
         else
         {
           found_one = 1;
           printf("%s is potentially vulnerable to a known sendmail bug "
                  "(CVE-1999-0130)\n", host);
         }
       }

       /* <= 8.7.5 */
       else if (version[0] < '8' || (!strncmp(version, "8.", 2) && 
               (version[2] < '7' || (version[2] == '7' && version[4] <= '5'))))
       {
         if (verbostic) sendmail_info("CVE-1999-0131");
         else
         {
            found_one = 1;
            printf("%s is potentially vulnerable to a known sendmail bug "
                   "(CVE-1999-0131)\n", host);
         }
       }

       else if (!strncmp(version, "5.", 2))
       {
         if (verbostic) sendmail_info("CVE-1999-0203");
         else
         {
            found_one = 1;
            printf("%s is potentially vulnerable to a known sendmail bug "
                   "(CVE-1999-0203)\n", host);
         }
       }
 
       else if (!strncmp(version, "8.6.9", 5))
       {
         if (verbostic) sendmail_info("CVE-1999-0204");
         else
         {
            found_one = 1;
            printf("%s is potentially vulnerable to a known sendmail bug "
                   "(CVE-1999-0204)\n", host);
         }
       }

       else if (!strncmp(version, "8.8.", 4) && version[4] <= '1') 
       {
         if (verbostic) sendmail_info("CVE-1999-0206");
         else
         {
            found_one = 1;
            printf("%s is potentially vulnerable to a known sendmail bug "
                   "(CVE-1999-0206)\n", host);
         }
       }
     }
   }

   if (sansmode)
   {
      if (found_one)
         printf("  Possible Sendmail version found. Refer to Docs/Sendmail.html\n\n");
      else printf("  No problems related to #5\n\n");
   }

   close(sockfd), free(host);
   return 0;
}
