
/*
	VoIPong Voice Over IP Sniffer
	Copyright (C) 2004 Murat Balaban <murat || enderunix.org>
	All rights reserved.

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <pthread.h>
#include <signal.h>
#include <errno.h>
#include <syslog.h>
#include <time.h>
#include <sysexits.h>
#include <pcap.h>
#include <fcntl.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <voipong.h>
#include <voipongsign.h>
#include <voipongsock.h>
#include <voipongpcap.h>
#include <voipongrtcp.h>
#include <voipongrtp.h>

#include <miscutil.h>
#include <conf.h>

/* GLOBALS	*/
config cfg;
int gfg = 0;
int gdbg = 0;
char gcfgfile[128];
char gclientlist[128];
int grtp_idle_time = 0;
int gthisday = 0;
int gthismon = 0;
time_t gstarttime = 0;
int mgmt_client = 0;
int gmgmtport = 0;
char gmgmt_password[128];
char gsoxpath[256];
char gpidfile[128];
char goutdir[256];
int pcapfd = -1, mgmtfd = -1;
char gdevice[256];
char gfilter[1024];
int gpromisc = 1;
int gsnaplen = 1;
int greadtmt = 1;
rtp_session *rtps = NULL;

void
worker_remove(rtp_session *in)
{
	rtp_session *rtp;
	rtp_session *prev;

	for (rtp = rtps; rtp != NULL; rtp = rtp->next) {
		if (rtp->pid == in->pid) {
			if (prev == NULL)
				rtps = rtp->next;
			else
				prev->next = rtp->next;
		}
		prev = rtp;
	}
}

int
worker_isexist(int ip1, int ip2, int port1, int port2)
{
	rtp_session *rtp;

	for (rtp = rtps; rtp != NULL; rtp = rtp->next) {
		if (ip1 == rtp->ip2)
			if (port1 == rtp->port2)
				return 1;
		if (ip1 == rtp->ip1)
			if (port1 == rtp->port1)
				return 1;
	}
	return 0;
}

void
worker_create(rtp_session **rtp)
{
	pid_t pid;

	switch((pid = fork())) {
		case -1:
			misc_debug(0, "worker_create: fork: %s\n", strerror(errno));
			break;
		case 0: /* cocuk */
			childmain(*rtp);
			exit(0);
			break;
		default:
			(*rtp)->pid = pid;
			(*rtp)->next = rtps;
			rtps = *rtp;
			break;
	}
}

rtp_session *
getworkerbypid(pid_t pid)
{
	rtp_session *rtp;

	for(rtp = rtps; rtp != NULL; rtp = rtp->next)
		if (rtp->pid == pid)
			return rtp;

	return NULL;
}

int
daemon_init(void)
{
	FILE *lckp = NULL;
	pid_t pid = 0;
	int i = 0;

	if (gfg == 0) {
		if ((pid = fork()) < 0)
			return -1;
		else if (pid != 0)
			exit(0);
	
		setsid();
		for (i = getdtablesize(); i >= 0; i--)
			close(i);
		if ((i = open("/dev/null", O_RDWR | O_CREAT, 0640)) == -1)
			exit(1);
		dup(i); /* stdin */
		dup(i); /* stdout */
		dup(i); /* stderr */
	}
	umask(027);
	openlog("voipong", 0, LOG_DAEMON);
	if ((lckp = fopen(gpidfile, "w")) == NULL) {
		fprintf(stderr, "cannot open pidfile: %s\n", strerror(errno));
		syslog(LOG_WARNING, "cannot open pidfile: %s\n", strerror(errno));
		exit(1);
	}
	if (lockf(fileno(lckp), F_TLOCK, 0)) {
		fprintf(stderr, "cannot lock pidfile: %s, may be another copy running?\n", strerror(errno));
		syslog(LOG_WARNING, "cannot lock pidfile: %s, may be another copy running?\n", strerror(errno));
		exit(1);
	}
	/* write pid */
	fprintf(lckp, "%d\n", getpid());
	setuid(getuid());
	setgid(getgid());
	return 0;
}
	
void 
wexit(int c)
{
	char tmp[1024];

	misc_debug(0, "PID %d [parent: %d]: exited with code: %d. uptime: %s.\n", getpid(), getppid(), c, 
								misc_getuptimestr(tmp, 1024, gstarttime));
	if (unlink(gpidfile) == -1) {
		syslog(LOG_ERR, "can't remove  pidfile[%s]: %s\n", gpidfile, strerror(errno));
	}
	misc_closelog();
	exit(c);
}

int 
main(int argc, char **argv)
{
	extern char *optarg;
	int error = 0;
	int c = 0;
	char tmp[512];
	struct tm tm;
	struct sigaction sa;
	struct sigaction sa_old;
	char errbuf[ERRBUFSIZ];

	sa.sa_handler = sighandler;
	sigemptyset(&sa.sa_mask);
	sa.sa_flags = SA_SIGINFO;
	sigaction(SIGHUP, &sa, &sa_old);
	sigaction(SIGUSR2, &sa, &sa_old);
	sigaction(SIGTERM, &sa, &sa_old);
	sigaction(SIGINT, &sa, &sa_old);
	sigaction(SIGSTOP, &sa, &sa_old);
	sigaction(SIGQUIT, &sa, &sa_old);
	sigaction(SIGALRM, &sa, &sa_old);

	/* Program baslarken bugunku tarihi alalim	*/
	time(&gstarttime);
	localtime_r(&gstarttime, &tm);
	gthisday = tm.tm_mday;
	gthismon = tm.tm_mon;

	strcpy(gcfgfile, "/usr/local/etc/voipong.conf");
	while (!error && (c = getopt(argc, argv, "c:d:hvf")) != -1) {
		switch(c) {
			case 'v':
				printf("%s %s\n", PROGRAM, VERSION);
			        printf("Copyright (C) 2004 Murat Balaban <murat || enderunix.org>\n"
				       "All rights reserved.\n\n"
				       "This program is free software; you can redistribute it and/or\n"
				       "modify it under the terms of the GNU General Public License\n"
				       "as published by the Free Software Foundation; either version 2\n"
				       "of the License, or (at your option) any later version.\n");
				exit(0);
				break;
			case 'h':
				usage();
				exit(0);
				break;
			case 'f':
				gfg = 1;
				break;
			case 'd':
				gdbg = atoi(optarg);
				break;
			case 'c':
				strncpy(gcfgfile, optarg, 127);
				break;
			default:
				printf("invalid option: %c, try -h for help\n", c);
				exit(EX_USAGE);

		}
	}
	init_config();
	misc_setlogtype(gfg);
	misc_setlogdir(config_getval(&cfg, "GENEL", "logdir"));
	misc_setlogfile(config_getval(&cfg, "GENEL", "logfile"));
	misc_setloglevel(gdbg);
	printf("%s starting...\n", PROGRAM);
	printf("%s, running on %s\n\n", VERSION, misc_getunamestr(tmp, 512));
	printf("%s\n", COPYRIGHT);
	daemon_init();
	if (misc_openlog() < 0) {
		syslog(LOG_ERR, "misc_openlog: error!: %s", strerror(errno));
		wexit(1);
	}
	misc_debug(0, "%s starting...\n", PROGRAM);
	misc_debug(0, "%s running on %s. %s [pid: %d]\n", VERSION, misc_getunamestr(tmp, 512), COPYRIGHT, getpid());

	if ((pcapfd = initpcap(errbuf)) == -1) {
		misc_debug(0, "libpcap start failure: %s\n", errbuf);
		wexit(1);
	}
	add_to_select_set(pcapfd);
	if (open_server_socket(gmgmtport, &mgmtfd) == -1) {
		misc_debug(0, "mgmt socket open failure!\n");
		wexit(1);
	}

	sockets_run();

	wexit(0);
	return 0;
}

void 
graceful_shutdown()
{
	rtp_session *rtp;

	for (rtp = rtps; rtp != NULL; rtp = rtp->next) {
		misc_debug(0, "session [%d] terminating...\n", rtp->pid);
		kill(rtp->pid, 15);
	}

	wexit(0);
}

void 
reload()
{


}

void 
init_config()
{
	char errbuf[CONFERRBUFSIZ];

	if ((config_load(&cfg, gcfgfile, errbuf)) == NULL) {
		fprintf(stderr, "init_config: %s\n", errbuf);
		wexit(1);
	}
	get_initcfgvals();
}

void
get_initcfgvals()
{
	char *tmp;

	/* First get listen_port number	*/
	tmp = (char *)config_getval(&cfg, "GENEL", "mgmt_client");
	if (strlen(tmp) == 0) {
		printf("cannot get mgmt_client from configfile, shutting down......\n");
		exit(1);
	}
	mgmt_client = misc_inet_addr(tmp);
	if (config_getstr(&cfg, "GENEL", "mgmt_password", gmgmt_password, sizeof(gmgmt_password)) == 0) {
		printf("cannot get mgmt_password from configfile, shutting down......\n");
		exit(1);
	}
	if ((gmgmtport = config_getint(&cfg, "GENEL", "mgmt_port", -1)) == -1) {
		printf("cannot get mgmt_port from configfile, shutting down......\n");
		exit(1);
	}
	if (config_getstr(&cfg, "GENEL", "pidfile", gpidfile, sizeof(gpidfile)) == 0) {
		printf("cannot get pidfile from configfile, shutting down......\n");
		exit(1);
	}
	if ((grtp_idle_time = config_getint(&cfg, "GENEL", "rtp_idle_time", -1)) == -1) {
		printf("cannot get rtp_idle_time from configfile, shutting down......\n");
		exit(1);
	}

	if (config_getstr(&cfg, "GENEL", "clientlist", gclientlist, sizeof(gclientlist)) == 0) {
		printf("cannot get clientlist from configfile, shutting down......\n");
		exit(1);
	}
	if (config_getstr(&cfg, "GENEL", "outdir", goutdir, sizeof(goutdir)) == 0) {
		printf("cannot get outdir from configfile, shutting down......\n");
		exit(1);
	}
	if (config_getstr(&cfg, "GENEL", "soxpath", gsoxpath, sizeof(gsoxpath)) == 0) {
		printf("cannot get soxpath from configfile, shutting down......\n");
		exit(1);
	}
	config_getstr(&cfg, "GENEL", "filter", gfilter, sizeof(gfilter));
	config_getstr(&cfg, "GENEL", "device", gdevice, sizeof(gdevice));

	if ((gpromisc = config_getint(&cfg, "GENEL", "promisc", -1)) == -1) {
		printf("cannot get promisc value from configfile, shutting down......\n");
		exit(1);
	}

	if ((gsnaplen = config_getint(&cfg, "GENEL", "snaplen", -1)) == -1) {
		printf("cannot get snaplen value from configfile, shutting down......\n");
		exit(1);
	}
	if ((greadtmt = config_getint(&cfg, "GENEL", "readtmt", -1)) == -1) {
		printf("cannot get readtmt value from configfile, shutting down......\n");
		exit(1);
	}

}


void
usage()
{
        printf("usage: voipong [options]\n");
        printf("\toptions:\n");
        printf("\t\t-h this screen\n");
        printf("\t\t-v version info\n");
        printf("\t\t-f run in foreground (don't become a daemon)\n");
        printf("\t\t-d debug level. Valid levels are 0 through 4. Default: 0\n");
        printf("\t\t-c config file path\n");

        printf("\n");
}

void
process_deadchild()
{
	pid_t pid;
	int stat = 0;
	int termsig = 0, exitcode = 0;
	int serrno = 0;
	rtp_session *rtp;

	for ( ; ; ) {
		if ((pid = wait(&stat)) == -1) {
			serrno = errno;
			if (serrno == EINTR)
				continue;
			if (serrno == ECHILD) {
				misc_debug(0, "no child processes to wait for, shutting down...!\n");
				break;
			}
			misc_debug(0, "waitforevents: wait error: %s\n", strerror(errno));
		}
		if ((rtp = getworkerbypid(pid)) == NULL) {
			misc_debug(0, "I dont have a child with pid %d\n", pid);
			continue;
		}
		if (WIFEXITED(stat)) {
			misc_debug(0, "child [pid: %d] terminated normally [exit code: %d]\n", pid, (exitcode = WEXITSTATUS(stat)));
		}
		else
		if (WIFSIGNALED(stat)) {
			misc_debug(0, "child [pid: %d] terminated by signal %d\n", pid, (termsig = WTERMSIG(stat)));
		}
		worker_remove(rtp);
	}
}
