
/*
	VoIPong Voice Over IP Sniffer
	Copyright (C) 2004 Murat Balaban <murat || enderunix.org>
	All rights reserved.

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include <pcap.h>

#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>

  
#include <netinet/in_systm.h>
#include <netinet/in.h>
#define __USE_BSD 1
#define __FAVOR_BSD 1
#include <netinet/ip.h>
#include <netinet/udp.h>

#include <arpa/inet.h>

#include <g711.h>

#include <voipongpcap.h>
#include <voipongrtp.h>
#include <miscutil.h>
#include <conf.h>

extern int grtp_idle_time;
extern char goutdir[];
extern char gsoxpath[];

static int mypid = 0;
static rtp_session *curses = NULL;

static char *rtp_pt_desc[] = {"0-PCMU-8KHz",
	"1-1016-8KHz",
	"2-G726-32-8KHz",
	"3-GSM-8KHz",
	"4-G723.1-8KHz",
	"5-DVI4-8KHz",
	"6-DVI4-16KHz",
	"7-LPC-8KHz",
	"8-PCMA-8KHz",
	"9-G722-8KHz",
	"10-L16-44.1KHz",
	"11-L16-44.1KHz",
	"12-QCELP-8KHz",
	"13-CN-8KHz",
	"14-MPA-90KHz",
	"15-G728-8KHz",
	"16-DVI4-11KHz",
	"17-DVI4-22KHz",
	"18-G729-8KHz",
	"19-reserved",
	"20-unassigned",
	"21-unassigned",
	"22-unassigned",
	"23-unassigned",
	"24-unassigned",
	"25-CelB-90KHz",
	"26-JPEG-90KHz",
	"27-unassigned"
};

int
create_path()
{
	char tfmt[64];
	char filefmt[512];
	struct stat st;

	if (stat(goutdir, &st) == -1)
		if (errno == ENOENT) {
			if (mkdir(goutdir, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH) < 0) {
				misc_debug(0, "cannot create outputdir %s: %s\n", goutdir, strerror(errno));
				return -1;
			}
		}

	misc_strftime(tfmt, sizeof(tfmt) - 1, "%Y%m%d");
	snprintf(filefmt, sizeof(filefmt) - 2, "%s/%s", goutdir, tfmt);

	if (stat(filefmt, &st) == -1)
		if (errno == ENOENT) {
			if (mkdir(filefmt, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH) < 0) {
				misc_debug(0, "cannot create daily outputdir %s: %s\n", filefmt, strerror(errno));
				return -1;
			}
		}
	return 0;
}

int
create_wave()
{
	char outfmt[512];
	pid_t pid, wpid;
	int stat;
	char inrate[32];
	char insize[4];
	char indenc[4];
	int exitcode = 0;

	switch(curses->enc) {
		case 0:
			snprintf(inrate, sizeof(inrate) - 1, "%d", curses->rate);
			strcpy(insize, "-w");
			strcpy(indenc, "-s");
			break;
		default:
			misc_debug(0, "[%d] create_wave: encoding type %d is not supported yet.", mypid, curses->enc);
			break;
	}

	strncpy(outfmt, curses->filefmt1, 510);
	strncpy(outfmt + (strlen(outfmt) - 4), ".wav", sizeof(outfmt) -1);
	switch((pid = fork())) {
		case -1:
			misc_debug(0, "[%d] cannot fork a new process to create .WAV file: %s\n", mypid, outfmt);
			return -1;
			break;
		case 0:
			if (execl(gsoxpath, gsoxpath, "-r", inrate, insize, indenc, curses->filefmt1, outfmt, NULL) == -1) {
				misc_debug(0, "%s %s -r %s %s %s %s error: %s\n", gsoxpath, gsoxpath, inrate, insize, indenc, curses->filefmt1, outfmt,
						strerror(errno));
				return -1;
			}
			exit(1);
		default:
			if ((wpid = wait(&stat)) == -1) {
				misc_debug(0, "create_wave: wait: %s\n", strerror(errno));
				return -1;
			}
			if ((exitcode = WEXITSTATUS(stat)) != 0) 
				misc_debug(0, "create_wave: sox helper process failed, return value: %d\n", exitcode);
			else
				misc_debug(0, ".WAV file [%s] has been created successfully.\n", outfmt);
	}
	if (unlink(curses->filefmt1) == -1) {
		misc_debug(0, "Cannot remove file [%s]: %s\n", curses->filefmt1, strerror(errno));
		return -1;
	}

	strncpy(outfmt, curses->filefmt2, 510);
	strncpy(outfmt + (strlen(outfmt) - 4), ".wav", sizeof(outfmt) - 1);
	switch((pid = fork())) {
		case -1:
			misc_debug(0, "[%d] cannot fork a new process to create .WAV file: %s\n", mypid, outfmt);
			return -1;
			break;
		case 0:
			if (execl(gsoxpath, gsoxpath, "-r", inrate, insize, indenc, curses->filefmt2, outfmt, NULL) == -1) {
				misc_debug(0, "%s %s -r %s %s %s %s error: %s\n", gsoxpath, gsoxpath, inrate, insize, indenc, curses->filefmt2, outfmt,
						strerror(errno));
				return -1;
			}
			exit(1);
		default:
			if ((wpid = wait(&stat)) == -1) {
				misc_debug(0, "create_wave: wait: %s\n", strerror(errno));
				return -1;
			}
			if ((exitcode = WEXITSTATUS(stat)) != 0) 
				misc_debug(0, "create_wave: sox helper process failed, return value: %d\n", exitcode);
			else
				misc_debug(0, ".WAV file [%s] has been created successfully.\n", outfmt);
	}
	if (unlink(curses->filefmt2) == -1) {
		misc_debug(0, "Cannot remove file [%s]: %s\n", curses->filefmt2, strerror(errno));
		return -1;
	}
	return 0;
}



void 
probertp(u_char *udata, const struct pcap_pkthdr *pkthdr, const u_char *packet)
{
	u_data *ud = (u_data *)udata;
	struct ip *ip;
	struct udphdr *udp;
	struct rtphdr *rtp;
	u_char *pl;
	int i;
	u_int16_t buf[1500];
	int offset;
	int plen;
	char filter[512];
	char pt[256];
	char filefmt[512];
	char tfmt[64];

	misc_strftime(tfmt, sizeof(tfmt) - 1, "%Y%m%d");
	snprintf(filefmt, sizeof(filefmt) - 2, "%s/%s/session-enc%%s-%%s,%%d-%%s,%%d.raw", goutdir, tfmt);

	offset = ud->dloffset;
	ip = (struct ip *) (packet + offset);
	udp = (struct udphdr *)(packet + offset + 20);
	rtp = (struct rtphdr *)(packet + offset + 20 + 8);
	pl = (u_char *)(packet + offset + 20 + 8 + sizeof(struct rtphdr));
	plen = pkthdr->len - (offset + 20 + 8 + sizeof(struct rtphdr));

	if (ntohs(rtp->seq) == ud->last_req1 || ntohs(rtp->seq) == ud->last_req2)
		return;

	if (rtp->pt > 27)
		snprintf(pt, sizeof(pt) - 1,  "%d-unknown", rtp->pt);
	else
		snprintf(pt, sizeof(pt) - 1, "%s", rtp_pt_desc[rtp->pt]);


	if (ip->ip_src.s_addr == ud->ip_arr[0]) {
		if (ud->fd[0] == -1) {
			misc_debug(0, "[%d] Encoding: %s\n", mypid, pt);
			snprintf(filter, sizeof(filter) - 1,  filefmt, pt, 
					ud->ip1, ntohs(udp->uh_sport), 
					ud->ip2, ntohs(udp->uh_dport));
			if ((ud->fd[0] = creat(filter, S_IRWXU)) == -1) {
				misc_debug(0, "[%d] Cannot create raw voice output file %s: %s\n", mypid, filter, strerror(errno));
				exit(1);
			}
			strncpy(curses->filefmt1, filter, 510);
		}
	} else {
		if (ud->fd[1] == -1) {
			snprintf(filter, sizeof(filter) - 1, filefmt, pt, 
					ud->ip2, ntohs(udp->uh_sport), 
					ud->ip1, ntohs(udp->uh_dport));
			if ((ud->fd[1] = creat(filter, S_IRWXU)) == -1) {
				misc_debug(0, "[%d] Cannot create raw voice output file [%s]: %s\n", mypid, filter, strerror(errno));
				exit(1);
			}
			strncpy(curses->filefmt2, filter, 510);
		}
	}

	switch(rtp->pt) {
		case PT_ULAW:
			for (i = 0; i < plen; i++)
				buf[i] = ulaw2linear(pl[i]);
			if (ip->ip_src.s_addr == ud->ip_arr[0]) {
				write(ud->fd[0], buf, plen * sizeof(u_int16_t));
				ud->last_req1 = ntohs(rtp->seq);
			}
			else {
				write(ud->fd[1], buf, plen * sizeof(u_int16_t));
				ud->last_req2 = ntohs(rtp->seq);
			}
			break;
		default:
			if (ip->ip_src.s_addr == ud->ip_arr[0]) {
				write(ud->fd[0], pl, plen);
				ud->last_req1 = ntohs(rtp->seq);
			}
			else {
				write(ud->fd[1], pl, plen);
				ud->last_req2 = ntohs(rtp->seq);
			}
			break;
	}
}

void 
childmain(rtp_session *rtpsess)
{
	pcap_t *pd;
	u_data ud;
	bpf_u_int32 netp;
	char perrbuf[PCAP_ERRBUF_SIZE];
	struct bpf_program fprog;
	char *filterfmt = "(host %s and port %d) and (host %s and port %d) and udp";	
	char filter[512];
	char ip1[16];
	char ip2[16];
	struct in_addr in;

	mypid = getpid();
	curses = rtpsess;
	if ((pd = pcap_open_live(rtpsess->dev, SNAPLEN, 1, 500, perrbuf)) == NULL) {
		misc_debug(0, "[%d] pcap_open_live error: %s\n", mypid, perrbuf);
		exit(1);
	}
	create_path();
	ud.ip_arr[0] = rtpsess->ip1;
	ud.ip_arr[1] = rtpsess->ip2;
	in.s_addr = rtpsess->ip1;
	inet_ntop(AF_INET, (void *)&in, ip1, 16);
	in.s_addr = rtpsess->ip2;
	inet_ntop(AF_INET, (void *)&in, ip2, 16);
	strncpy(ud.ip1, ip1, 20);
	strncpy(ud.ip2, ip2, 20);
	snprintf(filter, sizeof(filter) - 1,  filterfmt, ip1, ntohs(rtpsess->port1), ip2, ntohs(rtpsess->port2));
	misc_debug(0, "[%d] VoIP call has been detected.\n", mypid);
	misc_debug(0, "[%d] %s:%d <--> %s:%d\n", mypid, ip1, ntohs(rtpsess->port1), ip2, ntohs(rtpsess->port2));
	pcap_compile(pd, &fprog, filter, 0, netp);
	pcap_setfilter(pd, &fprog);
	pcap_freecode(&fprog);
	ud.fd[0] = -1;
	ud.fd[1] = -1;
	ud.dloffset = getdllen(pd);
        if (pcap_setnonblock(pd, 1, perrbuf) == -1) {
                misc_debug(0, "[%d] pcap_netnonblock: %s\n", mypid, perrbuf);
                exit(1);
        }
	if (child_loop(pd, -1, probertp, (u_char *)&ud) < 0) {
		misc_debug(0, "pcap_loop: %s\n", pcap_geterr(pd));
		exit(1);
	}
	return;
}

int
child_loop(pcap_t *pd, int cnt, pcap_handler callback, u_char *user)
{
	fd_set set;
	int fd = pcap_fileno(pd);
	time_t tnow, tidle;
	struct timeval tv;

	tv.tv_sec = grtp_idle_time;
	tv.tv_usec = 0;
	time(&tnow);
	tidle = tnow;
	for ( ; ; ) {
		FD_ZERO(&set);
		FD_SET(fd, &set);
		switch(select(fd + 1, &set, NULL, NULL, &tv)) {
			case -1:
				misc_debug(0, "[%d] select: error: %s\n", mypid, strerror(errno));
				exit(1);
				break;
			case 0:
				time(&tnow);
				if ((tnow - tidle) > grtp_idle_time) {
					misc_debug(0, "[%d] maximum waiting time [%d secs] has been elapsed for this call asildi, the call might have been ended.\n", 
							mypid, grtp_idle_time);
					create_wave();
					exit(0);
				}
				break;
			default:
				time(&tidle);
			  	while (pcap_dispatch(pd, cnt, callback, user) != 0)
					;
				break;
		}
	}
}
