;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; CRC32.DLL, version 2
;;
;; A sample program showing how to program DLLs for Windows95/NT
;; in assembly language.
;;
;; Copyright (c) 1997, 1999 G. Adam Stanislav
;; All rights reserved.
;;
;; e-mail: whizkid@bigfoot.com
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; This code, along with its makefile, illustrates how to create a DLL,
;; how to initialize its data, and how to share the same data among
;; all processes using the same DLL. The file CRCDEMO.ASM contains a very
;; simple program which illustrates how to call the routines in this DLL.
;;
;; Here is a philosophical note: Most code for Windows 95/NT is written
;; in a HLL, such as C or C++. Therefore, for a DLL to be useful to
;; other programmers, it should contain functions callable from C and
;; other HLLs.
;;
;; At the same time, to an assembly language programmer, the HLL interface
;; wastes precious time by pushing parameters on the stack, saving the
;; stack pointer, changing it, getting the parameters off the stack,
;; restoring the stack pointers, and popping off the parameters. It is
;; much simpler to pass parameters in the registers.
;;
;; CRC-32 is a classical example how it is MUCH more efficient to use
;; registers: CRC-32 is typically called from a big loop and the return
;; value of the CRC-32 calculation is passed as a parameter to the next
;; CRC-32 calculation. Using EAX for both the return value and the
;; parameter to the next call means you do not have to worry about
;; passing the parameter at all.
;;
;; This presents a dilemma: Should DLLs written in assembly language use
;; a HLL interface to be useful to as many programmers as possible, or
;; should they pass parameters in registers?
;;
;; The answer shown here is simple: Use both. Code it for assembly language
;; interface, but also include functions callable from HLL.
;;
;; As it turns out, you can just pop the parameters off the stack and fall
;; through to the fast routines. This is thanks to the STDCALL HLL interface
;; used in Windows 95, in which the callee clears the stack, not the caller.
;;
;; You can even give the HLL and assembly language functions the same name,
;; with capitalization being the only difference. The STDCALL convention,
;; after all, is case sensitive.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; While the main purpose of this code is to help others learn about DLL
;; programming in assembly language, this DLL may be quite useful for
;; anyone who needs to calculate CRC-32.
;;
;; So, if you need it for your programs, go ahead and use it. I just ask
;; that you do not modify it in any way: Other programmers may be using it
;; as well, and your modifications might cause their programs to work
;; differently from what they expected. I also ask that you have an
;; "About CRC32.DLL" item on your menu linked to the AboutCrc32 or
;; aboutcrc32 procedure at end of this code.
;;
;; And if you do use it, please send me a postcard. My address:
;;
;;      G. Adam Stanislav
;;      Whiz Kid Technomagic
;;      Rhinelander, WI 54501
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

.386
.model flat

option prologue:none
option epilogue:none
option dotname

extern __imp__MessageBoxA@16:dword

MB_OK			EQU	000h
MB_OKCANCEL		EQU	001h
MB_ABORTRETRYIGNORE	EQU	002h
MB_YESNOCANCEL		EQU	003h
MB_YESNO		EQU	004h
MB_RETRYCANCEL		EQU	005h
MB_ICONERROR		EQU	010h
MB_ICONQUESTION         EQU     020h
MB_ICONWARNING		EQU	030h
MB_ICONINFORMATION      EQU     040h

IDOK			EQU	01h
IDCANCEL		EQU	02h
IDABORT			EQU	03h
IDRETRY			EQU	04h
IDIGNORE		EQU	05h
IDYES			EQU	06h
IDNO			EQU	07h
IDCLOSE			EQU	08h
IDHELP			EQU	09h

DLL_PROCESS_ATTACH	EQU	01h
DLL_THREAD_ATTACH	EQU	02h
DLL_THREAD_DETACH	EQU	03h
DLL_PROCESS_DETACH	EQU	00h

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Unless directed otherwise, Windows95/NT will create a separate data
;; segment for each process using the same DLL. This is to prevent one
;; process from corrupting the data of another process.
;;
;; However, all we want to do here is initialize the table once and then
;; read it without ever changing it. It makes little sense to create
;; multiple copies of the same data.
;;
;; Therefore, we create our own segment, and call it .crc, or whatever
;; else we want. We tell the linker to share the data in .crc among all
;; processes with the following command line option:
;;
;;      link [other options] -section:.crc,rws
;;
;; The "rws" stands for: read, write, share.
;;
;; Incidentally, the program could be improved by having the crc-32 table
;; hardcoded instead of creating it when the DLL is first loaded. This code
;; creates the table on the fly simply to illustrate how to run some code
;; only once, when the DLL is first loaded.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
.crc    segment
align DWORD
crc32tbl        DWORD   256 dup(0)
.crc    ends

.const
about   db      "About CRC32.DLL", 0
msg     db      "CRC-32 library (crc32.dll), version 2"
        db      0Ah, 0Ah, "Copyright ", 169, " 1997, 1999 G. Adam Stanislav", 0Ah
        db      "All rights reserved.", 0Ah, 0Ah
        db      "For more information send e-mail to", 0Ah
        db      "whizkid@bigfoot.com", 0

.code
align DWORD
dll     proc    stdcall public, instance:DWORD, reason:DWORD, reserved:DWORD

        ; Do this only if the DLL is being attached to a process
        cmp     DWORD PTR [esp+8], DLL_PROCESS_ATTACH
        jne     @F

        ; Do this only once. How do we know? When first loaded, crc32tbl
        ; is all zeros. Once initialized, the first DWORD = 0, everything
        ; else is non-zero. So, all we need to do, is compare one of
        ; them to a zero. If it is NOT a zero, the table has been
        ; initialized by another process.
        cmp     DWORD PTR crc32tbl[4], 0
        jne     @F

        call    init?table

@@:
        mov     al, 1   ; Must return something other than 0 in EAX
        ret     12

dll     endp

align DWORD
init?table      proc    syscall private

        ; Here is what it would look like in C:
        ;
        ;   for (i = 0; i < 256; i++) {
        ;      crc = i;
        ;      for (j = 0; j < 8; j++) {
        ;         if (crc & 1)
        ;            crc = (crc >> 1) ^ 0xEDB88320;
        ;         else
        ;            crc >>= 1;
        ;         crc32tbl[i][j] = crc;
        ;      }
        ;   }
        ;
        ; And here is a much more elegant way of doing it:
        mov     ecx, 256        ; repeat for every DWORD in table
        mov     edx, 0EDB88320h

$BigLoop:
        lea     eax, [ecx-1]
        push    ecx
        mov     ecx, 8

$SmallLoop:
        shr     eax, 1
        jnc     @F
        xor     eax, edx

@@:
        loop    $SmallLoop
        pop     ecx
        mov     crc32tbl[ecx*4][-4], eax
        loop    $BigLoop

        ret

init?table      endp

align DWORD
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Two versions of the CRC-32 function are present here, one suitable
;; for calls from HLL like C or C++, the other for calls from assembly
;; language programs.
;;
;; The ASM version is much faster since its parameters are passed in
;; CPU registers rather than on the stack. Also, since typically this
;; function is called in a loop in which the result of the previous
;; call passed as a parameter to the next, the assembly-callable
;; routine expects that parameter to stay in the same register (EAX),
;; thus considerably reducing caller's own overhead.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
Crc32   proc    stdcall public, oldCrc:DWORD, aByte:DWORD

        ; This can be called from 32-bit C programs. It would be
        ; declared something like this:
        ;
        ;       UINT WINAPI Crc32(UINT oldCrc, UINT aByte);
        ;
        ; Where UINT is a 32-bit unsigned integer.
        ; "abyte" is a byte (8-bits) passed on 32-bit stack.
        ; "oldcrc" is the result of the previous call to this
        ; function, or 0xFFFFFFFF in the first call to it.
        ;
        ; A slower version of this procedure would look
        ; something like this:
        ;
        ;   push    ebp
        ;   mov     ebp, esp
        ;   mov     edx, [ebp+8]
        ;   mov     eax, [ebp+12]
        ;   call    asmcrc32
        ;   leave
        ;   ret     8
        ;
        ; stdcall procedures expect parameters to be pushed from
        ; right to left (like C functions) and to clean up the
        ; stack (like Pascal procedures and functions).
        ;
        ; Rather than adding the overhead of another call,
        ; we just move the parameters to the registers required
        ; by asmcrc32 while cleaning up the stack, and continue.

        pop     ecx     ; Save return address
        pop     eax     ; oldCrc
        pop     edx     ; aByte
        push    ecx     ; Restore return address

        ; Fall through

Crc32   endp

; I could put another align DWORD here, but since the above proc is
; exactly 4 bytes long, we are already aligned.
crc32   proc    syscall public

        ; This is a simple routine that calculates CRC-32 quickly
        ; by using a look-up table.
        ;
        ; Since it is an assembly language routine, we do not need
        ; to pass parameters on the stack. We just pass oldcrc in EAX,
        ; the byte in DL.
        ;
        ; In C, it would look something like:
        ;
        ;   temp = (oldcrc ^ abyte) & 0x000000FF;
        ;   crc  = (( oldcrc >> 8) & 0x00FFFFFF) ^ crc32tbl[temp];
        ;   return crc;
        ;
        ; On entry:
        ;       EAX = old CRC-32
        ;        DL = a byte
        ; On exit
        ;       EAX = new CRC-32
        ;       EDX = ?
        xor     dl, al
        movzx   edx, dl
        shr     eax, 8
        xor     eax, crc32tbl[edx*4]

        ret

crc32   endp

align DWORD
InitCrc32       proc    stdcall public

        ; Call from HLL to set the initial value of CRC-32:
        ;       UINT WINAPI InitCrc32(void);

        ; Fall through

InitCrc32       endp

initcrc32       proc    syscall public

        ; Call from assembly language programs

        mov     eax, -1
        ret

initcrc32       endp

        ; It is a good idea to align fall-through procs
        ; just ahead of the proc they are falling through to.
        ; Otherwise the assembler will insert dead code
        ; AFTER them, and slow down their execution.
align DWORD
nop
FinishCrc32     proc    stdcall public, oldCrc:DWORD

        ; Call from HLL to get the final value of CRC-32:
        ;       UINT WINAPI FinishCrc32(UINT oldCrc);
        pop     edx     ; Save return address
        pop     eax     ; oldCrc
        push    edx     ; Restore return address

        ; Fall through

FinishCrc32     endp

align DWORD
finishcrc32     proc    syscall public

        ; Call from assembly language programs
        not     eax
        ret

finishcrc32     endp

align DWORD
ArrayCrc32      proc    stdcall public, arrayPtr:DWORD, arraySize:DWORD
        ; Calculate CRC-32 of an entire array of bytes:
        ;       UINT WINAPI ArrayCrc32(BYTE *arrayPtr, UINT arraySize);
        ;
        ; The first parameter is a pointer to an array of bytes,
        ; the second one contains the number of the bytes in the array.
        ;
        ; In C, this function would look something like this:
        ;
        ;       UINT i;
        ;       UINT crc = InitCrc();
        ;
        ;       for (i = 0; i < arraySize; i++)
        ;          crc = Crc32(crc, arrayPtr[i];
        ;
        ;       return FinishCrc32(crc);
        ;
        pop     edx     ; Save return address
        pop     ebx     ; arrayPtr
        pop     ecx     ; arraySize
        push    edx     ; Restore return address

        ; Fall through

ArrayCrc32      endp

arraycrc32      proc    syscall public

        ; Calculate the CRC-32 of an array of bytes
        ; On entry:
        ;       EBX = address of first byte
        ;       ECX = number of bytes in array
        ; On exit:
        ;       EAX = CRC-32 of the entire array
        ;       EBX = ?
        ;       ECX = 0
        ;       EDX = ?

        mov     eax, -1 ; Faster than calling initcrc32
        jecxz   $Done   ; Just in case the array is empty
        or      ebx, ebx
        jz      $Done   ; Avoid NULL pointers

@@:
        mov     dl, [ebx]
        call    crc32
        inc     ebx
        loop    @B

        not     eax     ; Faster than calling finishcrc32

$Done:
        ret

arraycrc32      endp

; align this just ahead of the assembly language function
mov eax, eax
nop
PartialCrc32    proc    stdcall public, oldCrc: DWORD, arrayPtr:DWORD, arraySize:DWORD
        ; Use this function when you cannot fit all data into
        ; a single array. The code here is identical to that of ArrayCrc32,
        ; except it does not initialize the CRC to FFFFFFFF, and it does not
        ; do the final notting.
        ;       UINT WINAPI PartialCrc32(UINT oldCrc, BYTE *arrayPtr, UINT arraySize);
        ;
        ; To calculate the CRC-32 of data you cannot fit into a single data,
        ; do something like this:
        ;
        ; UINT crc = InitCrc32();
        ; while (more_data_coming) {
        ;   nbytes = GetMoreData(array);
        ;   crc = PartialCrc32(crc, array, nbytes);
        ; }
        ; crc = FinishCrc32();
        pop     edx     ; Save return address
        pop     eax     ; oldCrc
        pop     ebx     ; arrayPtr
        pop     ecx     ; arraySize
        push    edx     ; Restore return address

        ; Fall through

PartialCrc32    endp

align DWORD
partialcrc32    proc    syscall public

        ; On entry:
        ;       EAX = old CRC-32
        ;       EBX = address of array
        ;       ECX = number of bytes in array
        ; On exit:
        ;       EAX = new CRC-32
        ;       EBX = ?
        ;       ECX = 0
        ;       EDX = ?
        jecxz   $Done   ; Just in case the array is empty
        or      ebx, ebx
        jz      $Done   ; Avoid NULL pointers

@@:
        mov     dl, [ebx]
        call    crc32
        inc     ebx
        loop    @B

$Done:
        ret

partialcrc32    endp

; This just happens to be aligned where we want it: just ahead of aboutcrc32
AboutCrc32      proc    stdcall public, hwnd:DWORD

        ; Display "about" information about this DLL
        ;       void WINAPI AboutCrc32(HWND hwnd);
        pop     edx     ; Save return address
        pop     eax     ; Get window handle
        push    edx     ; Restore return address

        ; Fall through

AboutCrc32      endp

align DWORD
aboutcrc32      proc    syscall public

        ; Display information about this DLL
        ; On entry:
        ;       EAX = hwnd
        ; On exit
        ;       EAX = return value of MessageBox
        push    MB_OK or MB_ICONINFORMATION
        push    NEAR PTR about
        push    NEAR PTR msg
        push    eax
        call    DWORD PTR [__imp__MessageBoxA@16]
        ret

aboutcrc32      endp

end
