/*************************************************************************
 *   tracer driver Copyright (c) 2008 deroko of ARTeam
 *   This file is part of tracer driver.
 *
 *   xtracer is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   xtracer is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with xtracer.  If not, see <http://www.gnu.org/licenses/>.
 *
 *************************************************************************/
 
/*****************************************************************************
 * spinlock.c 
 *
 * This routine implements stuff needed to allow only one event at the time to be
 * reported to ring3 application.
 *****************************************************************************/
#include "defs.h"


extern TRACER_DATA td;
extern ULONG       tracer_init;
extern PKEVENT     ring3event;
extern FAST_MUTEX  fast_mutex;
extern ULONG       traced_pid;
extern ULONG   rdr0, rdr1, rdr2, rdr3, rdr6, rdr7;
KEVENT spin_lock_event;
ULONG  internal_lock = 0;

/*****************************************************************************
 * this procedure is responsible for cleanup of all internally used data...
 * at this point user doesn't care anymore about tracing, either he used
 * STOP_TRACER or he used CloseHandle() on this device object...
 *****************************************************************************/
VOID clean_up(){
        if (!tracer_init) return;
        
        ExAcquireFastMutex(&fast_mutex);
        InterlockedExchange(&tracer_init, 0);
        
        while (internal_lock){
                // if we are in exception s_default will signal 
                // to call default handler... which could lead to
                // application crash... or exit... if handler is
                // busy... for callbacks this has no practical meaning
                // as those will simply return...
                KeSetEvent(&spin_lock_event, IO_NO_INCREMENT, FALSE);
                InterlockedExchange(&td.dwStatus, StatusDefault);  
                ZwYieldExecution();          
        }       
        InterlockedExchange(&traced_pid, 0xFFFFFFFF);
        ObDereferenceObject(ring3event);
        // also clear all debug registers which might have been used or
        // setup by the user... doh...
        rdr0 = rdr1 = rdr2 = rdr3 = 0;
        rdr6 = 0xFFFF0FF0;
        // reserved bit 10 - set to 1
        // when debugger is not active clear LE and GE
        rdr7 = 0x400;                   
        fire_dpc();
        ExReleaseFastMutex(&fast_mutex);
        
} 
/******************************************************************
 * spin_lock which only waits for StatusContinue, StatusDefault and
 * StatusCallSoftice aren't set...
 ******************************************************************/
VOID spin_loop(){
        while (td.dwStatus != StatusContinue && td.dwStatus != StatusDefault && td.dwStatus != StatusExecuteSoftice)
                ZwYieldExecution();
}

VOID InitializeInternalLock(){
        // note that state of the event is set to signaled so first to
        // call KeWaitForSingleObject will immidiately return, and continue
        // but later calls will wait untill event isn't signaled again!!!
        KeInitializeEvent(&spin_lock_event, SynchronizationEvent, TRUE);
        internal_lock = 0;
}


VOID AcquireInternalLock(){
        InterlockedIncrement(&internal_lock);
        KeWaitForSingleObject(&spin_lock_event, Executive, KernelMode, FALSE, NULL);
}

VOID ReleaseInternalLock(){
        InterlockedDecrement(&internal_lock);
        KeSetEvent(&spin_lock_event, IO_NO_INCREMENT, FALSE);
}

