/*
    File:   eq.c
    Author: Jonathan Dale Kirwan

    Creation Date:  Fri 30-Dec-2005 14:34:49
    Last Modified:  Mon 13-Feb-2006 13:58:35

    Copyright 2005, 2006, Jonathan Dale Kirwan, All Rights Reserved.

    URL:    http://users.easystreet.com/jkirwan/new/c_expr.html
    email:  jkirwan@easystreet.com


    DESCRIPTION

    This program is a command line calculator and supports looping, for the
    generation of a series of calculations.  The command line arguments may be
    any of the following:

        expression          4*exp(-t/5)
        statement           x=2*sin(2*pi*hz)
        loop start          {expression -or- {statement
        loop end            }
        record output       ":two numbers &v1, &v2"

    An expression is calculated and printed, one per line.  A statement is
    simply calculated and assigned to the variable, but nothing is printed.
    A loop start begins with an open-setsign and allows either an expression
    or a statement to immediately follow it -- the value of which will be the
    number of iterations for the loop.  A loop end is simply a close-setsign.
    Anything else after the close set-sign on that argument is ignored (but
    that doesn't mean following arguments are ignored.)  And a record output
    simply writes out the text, with variable value substitutions inserted
    where there are &vars listed in it.  That's about it.

    An example of the arguments to print out a list of sine values from 0 to
    PI/2 would be:

        n=10 a=0 {n+1 sin(a) a=a+pi/2/n }

    If there's a need to print out two values (or more) per line, then the
    record output can be used.  For example, the above example can be changed
    to display both the angle and the sine, separated by a comma for easy
    inclusion into programs like Excel:

        n=10 a=0 {n+1 s=sin(a) ":&a,&s" a=a+pi/2/n }

    The only requirement is to place the value of the sine into a variable,
    so that it can be referenced in the record output field.  The double
    quotes aren't always needed, but it's good practice to get used to using
    them because of including spaces and the like.

    The following example illustrates the use of a significance specifier:

        sin(.5)#5

    That prints out the sine to 5 significant places.  The specifier can also
    be used in record outputs.  As in:

        r=sin(.5) :&r#3

    Which will print out the value of 'r' to 3 significant places.

    No specifier is allowed for the start loop argument or for statements.
    More details can be found at the web site mentioned near the top of this
    module.


    TARGET COMPILER

    Microsoft 16-bit Visual C/C++, IDE version 1.52C (DOS version 8.00c)


    COPYRIGHT NOTICE

    You are granted a non-transferable, non-exclusive, royalty-free worldwide
    license to use, copy, modify, prepare derivative works of and distribute
    this software, subject to your agreement that you acquire no ownership
    right, title, or interest in this software and your agreement that this
    software is research work that is provided 'as is.'

    Jonathan Dale Kirwan disclaims all warranties with regard to this source
    code software, including any and all implied warranties of merchantability
    and fitness of purpose.  In no event shall Jonathan Dale Kirwan be liable
    for any direct, indirect, consequential or special damages or any damages
    whatsoever resulting from loss of use, loss of data or profits, whether in
    an action of contract, negligence or other tortious action, arising out of
    or in connection with the use or performance of this software.

 
    MODIFICATIONS
 
    Wed 04-Jan-2006 00:23:33    jk  Added gnumber( ) and various comments.
    ----------------------------------------------------------------------
        The older output routine didn't handle exponential values, so I
        added a routine using _gcvt( ), instead.  Also added fuller comments
        to the code.  Also changed the variable indicator for record output
        fields from % to &, because of the special meaning of % under DOS.

    Wed 04-Jan-2006 13:08:54    jk  Loop limit modified.
    ----------------------------------------------------------------------
        The expression after the '{' specifier can be empty.  If so, the
        expression is zero and no iterations took place.  I changed this to
        set the default to 1 for anything less than 1 in the expression.
        This has the effect of passing through at least once, which makes
        more sense to a user, I think.

    Thu 05-Jan-2006 02:43:55    jk  Added error checking, fixed bug.
    ----------------------------------------------------------------------
        After modifying expr.c to support it, I added support here to check
        the parsing of the expressions to make sure that all of the argument
        was processed -- otherwise, it's an error.  Also moved the program
        name variable from inside of main() to a module-static so that all
        the routines can access it for error messages.  I also fixed a minor
        bug where the block_t array wasn't freed up in freepreparedblock( ).

    Thu 05-Jan-2006 16:54:14    jk  Added significance specifiers.
    ----------------------------------------------------------------------
        Added the '#' optional specifier for expressions and record output
        arguments.  A maximum of 17 digits is allowed, as that is the limit
        for double values.  Also made it an error to have characters follow
        the loop end argument.

    Fri 06-Jan-2006 01:06:28    jk  Removed _gcvt( ) call and fixed bug.
    ----------------------------------------------------------------------
        Fixed a bug in gnumber( ) and removed the use of the non-standard
        library call, _gcvt( ).  Also added code to handle a few more cases
        of output after switching to sprintf( ) -- cases that are NOT likely
        but not expensive to add and worth it just in case there are slight
        variations in different libraries (incompatibilities.)

    Sun 15-Jan-2006 17:13:59    jk  Added some exception handling code.
    ----------------------------------------------------------------------
        Added except.c and assert.c to the project and modified eq.c so it
        uses the exception handler when malloc fails to provide a handle to
        requested memory.  No exception handler was added to main(), so such
        errors currently display the default message.  Also added a program
        version number (1.006 now.)

    Mon 13-Feb-2006 13:54:23    jk  Added clipboard features and help.
    ----------------------------------------------------------------------
        Added clipbrd.c and clipbrd.h to the project and modified main.c
        to support clipboard pasting.  This uses the -c option.  Also added
        help using the -h option.
*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <string.h>
#include <ctype.h>

#include "expr.h"
#include "rnd.h"
#include "gaussrnd.h"
#include "assert.h"
#include "except.h"
#include "clipbrd.h"

// #include "malchk.h"      /* for malloc() and free() checking */

#define VERSION     "1.1"
#define MAXCLIP     40000

#define PI          (3.1415926535897932384)
#define DEFSIG      (10)        /* default significance */


typedef enum blocktype_t { tLOOP, tRETURN, tEXPR, tSTMT, tHEADER } blocktype_t;
typedef struct block_t {
    int type;
    int significance;
  union {
    pushpop_t expression;
    char *headerlist;
  };
} block_t;


int main( int argc, char *argv[] );
static block_t * prepareargs( char *argv[], int argc );
static void freepreparedblock( block_t *bb, int count );
static char * execute( block_t *blks, int count, char *clip );
static char * programname( const char *s );
static char * gnumber( double fv, int decimal );
static int significance( char **p );
static void unhandledexception( const char *msg );
static void displayhelp( FILE *f );


static const except_t memoryfailure= { "memory allocation failed" };
static char *pn= 0;         /* globally available copy of the program name */
static char cliptext[MAXCLIP];
#define addc(p,a)   do { if ( (p) && (p) < &cliptext[MAXCLIP-1] ) *(p)++= (a); } while (0)
#define adds(p,a)   do { if ( p ) { auto char *z, c; for ( z= (a); (c= *z++) != '\0'; ) \
                    if ( (p) < &cliptext[MAXCLIP-1] ) *(p)++= c; else break; } } while (0)


/*  ----------------------------------------------------------------------  */
int main( int argc, char *argv[] ) {
/*  ----------------------------------------------------------------------
    This is the program-start routine.  The arguments are processed as a
    list of math statements or expressions, along with a few special cases
    that support looping and printing out special headers or messages.

    See the file header for a fuller explanation of the tasks to be done
    here.
*/
    auto block_t *bb;
    auto int clipboard, help;

        UNHANDLED( unhandledexception );

        /* Get the program name. */
        pn= programname( argv[0] );

        /* Initialize the random number generators. */
        randomize( 5489UL * (unsigned long) time( 0 ) );
        rndgauss_init( );

        /* Initialize the variable table. */
        initvars( );
        evaluate( "pi = 4 * atan( 1 )", 0 );
        evaluate( "e  = exp( 1 )", 0 );
        evaluate( "version=" VERSION, 0 );

        /* Process the optional first argument, -c as clipboard copy flag, -h for help. */
        clipboard= help= 0;
        if ( strcmp( argv[1], "-c" ) == 0 ) {
            clipboard= 1;
            ++argv;
            --argc;
        } else if ( strcmp( argv[1], "-h" ) == 0 ) {
            help= 1;
            ++argv;
            --argc;
        }

        /* Put out version, if no command line arguments provided. */
        if ( help ) {
            displayhelp( stdout );
            free( pn );
            return 0;
        } else if ( argc < 2 ) {
            fprintf( stderr, "%s, version " VERSION ".  Compiled: " __DATE__ " " __TIME__ ".  Use '%s -h' for help.\n", pn, pn );
            free( pn );
            return 0;
        }

        /* Process the arguments as equations. */
        bb= prepareargs( argv + 1, argc - 1 );
        if ( bb != 0 ) {
            *execute( bb, argc - 1, clipboard? cliptext : 0 )= '\0';
            freepreparedblock( bb, argc - 1 );
            if ( clipboard ) {
                clpOpen( );
                clpEmpty( );
                clpCopyText( cliptext, strlen( cliptext ) );
                clpClose( );
            }
        }

        /* Free up the existing variable table. */
        initvars( );

        /* Free up the program name. */
        free( pn );

    return 0;
}


/*  ----------------------------------------------------------------------  */
static char * programname( const char *s ) {
/*  ----------------------------------------------------------------------
    This routine extracts the simple program name invoked at the command
    line from the first argv[] argument (passed as 's'.)  The returned
    string is allocated via malloc().
*/
    auto char dv[_MAX_DRIVE], dr[_MAX_DIR], fn[_MAX_FNAME], ex[_MAX_EXT];
    auto size_t slen;
    auto char *r;

        _splitpath( s, dv, dr, fn, ex );
        slen= strlen( fn );
        if ( slen < 1 ) {
            strcpy( fn, "DEMO" );
            slen= strlen( fn );
        }
        r= (char*) malloc( slen + 1 );
        errorif( !r, memoryfailure );
        strcpy( r, fn );
        _strlwr( r );

    return r;
}


/*  ----------------------------------------------------------------------  */
static block_t * prepareargs( char *argv[], int argc ) {
/*  ----------------------------------------------------------------------
    This routine processes each command line argument into a list of
    pre-compiled expressions and statements for execution as a basic
    block, of sorts.  Execution isn't handled here, just pre-compiling.
*/
    auto int i, count;
    auto block_t *b;

        /* allocate an array of compiled entries */
        b= (block_t *) malloc( argc * sizeof( block_t ) );
        errorif( !b, memoryfailure );

        /* initialize the pointers to 0, just in case we have to abort */
        for ( i= 0; i < argc; ++i )
            b[i].expression= 0;

        /* process each argument into its entry */
        for ( count= 0, i= 0; i < argc; ++i, ++count ) {
          auto char *endp;
          auto char c= *argv[i];
            switch ( c ) {

            /*  LOOP:  Compile the expression or statement for later.
            */
            case '{':
                b[count].type= tLOOP;
                b[count].expression= translate( argv[i]+1, &endp );
                if ( endp != 0 && *endp != '\0' ) {
                    printf( "%s: extra characters in loop start expression '%s'\n", pn, argv[i] );
                    freepreparedblock( b, i+1 );
                    return 0;
                }
                break;

            /*  RETURN:  Nothing else to worry about, except noting it.
            */
            case '}':
                b[count].type= tRETURN;
                b[count].expression= 0;
                if ( argv[i][1] != '\0' ) {
                    printf( "%s: extra characters in loop end '%s'\n", pn, argv[i] );
                    freepreparedblock( b, i+1 );
                    return 0;
                }
                break;

            /*  HEADER:  Save a pointer to the literal string.
            */
            case ':':
                b[count].type= tHEADER;
                b[count].headerlist= argv[i]+1;
                break;

            /*  STMT or EXPR:  Check for '=' to determine which of
                the two it is.  Compile and save the code for later.
            */
            default:
                if ( strchr( argv[i], '=' ) == 0 ) {
                    b[count].type= tEXPR;
                    b[count].expression= translate( argv[i], &endp );
                    b[count].significance= significance( &endp );       /* specifier allowed */
                } else {
                    b[count].type= tSTMT;
                    b[count].expression= translate( argv[i], &endp );
                    b[count].significance= DEFSIG;                      /* specifier not allowed */
                }
                if ( endp != 0 && *endp != '\0' ) {
                    printf( "%s: extra characters in %s '%s'\n", pn, b[count].type == tEXPR? "expression":"statement", argv[i] );
                    freepreparedblock( b, i+1 );
                    return 0;
                }
                break;
            }
        }

    return b;
}


/*  ----------------------------------------------------------------------  */
static void freepreparedblock( block_t *bb, int count ) {
/*  ----------------------------------------------------------------------
    Just free up any allocated memory from the compilations stored in the
    block.
*/
    auto int i;

        if ( bb == 0 )
            return;

        for ( i= 0; i < count; ++i )
            switch( bb[i].type ) {
            case tLOOP:
            case tEXPR:
            case tSTMT:
                freepp( bb[i].expression );
                break;
            default:
                break;
            }

        free( bb );

    return;
}


#define outc(a)     do { addc( clip, a ); putchar( a ); } while (0)
#define outs(a)     do { adds( clip, a ); printf( "%s", (a) ); } while (0)
#define newline()   do { addc( clip, '\r' ); addc( clip, '\n' ); putchar( '\n' ); } while (0)

/*  ----------------------------------------------------------------------  */
static char * execute( block_t * blks, int count, char *clip ) {
/*  ----------------------------------------------------------------------
    This routine executes the block of processed arguments.  Of particular
    interest will be the method used to handle looping.
*/
    auto int i, z, limit, lc;
    auto double v;
    auto char c, *s, *t, *var, *varv;

        for ( i= 0; i < count; ++i ) {
            switch ( blks[i].type ) {

            /*  In the case of the loop statement (one whose argument
                began with '{'), the idea is to evaluate the expression
                or statement given to get the loop count required and to
                then recursively call this routine to execute each loop,
                starting though at the next list entry.  Such calls can
                handle nexted loops.  When the loop count is completed,
                the code then searches for its matching RETURN ('}') and
                allows execution to continue at that point.
            */
            case tLOOP:
                /* get count */
                limit= (int) floor( evaluatepp( blks[i].expression ) );
                if ( limit < 1 )
                    limit= 1;
                /* execute a loop with that number of iterations */
                for ( z= 1; z <= limit; ++z )
                    clip= execute( blks + i + 1, count - i - 1, clip );
                /* seach for the corresponding '}' or RETURN, and continue */
                for ( lc= 1; lc > 0; ) {
                    if ( ++i >= count )
                        return clip;    /* no more in the list, assume return */
                    if ( blks[i].type == tRETURN )
                        --lc;
                    else if ( blks[i].type == tLOOP )
                        ++lc;
                }
                break;

            /*  In this case, the '}' or RETURN, simply return to the
                caller.  This should normally only happen if there was
                a starting loop before.  But if the arguments didn't
                happen to include a loop-start, then this return will
                simply end all the execution and no more arguments will
                be processed.
            */
            case tRETURN:
                return clip;

            /*  The header case simply prints out the indicated message
                while replacing any variables within it with their
                current values.  A variable name is preceded by a '&'
                character.
            */
            case tHEADER:
                for ( s= blks[i].headerlist; (c= *s) != '\0'; )
                    if ( c == '&' && s[1] != '\0' && isalpha( s[1] ) ) {
                        for ( t= s+2; *t != '\0'; ++t )
                            if ( !isalnum( *t ) )
                                break;
                        var= (char *) malloc( t - s );
                        errorif( !var, memoryfailure );
                        strncpy( var, s + 1, t - s - 1 );
                        var[t - s - 1]= '\0';
                        varv= gnumber( evaluate( var, 0 ), significance( &t ) );
                        outs( varv );
                        free( varv );
                        free( var );
                        s= t;
                    } else
                        outc( *s++ );
                newline( );
                break;

            /*  Expressions are evaluated and printed.
            */
            case tEXPR:
                v= evaluatepp( blks[i].expression );
                s= gnumber( v, blks[i].significance );
                outs( s );
                free( s );
                newline( );
                break;

            /*  Statements are simply evaluated.
            */
            case tSTMT:
                evaluatepp( blks[i].expression );
                break;

            /*  The rest is ignored, not knowing what it may be.
            */
            default:
                break;
            }
        }

    return clip;
}


/*  ----------------------------------------------------------------------  */
static char * gnumber( double fv, int decimal ) {
/*  ----------------------------------------------------------------------
    This routine generates an ASCII number version of the given floating
    point value in 'fv', with the number of significant decimal digits
    indicated by 'decimal'.  The returned string is allocated via malloc().
*/
    auto char c, *p, buf[30];

        /* Handle non-error computation results. */
        if ( fv != HUGE_VAL ) {
          auto char spec[10];
            /* produce the specification for sprintf() and then apply it */
            sprintf( spec, "%%.%dlg", decimal );
            sprintf( buf, spec, fv );
            /* though, could have used the non-standard, non-portable form,
                _gcvt( fv, decimal, buf );
            */
            /* handle non-exponential formats */
            if ( (p= strchr( buf, 'e' )) == 0 ) {
                if ( strchr( buf, '.' ) != 0 ) {
                    p= & buf[ strlen( buf ) ];
                    while ( (c= *--p) != '.' )
                        if ( c != '0' ) {
                            ++p;
                            break;
                        }
                    *p= '\0';
                }
            }
            /* handle exponential formats */
            else {
              auto char *x;
                if ( p[-1] == '.' )
                    strcpy( p-1, p );
                else
                    ++p;
                if ( (c= *p) == '+' || c == '-' )
                    ++p;
                for ( x= p; *x == '0'; ++x )
                    ;
                strcpy( p, x );
            }
        }

        /* Must be an error. */
        else
            strcpy( buf, "???" );

        /* Allocate a new string and copy the formatted number into it. */        
        p= (char *) malloc( strlen( buf ) + 1 );
        errorif( !p, memoryfailure );
        strcpy( p, buf );

    return p;
}


/*  ----------------------------------------------------------------------  */
static int significance( char **p ) {
/*  ----------------------------------------------------------------------
    This routine accepts a pointer to a character pointer and processes a
    significance count, if started by '#'.  The character pointer is
    advanced across that specifier for significance.
*/
    auto int digits;
    auto char c;

        digits= DEFSIG;
        if ( p && *p && **p == '#' ) {
            digits= 0;
            for ( ++(*p); (c= **p) != '\0'; ++(*p) ) {
                if ( !isdigit( c ) )
                    break;
                digits= 10 * digits + (c - '0');
                if ( digits > 17 )
                    digits= 17;
            }
            if ( digits < 1 )
                digits= DEFSIG;
        }

    return digits;
}


/*  ----------------------------------------------------------------------  */
static void unhandledexception( const char *msg ) {
/*  ----------------------------------------------------------------------
    This routine simply displays the exception message, in the case of
    unhandled exceptions.
*/
        if ( pn )
            fprintf( stderr, "%s: %s\n", pn, msg );
        else
            fprintf( stderr, "eq: %s\n", msg );

    return;
}


/*  ----------------------------------------------------------------------  */
static void displayhelp( FILE *f ) {
/*  ----------------------------------------------------------------------
    This routine simply displays help.
*/

        fprintf( f, "%s, version " VERSION ".  Compiled: " __DATE__ " " __TIME__ "\n", pn );
        fprintf( f, "\n"
            "%s takes a series of command line arguments and processes them as\n"
            "expressions or statements (which are just silent variable assignments.)\n"
            "It supports the use of looping, including nested looping, and writes\n"
            "out the results of expression calculations or record outputs.\n", pn );
        fprintf( f, "\n"
            "An expression or statement, though, may not include spaces because the\n"
            "space character is what separates one argument from another.  So if you\n"
            "want to include spacing in your expressions or statements, you'll need\n"
            "to use double-quote marks to surround them.\n" );
        fprintf( f, "\n"
            "An expression (one that doesn't assign a value to a variable) generates\n"
            "output to the screen, displaying the resulting value.  A statement does\n"
            "not, instead simply silently calculating the value and changing the\n"
            "variable's value, as indicated.  The only case where an expression does\n"
            "not generate output to the screen is when it is used to set the count\n"
            "of iterations for a loop.\n" );
        fprintf( f, "\n"
            "A loop is started by using an open-set-sign followed by an expression\n"
            "or statement in one command line argument.  If the open-set-sign is used\n"
            "without an expression or statement, that isn't an error and only one\n"
            "iteration of the loop will occur.  (Might as well not use it, though.)\n"
            "A loop is closed using the close-set-sign for an argument.\n" );
        fprintf( f, "\n"
            "There is also a type which permits arbitrary text mixed with values to\n"
            "be printed, which is the record output argument.  This argument starts\n"
            "with a colon to denote it.  Because of the limitation of spaces already\n"
            "mentioned, it is advisable to consistently enclose record outputs with\n"
            "double quote marks as a matter of practice.  A variable name may be\n"
            "included within the record output fields by using the ampersand with\n"
            "the variable name following it.  The program will fill in the value.\n" );
        fprintf( f, "\n"
            "An expression or an ampersand'd field in an output record can have the\n"
            "desired number of digits displayed by using a # character after the\n"
            "expression or variable name (in the output record), followed by a\n"
            "number to specify the significant digits to use.\n" );
        fprintf( f, "\n"
            "Here is a quick list of the argument types and some examples:\n"
            "\n"
            "  Argument Type       An Example or General Case\n"
            "  ---------------     ---------------------------\n"
            "  expression          4*exp(-t/5)\n"
            "  statement           x=2*sin(2*pi*hz)\n"
            "  loop start          {expression -or- {statement\n"
            "  loop end            }\n"
            "  record output       \":two numbers &v1, &v2\"\n" );
        fprintf( f, "\n"
            "Illustrative examples to try on your own:\n"
            "\n"
            "  %s sin(pi/3)\n"
            "  %s n=10 a=0 \":Sines n=&n\" {n+1 s=sin(a) \":&a,&s\" a=a+pi/2/n } \":End\"\n"
            "  %s n=10 a=0 {n+1 s=sin(a) \":&a#5,&s#4\" a=a+pi/2/n }\n", pn, pn, pn );
        fprintf( f, "\n"
            "And, by now, you may have already noticed the use of pi in the\n"
            "expressions.  Both 'pi' and 'e' are pre-defined values, as is\n"
            "'version'.\n" );
        fprintf( f, "\n"
            "Finally, the program also allows you to copy its output to the windows\n"
            "clipboard, if desired.  Use the -c option as the first argument, to get\n"
            "this feature.  For example:\n"
            "\n"
            "  %s -c sin(pi/3)\n"
            "\n"
            "That will place the value of the sine of pi/3 onto the windows clipboard\n"
            "as text that you can paste into other programs.  This feature also works\n"
            "with looping active, so you can get long outputs placed onto the clipboard.\n", pn );

    return;
}
