/*
    File:   except.c
    Author: Jonathan Dale Kirwan
 
    Creation Date:  Fri 08-Apr-2005 02:27:26
    Last Modified:  Mon 16-Jan-2006 00:07:09

    Copyright 2005, 2006, Jonathan Dale Kirwan, All Rights Reserved.

    URL:    http://users.easystreet.com/jkirwan/new/c_expr.html
    email:  jkirwan@easystreet.com


    DESCRIPTION

    This module provides a simple exception handler.  The routines provided
    here are usually not specifically called by user code, which should use
    the macros defined in except.h, instead.  See except.h for instructions.


    DESIGN

    I've broken this file into three main parts.  The first is the primary
    code for exception handling once an exception is raised.  The second and
    third parts, though, may be moved into a thread handler if the program
    is running multi-threaded, since in those cases separate static state is
    required for each thread.


    CREDITS

    The approach taken here was heavily drawn from David Hanson's book, "C
    Interfaces and Implementations."  In particular, Chapter 4.  I've used
    a different version of these routines in a multi-threaded application,
    so some of what I learned from that has been folded back into this code.


    TARGET COMPILER

    Microsoft 16-bit Visual C/C++, IDE version 1.52C (DOS version 8.00c)


    COPYRIGHT NOTICE

    You are granted a non-transferable, non-exclusive, royalty-free worldwide
    license to use, copy, modify, prepare derivative works of and distribute
    this software, subject to your agreement that you acquire no ownership
    right, title, or interest in this software and your agreement that this
    software is research work that is provided 'as is.'

    Jonathan Dale Kirwan disclaims all warranties with regard to this source
    code software, including any and all implied warranties of merchantability
    and fitness of purpose.  In no event shall Jonathan Dale Kirwan be liable
    for any direct, indirect, consequential or special damages or any damages
    whatsoever resulting from loss of use, loss of data or profits, whether in
    an action of contract, negligence or other tortious action, arising out of
    or in connection with the use or performance of this software.
 
 
    MODIFICATIONS
 
    Original source.
*/


#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "assert.h"
#include "except.h"

#define T except_t
#define U exceptframe_t


/*  ======================================================================
    PRIMARY MODULE SECTION
    ======================================================================
    Support for exception handling.
*/


/*  ----------------------------------------------------------------------  */
#if ( EXCEPT_DETAILS == 1 )
void exceptraise( const T *e, const char *file, int line ) {
#else
void exceptraise( const T *e ) {
#endif
/*  ----------------------------------------------------------------------
    This routine raises an exception event.  If 'exceptstack' is 0, then
    the exception was uncaught and this routine provides a default error
    message and then exits the program.

    Note that this code is _not_ safe from pre-emption, as it uses a
    static buffer, temporarily, in preparing a message.  It will support
    co-operating threading, though.
*/
    auto U *p= pexceptstack( );

        assert( e );

        /* Handle uncaught exceptions */
        if ( p == 0 ) {
          auto void (*func)( const char *msg );
          static char bufr[200];        /* beware of pre-emption here */
            strcpy( bufr, "uncaught exception:\n" );
            if ( e->reason )
                sprintf( bufr+strlen( bufr ), "    %s\n", e->reason );
            else
                sprintf( bufr+strlen( bufr ), "    at 0x%p\n", e );
#if ( EXCEPT_DETAILS == 1 )
            if ( file && line > 0 )
                sprintf( bufr+strlen( bufr ), "    raised at %s, line %d\n", file, line );
#endif
            sprintf( bufr+strlen( bufr ), "    --- aborting ---" );
            if ( (func= punhandled( )) ) {
              auto char *emsg;
                emsg= (char *) malloc( strlen( bufr ) + 1 );
                if ( emsg ) {
                    strcpy( emsg, bufr );
                    (*func)( emsg );
                    free( emsg );
                } else
                    (*func)( bufr );
            } else
                fprintf( stderr, "%s\n", bufr );
            exit( 1 );
        }

        /* Otherwise, handle caught ones */
        p->exception= e;
#if ( EXCEPT_DETAILS == 1 )
        p->file= file;
        p->line= line;
#endif
        psetexceptstack( p->prev );
        longjmp( p->env, EXCEPTRAISED );

    return;
}


/*  ======================================================================
    EXCEPTION STACK SECTION
    ======================================================================
    Normally, all that is needed for the exception stack is a single
    static instance of a pointer to the bottom of the exception stack (a
    linked FILO list) in order to serve an entire program's exception
    handling requirements.  For this, I could have simply defined that
    instance here and be done with it.

    However, for multi-threaded programming and each thread with its own
    stack, the setjump and longjump routines must be treated with some
    care.  It's not appropriate to be unwinding the stack of some thread
    into the middle of another thread's stack.  So each thread must have
    its own instance of the pointer.

    So I've divided it this way so that it is easy to port this for code
    written in a multi-threaded environment.  In such cases, there should
    be a separate instance of the exception stack for each thread and the
    following two routines should be re-coded, appropriately, so that they
    access the thread-specific exception stack rather than a global one.

    A likely approach might be for the thread handler to keep an instance
    in its process/thread data structures and to replace the routines here
    with routines from the thread handler that are aware of an appropriate
    instance to provide.
*/


/*  ----------------------------------------------------------------------  */
static U *exceptstack= 0;
/*  ----------------------------------------------------------------------
    This definition provides the start of the exception stack for a single
    threaded program.
*/


/*  ----------------------------------------------------------------------  */
void psetexceptstack( U *stack ) {
/*  ----------------------------------------------------------------------
    This function sets the exception stack for a single threaded program
    to point to the latest indicated entry.
*/
        exceptstack= stack;

    return;
}


/*  ----------------------------------------------------------------------  */
U *pexceptstack( void ) {
/*  ----------------------------------------------------------------------
    This function returns the most recent entry of the exception stack for
    a single threaded program.
*/
    return exceptstack;
}


/*  ======================================================================
    EXCEPTION MESSAGE CALLBACK SECTION
    ======================================================================
    Normally, all that is needed for the exception stack is a single
    static instance of a callback pointer that can be used in cases where
    there is an unhandled exception.  Tentative text for an error message
    is first created for unhandled exceptions and then provided to the
    callback function.  The callback may decide to print its own message
    or use the provided message; or some combination.

    As before, for multi-threaded programming and each thread with its own
    stack, separate callback handlers may be required so that thread-aware
    handling is supported.

    So I've divided it this way so that it is easy to port this for code
    written in a multi-threaded environment.  In such cases, there should
    be a separate instance of the callback handler pointer for each thread
    and the following two routines should be re-coded, appropriately, so
    that they access the thread-specific, unhandled exception callback
    handler rather than a global one.
*/


/*  ----------------------------------------------------------------------  */
static void (*callback)( const char *msg )= 0;
/*  ----------------------------------------------------------------------
    This definition provides a pointer to an unhandled exception message
    handler for a single-threaded program.  Normally, this handler should
    just output an appropriate message.
*/


/*  ----------------------------------------------------------------------  */
void psetunhandled( void (*handler)( const char *msg ) ) {
/*  ----------------------------------------------------------------------
    This function sets the pointer to the unhandled exception message
    handler for a single-threaded program.
*/
        callback= handler;

    return;
}


/*  ----------------------------------------------------------------------  */
void (* punhandled( void ))( const char *msg ) {
/*  ----------------------------------------------------------------------
    This function gets the pointer to the unhandled exception message
    handler for a single-threaded program.
*/
    return callback;
}
