/*
    File:   expr.c
    Author: Jonathan Dale Kirwan
 
    Creation Date:  Sat 13-Mar-2004 02:31:10
    Last Modified:  Mon 16-Jan-2006 00:47:53
         Modified:  Wed 04-Jan-2006 01:54:26
 
    Copyright 2004, 2006  Jonathan Dale Kirwan, All Rights Reserved.
 
 
    DESCRIPTION

    This module provides a recursive descent parser for simple expressions,
    in algebraic like notation.  I've commented the code only here and not
    so much in the routines themselves.

 
    DESIGN
 
    Normally, the expression to evaluate breaks into these terms, leaving out
    for now the definition of "number":
 
        expression  :=  term | expression + term | expression - term
        term        :=  factor | term * factor | term / factor
        factor      :=  element | factor ^ element
        element     :=  number | ( expression )
 
    The | means OR, so an expression is either a "term" or else it is an
    "expression + term" or else it is an "expression - term".
 
    The problem with the above description is that it is left-recursive.  If
    you write a routine for "expression" and if that routine decides to call
    itself for "expression + term", for example, it will call itself over and
    over again until the stack blows up.  Bad news!
 
    There's a reason that just using "term + expression" above won't work,
    either.  The order of the operations should be left to right, not right to
    left, so rewriting it the other way to fix the recursion problem would
    cause "5 - 4 - 3" to compute "5 - (4 - 3)" instead of "(5 - 4) - 3".  So
    just reordering things to avoid the infinite recursion isn't the answer,
    either.
 
    The idea is to rewrite it in a more implementable form, where the
    recursive call occurs at the end.  That's called right-recursive.  You
    start with the original definitions shown above, which properly handle
    "5 - 4 - 3" as "(5 - 4) - 3", but rework them into a right-handed form
    that still retains the left-to-right interpretations.
 
    Doing so, the above four items are now seven:
 
        expression  :=  term moreterms
        term        :=  factor morefactors
        factor      :=  element moreelements
        element     :=  number | ( expression )
        moreterms   :=  + term moreterms | - term moreterms | <null>
        morefactors :=  * factor morefactors | / factor morefactors | <null>
        moreelements:=  ^ element moreelements | <null>
 
    Plus, here's the definition for number:
 
        number      :=  sign mantissa scale
        sign        :=  + | - | <null>
        mantissa    :=  digit digits | digit digits . digits | . digit digits
        digit       :=  0 | 1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | 9
        digits      :=  digit digits | <null>
        scale       :=  scaleid sign digit digits | <null>
        scaleid     :=  e | E

    I've further extended the definitions:

        statement   :=  expression | mnemonic = expression
        expression  :=  term moreterms
        term        :=  factor morefactors
        factor      :=  element moreelements
        element     :=  sign mnemonic | number | sign unary ( expression )
        moreterms   :=  + term moreterms | - term moreterms | <null>
        morefactors :=  * factor morefactors | / factor morefactors | <null>
        moreelements:=  ^ element moreelements | <null>
        unary       :=  SIN | COS | TAN | SEC | CSC | COT | ATAN | ASIN |
                        ACOS | ASEC | ACSC | ACOT | ABS | SGN | INT | SQRT |
                        LN | LOG | EXP | RND | RNDG | SINH | COSH | TANH |
                        SECH | CSCH | COTH | DEG | RAD | CEIL | FLOOR | <null>

 
    Where a variable name or function name is:
 
        alpha := A | B | C | D | E | F | G | H | I | J | K | L | M |
                 N | O | P | Q | R | S | T | U | V | W | X | Y | Z |
                 a | b | c | d | e | f | g | h | i | j | k | l | m |
                 n | o | p | q | r | s | t | u | v | w | x | y | z
        alphanumeric := alpha | digit
        alphanumerics := alphanumeric alphanumerics | <null>
        mnemonic := alpha alphanumerics

    That's the general model for the recursive descent code illustrated below.

    The code checks for impossible values that are too large, division by 0,
    and the like.  Hopefully, no unhandled errors can occur.


    TARGET COMPILER

    Microsoft 16-bit Visual C/C++, IDE version 1.52C (DOS version 8.00c)


    COPYRIGHT NOTICE

    You are granted a non-transferable, non-exclusive, royalty-free worldwide
    license to use, copy, modify, prepare derivative works of and distribute
    this software, subject to your agreement that you acquire no ownership
    right, title, or interest in this software and your agreement that this
    software is research work that is provided 'as is.'

    Jonathan Dale Kirwan disclaims all warranties with regard to this source
    code software, including any and all implied warranties of merchantability
    and fitness of purpose.  In no event shall Jonathan Dale Kirwan be liable
    for any direct, indirect, consequential or special damages or any damages
    whatsoever resulting from loss of use, loss of data or profits, whether in
    an action of contract, negligence or other tortious action, arising out of
    or in connection with the use or performance of this software.
 
 
    MODIFICATIONS

    Wed 04-Jan-2006 00:23:33    jk  Added push-pop compiled code + statements.
    ----------------------------------------------------------------------
        I decided to add statements, a variable name table (no hashing, for
        now, just a simple list to scan), and the ability to pre-compile the
        expressions or statements into push-pop code.  The push-pop code will
        convert used variable names into direct memory references to the
        variables in the list (no searching required.)  Pre-defined function
        name references are also converted into direct function pointers, etc.
        In the process of this conversion, the expression or statement is
        also calculated with whatever values are available at the time.

    Wed 04-Jan-2006 01:53:03    jk  Fixed (t/5) equals (-t/5) bug.
    ----------------------------------------------------------------------
        Discovered a bug where a minus sign was parsed, but functionally
        ignored, if it appeared before a variable as part of an expression.

    Wed 04-Jan-2006 18:39:50    jk  Fixed skipspaces( ) bug.
    ----------------------------------------------------------------------
        The skipspaces( ) function would advance past the end of a string,
        which could lead the parsing to examine unknown memory.  Fixed.  I
        also moved around a few bits of code that adds push-pop code to the
        list, just to clean up the approach a little (so that executing the
        push-pop code doesn't even implicitly attempt to add code to a new
        list.)

    Wed 04-Jan-2006 19:07:57    jk  Added more math error checking.
    ----------------------------------------------------------------------
        Modified the _matherr( ) routine from its example code which wasn't
        used at all into some very simple code that will probably not hurt
        things too much.  This meant changing the project to use /NOE.

    Thu 05-Jan-2006 02:46:18    jk  Added support for parsing.
    ----------------------------------------------------------------------
        I added a second parameter to each of evaluate( ) and translate( ).
        This parameter acts like the second parameter to strtod( ), where
        if a pointer to a string pointer is provided, then that pointer is
        updated to point to the first character that couldn't be parsed as
        part of either an expression or statement.  This allows the caller
        to do further processing on the remainder, if desired, or to add
        error messages.  Without this addition, the caller was blinded to
        where the parsing stopped.  Also corrected some comments and made
        slight adjustments to the code for clearity and to remove redundancy
        in alphanumeric( ).  Finally, switched from #defines to enum for the
        pushpopid_t.
*/

#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "rnd.h"
#include "gaussrnd.h"
#include "expr.h"
// #include "malchk.h"      /* for malloc() and free() checking */


/*  ----------------------------------------------------------------------  */
typedef int bool_t;
/*  ----------------------------------------------------------------------
    Return type for functions analyzing expressions and statements -- it
    indicates whether or not the parsing syntax was satisfied.  0 = fail,
    otherwise it succeeded.  Any parsing that fails resets the pointer
    into the algebraic expression back to the starting place before the
    failure.
*/


/*  ----------------------------------------------------------------------  */
typedef enum pushpopid_t {
/*  ----------------------------------------------------------------------
    Push-pop statement types (or IDs.)
*/
    START= 0,               /* first push-pop code for compiled code */
    PUSHVALUE,              /* push double value operand */
    ADD,                    /* add top two on stack, push result */
    SUBTRACT,               /* subtract top two on stack, push result */
    MULTIPLY,               /* multiply top two on stack, push result */
    DIVIDE,                 /* divide top two on stack, push result */
    POWER,                  /* x^y, top two on stack, push result */
    NEG,                    /* negate top value on stack */
    SETVAR,                 /* copy top of stack into variable */
    PUSHVAR,                /* push variable operand */
    UNARY,                  /* perform function in operand */
    POPVAL                  /* last push-pop code, pop top value on stack */
} pushpopid_t;


/*  ----------------------------------------------------------------------  */
struct pushpop_t {
/*  ----------------------------------------------------------------------
    Push-pop statement node.  A linked list of these are generated when
    compiling expressions/statements.
*/
    pushpop_t p;                /* pointer to the next push-pop statement */
    void *operand;              /* pointer to an associated operand or 0 */
    pushpopid_t op;             /* push-pop ID -- types shown below here */
};


/*  ----------------------------------------------------------------------  */
typedef struct value_t {
/*  ----------------------------------------------------------------------
    Value stack node.  A linked list of these are generated and modified
    during the execution of an expression or statement.  Used to aid the
    computation process.  This list is maintained as a true stack; FILO.
*/
    struct value_t *p;          /* pointer to the next value */
    double v;                   /* value of the node */
} value_t;


/*  ----------------------------------------------------------------------  */
typedef struct expr_t {
/*  ----------------------------------------------------------------------
    Expression header.  One of these is needed during processing.  It
    holds the start of the value stack (linked list of value nodes) and
    the start of the generated push-pop code (also a linked list.)  The
    value of both may be 0 to start.  The value stack will simply be
    added to, as needed, so a null pointer for it does not affect the
    operation.  However, if the pointer to the list of push-pop code is
    null, then no push-pop code is ever added.  Push-pop code is added
    only if the linked list actually points to some push-pop code that
    already exists.  This is the reason for the START push-pop code item.
    This list is maintained as a FIFO list -- push-pop code is added to
    the end of the list.
*/
    struct value_t *s;          /* pointer to value stack */
    pushpop_t e;                /* pointer to push-pop code */
} expr_t;


/*  ----------------------------------------------------------------------  */
typedef struct vars_t {
/*  ----------------------------------------------------------------------
    Variable name table entry.  One of these is used for each variable.
    It is maintained as a linked list, with new variables added to the
    front of the list.  Searching is currently done sequentially; no
    hashing is used as this time (few variables are typical and since the
    support for compiled expressions has been added, with variable names
    replaced with direct pointers to their entries, there probably won't
    be much driving force for adding hashing.
*/
    struct vars_t *nxt;         /* pointer to next variable entry */
    char *name;                 /* variable name */
    double v;                   /* variable's value */
} vars_t;


/*  ----------------------------------------------------------------------  */
typedef struct unary_t {
/*  ----------------------------------------------------------------------
    Unary function table entry.  Unary functions are included in a table
    that keeps both their name (for searching) and a pointer to the actual
    function, itself.  This table is searched when evaluating expressions
    or statements.  The table follows.
*/
    const char *fname;          /* function name */
    void (*func)( expr_t * );   /* function pointer */
} unary_t;


/*  ----------------------------------------------------------------------
    LOCAL FUNCTIONS
    ----------------------------------------------------------------------
    Simply a list of prototypes for the various local functions in this
    module.
*/
static bool_t number( const char **expptr );
static bool_t sign( const char **expptr );
static bool_t mantissa( const char **expptr );
static bool_t digit( const char **expptr );
static bool_t digits( const char **expptr );
static bool_t scale( const char **expptr );
static bool_t scaleid( const char **expptr );
static bool_t alphanumeric( const char **expptr );
static bool_t alpha( const char **expptr );
static bool_t mnemonic( const char **expptr );
static bool_t statement( const char **expptr, expr_t *stkptr );
static bool_t expression( const char **expptr, expr_t *stkptr );
static bool_t term( const char **expptr, expr_t *stkptr );
static bool_t moreterms( const char **expptr, expr_t *stkptr );
static bool_t factor( const char **expptr, expr_t *stkptr );
static bool_t morefactors( const char **expptr, expr_t *stkptr );
static bool_t element( const char **expptr, expr_t *stkptr );
static bool_t moreelements( const char **expptr, expr_t *stkptr );
static void skipspaces( const char **expptr );
static bool_t match( const char c, const char **expptr );
static void push( const double v, expr_t *stkptr );
static double pop( expr_t *stkptr );
static void freestk( value_t *stkptr );
static void add( expr_t *stkptr );
static void subtract( expr_t *stkptr );
static void multiply( expr_t *stkptr );
static void divide( expr_t *stkptr );
static void power( expr_t *stkptr );
static void negate( expr_t *stkptr );
static void addop( int op, expr_t *stkptr, unsigned int size, void *p );
static void addvar( const char *name, double value );
static void setvar( const char *name, double value );
static double getvar( const char *name );
static vars_t * isvar( const char *name );
static void fn_sine( expr_t *stkptr );
static void fn_cosine( expr_t *stkptr );
static void fn_tangent( expr_t *stkptr );
static void fn_secant( expr_t *stkptr );
static void fn_cosecant( expr_t *stkptr );
static void fn_cotangent( expr_t *stkptr );
static void fn_arctangent( expr_t *stkptr );
static void fn_arcsine( expr_t *stkptr );
static void fn_arccosine( expr_t *stkptr );
static void fn_arcsecant( expr_t *stkptr );
static void fn_arccosecant( expr_t *stkptr );
static void fn_arccotangent( expr_t *stkptr );
static void fn_absolute( expr_t *stkptr );
static void fn_sign( expr_t *stkptr );
static void fn_integer( expr_t *stkptr );
static void fn_floor( expr_t *stkptr );
static void fn_ceiling( expr_t *stkptr );
static void fn_squareroot( expr_t *stkptr );
static void fn_logarithm( expr_t *stkptr );
static void fn_logbase10( expr_t *stkptr );
static void fn_radtodeg( expr_t *stkptr );
static void fn_degtorad( expr_t *stkptr );
static void fn_exponential( expr_t *stkptr );
static void fn_hyperbolicsine( expr_t *stkptr );
static void fn_hyperboliccosine( expr_t *stkptr );
static void fn_hyperbolictangent( expr_t *stkptr );
static void fn_hyperbolicsecant( expr_t *stkptr );
static void fn_hyperboliccosecant( expr_t *stkptr );
static void fn_hyperboliccotangent( expr_t *stkptr );
static void fn_random( expr_t *stkptr );
static void fn_gaussianrandom( expr_t *stkptr );


/*  ----------------------------------------------------------------------
    UNARY FUNCTION TABLE
    ----------------------------------------------------------------------
    The list of unary functions supported in expressions and statements.
*/
static unary_t predefined[]= {
    { "SIN", fn_sine },
    { "COS", fn_cosine },
    { "TAN", fn_tangent },
    { "SEC", fn_secant },
    { "CSC", fn_cosecant },
    { "COT", fn_cotangent },
    { "ATAN", fn_arctangent },
    { "ASIN", fn_arcsine },
    { "ACOS", fn_arccosine },
    { "ASEC", fn_arcsecant },
    { "ACSC", fn_arccosecant },
    { "ACOT", fn_arccotangent },
    { "ABS", fn_absolute },
    { "SGN", fn_sign },
    { "INT", fn_integer },
    { "FLOOR", fn_floor },
    { "CEIL", fn_ceiling },
    { "SQRT", fn_squareroot },
    { "LN", fn_logarithm },
    { "LOG", fn_logbase10 },
    { "DEG", fn_radtodeg },
    { "RAD", fn_degtorad },
    { "EXP", fn_exponential },
    { "SINH", fn_hyperbolicsine },
    { "COSH", fn_hyperboliccosine },
    { "TANH", fn_hyperbolictangent },
    { "SECH", fn_hyperbolicsecant },
    { "CSCH", fn_hyperboliccosecant },
    { "COTH", fn_hyperboliccotangent },
    { "RND", fn_random },
    { "RNDG", fn_gaussianrandom }
};


/*  ----------------------------------------------------------------------
    VARIABLE NAME TABLE
    ----------------------------------------------------------------------
    Since the variable name table may be used across expression/statement
    executions, it is held here in static form.  It may be useful, at some
    later time to provide the caller with this as an obscured structure,
    so that multiple variable tables might be maintained.  But I don't see
    all that much value for it, just now.  So only a single table is now
    supported and it is kept here in this module.
*/
static vars_t *vars= 0;


/*  ----------------------------------------------------------------------  */
static bool_t number( const char **expptr ) {
/*  ----------------------------------------------------------------------
    Evaluates whether or not a number follows.  If so, the pointer is
    advanced.  If not, it is returned the position prior to examination.

        number      :=  sign mantissa scale
*/
    auto const char *i;

        i= *expptr;
        /* sign and mantissa are mandatory */
        if ( sign( expptr ) && mantissa( expptr ) ) {
            /* but the scaling is optional */
            (void) scale( expptr );
            return 1;
        }
        *expptr= i;

    return 0;
}


/*  ----------------------------------------------------------------------  */
static bool_t sign( const char **expptr ) {
/*  ----------------------------------------------------------------------
    Accepts a + or -.  Always succeeds.

        sign        :=  + | - | <null>
*/

        (void) ( match( '+', expptr ) || match( '-', expptr ) );

    return 1;
}


/*  ----------------------------------------------------------------------  */
static bool_t mantissa( const char **expptr ) {
/*  ----------------------------------------------------------------------
    Evaluates whether or not a mantissa follows.  If so, the pointer is
    advanced.  If not, it is returned the position prior to examination.

        mantissa    :=  digit digits | digit digits . digits | . digit digits
*/
    auto const char *i;

        i= *expptr;
        if ( match( '.', expptr ) ) {
            if ( digit( expptr ) && digits( expptr ) )
                return 1;
            *expptr= i;
        } else if ( digit( expptr ) ) {
            (void) digits( expptr );
            (void) ( match( '.', expptr ) && digits( expptr ) );
            return 1;
        }

    return 0;
}


/*  ----------------------------------------------------------------------  */
static bool_t digit( const char **expptr ) {
/*  ----------------------------------------------------------------------
    Evaluates if a digit follows.  If so, the pointer is advanced.  If
    not, it is not advanced.

        digit       :=  0 | 1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | 9
*/
    auto char c;

        if ( (c= **expptr) == '\0' || strchr( "0123456789", c ) == 0 )
            return 0;
        ++(*expptr);

    return 1;
}


/*  ----------------------------------------------------------------------  */
static bool_t digits( const char **expptr ) {
/*  ----------------------------------------------------------------------
    Consumes any digits present.  Always succeeds.

        digits      :=  digit digits | <null>
*/

        while ( digit( expptr ) )
            ;

    return 1;
}


/*  ----------------------------------------------------------------------  */
static bool_t scale( const char **expptr ) {
/*  ----------------------------------------------------------------------
    Accepts a floating point scaling, if it follows.  If so, the pointer
    is advanced.  If not, it is returned the position prior to examining
    for it.  Always succeeds, either way.

        scale       :=  scaleid sign digit digits | <null>
*/
    auto const char *i;

        i= *expptr;
        if ( !scaleid( expptr ) )
            return 0;
        if ( sign( expptr ) && digit( expptr ) && digits( expptr ) )
            return 1;
        *expptr= i;

    return 0;
}


/*  ----------------------------------------------------------------------  */
static bool_t scaleid( const char **expptr ) {
/*  ----------------------------------------------------------------------
    Examines for the possibility of a scaling identifier.  If so, the
    pointer is advanced.  If not, no advancement of the pointer.

        scaleid     :=  e | E
*/

    return match( 'e', expptr ) || match( 'E', expptr );
}


/*  ----------------------------------------------------------------------  */
static bool_t alpha( const char **expptr ) {
/*  ----------------------------------------------------------------------
    Evaluates if a letter follows.  If so, the pointer is advanced.  If
    not, it is not advanced.

        alpha := A | B | C | D | E | F | G | H | I | J | K | L | M |
                 N | O | P | Q | R | S | T | U | V | W | X | Y | Z |
                 a | b | c | d | e | f | g | h | i | j | k | l | m |
                 n | o | p | q | r | s | t | u | v | w | x | y | z
*/
    auto char c;

        if ( (c= **expptr) == '\0' || strchr( "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz", c ) == 0 )
            return 0;
        ++(*expptr);

    return 1;
}


/*  ----------------------------------------------------------------------  */
static bool_t alphanumeric( const char **expptr ) {
/*  ----------------------------------------------------------------------
    Evaluates if an alphanumeric follows.  If so, the pointer is advanced.
    If not, it is not advanced.

        alphanumeric := alpha | digit
*/

    return alpha( expptr ) || digit( expptr );
}


/*  ----------------------------------------------------------------------  */
static bool_t mnemonic( const char **expptr ) {
/*  ----------------------------------------------------------------------
    Evaluates if a mnemonic follows.  If so, the pointer is advanced.  If
    not, it is not advanced.

        mnemonic := alpha alphanumerics
        alphanumerics := alphanumeric alphanumerics | <null>
*/

        if ( !alpha( expptr ) )
            return 0;
        while ( alphanumeric( expptr ) )
            ;

    return 1;
}


/*  ----------------------------------------------------------------------  */
static bool_t statement( const char **expptr, expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Evaluates a statement or expression.  Calculations may be performed in
    the process of parsing.  (True of routines with the stkptr parameter.)

    This routine first checks to see if there is a variable name followed
    by an '='.  If so, it is assumed to be a statement.  If not, it is
    assumed to be an expression.

        statement   :=  expression | mnemonic = expression
*/
    auto const char *i;

        skipspaces( expptr );
        i= *expptr;
        if ( mnemonic( expptr ) ) {
          auto unsigned int len;
          auto char *s;
          auto vars_t *var;
          auto double v;
            len= *expptr - i;
            s= (char *) malloc( len + 1 );
            strncpy( s, i, len );
            s[len]= '\0';
            skipspaces( expptr );
            if ( match( '=', expptr ) ) {
                if ( expression( expptr, stkptr ) ) {
                    v= pop( stkptr );
                    push( v, stkptr );
                    setvar( s, v );
                    var= isvar( s );
                    if ( stkptr->e != 0 )
                        addop( SETVAR, stkptr, sizeof( var ), &var );
                    free( s );
                    return 1;
                }
            }
            free( s );
        }
        *expptr= i;
        if ( expression( expptr, stkptr ) )
            return 1;

    return 0;
}


/*  ----------------------------------------------------------------------  */
static bool_t expression( const char **expptr, expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    This routine evaluates an expression as:

        expression  :=  term moreterms
*/
    auto const char *i;

        i= *expptr;
        if ( term( expptr, stkptr ) && moreterms( expptr, stkptr ) )
            return 1;
        *expptr= i;

    return 0;
}


/*  ----------------------------------------------------------------------  */
static bool_t term( const char **expptr, expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    This routine evaluates a term as:

        term        :=  factor morefactors
*/
    auto const char *i;

        i= *expptr;
        if ( factor( expptr, stkptr ) && morefactors( expptr, stkptr ) )
            return 1;
        *expptr= i;

    return 0;
}


/*  ----------------------------------------------------------------------  */
static bool_t factor( const char **expptr, expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    This routine evaluates a factor as:

        factor      :=  element moreelements
*/
    auto const char *i;

        i= *expptr;
        if ( element( expptr, stkptr ) && moreelements( expptr, stkptr ) )
            return 1;
        *expptr= i;

    return 0;
}

/*  ----------------------------------------------------------------------  */
static bool_t moreterms( const char **expptr, expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    This routine evaluates a moreterm as:

        moreterms   :=  + term moreterms | - term moreterms | <null>
*/
    auto const char *i;

        skipspaces( expptr );
        i= *expptr;
        if ( match( '+', expptr ) ) {
            if ( term( expptr, stkptr ) ) {
                add( stkptr );
                if ( stkptr->e != 0 )
                    addop( ADD, stkptr, 0, 0 );
                return moreterms( expptr, stkptr );
            }
            *expptr= i;
        } else if ( match( '-', expptr ) ) {
            if ( term( expptr, stkptr ) ) {
                subtract( stkptr );
                if ( stkptr->e != 0 )
                    addop( SUBTRACT, stkptr, 0, 0 );
                return moreterms( expptr, stkptr );
            }
            *expptr= i;
        }

    return 1;
}


/*  ----------------------------------------------------------------------  */
static bool_t morefactors( const char **expptr, expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    This routine evaluates a morefactors as:

        morefactors :=  * factor morefactors | / factor morefactors | <null>
*/
    auto const char *i;

        skipspaces( expptr );
        i= *expptr;
        if ( match( '*', expptr ) ) {
            if ( factor( expptr, stkptr ) ) {
                multiply( stkptr );
                if ( stkptr->e != 0 )
                    addop( MULTIPLY, stkptr, 0, 0 );
                return morefactors( expptr, stkptr );
            }
            *expptr= i;
        } else if ( match( '/', expptr ) ) {
            if ( factor( expptr, stkptr ) ) {
                divide( stkptr );
                if ( stkptr->e != 0 )
                    addop( DIVIDE, stkptr, 0, 0 );
                return morefactors( expptr, stkptr );
            }
            *expptr= i;
        }

    return 1;
}


/*  ----------------------------------------------------------------------  */
static bool_t moreelements( const char **expptr, expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    This routine evaluates a moreelements as:

        moreelements:=  ^ element moreelements | <null>
*/
    auto const char *i;

        skipspaces( expptr );
        i= *expptr;
        if ( match( '^', expptr ) ) {
            if ( element( expptr, stkptr ) ) {
                power( stkptr );
                if ( stkptr->e != 0 )
                    addop( POWER, stkptr, 0, 0 );
                return moreelements( expptr, stkptr );
            }
            i= *expptr;
        }

    return 1;
}


/*  ----------------------------------------------------------------------  */
static bool_t element( const char **expptr, expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    This routine handles the somewhat rather more complex meaning for an
    element.  This includes handling a mnemonic, which may either indicate
    a variable or a unary function.  It also must handle the case where an
    optional sign precedes either the variable or the unary function or a
    parenthetical expression.  The complexity isn't that bad, but it is
    important to get all the details.  Note that both the sign and the
    unary function name are optional!

        element     :=  sign mnemonic | number | sign unary ( expression )

    (Signs for numbers are already handled in the number parsing.)
*/
    auto const char *i;
    auto bool_t negflag;

        skipspaces( expptr );
        i= *expptr;

        /*  first check for a number */
        if ( number( expptr ) ) {
          auto size_t c;
          auto char *d;
          auto double v;
            c= *expptr - i;
            d= (char *) malloc( 1 + c );
            if ( d == 0 )
                return 0;
            strncpy( d, i, c );
            d[c]= '\0';
            v= strtod( d, 0 );
            free( d );
            push( v, stkptr );
            if ( stkptr->e != 0 )
                addop( PUSHVALUE, stkptr, sizeof( v ), &v );
            return 1;
        }

        /* not a number, so check for a sign before a variable or unary function */
        negflag= 0;
        if ( match( '+', expptr ) )
            skipspaces( expptr );
        else if ( match( '-', expptr ) ) {
            negflag= 1;
            skipspaces( expptr );
        }

        /* handle a parenthesized expression, if present */
        if ( match( '(', expptr ) ) {
            if ( expression( expptr, stkptr ) ) {
                skipspaces( expptr );
                if ( match( ')', expptr ) ) {
                    if ( negflag ) {
                        negate( stkptr );
                        if ( stkptr->e != 0 )
                            addop( NEG, stkptr, 0, 0 );
                    }
                    return 1;
                }
            }
        }

        /* otherwise, it should be either a variable or a unary function */
        else {
          auto const char *j;
            j= *expptr;
            if ( mnemonic( expptr ) ) {
              auto unsigned int len;
              auto char *s;
              auto vars_t *var;
                /* make a copy of the variable or unary function name */
                len= *expptr - j;
                s= (char *) malloc( len + 1 );
                strncpy( s, j, len );
                s[len]= '\0';
                /* skip spaces and see if there is a '(' to indicate a function */
                skipspaces( expptr );
                /* handle unary functions */
                if ( match( '(', expptr ) ) {
                  auto int idx;
                    for ( idx= sizeof( predefined ) / sizeof( predefined[0] ); --idx >= 0; )
                        if ( stricmp( s, predefined[idx].fname ) == 0 )
                            break;
                    if ( idx >= 0 ) {
                        if ( expression( expptr, stkptr ) ) {
                            skipspaces( expptr );
                            if ( match( ')', expptr ) ) {
                              auto void (*func)( expr_t *);
                                func= predefined[idx].func;
                                func( stkptr );
                                if ( stkptr->e != 0 )
                                    addop( UNARY, stkptr, sizeof( func ), &func );
                                if ( negflag ) {
                                    negate( stkptr );
                                    if ( stkptr->e != 0 )
                                        addop( NEG, stkptr, 0, 0 );
                                }
                                free( s );
                                return 1;
                            }
                        }
                    }
                }
                /* otherwise handle variables, if present -- if not, its an error */
                else if ( (var= isvar( s )) != 0 ) {
                  auto double v;
                    v= getvar( s );
                    push( v, stkptr );
                    if ( stkptr->e != 0 )
                        addop( PUSHVAR, stkptr, sizeof( var ), &var );
                    if ( negflag ) {
                        negate( stkptr );
                        if ( stkptr->e != 0 )
                            addop( NEG, stkptr, 0, 0 );
                    }
                    free( s );
                    return 1;
                }
                free( s );
            }
        }
        *expptr= i;

    return 0;
}


/*  ----------------------------------------------------------------------  */
static void skipspaces( const char **expptr ) {
/*  ----------------------------------------------------------------------
    Skips any following spaces.
*/
    auto char c;
    auto const char *p;

        for ( p= *expptr; (c= *p) != '\0'; ++p )
            if ( c != ' ' )
                break;
        *expptr= p;

    return;
}


/*  ----------------------------------------------------------------------  */
static bool_t match( const char c, const char **expptr ) {
/*  ----------------------------------------------------------------------
    Matches the given character by advancing the pointer.
*/

        if ( **expptr == '\0' )
            return 0;
        if ( **expptr == c ) {
            ++(*expptr);
            return 1;
        }

    return 0;
}


/*  ----------------------------------------------------------------------  */
static void push( const double v, expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Pushes the given value onto the value stack.
*/
    auto value_t *e;

        e= (value_t *) malloc( sizeof( value_t ) );
        if ( e == 0 )
            return;
        e->p= stkptr->s;
        e->v= v;
        stkptr->s= e;

    return;
}


/*  ----------------------------------------------------------------------  */
static double pop( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Pops the top of the value stack and returns the value.
*/
    auto value_t *e;

        if ( (e= stkptr->s) != 0 ) {
          double v;
            v= e->v;
            stkptr->s= e->p;
            free( e );
            return v;
        }

    return 0.0;
}


/*  ----------------------------------------------------------------------  */
static void add( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Pops the top two entries on the value stack, adds them, and pushes the
    result back onto the value stack.
*/
    auto double addend1, addend2;

        addend2= pop( stkptr );
        addend1= pop( stkptr );
        if ( addend1 != HUGE_VAL && addend2 != HUGE_VAL )
            push( addend1 + addend2, stkptr );
        else
            push( HUGE_VAL, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void subtract( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Pops the top two entries on the value stack, subtracts them, and
    pushes the result back onto the value stack.
*/
    auto double subtrahend, minuend;

        subtrahend= pop( stkptr );
        minuend= pop( stkptr );
        if ( minuend != HUGE_VAL && subtrahend != HUGE_VAL )
            push( minuend - subtrahend, stkptr );
        else
            push( HUGE_VAL, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void multiply( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Pops the top two entries on the value stack, multiplies them, and
    pushes the result back onto the value stack.
*/
    auto double multiplicand1, multiplicand2;

        multiplicand2= pop( stkptr );
        multiplicand1= pop( stkptr );
        if ( multiplicand1 != HUGE_VAL && multiplicand2 != HUGE_VAL )
            push( multiplicand1 * multiplicand2, stkptr );
        else
            push( HUGE_VAL, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void divide( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Pops the top two entries on the value stack, divides them, and pushes
    the result back onto the value stack.
*/
    auto double dividend, divisor;

        divisor= pop( stkptr );
        dividend= pop( stkptr );
        if ( divisor != HUGE_VAL && dividend != HUGE_VAL && divisor != 0.0 )
            push( dividend / divisor, stkptr );
        else
            push( HUGE_VAL, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void power( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Pops the top two entries on the value stack, raises one to the power
    of the other, and pushes the result back onto the value stack.
*/
    auto double x, y;

        y= pop( stkptr );
        x= pop( stkptr );
        if ( x != HUGE_VAL && y != HUGE_VAL && ( x != 0.0 || y != 0.0 ) && (x >= 0.0 || y == floor( y ) ) )
            push( pow( x, y ), stkptr );
        else if ( x == 0.0 && y == 0.0 )
            push( 0.0, stkptr );
        else
            push( HUGE_VAL, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void negate( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Negates the top entry on the value stack.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL )
            v= -v;
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_sine( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its sine.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL )
            v= sin( v );
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_cosine( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its cosine.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL )
            v= cos( v );
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_tangent( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its tangent.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL )
            v= tan( v );
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_secant( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its secant.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL ) {
            v= cos( v );
            if ( v != 0.0 )
                v= 1.0 / v;
            else
                v= HUGE_VAL;
        }
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_cosecant( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its cosecant.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL ) {
            v= sin( v );
            if ( v != 0.0 )
                v= 1.0 / v;
            else
                v= HUGE_VAL;
        }
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_cotangent( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its cotangent.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL ) {
            v= tan( v );
            if ( v != 0.0 )
                v= 1.0 / v;
            else
                v= HUGE_VAL;
        }
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_arctangent( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its arc-tangent.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL )
            v= atan( v );
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_arcsine( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its arc-sine.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL ) {
            if ( v >= -1.0 && v <= 1.0 )
                v= asin( v );
            else
                v= HUGE_VAL;
        }
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_arccosine( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its arc-cosine.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL ) {
            if ( v >= -1.0 && v <= 1.0 )
                v= acos( v );
            else
                v= HUGE_VAL;
        }
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_arcsecant( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its arc-secant.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL ) {
            if ( v <= -1.0 )
                v= 4.0 * atan( 1.0 ) - atan( sqrt( v*v-1.0 ) );
            else if ( v >= 1.0 )
                v= atan( sqrt( v*v-1.0 ) );
            else
                v= HUGE_VAL;
        }
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_arccosecant( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its arc-cosecant.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL ) {
            if ( v <= -1.0 )
                v= -acos( sqrt( v*v-1.0 ) / v );
            else if ( v >= 1.0 )
                v= acos( sqrt( v*v-1.0 ) / v );
            else
                v= HUGE_VAL;
        }
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_arccotangent( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its arc-cotangent.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL ) {
            if ( v < 0.0 )
                v= -2.0 * atan( 1.0 ) - atan( v );
            else
                v= 2.0 * atan( 1.0 ) - atan( v );
        }
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_absolute( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its absolute.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL )
            v= fabs( v );
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_sign( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its sign-only.  This is
    +1 for positive values, -1 for negative values, and 0 for 0.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL ) {
            if ( v < 0.0 )
                v= -1.0;
            else if ( v > 0.0 )
                v= 1.0;
            else
                v= 0.0;
        }
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_integer( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with the integer that is
    equal to it or else the one next closer to zero.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL )
            if ( v >= 0.0 )
                v= floor( v );
            else
                v= -floor( -v );
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_floor( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with the integer that is
    less than or equal to it.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL )
            v= floor( v );
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_ceiling( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with the integer that is
    greater than or equal to it.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL )
            v= ceil( v );
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_squareroot( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its square root.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL ) {
            if ( v >= 0.0 )
                v= sqrt( v );
            else
                v= HUGE_VAL;
        }
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_logarithm( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its natural logarithm.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL ) {
            if ( v > 0.0 )
                v= log( v );
            else
                v= HUGE_VAL;
        }
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_logbase10( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its base-10 logarithm.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL ) {
            if ( v > 0.0 )
                v= log10( v );
            else
                v= HUGE_VAL;
        }
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_radtodeg( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack, assuming it is in radians,
    with degrees.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL )
            v *= (45.0 / atan( 1.0 ));
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_degtorad( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack, assuming it is in degrees,
    with radians.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL )
            v *= (atan( 1.0 ) / 45.0);
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_exponential( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with that power of e.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL )
            v= exp( v );
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_hyperbolicsine( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its hyperbolic sine.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL )
            v= sinh( v );
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_hyperboliccosine( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its hyperbolic cosine.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL )
            v= cosh( v );
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_hyperbolictangent( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its hyperbolic tangent.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL )
            v= tanh( v );
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_hyperbolicsecant( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its hyperbolic secant.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL ) {
            v= cosh( v );
            if ( v != 0.0 )
                v= 1.0 / v;
            else
                v= HUGE_VAL;
        }
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_hyperboliccosecant( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its hyperbolic cosecant.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL ) {
            v= sinh( v );
            if ( v != 0.0 )
                v= 1.0 / v;
            else
                v= HUGE_VAL;
        }
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_hyperboliccotangent( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with its hyperbolic cotangent.
*/
    auto double v;

        v= pop( stkptr );
        if ( v != HUGE_VAL ) {
            v= tanh( v );
            if ( v != 0.0 )
                v= 1.0 / v;
            else
                v= HUGE_VAL;
        }
        push( v, stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_random( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with a uniform deviate.
*/

        pop( stkptr );
        push( rnd( ), stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void fn_gaussianrandom( expr_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Replaces the top entry on the value stack with a gaussian deviate.
*/

        pop( stkptr );
        push( rndgauss( ), stkptr );

    return;
}


/*  ----------------------------------------------------------------------  */
static void freestk( value_t *stkptr ) {
/*  ----------------------------------------------------------------------
    Completely frees up all allocated memory in the given value stack.
*/

        while ( stkptr != 0 ) {
          value_t *np= stkptr->p;
            free( stkptr );
            stkptr= np;
        }

    return;
}


/*  ----------------------------------------------------------------------  */
static void addop( int op, expr_t *stkptr, unsigned int size, void *p ) {
/*  ----------------------------------------------------------------------
    Adds a push-pop code entry to the code list.  If a non-zero value is
    provided for size and the memory pointer p, then space is allocated
    for that and it is linked into the push-pop code entry.  Otherwise,
    that pointer entry in the push-pop code entry is set to 0 (NULL.)
*/
    auto pushpop_t pp, *c, e;

        /* Construct a new operator/operand. */
        pp= (pushpop_t) malloc( sizeof( struct pushpop_t ) );
        if ( pp == 0 )
            return;
        pp->op= op;
        pp->p= 0;
        if ( size > 0 && p != 0 ) {
          auto void *e;
            e= malloc( size );
            if ( e == 0 ) {
                free( pp );
                return;
            }
            memcpy( e, p, size );
            pp->operand= e;
        } else
            pp->operand= 0;

        /* Append it to the list. */
        for ( c= &stkptr->e, e= *c; e != 0; c= &e->p, e= e->p )
            ;
        *c= pp;

    return;
}


/*  ----------------------------------------------------------------------  */
void freepp( pushpop_t ppptr ) {
/*  ----------------------------------------------------------------------
    Completely frees up all memory allocated for the push-pop code list.
    This routine frees up any malloc'd memory used in generating the given
    push-pop code and should be used once the compiled code is no longer
    needed.
*/

        while ( ppptr != 0 ) {
          pushpop_t np= ppptr->p;
            if ( ppptr->operand != 0 )
                free( ppptr->operand );
            free( ppptr );
            ppptr= np;
        }

    return;
}


/*  ----------------------------------------------------------------------  */
double evaluate( const char *algebraic, char **endptr ) {
/*  ----------------------------------------------------------------------
    This routine accepts an algebraic expression or assignment statement
    in ASCII text and returns the calculated value.  If 'endptr' is given,
    the string pointer it references is updated to point at the first
    character that stopped the scanning.

    The special value of HUGE_VAL (in <math.h>) is returned in cases where
    the expression had domain errors or otherwise couldn't be calculated.
    As a side effect, this routine may update the variable table, if
    statements are fed to it.  Expressions do not have such side effects.
*/
    auto expr_t stack= { 0, 0 };
    auto double v;
    auto const char *s;

        s= algebraic;
        (void) statement( &s, &stack );
        v= pop( &stack );
        freestk( stack.s );

        /* update endptr, if desired */
        if ( endptr != 0 ) {
            skipspaces( &s );
            *endptr= (char *) s;
        }

    return v;
}


/*  ----------------------------------------------------------------------  */
pushpop_t translate( const char *algebraic, char **endptr ) {
/*  ----------------------------------------------------------------------
    This routine accepts an algebraic expression or assignment statement
    in ASCII text and returns a malloc'd, compiled structure for faster
    execution of the expression.  If 'endptr' is given, the string pointer
    it references is updated to point at the first character that stopped
    the scanning.
    
    The compiled code is "push-pop" and is not machine code.  The push-pop
    code converts used variable names into direct memory references to the
    variables and also pre-defined function name references into direct
    function pointers, etc.  In the process of this conversion, the
    expression or statement is actually calculated, just as with evaluate( ),
    but no value is returned.  It's not important to do so, because the
    variable values may not yet exist or may be different, when the
    expression is later executed.

    The special value of HUGE_VAL (in <math.h>) is returned in cases where
    the expression had domain errors or otherwise couldn't be calculated.
    As a side effect, this routine may update the variable table, if
    statements are fed to it.  Expressions do not have such side effects.
*/
    auto expr_t stack= { 0, 0 };
    auto const char *s;

        s= algebraic;
        addop( START, &stack, 0, 0 );
        (void) statement( &s, &stack );
        addop( POPVAL, &stack, 0, 0 );
        freestk( stack.s );

        /* update endptr, if desired */
        if ( endptr != 0 ) {
            skipspaces( &s );
            *endptr= (char *) s;
        }

    return stack.e;
}


/*  ----------------------------------------------------------------------  */
double evaluatepp( pushpop_t pp ) {
/*  ----------------------------------------------------------------------
    This routine accepts compiled push-pop code and performs the statement
    or expression, as indicated.

    The special value of HUGE_VAL (in <math.h>) is returned in cases where
    the expression had domain errors or otherwise couldn't be calculated.
    As a side effect, this routine may update the variable table, if
    statements are fed to it.  Expressions do not have such side effects.
*/
    auto expr_t stack= { 0, 0 };
    auto double v;

        v= 0.0;
        while ( pp != 0 ) {
            switch( pp->op ) {
            case START:
                v= 0.0;
                break;
            case PUSHVALUE:
                push( *(double *)pp->operand, &stack );
                break;
            case ADD:
                add( &stack );
                break;
            case SUBTRACT:
                subtract( &stack );
                break;
            case MULTIPLY:
                multiply( &stack );
                break;
            case DIVIDE:
                divide( &stack );
                break;
            case POWER:
                power( &stack );
                break;
            case NEG:
                negate( &stack );
                break;
            case PUSHVAR:
                push( (* (vars_t **) pp->operand)->v, &stack );
                break;
            case POPVAL:
                v= pop( &stack );
                break;
            case SETVAR:
                push( (* (vars_t **) pp->operand)->v= pop( &stack ), &stack );
                break;
            case UNARY:
                (* (void(**)(expr_t*)) pp->operand)( &stack );
                break;
            default:
                break;
            }
            pp= pp->p;
        }
        freestk( stack.s );

    return v;
}


/*  ----------------------------------------------------------------------  */
static void addvar( const char *name, double value ) {
/*  ----------------------------------------------------------------------
    Adds a new variable name to the variable name list with the given
    value.  No checking is done to verify that there already isn't an
    entry with the same name.
*/
    auto vars_t *node;

        node= (vars_t *) malloc( sizeof( struct vars_t ) );
        if ( node == 0 )
            return;
        node->name= (char *) malloc( strlen( name ) + 1 );
        if ( node->name == 0 ) {
            free( node );
            return;
        }
        strcpy( node->name, name );
        strlwr( node->name );
        node->v= value;
        node->nxt= vars;
        vars= node;

    return;
}


/*  ----------------------------------------------------------------------  */
static void setvar( const char *name, double value ) {
/*  ----------------------------------------------------------------------
    Updates a variable name's value in the variable name list.  If the
    variable doesn't already exist, it is created.  If it does exist, then
    the value is simply updated.
*/
    auto vars_t *var;

        if ( (var= isvar( name )) == 0 )
            addvar( name, value );
        else
            var->v= value;

    return;
}


/*  ----------------------------------------------------------------------  */
static double getvar( const char *name ) {
/*  ----------------------------------------------------------------------
    Gets the value for the indicated variable name.  If no such variable
    name exists, the value defaults to HUGE_VAL.
*/
    auto vars_t *p;

        for ( p= vars; p != 0; p= p->nxt )
            if ( stricmp( p->name, name ) == 0 )
                return p->v;

    return HUGE_VAL;
}


/*  ----------------------------------------------------------------------  */
static vars_t * isvar( const char *name ) {
/*  ----------------------------------------------------------------------
    Checks to see if the indicated variable name exists in the table.  If
    it does, it returns a pointer to the variable.  If it doesn't, the
    NULL pointer, 0, is returned.
*/
    auto vars_t *p;

        for ( p= vars; p != 0; p= p->nxt )
            if ( stricmp( p->name, name ) == 0 )
                return p;

    return 0;
}


/*  ----------------------------------------------------------------------  */
void initvars( void ) {
/*  ----------------------------------------------------------------------
    This routine initializes the variable name/value table and empties it.
    Technically, it isn't strictly necessary to call initvars( ) before
    using the other routines.  A static variable has an initialized value
    and is sufficient.  But if the variable list needs to be cleared out
    and the memory returned to malloc(), then it should be called.  My
    recommendation is to call it once at init-time and once again when you
    are finished with using any of these routines, so that the malloc'd
    memory can be returned.
*/
    auto vars_t *p;

        p= vars;
        while ( p != 0 ) {
          auto vars_t *n;
            n= p->nxt;
            if ( p->name != 0 )
                free( p->name );
            free( p );
            p= n;
        }        
        vars= 0;

    return;
}


#if 1
/*  ----------------------------------------------------------------------  */
int _matherr( struct _exception *except )
/*  ----------------------------------------------------------------------
    This function catches errors thrown by the math functions (it does not
    catch division by zero, for example.)  For now, only some errors are
    handled and only without any attempt to investigate or remedy them with
    more reasoning.  Someday, this may need some expansion, when I get more
    serious on numeric error checking.  For now, this will have to do.
    
    If this routine is enabled, the switch option /NOE is required on the
    linker.
*/
{
        switch( except->type ) {
        case _DOMAIN:
        case _SING:
        case _OVERFLOW:
            except->retval= HUGE_VAL;
            return 1;
        case _UNDERFLOW:
            except->retval= 0.0;
            return 1;
        case _PLOSS:
        case _TLOSS:
            /* no idea what to do here */
            break;
        default:
            break;
        }

    return 0;
}
#endif
