/*
    File:   expr.h
 
    Creation Date:  Sat 13-Mar-2004 02:31:10
    Last Modified:  Thu 05-Jan-2006 02:50:01
 
 
    DESCRIPTION

    This file provides an interface to a recursive descent parser for simple
    expressions, written in algebraic-like notation.

    
    EXAMPLES

    (1)
        printf( "%lf\n", evaluate( "4 * atan( 1 )" ) );

    (2)
        initvars( );
        evaluate( "pi = 4 * atan( 1 )" );
        evaluate( "e  = exp( 1 )" );
        printf( "%lf\n", evaluate( "- ( pi * e )" ) );
        initvars( );

    (3)
        {
        auto pushpop_t ppcode;
        initvars( );
        evaluate( "pi = 4 * atan( 1 )" );
        evaluate( "e  = exp( 1 )" );
        evaluate( "t = 2" );
        ppcode= translate( "t*exp(t*pi)" );
        printf( "%lf\n", evaluatepp( ppcode ) );
        evaluate( "t = 3" );
        printf( "%lf\n", evaluatepp( ppcode ) );
        freepp( ppcode );
        initvars( );
        }


    MODIFICATIONS
 
    30-Dec-2005 15:24:47    jk      Added file header.
*/

#ifndef EXPR_H
#define EXPR_H

#define T pushpop_t
struct T;
typedef struct T* T;

/*  ----------------------------------------------------------------------  */
extern double evaluate( const char *algebraic, char **endptr );
/*  ----------------------------------------------------------------------
    This routine accepts an algebraic expression or assignment statement
    in ASCII text and returns the calculated value.  If 'endptr' is given,
    the string pointer it references is updated to point at the first
    character that stopped the scanning.

    The special value of HUGE_VAL (in <math.h>) is returned in cases where
    the expression had domain errors or otherwise couldn't be calculated.
    As a side effect, this routine may update the variable table, if
    statements are fed to it.  Expressions do not have such side effects.
*/

/*  ----------------------------------------------------------------------  */
extern T translate( const char *algebraic, char **endptr );
/*  ----------------------------------------------------------------------
    This routine accepts an algebraic expression or assignment statement
    in ASCII text and returns a malloc'd, compiled structure for faster
    execution of the expression.  If 'endptr' is given, the string pointer
    it references is updated to point at the first character that stopped
    the scanning.
    
    The compiled code is "push-pop" and is not machine code.  The push-pop
    code converts used variable names into direct memory references to the
    variables and also pre-defined function name references into direct
    function pointers, etc.  In the process of this conversion, the
    expression or statement is actually calculated, just as with evaluate( ),
    but no value is returned.  It's not important to do so, because the
    variable values may not yet exist or may be different, when the
    expression is later executed.

    The special value of HUGE_VAL (in <math.h>) is returned in cases where
    the expression had domain errors or otherwise couldn't be calculated.
    As a side effect, this routine may update the variable table, if
    statements are fed to it.  Expressions do not have such side effects.
*/

/*  ----------------------------------------------------------------------  */
extern double evaluatepp( T ppcode );
/*  ----------------------------------------------------------------------
    This routine accepts compiled push-pop code and performs the statement
    or expression, as indicated.

    The special value of HUGE_VAL (in <math.h>) is returned in cases where
    the expression had domain errors or otherwise couldn't be calculated.
    As a side effect, this routine may update the variable table, if
    statements are fed to it.  Expressions do not have such side effects.
*/

/*  ----------------------------------------------------------------------  */
extern void freepp( T ppcode );
/*  ----------------------------------------------------------------------
    This routine frees up any malloc'd memory used in generating the given
    push-pop code.  Use this routine when you are finished with the code.
*/

/*  ----------------------------------------------------------------------  */
extern void initvars( void );
/*  ----------------------------------------------------------------------
    This routine initializes the variable name/value table and empties it.
    Technically, it isn't strictly necessary to call initvars( ) before
    using the other routines.  A static variable has an initialized value
    and is sufficient.  But if the variable list needs to be cleared out
    and the memory returned to malloc(), then it should be called.  My
    recommendation is to call it once at init-time and once again when you
    are finished with using any of these routines, so that the malloc'd
    memory can be returned.
*/

#undef T

#endif
