/*
    File:   rnd.c
    Contributing Author: Jon Kirwan

    Creation Date:  Fri 30-Dec-2005 14:27:49
    Last Modified:  Fri 30-Dec-2005 15:10:26


    DESCRIPTION

    This module provides a uniform deviate random number, from 0.0 to 1.0
    (not including 1.0.)


    TARGET COMPILER

    Microsoft 16-bit Visual C/C++, IDE version 1.52C (DOS version 8.00c)
 
 
    MODIFICATIONS
 
    Original source.


    ORIGINAL AUTHOR'S NOTICE

    A C-program for MT19937, with initialization improved 2002/1/26.
    Coded by Takuji Nishimura and Makoto Matsumoto.

    Before using, initialize the state by using init_genrand(seed)  
    or init_by_array(init_key, key_length).

    Copyright (C) 1997 - 2002, Makoto Matsumoto and Takuji Nishimura,
    All rights reserved.                          

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

     1. Redistributions of source code must retain the above copyright
        notice, this list of conditions and the following disclaimer.

     2. Redistributions in binary form must reproduce the above copyright
        notice, this list of conditions and the following disclaimer in the
        documentation and/or other materials provided with the distribution.

     3. The names of its contributors may not be used to endorse or promote 
        products derived from this software without specific prior written 
        permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
    "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
    LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
    A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT
    OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
    SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
    TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
    PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
    LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
    NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
    SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

    Any feedback is very welcome.
    http://www.math.sci.hiroshima-u.ac.jp/~m-mat/MT/emt.html
    email: m-mat @ math.sci.hiroshima-u.ac.jp (remove space)
*/

#include "rnd.h"

/* Period parameters */  
#define N 624
#define M 397
#define MATRIX_A 0x9908b0dfUL   /* constant vector a */
#define UPPER_MASK 0x80000000UL /* most significant w-r bits */
#define LOWER_MASK 0x7fffffffUL /* least significant r bits */

static unsigned long mt[N]; /* the array for the state vector  */
static int mti=N+1; /* mti==N+1 means mt[N] is not initialized */


/*  ----------------------------------------------------------------------  */
void randomize( unsigned long s ) {
/*  ----------------------------------------------------------------------
    This routine initializes the array, mt[N], with a seed.
*/
        mt[0]= s & 0xffffffffUL;
        for ( mti= 1; mti<N; ++mti ) {
            mt[mti] = 
            (1812433253UL * (mt[mti-1] ^ (mt[mti-1] >> 30)) + mti); 
            /* See Knuth TAOCP Vol2. 3rd Ed. P.106 for multiplier. */
            /* In the previous versions, MSBs of the seed affect   */
            /* only MSBs of the array mt[].                        */
            /* 2002/01/09 modified by Makoto Matsumoto             */
            mt[mti] &= 0xffffffffUL;
            /* for >32 bit machines */
        }

    return;
}


/*  ----------------------------------------------------------------------  */
unsigned long rndint32( void ) {
/*  ----------------------------------------------------------------------
    This routine generates a random number on the interval, [0,0xffffffff]
*/
    auto unsigned long y;
    static unsigned long mag01[2]= { 0x0UL, MATRIX_A };
                                /* mag01[x] = x * MATRIX_A  for x=0,1 */

        if ( mti >= N ) {           /* generate N words at one time */
          auto int kk;
            if ( mti == N+1 )       /* if init_genrand() has not been called, */
                randomize( 5489UL ); /* a default initial seed is used */
            for ( kk= 0; kk<N-M; ++kk ) {
                y= ( mt[kk] & UPPER_MASK ) | ( mt[kk+1] & LOWER_MASK );
                mt[kk]= mt[kk+M] ^ (y >> 1) ^ mag01[y & 0x1UL];
            }
            for ( ; kk<N-1; ++kk ) {
                y= ( mt[kk] & UPPER_MASK ) | ( mt[kk+1] & LOWER_MASK );
                mt[kk]= mt[kk+(M-N)] ^ (y >> 1) ^ mag01[y & 0x1UL];
            }
            y= ( mt[N-1] & UPPER_MASK ) | ( mt[0] & LOWER_MASK );
            mt[N-1]= mt[M-1] ^ (y >> 1) ^ mag01[y & 0x1UL];
            mti= 0;
        }
        y= mt[mti++];

        /* Tempering */
        y ^= (y >> 11);
        y ^= (y << 7) & 0x9d2c5680UL;
        y ^= (y << 15) & 0xefc60000UL;
        y ^= (y >> 18);

    return y;
}


/*  ----------------------------------------------------------------------  */
double rnd( void ) { 
/*  ----------------------------------------------------------------------
    This routine generates a random number on [0,1) with 53-bit resolution.
    This real version due to Isaku Wada, 2002/01/09 added.
*/
    auto unsigned long a= rndint32( ) >> 5, b= rndint32( ) >> 6;

    return ( a*67108864.0 + b ) * ( 1.0/9007199254740992.0 ); 
}
