' ͻ
'  File:   fracalc.bas                                                   
'  Creation Date:  Mon 12-Apr-1999 23:00:08        Jonathan D. Kirwan    
'  Last Modified:  Sat 01-May-1999 00:03:23        Initial version.      
' ͹
'  Copyright (C) 1999 Jonathan Dale Kirwan                               
'  All Rights Reserved: See COPYRIGHT NOTICE below for a description.    
' ͼ
'
'
'   DESCRIPTION
'
'   This program is designed to illustrate a spectrum of techniques that
'   include: state-table parsing, recursive descent parsing, expression
'   evaluation, and fractions.  It's rather a hodge-podge, since I built
'   it in about 2 hours from some odd pieces I'd written elsewhere.  At
'   first, I considered leaving the fraction code in a separate module
'   where I'd originally had it.  But that would make this incompatible
'   with QBASIC, so I pasted it all together into one big file, instead.
'
'   Because I did write some documentation on the fraction code, I'm
'   including it here, below.  The rest is rather sparse, so far.  The
'   function, Parse, is a general purpose state table parser, and isn't
'   the least bit limited to parsing numbers, as it is used here.  But
'   for this program, I also wanted to use recursive descent parsing
'   for the rest of the expression, so the state-table parsing is limited
'   to just parsing floating point numbers.  The mix of techniques can
'   provide several different techniques to learn from.
'
'   Normally, the expression to evaluate breaks into these terms:
'
'       expression  :=  term | expression + term | expression - term
'       term        :=  factor | term * factor | term / factor
'       factor      :=  number | ( expression )
'
'   The | means OR, so an expression is either a "term" or it is an
'   "expression + term" or it is an "expression - term".  I don't
'   define the syntax for "number" above, but it basically follows the
'   general form of a decimal number with an optional scale or power
'   (such as 1.2E5).
'
'   The problem with the above description is that it is left-recursive,
'   making parsing ambiguous.  A more concrete form, in right-recursive
'   form is:
'
'       expression  :=  term moreterms
'       term        :=  factor morefactors
'       factor      :=  number | ( expression )
'       moreterms   :=  + term moreterms | - term moreterms | <null>
'       morefactors :=  * factor morefactors | / factor morefactors | <null>
'
'   That's the model for the recursive descent code illustrated below.
'
'   I've chosen to set up the fractional computations so that they work
'   fine with numerators and denominators of zero.  All fractions with a
'   numerator of zero are reduced to the fraction, 0/1, and all fractions
'   with a denominator of zero are reduced to the fraction, 1/0.
'
'   Operations on such values will always succeed.  For example, (3/4) /
'   (4/0) produces a fraction value of 0/1.  However, if you switch to
'   decimal mode, this will cause a big problem that I haven't troubled
'   myself to fix, yet.  Live with it.
'
'
'   SOME HISTORY
'
'   Eudoxus developed a valid theory of proportions and ratios and
'   supplied some long-sought after proofs that had been handily
'   demolished when Hippasus had earlier demonstated incommensurable
'   ratios and thus began the "logical scandal of Greek geometry."
'   Eudoxus' theory was included as Book V of Euclid's "Elements."
'
'   Pythagoras was born in Samos around 572 BC.  Some legends about him
'   say that he studied with Thales, himself.  But when Polycrates took
'   power, Pythagoras fled to Crotona (a Greek town in southern Italy)
'   where he founded the Pythagorean brotherhood (no sisters, I guess.)
'
'   The Pythagoreans contemplated music, astronomy, and philosophy.  I
'   really don't know what their life was truly like, but I imagine a
'   kind of "life of the mind."  Mathematics was a developing idea, at
'   the time, and the Pythagoreans believed that whole numbers were the
'   basis, the most critical foundation, of all natural phenomena.  They
'   saw ratios of these whole numbers as being ideal, since they felt
'   that all relationships in nature could be described by ratios of
'   these whole numbers.
'
'   For example, all musical notes were developed from different strand
'   lengths.  In comparing these strands, say for middle C and middle E,
'   they believed that you could always find some smaller length that
'   would evenly and exactly divide both larger lengths.
'
'   In fact, you could play a game with a friend, saying:  "Take this
'   ball of string and cut two pieces from it.  Cut them to any length
'   you want."  With majestic flourishes and mystical hand-waving, you
'   then proceed to cut off a third length of string from the ball that
'   can be used to evenly measure both the lengths your friend cut off.
'
'   Without thinking about it much, your friend might be surprised or
'   might not be.  To some, this might be a real trick and surprising.
'   To others, they might just shrug it off as "obvious."  But without
'   thinking, anything can be believed or not believed.  That's no
'   trick at all.
'
'   But when you think more deeply about it, you can certainly argue
'   yourself into accepting the idea that there is always some short
'   length that can be fashioned in such a way that it evenly divides
'   any two given lengths.  For example, if I ask what measure best
'   fits 12 and 15, you'd say 3.  Both 12 and 15 can be expressed as
'   4*3 and 5*3, respectively.  So you could say that 12 measures
'   exactly 4 of the 3-length units and 15 measures exactly 5 of them.
'   When I ask about 4.1 and 7.1, you could respond with 0.1, since
'   4.1 is 41 such units and 7.1 is 71 of them.  If I asked about
'   4.003277 and 9.000012, you could answer 0.000001 without any
'   hesitation I suspect.
'
'   In fact, we could play this game all day long and you'd always
'   stay ahead of me without any effort, really.  So, perhaps after
'   reflecting on this for a bit, you might very well conclude that
'   there always is some smaller length that can always be fashioned
'   with just the right size to evenly divide any pair of numbers I
'   can come up with.
'
'   This is what the Pythagoreans thought, too.  And they believed
'   that this idea was significant.
'
'   Mathematicians now use a special term for this idea.  If it is
'   possible to fashion a third value that evenly divides a pair of
'   numbers, then that pair is said to be "commensurable."  Of course,
'   I did say "if," didn't I?  Well, I suppose that probably gives it
'   away -- not all pairs of numbers are commensurable!
'
'   You remember the famous Pythagorean theorem?  The one that relates
'   the two sides of a rectangle or a square to it's "hypotenuse" or
'   diagonal?  "The hypotenuse squared is equal to the sum of the
'   squares of the two sides."  If R is the hypotenuse, X is one side
'   and Y is the other side of a rectangle, then R*R = X*X + Y*Y.
'
'   Pythagorus is said to have come up with this and then sacrificed
'   an ox to the gods to celebrate his discovery!
'
'   Well, another Pythagorean, Hippasus, was able to show that the
'   hypotenuse of a square and its side are not commensurable at all!
'   This is true for any square!
'
'   For squares, X = Y, so the Pythagorean theorem above reduces to
'   just R*R = 2*X*X.  Now in today's terms, R and X can be any real
'   value, but if we assume that X and R have some common measure
'   between them, we can call that common measure, D.  Given that
'   assumption, R = r*D where r is a whole number and X = x*D where
'   x is also a whole number.  We are allowing D to be any value that
'   evenly divides both R and X, such that r and x are whole numbers.
'
'   So let's replace R and X in our equation, using r*D and x*D instead
'   of R and X:  (r*D)*(r*D) = 2*(x*D)*(x*D) or r*r*D*D = 2*x*x*D*D.
'   We can remove the common factor, D*D, on both sides, leaving us with
'   r*r = 2*x*x.  However, recall that r and x must be positive, whole
'   numbers because we've assumed we could devise a proper D to allow
'   it.
'
'   Here's a sample table of positive whole numbers for x:
'
'        x   2*x*x
'       ---  -----
'        1     2
'        2     8
'        3    18
'        4    32
'        5    50
'
'   However, no matter how many x values you try out, it just isn't
'   possible to find a positive, whole number r that equals 2*x*x.
'   What r, when squared, equals 2?  Or 8, 18, 32, or 50?
'
'   What this shows is that our assumption, that there is a value D
'   that evenly divides both the hypotenuse and the side of a square,
'   is simply wrong.
'
'   This discovery by Hippasus demolished some of the proofs that had
'   been devised before his time and was known as the "logical scandal
'   of Greek geometry."
'
'   We now know that the relationship between the hypotenuse and the
'   side of a square is just the square-root of 2.  And we call this
'   relationship "irrational" and such numbers are irrational numbers.
'
'   All pairs of rational numbers are commensurable, but all pairs formed
'   from one rational number and one irrational number are not.  Pairs of
'   irrational numbers may or may not be commensurable.  For example, the
'   SQRT(12) and the SQRT(3) are each irrational but commensurable, if D
'   is the SQRT(3), since SQRT(12)/SQRT(3) = 2 and SQRT(3)/SQRT(3) = 1,
'   where both answers are positive, whole numbers.  But the SQRT(12) and
'   the SQRT(15) are both irrational while not commensurable.
'
'
'   DESIGN
'
'   Although both the numerator and denominator can be negative, it can
'   be disconcerting.  Fractions should have only one sign.  The design
'   here is to only allow the numerator to take on negative values to
'   signify a negative fraction.  All functions adhere to this rule.
'
'   The basic equations to compute the result of the four math operations
'   on fractions are easily applied.  If that was all there was to this
'   program, it would be quite short indeed.  The modest complexity in
'   this code results from a desire to keep all fractional results in
'   their "lowest terms" (no common factors between the numerator and
'   denominator, except 1.)  This derives from a desire to minimize the
'   risk of a computation overflow while attempting to compute the
'   fractional results.
'
'   To reduce a fraction to its lowest terms, we need to compute the
'   greatest divisor that divides evenly into both the numerator and the
'   denominator of a fraction.  There is an algorithm, called Euclid's
'   Method.
'
'   Several design issues are handled by the gcd function, too.  It must
'   be impossible to return a common divisor value less than one, since
'   these results are used to divide other values and it's important to
'   prevent division by zero or to accidently cause a sign change if
'   negative values were returned.
'
'   This "understanding" is relied upon by the other routines.
'
'   Also, except where both the numerator and the denominator are zero,
'   if the numerator is zero the denominator is returned and if the
'   denominator is zero the numerator is returned.  This follows from
'   the concept that the largest common divisor between zero and any
'   other value is simply the other value, since that other value divides
'   zero evenly, too.
'
'   This "understanding" is useful.  It causes any fraction with a zero
'   in the numerator or denominator to be reduced to 0/1 or 1/0.  It is
'   very useful effect for comparing two values, when 0/1, 0/2. 0/3, etc.
'   since they are all reduced to 0/1.
'
'
'   A SPECIFIC EXAMPLE
'
'   Assume that there *IS* a common divisor between two numbers, Q and R,
'   and that Q is less than R.  If there is such a divisor, then it's
'   also clear that the same common divisor must also evenly divide the
'   difference between Q and R.  This is clearly shown by representing
'   Q and R as lengths of string, marked off by their common divisor:
'
'   Q=  |--|--|--|--|--|--|--|--|--|--|--|--|--|--|--|
'   R=  |--|--|--|--|--|--|--|--|--|--|--|--|--|--|--|--|--|--|--|
' R-Q=                                               |--|--|--|--|
'   D=  |--|
'
'   You can see that the common element, D, divides evenly into Q and R
'   and you can also see that it must also evenly divide into the dif-
'   ference of their lengths, R-Q.
'   of their lengths.  Also notice, in this example, that the difference
'   is smaller than either Q or R.  Let's call this difference, D.
'
'   Euclid's algorithm helps us find the greatest common divisor, D,
'   given any pair of numbers, Q and R, if there exists a number D and
'   two positive, whole numbers, q and r, such that Q = q * D and
'   R = r * D.  This is always true for any pair of rational numbers,
'   sometimes true for a pair of irrational numbers, and never true
'   when one of the numbers is rational and the other is irrational.
'
'   Although I think we already understand that D must also evenly
'   divide R-Q, it's easy to show it with algebra.  Since (R-Q) = r*D
'   - q*D = (r-q) * D, it is obvious that D is also a factor for the
'   quantity, (R - Q).  So now you know that D *must* divide evenly
'   into all three, R, Q, and (R - Q), [given our assumption that
'   such a D actually does exist, of course.]
'
'   If you take the smaller of the three values, D, Q, and R, and repeat
'   the subtraction process to compute a new value for D, eventually you
'   will arrive at two values that are equal to each other.  That value
'   is the greatest common divisor.
'
'   For example, start with 45 and 57.  Their difference is 12.  Now take
'   the difference between 12 and 45, and so on until the smallest two of
'   the three values are the same:
'
'       57 - 45 = 12
'       45 - 12 = 33
'       33 - 12 = 21
'       21 - 12 = 9
'       12 - 9 = 3
'       9 - 3 = 6
'       6 - 3 = 3               smallest two are both 3
'       3 - 3 = 0               and their difference is 0
'
'   There you have the common divisor, 3.  Let's repeat this using division
'   and taking the remainder, instead of successive subtraction.  (We'll
'   ignore the quotient and only look at the remainder, instead.)
'
'       57/45 = 1 r 12
'       45/12 = 3 r 9
'       12/9 = 1 r 3
'       9/3 = 3 r 0             answer is divisor, when remainder is 0
'
'   Again, 3 is the answer, taken from the last divisor we used, when we
'   got a remainder of 0.  (Ignore the quotient, whose value of 3 is a
'   simple coincidence in this example.)
'
'   The point is that as we continue to take the smaller two of three
'   values, all three still have the same divisor in common, throughout
'   the process.  Taking the smaller two means we simply take the two
'   values closer to the common divisor.  If we repeat this over and
'   over, we eventually arrive at the common divisor, itself.
'
'
'   TARGET COMPILER
'
'   This module is designed to be used with Microsoft QBASIC v1.1,
'   QuickBASIC 4.5, PDS 7.0 and 7.1, and VB-DOS 1.0.  OPTION EXPLICIT
'   was used under VB-DOS in order to ensure that every variable has
'   an explicit declaration of its type.
'
'
'   MODIFICATIONS
'
'   No modifications.
'
'
'   COPYRIGHT NOTICE
'
'   Jonathan Dale Kirwan grants you a non-transferable, non-exclusive,
'   royalty-free worldwide license to use, copy, modify, prepare deriva-
'   tive works of and distribute this software, subject to your agreement
'   that you acquire no ownership right, title, or interest in this soft-
'   ware and your agreement that this software is research work which is
'   provided 'as is', where Jonathan Dale Kirwan disclaims all warranties
'   with regard to this software, including all implied warranties of
'   merchantability and fitness of purpose.  In no event shall Jonathan
'   Dale Kirwan be liable for any direct, indirect, consequential or
'   special damages or any damages whatsoever resulting from loss of use,
'   data or profits, whether in an action of contract, negligence or
'   other tortious action, arising out of or in connection with the use
'   or performance of this software.

TYPE FRACTION
    numerator AS LONG
    denominator AS LONG
END TYPE

DECLARE SUB Evaluate (expressiontext AS STRING)
DECLARE FUNCTION Expression% (idx AS INTEGER, text AS STRING, count AS INTEGER)
DECLARE FUNCTION Term% (idx AS INTEGER, text AS STRING, count AS INTEGER)
DECLARE FUNCTION Factor% (idx AS INTEGER, text AS STRING, count AS INTEGER)
DECLARE FUNCTION Number% (idx AS INTEGER, text AS STRING, count AS INTEGER)
DECLARE FUNCTION Match% (char AS STRING, idx AS INTEGER, text AS STRING)
DECLARE FUNCTION MoreTerms% (idx AS INTEGER, text AS STRING, count AS INTEGER)
DECLARE FUNCTION MoreFactors% (idx AS INTEGER, text AS STRING, count AS INTEGER)
DECLARE FUNCTION Parse% (idx AS INTEGER, cstr AS STRING, istate AS INTEGER, state AS INTEGER, token AS INTEGER, clist AS STRING, ilist() AS INTEGER, istates() AS INTEGER)
DECLARE SUB DisplayResult ()

DECLARE SUB FracSet (f AS FRACTION, n AS LONG, d AS LONG)
DECLARE SUB FracAdd (r AS FRACTION, a1 AS FRACTION, a2 AS FRACTION)
DECLARE SUB FracSub (r AS FRACTION, m AS FRACTION, s AS FRACTION)
DECLARE SUB FracMul (r AS FRACTION, m1 AS FRACTION, m2 AS FRACTION)
DECLARE SUB FracDiv (r AS FRACTION, n AS FRACTION, d AS FRACTION)
DECLARE FUNCTION gcd& (n AS LONG, d AS LONG)
DECLARE FUNCTION gcd0& (n AS LONG, d AS LONG)
DECLARE FUNCTION fgcd# (f AS DOUBLE, precision AS DOUBLE)

CONST FRACT% = 1
CONST DECIMAL% = 2

    DIM SHARED ErrLoc AS INTEGER
    DIM SHARED ErrType AS INTEGER
    DIM SHARED DStack(0 TO 50) AS DOUBLE
    DIM SHARED FStack(0 TO 50) AS FRACTION
    DIM SHARED mode AS INTEGER
    DIM SHARED precision AS DOUBLE

    DIM vstr AS STRING, istr AS INTEGER, value AS DOUBLE, dv AS DOUBLE
    DIM r AS FRACTION

        CLS

        PRINT "CALCULATOR, v1.1"
        PRINT
        PRINT "  Mode is DECIMAL"
        PRINT "  Use FRAC to switch to FRACTION mode, DEC to switch to DECIMAL."
        PRINT

        LET precision = .0000001#
        LET mode = DECIMAL
        INPUT "? ", vstr
        LET vstr = LTRIM$(RTRIM$(UCASE$(vstr)))
        DO WHILE vstr <> ""
            IF vstr = "FRAC" THEN
                IF mode = DECIMAL THEN
                    LET mode = FRACT
                    PRINT
                    PRINT "  Mode is FRACTION"
                    PRINT
                    LET dv = fgcd(DStack(0), precision)
                    FracSet r, INT((DStack(0) / dv) + .5#), INT((1# / dv) + .5#)
                    LET FStack(0) = r
                    DisplayResult
                END IF
            ELSEIF vstr = "DEC" THEN
                IF mode = FRACT THEN
                    LET mode = DECIMAL
                    PRINT
                    PRINT "  Mode is DECIMAL"
                    PRINT
                    LET DStack(0) = FStack(0).numerator / CDBL(FStack(0).denominator)
                    DisplayResult
                END IF
            ELSE
                Evaluate vstr
                SELECT CASE ErrType
                CASE 0
                    DisplayResult
                CASE 1
                    PRINT SPACE$(1 + ErrLoc); "^ Missing value"
                CASE 2
                    PRINT SPACE$(1 + ErrLoc); "^ Missing closing parenthesis"
                CASE 3
                    PRINT SPACE$(1 + ErrLoc); "^ Invalid or extraneous characters"
                END SELECT
            END IF
            INPUT "? ", vstr
            LET vstr = LTRIM$(RTRIM$(UCASE$(vstr)))
        LOOP

        END

DecParse:
        DATA "+-0123456789.DEde "
        DATA 7, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 4, 4, 4, 4, 5, 6
        DATA  2,  4,  3, -1,  1, -1, -1
        DATA -1,  4,  3, -1, -1, -1, -1
        DATA -1,  5, -1, -1, -1, -1, -1
        DATA -1,  4,  5,  6,  9,  0, -1
        DATA -1,  5, -1,  6,  9,  0, -1
        DATA  7,  8, -1, -1, -1, -1, -1
        DATA -1,  8, -1, -1, -1, -1, -1
        DATA -1,  8, -1, -1,  9,  0, -1
        DATA -1, -1, -1, -1,  9,  0, -1

SUB DisplayResult

        SELECT CASE mode
        CASE FRACT
            IF FStack(0).denominator <> 1 THEN
                PRINT LTRIM$(STR$(FStack(0).numerator)) + "/" + LTRIM$(STR$(FStack(0).denominator))
            ELSE
                PRINT LTRIM$(STR$(FStack(0).numerator))
            END IF
        CASE DECIMAL
            PRINT LTRIM$(STR$(DStack(0)))
        END SELECT
        PRINT

END SUB

SUB Evaluate (expressiontext AS STRING)

    DIM count AS INTEGER, idx AS INTEGER

        LET ErrType = 0

        LET idx = 1
        LET count = Expression(idx, expressiontext, 0)

        IF idx <= LEN(expressiontext) THEN
            LET ErrType = 3
            LET ErrLoc = idx
        END IF

END SUB

FUNCTION Expression% (idx AS INTEGER, text AS STRING, count AS INTEGER)

    DIM c AS INTEGER

        LET c = Term(idx, text, count)
        LET Expression = MoreTerms(idx, text, c)

END FUNCTION

FUNCTION Factor% (idx AS INTEGER, text AS STRING, count AS INTEGER)

    DIM c AS INTEGER

        IF ErrType <> 0 THEN
            LET Factor = count
            EXIT FUNCTION
        END IF

        IF Match("(", idx, text) THEN
            LET c = Expression(idx, text, count)
            IF NOT Match(")", idx, text) THEN
                LET ErrType = 2
                LET ErrLoc = idx
            END IF
            LET Factor = c
        ELSE
            LET Factor = Number(idx, text, count)
        END IF

END FUNCTION

FUNCTION fgcd# (f AS DOUBLE, precision AS DOUBLE)
'   
'   Here's an equivalent version of the gcd() function for converting
'   floating-point values into fractions.  You'll note the similarities,
'   I suspect.  The 'precision' parameter lets you set how closely the
'   fraction must approximate the floating-point value and affects the
'   returned divisor.  Use the function like this:
'
'       LET value = 1.14#
'       LET dv = fgcd(value, .000001#)      ' dv = 0.02 in this example
'       FracSet f, INT((value / dv) + .5#), INT((1# / dv) + .5#)
'       PRINT f.numerator, f.denominator
'
'   Fraction f is then 57/50.
'
'   Note that the numerator is computed by rounding the result from
'   dividing the original floating-point value by the computed divisor
'   and that the denominator is computed by dividing 1 by that same
'   divisor.  This is because the original "fraction" was (value/1) and
'   you must compute the new fraction as (value/dv)/(1/dv), with rounding.
'
'   The computed divisor, from fgcd(), assumes you'll use it this way.
'
'   Finally, even after recomputing the new numerator and denominator, you
'   need to make one last call to the integer version of gcd(), since the
'   computed numerator and denominator may have rounded in such a way to
'   still contain another common divisor that should be removed.  Using
'   FracSet makes sure this is done.
'   

    DIM x AS DOUBLE, y AS DOUBLE, r AS DOUBLE

        IF f = 0# THEN
            LET fgcd = 1#
        ELSE
            LET x = ABS(f)
            LET y = 1#
            LET r = x - INT(x)
            DO WHILE r > precision
                LET x = y
                LET y = r
                LET r = x - INT(x / y) * y
            LOOP
            LET fgcd = y
        END IF

END FUNCTION

SUB FracAdd (r AS FRACTION, a1 AS FRACTION, a2 AS FRACTION)
'   
'   Add fractions a1 and a2, giving fraction r.
'   

    DIM a AS LONG, a1d AS LONG

        LET a = gcd(a1.denominator, a2.denominator)
        LET a1d = a1.denominator / a

        FracSet r, a1.numerator * (a2.denominator / a) + a2.numerator * a1d, a1d * a2.denominator

END SUB

SUB FracDiv (r AS FRACTION, n AS FRACTION, d AS FRACTION)
'   
'   Divide fraction n by fraction d, giving fraction r.
'   

    DIM a AS LONG, b AS LONG

        IF d.numerator >= 0& THEN
            LET a = gcd(n.numerator, d.numerator)
        ELSE
            LET a = -gcd(n.numerator, d.numerator)
        END IF
        LET b = gcd(n.denominator, d.denominator)
        LET r.numerator = (n.numerator / a) * (d.denominator / b)
        LET r.denominator = (n.denominator / b) * (d.numerator / a)

END SUB

SUB FracMul (r AS FRACTION, m1 AS FRACTION, m2 AS FRACTION)
'   
'   Multiply fractions m1 and m2, giving fraction r.
'   

    DIM a AS LONG, b AS LONG

        LET a = gcd(m1.numerator, m2.denominator)
        LET b = gcd(m2.numerator, m2.denominator)
        LET r.numerator = (m1.numerator / a) * (m2.numerator / b)
        LET r.denominator = (m1.denominator / b) * (m2.denominator / a)

END SUB

SUB FracSet (f AS FRACTION, n AS LONG, d AS LONG)
'   
'   Initialize fraction f from numerator n and denominator d.
'   

    DIM y AS LONG, n0 AS LONG, d0 AS LONG

        LET y = gcd(n, d)
        LET n0 = n / y
        LET d0 = d / y

        IF d0 < 0& THEN
            LET n0 = -n0
            LET d0 = -d0
        END IF

        LET f.numerator = n
        LET f.denominator = d

END SUB

SUB FracSub (r AS FRACTION, m AS FRACTION, s AS FRACTION)
'   
'   Subtract fraction s from fraction m, giving fraction r.
'   

    DIM a AS LONG, md AS LONG

        LET a = gcd(m.denominator, s.denominator)
        LET md = m.denominator / a

        FracSet r, m.numerator * (s.denominator / a) - s.numerator * md, md * s.denominator

END SUB

FUNCTION gcd& (n AS LONG, d AS LONG)
'   
'   Compute the greatest common divisor between two values.
'
'   This gcd algorithm uses division to iterate quickly to the answer.
'   

    DIM x AS LONG, y AS LONG, r AS LONG

        IF n = 0& AND d = 0& THEN
            LET gcd = 1&
        ELSEIF d = 0& THEN
            LET gcd = ABS(n)
        ELSE
            LET x = ABS(n)
            LET y = ABS(d)
            LET r = x MOD y
            DO WHILE r > 0&
                LET x = y
                LET y = r
                LET r = x MOD y
            LOOP
            LET gcd = y
        END IF

END FUNCTION

FUNCTION gcd0& (n AS LONG, d AS LONG)
'   
'   Compute the greatest common divisor between two values.
'
'   This gcd algorithm uses subtraction to find the answer.
'   

    DIM x AS LONG, y AS LONG

        IF n = 0& AND d = 0& THEN
            LET gcd0 = 1&
        ELSEIF d = 0& THEN
            LET gcd0 = ABS(n)
        ELSE
            LET x = ABS(n)
            LET y = ABS(d)
            IF x <> 0& THEN
                DO
                    IF x < y THEN
                        LET y = y - x
                    ELSEIF x > y THEN
                        LET x = x - y
                    ELSE
                        EXIT DO
                    END IF
                LOOP
            END IF
            LET gcd0 = y
        END IF

END FUNCTION

FUNCTION Match% (char AS STRING, idx AS INTEGER, text AS STRING)

    DIM i AS INTEGER

        IF ErrType <> 0 THEN
            LET Match = 0
            EXIT FUNCTION
        END IF

        LET i = idx
        DO WHILE i <= LEN(text)
            IF MID$(text, i, 1) <> " " THEN
                EXIT DO
            END IF
            LET i = i + 1
        LOOP
        IF i <= LEN(text) THEN
            IF MID$(text, i, 1) = char THEN
                LET idx = i + 1
                LET Match = -1
                EXIT FUNCTION
            END IF
        END IF
        LET Match = 0

END FUNCTION

FUNCTION MoreFactors% (idx AS INTEGER, text AS STRING, count AS INTEGER)

    DIM c AS INTEGER, r AS FRACTION

        IF Match("*", idx, text) THEN
            LET c = Factor(idx, text, count)
            IF c > count THEN
                SELECT CASE mode
                CASE FRACT
                    FracMul r, FStack(c - 2), FStack(c - 1)
                    LET FStack(c - 2) = r
                CASE DECIMAL
                    LET DStack(c - 2) = DStack(c - 2) * DStack(c - 1)
                END SELECT
                LET c = MoreFactors(idx, text, c - 1)
            END IF
            LET MoreFactors = c
        ELSEIF Match("/", idx, text) THEN
            LET c = Factor(idx, text, count)
            IF c > count THEN
                SELECT CASE mode
                CASE FRACT
                    FracDiv r, FStack(c - 2), FStack(c - 1)
                    LET FStack(c - 2) = r
                CASE DECIMAL
                    LET DStack(c - 2) = DStack(c - 2) / DStack(c - 1)
                END SELECT
                LET c = MoreFactors(idx, text, c - 1)
            END IF
            LET MoreFactors = c
        ELSE
            LET MoreFactors = count
        END IF

END FUNCTION

FUNCTION MoreTerms% (idx AS INTEGER, text AS STRING, count AS INTEGER)

    DIM c AS INTEGER, r AS FRACTION

        IF Match("+", idx, text) THEN
            LET c = Term(idx, text, count)
            IF c > count THEN
                SELECT CASE mode
                CASE FRACT
                    FracAdd r, FStack(c - 2), FStack(c - 1)
                    LET FStack(c - 2) = r
                CASE DECIMAL
                    LET DStack(c - 2) = DStack(c - 2) + DStack(c - 1)
                END SELECT
                LET c = MoreTerms(idx, text, c - 1)
            END IF
            LET MoreTerms = c
        ELSEIF Match("-", idx, text) THEN
            LET c = Term(idx, text, count)
            IF c > count THEN
                SELECT CASE mode
                CASE FRACT
                    FracSub r, FStack(c - 2), FStack(c - 1)
                    LET FStack(c - 2) = r
                CASE DECIMAL
                    LET DStack(c - 2) = DStack(c - 2) - DStack(c - 1)
                END SELECT
                LET c = MoreTerms(idx, text, c - 1)
            END IF
            LET MoreTerms = c
        ELSE
            LET MoreTerms = count
        END IF

END FUNCTION

FUNCTION Number% (idx AS INTEGER, text AS STRING, count AS INTEGER)

    STATIC isloaded AS INTEGER
    STATIC DecimalCharMap AS STRING
    STATIC DecimalStateTable() AS INTEGER
    STATIC DecimalTokenMap() AS INTEGER

    DIM i AS INTEGER, j AS INTEGER, s AS INTEGER, t AS INTEGER
    DIM r AS FRACTION, dv AS DOUBLE, value AS DOUBLE

        IF ErrType <> 0 THEN
            LET Number = count
            EXIT FUNCTION
        END IF

        IF NOT isloaded THEN
            DIM DecimalStateTable(1 TO 9, 1 TO 7) AS INTEGER
            DIM DecimalTokenMap(0 TO 19) AS INTEGER
            RESTORE DecParse
            READ DecimalCharMap
            FOR i = 0 TO 19
                READ DecimalTokenMap(i)
            NEXT i
            FOR i = 1 TO 9
                FOR j = 1 TO 7
                    READ DecimalStateTable(i, j)
                NEXT j
            NEXT i
            LET isloaded = -1
        END IF

        LET i = Parse(idx, text, 1, s, t, DecimalCharMap, DecimalTokenMap(), DecimalStateTable())

        IF i = 0 AND DecimalStateTable(s, t) >= 0 THEN
            LET j = LEN(text) + 1
        ELSEIF i = 0 THEN
            LET j = LEN(text) + 1
        ELSE
            LET j = i
        END IF

        IF idx = j THEN
            LET ErrType = 1
            LET ErrLoc = idx
            LET Number = count
            EXIT FUNCTION
        END IF

        SELECT CASE mode
        CASE FRACT
            LET value = VAL(MID$(text, idx, j - idx))
            LET dv = fgcd(value, precision)
            FracSet r, INT((value / dv) + .5#), INT((1# / dv) + .5#)
            LET FStack(count) = r
        CASE DECIMAL
            LET DStack(count) = VAL(MID$(text, idx, j - idx))
        END SELECT
        LET idx = j
        LET Number = count + 1

END FUNCTION

FUNCTION Parse% (idx AS INTEGER, cstr AS STRING, istate AS INTEGER, state AS INTEGER, token AS INTEGER, clist AS STRING, ilist() AS INTEGER, istates() AS INTEGER)

    DIM i AS INTEGER, j AS INTEGER

        LET state = istate
        FOR i = idx TO LEN(cstr)
            LET token = ilist(INSTR(clist, MID$(cstr, i, 1)))
            LET j = istates(state, token)
            IF j <= 0 THEN
                LET Parse = i
                EXIT FUNCTION
            END IF
            LET state = j
        NEXT i
        LET token = ilist(UBOUND(ilist, 1))

        LET Parse = 0

END FUNCTION

FUNCTION Term% (idx AS INTEGER, text AS STRING, count AS INTEGER)

    DIM c AS INTEGER

        LET c = Factor(idx, text, count)
        LET Term = MoreFactors(idx, text, c)

END FUNCTION
