      
#include "Chip8.h"
#include "DirectInput.h"

typedef struct tagPluginInfo{

    HWND* Parent_hWnd;
    BYTE* FilePtr;
    DWORD FileSize;
    bool  DisasmReady;
    bool  LoadedPe;

}PLUGINFO; // Plugin Struct

#define    KeyDown(data, n)    ((data[n] & 0x80) ? true : false)
#define    KeyUp(data, n)    ((data[n] & 0x80) ? false : true)

extern void InitEmulator();
extern LPDIRECTINPUTDEVICE m_keyboard;
extern unsigned char keys[256];
extern bool EmulatorRunning;
extern PLUGINFO PlgData; 

char szFileName[MAX_PATH]="";

void LoadRom(HWND hWnd)
{
    OPENFILENAME ofn;
    HANDLE hFile;
    DWORD NumberOfBytesRead;
    WORD Code=0;     
    
    memset(&ofn,0,sizeof(OPENFILENAME));
    ofn.lStructSize = sizeof(OPENFILENAME);
    ofn.hwndOwner = hWnd;
    ofn.lpstrFilter = "All Files (*.*)\0*.*\0";
    ofn.lpstrFile = szFileName;
	ofn.nMaxFile = MAX_PATH;
	ofn.Flags = OFN_EXPLORER | OFN_FILEMUSTEXIST | OFN_HIDEREADONLY;	

    if(EmulatorRunning==TRUE)    
        PauseCpu=FALSE;    

    if(
        MessageBox(hWnd,"Would you like to Emulate the Disassembled File? \
        \nOtherwise select rom of your choise."\
        ,"Load Chip8 Rom",MB_YESNO|MB_DEFBUTTON1)==IDYES
      )
    {
        InitEmulator();
        Init_CPU();
        FileSize=PlgData.FileSize;
        memcpy(Memory+0x200,(char*)(PlgData.FilePtr),PlgData.FileSize);                
        EmulatorRunning=TRUE;                
        PauseCpu=FALSE;
    }
    else{    
        if(GetOpenFileName(&ofn)!=0)
        {             
            hFile=CreateFile(szFileName,
                GENERIC_READ,FILE_SHARE_READ,
                NULL,OPEN_EXISTING,FILE_ATTRIBUTE_NORMAL,NULL);
            
            if(hFile==INVALID_HANDLE_VALUE)
                return;       
            
            InitEmulator();
            Init_CPU();
            
            FileSize=GetFileSize(hFile,NULL);
            ReadFile(hFile,Memory+0x200,FileSize,&NumberOfBytesRead,NULL);
            CloseHandle(hFile);
            EmulatorRunning=TRUE;                
            PauseCpu=FALSE;
        }
        else {        
            return;      
        }
    }
        
}

void Emulate()
{               
    if(EmulatorRunning==TRUE && PauseCpu==FALSE)
    {    
        try{ 

            for(DWORD dwCPU=0; dwCPU<ClockSpeed*10000; dwCPU++); // Delay the CPU

            CPU();
        }
        catch(...) // Catch Exception
        {
            Init_CPU();
        }
        
        if (DelayTimer>0)
            --DelayTimer;
        
        // Generate Square_Wave Tone
        if (SoundTimer>0)
        { 
            if(SoundOn==TRUE && SoundTimer==1){                             
              Beep(150,40);
            }
            
            --SoundTimer;
        }  
        
    }
}

void Init_CPU()
{
    byte Font[80]={ 
            0xF0, 0x90, 0x90, 0x90, 0xF0,// 0
            0x20, 0x60, 0x20, 0x20, 0x70,// 1
            0xF0, 0x10, 0xF0, 0x80, 0xF0,// 2
            0xF0, 0x10, 0xF0, 0x10, 0xF0,// 3
            0x90, 0x90, 0xF0, 0x10, 0x10,// 4
            0xF0, 0x80, 0xF0, 0x10, 0xF0,// 5
            0xF0, 0x80, 0xF0, 0x90, 0xF0,// 6
            0xF0, 0x10, 0x20, 0x40, 0x40,// 7
            0xF0, 0x90, 0xF0, 0x90, 0xF0,// 8
            0xF0, 0x90, 0xF0, 0x10, 0xF0,// 9
            0xF0, 0x90, 0xF0, 0x90, 0x90,// A
            0xE0, 0x90, 0xE0, 0x90, 0xE0,// B
            0xF0, 0x80, 0x80, 0x80, 0xF0,// C
            0xE0, 0x90, 0x90, 0x90, 0xE0,// D
            0xF0, 0x80, 0xF0, 0x80, 0xF0,// E
            0xF0, 0x80, 0xF0, 0x80, 0x80 // F
    };
    
    int i;

    CodeIndex=0x200;
    DelayTimer=SoundTimer=0;
    RegI=SP=0;

    for(i=0;i<80;i++)
      Memory[i]=Font[i];
    

    for(i=0;i<MAX_REGS;i++)
        Registers[i]=0;

    for(i=0;i<SCREEN_SIZE;i++)
        Screen[i]=0;   

    for(i=0;i<MAX_STACK;i++)
        Stack[i]=0;
        
}

void CPU_STEP()
{    
    CPU();
    
    if (DelayTimer>0)
        --DelayTimer;
    
    // Generate Square_Wave Tone
    if (SoundTimer>0)
    { 
        if(SoundOn==TRUE && SoundTimer==1){                             
            Beep(150,40);
        }
        
        --SoundTimer;
    } 
}

void CPU()
{
    WORD wOpcode = Memory[CodeIndex],op;
    
    op = wOpcode = (wOpcode<<8)|Memory[CodeIndex+1];
    CHAR8 Opcode = (wOpcode & 0xF000)>>12;

    switch(Opcode) {

        case 0x00:
        {
            switch(wOpcode&0x00FF)
            {
                case 0xE0:  Chip8_cls();  break;
                case 0xEE:  Chip8_rts();  break;
                case 0xFB:  CodeIndex+=2; break;
                case 0xFC:  CodeIndex+=2; break;
                case 0xFE:  CodeIndex+=2; break;
                case 0xFF:  CodeIndex+=2; break;   
                default:    CodeIndex+=2; break;
            }
        }
    	break;

        case 0x01:  Chip8_jmp(wOpcode);                      break;
        case 0x02:  Chip8_jsr(wOpcode);                      break;
        case 0x03:  Chip8_Skip_Reg_eq_Constant(wOpcode);     break;
        case 0x04:  Chip8_Skip_Reg_not_eq_Constant(wOpcode); break;
        case 0x05:  Chip8_Skip_Reg_r_eq_Reg_y(wOpcode);      break;
        case 0x06:  Chip8_Mov_Reg_r_Constant(wOpcode);       break;
        case 0x07:  Chip8_Add_Reg_r_Constant(wOpcode);       break;
        case 0x08: 
        {
            switch(wOpcode&0x000F)
            {
              case 0x00: Chip8_Mov_Reg_y_to_Reg_r(wOpcode); break;
              case 0x01: Chip8_Or_Reg_y_to_Reg_r(wOpcode);  break;
              case 0x02: Chip8_And_Reg_y_to_Reg_r(wOpcode); break;
              case 0x03: Chip8_Xor_Reg_y_to_Reg_r(wOpcode); break;
              case 0x04: Chip8_Add_Reg_y_to_Reg_r(wOpcode); break;
              case 0x05: Chip8_Sub_Reg_y_to_Reg_r(wOpcode); break;
              case 0x06: Chip8_Shift_Left_Reg_r(wOpcode);   break;
              case 0x07: Chip8_Reg_r_rsb_Reg_y(wOpcode);    break;
              case 0x0E: Chip8_Shift_Right_Reg_r(wOpcode);  break;             
            }         
        }
        break;

        case 0x09:  Chip8_Skip_Reg_r_not_eq_Reg_y(wOpcode);           break;
        case 0x0A:  Chip8_Load_RegI_with_Constant(wOpcode);           break;
        case 0x0B:  Chip8_Jmp_To_Address_with_Reg0(wOpcode);          break;
        case 0x0C:  Chip8_Random_Number_less_or_eq_Constant(wOpcode); break;
        case 0x0D: 
        {
            switch(wOpcode&0x000F)
            {
              case 0x00:  Chip8_Put_16it_Pixel_On_Screen(wOpcode); break;
              default: 
              {
                  Chip8_Put_8Bit_Pixel_On_Screen(wOpcode);
              }
              break;
            }
        }
        break;

        case 0x0E:
        {
            switch(wOpcode&0x00FF)
            {
                case 0x9E:  Chip8_Skip_if_key_pressed(wOpcode);     break;
                case 0xA1:  Chip8_Skip_if_key_not_pressed(wOpcode); break;
            }
        }
        break;

        case 0x0F:
        {
            switch(wOpcode&0x00FF)
            {
                case 0x07: Chip8_Get_delay_timer(wOpcode);              break;
                case 0x0A: Chip8_Scan_Keys(wOpcode);                    break;
                case 0x15: Chip8_Set_delay_timer(wOpcode);              break;
                case 0x18: Chip8_Set_sound_timer(wOpcode);              break;
                case 0x1E: Chip8_Add_Reg_r_to_Reg_i(wOpcode);           break;
                case 0x29: Chip8_Set_i_to_hex_character(wOpcode);       break;
                case 0x30: Chip8_Set_i_to_hex_character_super(wOpcode); break;
                case 0x33: Chip8_Store_bcd_at_ram(wOpcode);             break;
                case 0x55: Chip8_Set_Regs_at_Memory(wOpcode);           break;
                case 0x65: Chip8_Load_Regs_from_Memory(wOpcode);        break;
            }
        }  
        break;
    }               
}

void Chip8_jsr(WORD wOpcode)
{
    Stack[SP]=CodeIndex;
    SP++;
    CodeIndex=wOpcode&0x0FFF;
}

void Chip8_jmp(WORD wOpcode)
{
    CodeIndex=wOpcode&0x0FFF;
}

void Chip8_Skip_Reg_eq_Constant(WORD wOpcode)
{
    if(Registers[(wOpcode&0x0F00)>>8] == (wOpcode&0x00FF) )
        CodeIndex+=4;
    else 
        CodeIndex+=2;
}

void Chip8_Skip_Reg_not_eq_Constant(WORD wOpcode)
{
    if(Registers[(wOpcode&0x0F00)>>8] != (wOpcode&0x00FF) )
        CodeIndex+=4;
    else 
        CodeIndex+=2;
}

void Chip8_Skip_Reg_r_eq_Reg_y(WORD wOpcode)
{
    if( Registers[(wOpcode&0x0F00)>>8] == Registers[(wOpcode&0x00F0)>>4] )
        CodeIndex+=4;
    else 
        CodeIndex+=2;
}

void Chip8_Mov_Reg_r_Constant(WORD wOpcode)
{
    Registers[(wOpcode&0x0F00)>>8]=wOpcode&0x00FF;
    CodeIndex+=2;
}

void Chip8_Add_Reg_r_Constant(WORD wOpcode)
{
    Registers[(wOpcode&0x0F00)>>8]+=wOpcode&0x00FF;
    CodeIndex+=2;
}

void Chip8_Mov_Reg_y_to_Reg_r(WORD wOpcode)
{
    Registers[(wOpcode&0x0F00)>>8]=Registers[(wOpcode&0x00F0)>>4];
    CodeIndex+=2;
}

void Chip8_Or_Reg_y_to_Reg_r(WORD wOpcode)
{
    Registers[(wOpcode&0x0F00)>>8]|=Registers[(wOpcode&0x00F0)>>4];
    CodeIndex+=2;
}

void Chip8_And_Reg_y_to_Reg_r(WORD wOpcode)
{
    Registers[(wOpcode&0x0F00)>>8]&=Registers[(wOpcode&0x00F0)>>4];
    CodeIndex+=2;
}

void Chip8_Xor_Reg_y_to_Reg_r(WORD wOpcode)
{
    Registers[(wOpcode&0x0F00)>>8]^=Registers[(wOpcode&0x00F0)>>4];
    CodeIndex+=2;
}

void Chip8_Add_Reg_y_to_Reg_r(WORD wOpcode)
{
    if(Registers[(wOpcode&0x00F0)>>4] >  (0xFF-Registers[(wOpcode&0x0F00)>>8]))
        Registers[0x0F]=1;
    else
        Registers[0x0F]=0;

    Registers[(wOpcode&0x0F00)>>8]+=Registers[(wOpcode&0x00F0)>>4];    
    CodeIndex+=2;
}

void Chip8_Sub_Reg_y_to_Reg_r(WORD wOpcode)
{
    if( (Registers[(wOpcode&0x0F00)>>8] < Registers[(wOpcode&0x00F0)>>4]))
        Registers[0x0F]=0;
    else
        Registers[0x0F]=1;

    Registers[(wOpcode&0x0F00)>>8]-=Registers[(wOpcode&0x00F0)>>4];
    
    CodeIndex+=2;
}

void Chip8_Shift_Right_Reg_r(WORD wOpcode)
{
    Registers[0x0F]=Registers[((wOpcode&0x0F00)>>8)]&0x01;
    Registers[(wOpcode&0x0F00)>>8]>>=1;
    
    CodeIndex+=2;
}

void Chip8_Reg_r_rsb_Reg_y(WORD wOpcode)
{
    
    if((Registers[(wOpcode&0x00F0)>>4]<Registers[(wOpcode&0x0F00)>>8]))
       Registers[0x0F]=1;
    else
       Registers[0x0F]=0;
        
    Registers[(wOpcode&0x0F00)>>8] = Registers[(wOpcode&0x00F0)>>4]-Registers[(wOpcode&0x0F00)>>8];
    
    CodeIndex+=2;
}

void Chip8_Shift_Left_Reg_r(WORD wOpcode)
{
    Registers[0x0F]=(((Registers[(wOpcode&0x0F00)>>8])&0x80)>>7);
    Registers[(wOpcode&0x0F00)>>8]<<=1;
    
    CodeIndex+=2;
}

void Chip8_Skip_Reg_r_not_eq_Reg_y(WORD wOpcode)
{
    if( Registers[(wOpcode&0x0F00)>>8] != Registers[(wOpcode&0x00F0)>>4] )
        CodeIndex+=4;
    else 
        CodeIndex+=2;
}

void Chip8_Load_RegI_with_Constant(WORD wOpcode)
{
    RegI = wOpcode&0x0FFF;
    CodeIndex+=2;
}

void Chip8_Jmp_To_Address_with_Reg0(WORD wOpcode)
{
  CodeIndex = Registers[0]+(wOpcode&0x0FFF);
  //CodeIndex+=2;
}

void Chip8_Random_Number_less_or_eq_Constant(WORD wOpcode)
{    
    Registers[(wOpcode&0x0F00)>>8] = rand()%(wOpcode&0x00FF);
    CodeIndex+=2;
}

void Chip8_cls()
{
  //  for(int i=0;i<SCREEN_SIZE-1;i++)
  //      Screen[i]=0;
    
    memset(&Screen,0,SCREEN_SIZE-1);
    CodeIndex+=2;
}

void Chip8_rts()
{
   --SP;
   CodeIndex = Stack[SP];
   CodeIndex+=2;
}

void Chip8_Get_delay_timer(WORD wOpcode)
{
    Registers[(wOpcode&0x0F00)>>8]=DelayTimer;
    CodeIndex+=2;
}

void Chip8_Set_delay_timer(WORD wOpcode)
{
    DelayTimer=Registers[(wOpcode&0x0F00)>>8];
    CodeIndex+=2;
}

void Chip8_Scan_Keys(WORD wOpcode)
{
    Registers[(wOpcode&0x0F00)>>8]=ScanKeys();    
    CodeIndex+=2;
}

void Chip8_Skip_if_key_pressed(WORD wOpcode)
{
    if(CheckKey(Keys[Registers[(wOpcode&0x0F00)>>8]])!=0)
        CodeIndex+=4;
    else
        CodeIndex+=2;
}

void Chip8_Skip_if_key_not_pressed(WORD wOpcode)
{
    if(CheckKey(Registers[(wOpcode&0x0F00)>>8])==0)
        CodeIndex+=4;
    else
        CodeIndex+=2;
}

int CheckKey(WORD key)
{       
    int KeyVal=0;

     
    m_keyboard->GetDeviceState(sizeof(keys), &keys);        
    
    try{    
        return KeyDown(keys, Keys[key]); 
    }
    catch(...)
    {
        return FALSE;
    }
}

CHAR8 ScanKeys()
{    
    int done = 0;   
       
    m_keyboard->GetDeviceState(sizeof(keys), &keys);
    
    
    if (KeyDown(keys, DIK_NUMPAD0))
        return 12; // Key's index

    if (KeyDown(keys, DIK_NUMPAD1))
        return 8;

    if (KeyDown(keys, DIK_NUMPAD2))
        return 9;

    if (KeyDown(keys, DIK_NUMPAD3))
        return 10;

    if (KeyDown(keys, DIK_NUMPAD4))
        return 4;

    if (KeyDown(keys, DIK_NUMPAD5))
        return 5;

    if (KeyDown(keys, DIK_NUMPAD6))
        return 6;

    if (KeyDown(keys, DIK_NUMPAD7))
        return 0;

    if (KeyDown(keys, DIK_NUMPAD8))
        return 1;

    if (KeyDown(keys, DIK_NUMPAD9))
        return 2;

    if (KeyDown(keys, DIK_NUMPADSLASH))
        return 3;

    if (KeyDown(keys, DIK_NUMPADSTAR))
        return 7;

    if (KeyDown(keys, DIK_NUMPADMINUS))
        return 11;

    if (KeyDown(keys, DIK_DELETE))
        return 13;

    if (KeyDown(keys, DIK_RETURN))
        return 14;

    if (KeyDown(keys, DIK_NUMPADPLUS))
        return 15;

    return -1;

}

void Chip8_Set_sound_timer(WORD wOpcode)
{
    SoundTimer=Registers[(wOpcode&0x0F00)>>8];
    CodeIndex+=2;
}

void Chip8_Add_Reg_r_to_Reg_i(WORD wOpcode)
{
    RegI+=Registers[(wOpcode&0x0F00)>>8];
    CodeIndex+=2;
}

void Chip8_Set_i_to_hex_character(WORD wOpcode)
{
  RegI = Registers[(wOpcode&0x0F00)>>8]*5;
  CodeIndex+=2;
}

void Chip8_Set_i_to_hex_character_super(WORD wOpcode)
{
  RegI = Registers[(wOpcode&0x0F00)>>8]*10;
  CodeIndex+=2;
}

void Chip8_Store_bcd_at_ram(WORD wOpcode)
{
    Memory[RegI]   = Registers[(wOpcode&0x0F00)>>8]/100;
    Memory[RegI+1] = (Registers[(wOpcode&0x0F00)>>8]/10)%10;
    Memory[RegI+2] = (Registers[(wOpcode&0x0F00)>>8]%100)%10;
    CodeIndex+=2;
}

void Chip8_Set_Regs_at_Memory(WORD wOpcode)
{
    for(int i=0;i<=((wOpcode&0x0F00)>>8);i++)
    {
        Memory[RegI+i]=Registers[i];
    }

    CodeIndex+=2;
}

void Chip8_Load_Regs_from_Memory(WORD wOpcode)
{
    for(int i=0;i<=((wOpcode&0x0F00)>>8);i++)
    {
        Registers[i]=Memory[RegI+i];
    }

    CodeIndex+=2;
}


void Chip8_Put_8Bit_Pixel_On_Screen(WORD wOpcode)
{
    int height,x,y;
    CHAR8 Pixel;

    height = wOpcode&0x000F;
    x=Registers[(wOpcode&0x0F00)>>8];
    y=Registers[(wOpcode&0x00F0)>>4];

    for(int row=0;row<height;row++)
    {
      Pixel = Memory[RegI+row]; // Byte Per Line
      for(int line=0;line<8;line++)
      {
          if((Pixel & (0x80 >> line))!=0)
          {
              if(Screen[(line+x+((y+row)*64))]==1)
              {
                  Registers[0x0F]=1;                                    
              }             

              Screen[line+x+((y+row)*64)]^=1;
          }
      }
    }

    CodeIndex+=2;
    DrawScene();
}

void Chip8_Put_16it_Pixel_On_Screen(WORD wOpcode)
{
    int x,y;
    CHAR8 Pixel;

    x=Registers[(wOpcode&0x0F00)>>8];
    y=Registers[(wOpcode&0x00F0)>>4];

    for(int row=0;row<16;row++)
    {
      Pixel = Memory[RegI+row]; // Byte Per Line
      for(int line=0;line<16;line++)
      {
          if((Pixel & (0x80 >> line))==1)
          {
              if(Screen[(line+x+((y+row)*64))]==1)
              {
                  Registers[0x0F]=1;
              }

              Screen[line+x+((y+row)*64)]^=1;
          }
      }
    }

    CodeIndex+=2;
    DrawScene();
}

void DrawScene()
{
    int y,x,posx,posy;     
    DDSURFACEDESC   ddsd;
    unsigned char * szSurface;
    int             offset; 
    
    ddsd.dwSize = sizeof(ddsd);
    // 32x64
    // height = 32
    // width  = 64
    
    try{          
        g_pDDSBack->Lock( NULL, &ddsd, DDLOCK_WAIT | DDLOCK_SURFACEMEMORYPTR, NULL );
        szSurface = (unsigned char*)ddsd.lpSurface;
        
        for(y=0;y<32;y++)
        {
            for(x=0;x<64;x++)
            {
                if(Screen[x+(y*64)]==1)
                {                                
                    // Make Pixel 8x big                
                    for(posx=x*STRETCHX;posx<(x*STRETCHX)+STRETCHX;posx++)
                    {
                        for(posy=y*STRETCHY;posy<(y*STRETCHY)+STRETCHY;posy++)
                        {
                            offset = (posy*ddsd.lPitch + posx);
                            *(szSurface+offset) = 0xFF; // White                        
                        }
                    }                
                    
                }
                else{
                    
                    for(posx=x*STRETCHX;posx<(x*STRETCHX)+STRETCHX;posx++)
                    {
                        for(posy=y*STRETCHY;posy<(y*STRETCHY)+STRETCHY;posy++)
                        {
                            offset = (posy*ddsd.lPitch + posx);
                            *(szSurface+offset) = 0x00; // Black                   
                        }
                    }                
                }
                
            }  
        }
        
        g_pDDSBack->Unlock(ddsd.lpSurface);
        CheckSurfaces();
        DDFlip(); // Double Buffer
    }
    catch(...)
    {
        MessageBox(NULL,"Failed in Drawing","Error",MB_OK);
    }
}

void Disasm8Chip(WORD wOpcode,char *Assembly)
{    
    CHAR8 Opcode = (wOpcode & 0xF000)>>12;

    switch(Opcode) {

        case 0x00:
        {
            switch(wOpcode&0x00FF)
            {
                case 0xE0: lstrcpy(Assembly,"cls");     break;
                case 0xEE: lstrcpy(Assembly,"rts");     break;
                case 0xFB: lstrcpy(Assembly,"scright"); break;
                case 0xFC: lstrcpy(Assembly,"scleft");  break;
                case 0xFE: lstrcpy(Assembly,"low");     break;
                case 0xFF: lstrcpy(Assembly,"high");    break;
                default: wsprintf(Assembly,"scdown %d",wOpcode&0x000F);
            }
        }
    	break;

        case 0x01: wsprintf(Assembly,"jmp %03X",wOpcode&0x0FFF);                              break;
        case 0x02: wsprintf(Assembly,"jsr %03X",wOpcode&0x0FFF);                              break;
        case 0x03: wsprintf(Assembly,"skeq v%d,%X",(wOpcode&0x0F00)>>8,wOpcode&0x00FF);       break;
        case 0x04: wsprintf(Assembly,"skne v%d,%X",(wOpcode&0x0F00)>>8,wOpcode&0x00FF);       break;
        case 0x05: wsprintf(Assembly,"skeq v%d,v%d",(wOpcode&0x0F00)>>8,(wOpcode&0x00F0)>>4); break;
        case 0x06: wsprintf(Assembly,"mov v%d,%X",(wOpcode&0x0F00)>>8,wOpcode&0x00FF);        break;
        case 0x07: wsprintf(Assembly,"add v%d,%X",(wOpcode&0x0F00)>>8,wOpcode&0x00FF);        break;
        case 0x08: 
        {
            switch(wOpcode&0x000F)
            {
              case 0x00: wsprintf(Assembly,"mov v%d,v%d",(wOpcode&0x0F00)>>8,(wOpcode&0x00F0)>>4);  break;
              case 0x01: wsprintf(Assembly,"or v%d,v%d",(wOpcode&0x0F00)>>8,(wOpcode&0x00F0)>>4);   break;
              case 0x02: wsprintf(Assembly,"and v%d,v%d",(wOpcode&0x0F00)>>8,(wOpcode&0x00F0)>>4);  break;
              case 0x03: wsprintf(Assembly,"xor v%d,v%d",(wOpcode&0x0F00)>>8,(wOpcode&0x00F0)>>4);  break;
              case 0x04: wsprintf(Assembly,"add v%d,v%d",(wOpcode&0x0F00)>>8,(wOpcode&0x00F0)>>4);  break;
              case 0x05: wsprintf(Assembly,"sub v%d,v%d",(wOpcode&0x0F00)>>8,(wOpcode&0x00F0)>>4);  break;
              case 0x06: wsprintf(Assembly,"shr v%d",(wOpcode&0x0F00)>>8);                          break;
              case 0x07: wsprintf(Assembly,"rsb v%d,v%d",(wOpcode&0x0F00)>>8,(wOpcode&0x00F0)>>4);  break;
              case 0x0E: wsprintf(Assembly,"shl v%d",(wOpcode&0x0F00)>>8);                          break;
             
            }         
        }
        break;

        case 0x09: wsprintf(Assembly,"skne v%d,v%d",(wOpcode&0x0F00)>>8,(wOpcode&0x00F0)>>4); break;
        case 0x0A: wsprintf(Assembly,"mvi %03X",wOpcode&0x0FFF);                              break;
        case 0x0B: wsprintf(Assembly,"jmi %03X",wOpcode&0x0FFF);                              break;
        case 0x0C: wsprintf(Assembly,"rand v%d,%03X",(wOpcode&0x0F00)>>8,wOpcode&0x0FFF);     break;
        case 0x0D: 
        {
            switch(wOpcode&0x000F)
            {
              case 0x00: wsprintf(Assembly, "xsprite %d,%d",(wOpcode&0x0F00)>>8, (wOpcode&0x00F0)>>4);                   break;
              default:   wsprintf(Assembly, "sprite %d,%d,%d",(wOpcode&0x0F00)>>8, (wOpcode&0x00F0)>>4,wOpcode&0x000F); break;
            }
        }
        break;

        case 0x0E:
        {
            switch(wOpcode&0x00FF)
            {
                case 0x9E: wsprintf(Assembly,"skpr %d",(wOpcode&0x0F00)>>8); break;
                case 0xA1: wsprintf(Assembly,"skup %d",(wOpcode&0x0F00)>>8); break;
            }
        }
        break;

        case 0x0F:
        {
            switch(wOpcode&0x00FF)
            {
                case 0x07: wsprintf(Assembly,"gdelay v%d",(wOpcode&0x0F00)>>8);  break;
                case 0x0A: wsprintf(Assembly,"key v%d",(wOpcode&0x0F00)>>8);     break;
                case 0x15: wsprintf(Assembly,"sdelay v%d",(wOpcode&0x0F00)>>8);  break;
                case 0x18: wsprintf(Assembly,"ssound v%d",(wOpcode&0x0F00)>>8);  break;
                case 0x1E: wsprintf(Assembly,"adi v%d",(wOpcode&0x0F00)>>8);     break;
                case 0x29: wsprintf(Assembly,"font v%d",(wOpcode&0x0F00)>>8);    break;
                case 0x30: wsprintf(Assembly,"xfont v%d",(wOpcode&0x0F00)>>8);   break;
                case 0x33: wsprintf(Assembly,"bcd v%d",(wOpcode&0x0F00)>>8);     break;
                case 0x55: wsprintf(Assembly,"str v0-v%d",(wOpcode&0x0F00)>>8);  break;
                case 0x65: wsprintf(Assembly,"ldr v0-v%d",(wOpcode&0x0F00)>>8);  break;
            }
        } 
        break;
        
        default: 
        {
            wsprintf(Assembly,"???");
        }
        break;
    }
}