//--------------------------------------------------------------------
//
// EFSDump
// Copyright (C) 1999 Mark Russinovich
// Systems Internals - http://www.sysinternals.com
//
// This program demonstrates the use of Win2K EFS APIs to dump
// DDF and DRF field entries for encrypted files.
//
//--------------------------------------------------------------------
#include <windows.h>
#include <winefs.h>
#include <tchar.h>

//
// Global variables
//
BOOLEAN		Silent = FALSE;
BOOLEAN		Recurse = FALSE;


//--------------------------------------------------------------------
//
// PrintWin32Error
// 
// Translates a Win32 error into a text equivalent
//
//--------------------------------------------------------------------
void PrintWin32Error( DWORD ErrorCode )
{
	LPVOID lpMsgBuf;

	if( ErrorCode == ERROR_INVALID_FUNCTION ) {

		wprintf(L"File is not encrypted\n" );
		return;
	}
 
	FormatMessage( FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
					NULL, ErrorCode, 
					MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
					(LPTSTR) &lpMsgBuf, 0, NULL );
	wprintf(L"%s\n", lpMsgBuf );
	LocalFree( lpMsgBuf );
}


//--------------------------------------------------------------------
//
// ProcessFile
//
// Queries a file to obtain stream information.
//
//--------------------------------------------------------------------
VOID ProcessFile( BOOLEAN Directory, WCHAR *FileName )
{
	PENCRYPTION_CERTIFICATE_HASH_LIST   users;
	PENCRYPTION_CERTIFICATE_HASH        user;
	WCHAR systemName[1024], userName[1024];
	DWORD systemNameLen, userNameLen;
	SID_NAME_USE sidUse;
	DWORD i, error;

	//
	// We can't examine directories because the EFS
	// API won't let us
	//
	if( Directory ) return;

	//
	// Get Data Decryption Field (DDF) information
	//
	if( error = QueryUsersOnEncryptedFile( FileName,
                            &users )) {

        if( !Silent ) {

			wprintf(L"Error querying %s: ", FileName);
            PrintWin32Error( error );
        }
		return;
	}
	wprintf(L"%s:\n", FileName );

	//
	// Dump information on the users
	//
	wprintf(L"DDF %s:\n",
		users->nCert_Hash > 1 ? L"Entries" : L"Entry" );

	for( i = 0; i < users->nCert_Hash; i++ ) {

		user = users->pUsers[i];

		userNameLen = sizeof( userName );
		systemNameLen = sizeof( systemName );
		if( LookupAccountSid( NULL,
					user->pUserSid,
					userName,
					&userNameLen,
					systemName,
					&systemNameLen,
					&sidUse )) {

			wprintf(L"    %s\\%s:\n", systemName, userName );

		} else {

			wprintf(L"    Unknown user:\n");
		}

		wprintf( L"        %s\n", user->lpDisplayInformation );
	}

	//
    // Get Data Recovery Field (DRF) information
    //
    if( error = QueryRecoveryAgentsOnEncryptedFile( FileName,
                                    &users )) {

		if( !Silent ) {
			wprintf(L"Error querying %s: ", FileName);
			PrintWin32Error( error );
		}
		return;
	}

	//
	// Dump information on the Recovery Agents
	//	
	wprintf(L"DRF %s:\n",
	users->nCert_Hash > 1 ? L"Entries" : L"Entry" );

	for( i = 0; i < users->nCert_Hash; i++ ) {
		
		user = users->pUsers[i];

		userNameLen = sizeof( userName );
		systemNameLen = sizeof( systemName );
		if( LookupAccountSid( NULL,
					user->pUserSid,
					userName,
					&userNameLen,
					systemName,
					&systemNameLen,
					&sidUse )) {

			wprintf(L"    %s\\%s:\n", systemName, userName );

		} else {

			wprintf(L"    Unknown user:\n");
		}

		wprintf( L"        %s\n", user->lpDisplayInformation );
	}	

	FreeEncryptionCertificateHashList( users );
	wprintf(L"\n");
}




//--------------------------------------------------------------------
//
// ProcessDirectory
// 
// Recursive routine that passes files to the stream analyzing 
// function.
//
//--------------------------------------------------------------------
void ProcessDirectory( WCHAR *PathName, WCHAR *SearchPattern )
{
	WCHAR			subName[MAX_PATH], fileSearchName[MAX_PATH], searchName[MAX_PATH];
	HANDLE			dirHandle, patternHandle;
	static BOOLEAN	firstCall = TRUE;
	WIN32_FIND_DATA foundFile;

	//
	// Scan the files and/or directories if this is a directory
	//
	if( firstCall ) {

		if( wcsrchr( PathName, '*' ) ) {
	
            if( wcsrchr( PathName, '\\' ) ) {

                swprintf( SearchPattern, wcsrchr( PathName, '\\' )+1 );
                wcscpy( searchName, PathName );
                wcscpy( wcsrchr( searchName, '\\')+1, L"*.*" );

            } else {
                
                swprintf( SearchPattern, PathName );
                wcscpy( searchName, PathName );
            }
            swprintf( fileSearchName, L"%s", PathName );

		} else {

			swprintf( SearchPattern, L"*.*" );
			swprintf( searchName, L"%s", PathName );
            swprintf( fileSearchName, L"%s", PathName );
		}

	} else {

		swprintf( searchName, L"%s\\*.*", PathName );
		swprintf( fileSearchName, L"%s\\%s", PathName, SearchPattern );
	}

	//
	// Process all the files, according to the search pattern
	//
	if( (patternHandle = FindFirstFile( fileSearchName, &foundFile )) != 
		INVALID_HANDLE_VALUE  ) {

		do {

			if( wcscmp( foundFile.cFileName, L"." ) &&
				wcscmp( foundFile.cFileName, L".." )) {

				wcscpy( subName, searchName );
				if( wcsrchr( subName, '\\' ) ) 
					wcscpy( wcsrchr( subName, '\\')+1, foundFile.cFileName );
				else
					wcscpy( subName, foundFile.cFileName );

				//
				// Do this file/directory
				//
				ProcessFile( (BOOLEAN) (foundFile.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY),
							subName );
			}
		} while( FindNextFile( patternHandle, &foundFile ));
		FindClose( patternHandle );
	}

	//
	// Now recurse if we're supposed to
	//
	if( Recurse ) {

        if( firstCall && !wcsrchr( searchName, L'\\') ) {

            if( wcsrchr( searchName, L'*' )) {

                if( (dirHandle = FindFirstFile( L"*.*", &foundFile )) == 
                    INVALID_HANDLE_VALUE  ) {

                    //
                    // Nothing to process
                    //
                    return;
                }
            } else {

                if( (dirHandle = FindFirstFile( searchName, &foundFile )) == 
                    INVALID_HANDLE_VALUE  ) {

                    //
                    // Nothing to process
                    //
                    return;
                }
            }
        } else {

            if( (dirHandle = FindFirstFile( searchName, &foundFile )) == 
                INVALID_HANDLE_VALUE  ) {

                //
                // Nothing to process
                //
                return;
            }
        }
        firstCall = FALSE;

		do {

			if( (foundFile.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) &&
				wcscmp( foundFile.cFileName, L"." ) &&
				wcscmp( foundFile.cFileName, L".." )) {

				wcscpy( subName, searchName );
				if( wcsrchr( subName, '\\' ) ) 
					wcscpy( wcsrchr( subName, '\\')+1, foundFile.cFileName );
				else
					wcscpy( subName, foundFile.cFileName );

				//
				// Go into this directory
				//
				ProcessDirectory( subName, SearchPattern );

			}
		} while( FindNextFile( dirHandle, &foundFile ));
	}

	FindClose( dirHandle );
}


int Usage( WCHAR *ProgramName ) 
{
    wprintf(L"usage: %s [-s] [-q] <file or directory>\n", ProgramName );
    wprintf(L"-s     Recurse subdirectories\n");
    wprintf(L"-q     Don't print errors (Quiet)\n\n");
    return -1;
}


int wmain( int argc, WCHAR *argv[] ) 
{
    WCHAR       searchPattern[MAX_PATH];
	WCHAR		searchPath[MAX_PATH];
	PWCHAR		filePart;
	int			i;

    //
    // Print banner and perform parameter check
    //
    wprintf(L"\nEFS Information Dumper v1.02\n" );
    wprintf(L"Copyright (C) 1999 Mark Russinovich\n");
    wprintf(L"Systems Internals - http://www.sysinternals.com\n\n");

    if( argc < 2 ) {

        return Usage( argv[0] );
    }

    for( i = 1; i < argc - 1; i++ ) {

	    if( !wcsicmp( argv[i], L"/s" ) ||
			!wcsicmp( argv[i], L"-s" )) {

			Recurse = TRUE;

		} else if( !wcsicmp( argv[i], L"/q" ) ||
				   !wcsicmp( argv[i], L"-q" )) {

			Silent = TRUE;

		} else {

			return Usage( argv[0] );
		}
    }

	GetFullPathName( argv[argc-1], MAX_PATH, searchPath, &filePart );

    //
    // Now go and process directories
    //
    ProcessDirectory( searchPath, searchPattern );
    return 0;
}

