/* 
RANDRW.C
What are the chances that a random pointer would not GP fault?
Appears to be a 1-2% chance that a random ptr in Win95 will trash memory;
Chances go up for each DOS box running.

Andrew Schulman
andrew@ora.com
http://www.ora.com/windows/
ftp://ftp.ora.com/pub/examples/windows/win95.update/schulman.html
August 1995

In Windows 95, with lots of DOS boxes and Win32 apps running:

1000000 attempts
15235 hits
1.52% hit rate
268 seconds
3731 attempts/second
56 hits/second

Megabyte   Attempts  Hits   % Hits     Megabyte   Attempts  Hits   % Hits
---------  --------  ----   ------     ---------  --------  ----   ------
00000000h     251     248    98.80     00100000h     241     123    51.04
00200000h     236     41     17.37     00400000h     251     29     11.55
00500000h     237     2      0.84      00600000h     218     12     5.50
80000000h     229     221    96.51     80100000h     208     200    96.15
80200000h     267     215    80.52     80300000h     267     183    68.54
80400000h     214     2      0.93      81500000h     258     23     8.91
81600000h     229     3      1.31      81A00000h     234     75     32.05
81B00000h     271     8      2.95      81C00000h     237     11     4.64
81D00000h     244     14     5.74      81E00000h     222     14     6.31
82000000h     234     52     22.22     82200000h     224     140    62.50
82300000h     258     8      3.10      82400000h     269     1      0.37
82500000h     245     22     8.98      86500000h     255     238    93.33
86600000h     245     245    100.00    86700000h     229     229    100.00
86800000h     245     245    100.00    86900000h     236     236    100.00
86A00000h     230     230    100.00    86B00000h     229     229    100.00
86C00000h     219     219    100.00    86D00000h     264     264    100.00
86E00000h     244     244    100.00    86F00000h     238     238    100.00
87000000h     243     243    100.00    87100000h     240     240    100.00
87200000h     241     241    100.00    87300000h     253     253    100.00
87400000h     249     249    100.00    87500000h     223     223    100.00
87600000h     275     275    100.00    87700000h     258     258    100.00
87800000h     262     262    100.00    87900000h     218     218    100.00
87A00000h     243     243    100.00    87B00000h     262     262    100.00
87C00000h     219     219    100.00    87D00000h     212     212    100.00
87E00000h     216     216    100.00    87F00000h     247     247    100.00
88000000h     239     239    100.00    88100000h     242     242    100.00
88200000h     270     270    100.00    88300000h     271     271    100.00
88400000h     272     272    100.00    88500000h     239     237    99.16
88600000h     250     250    100.00    88700000h     229     229    100.00
88800000h     231     231    100.00    88900000h     258     101    39.15
BFE00000h     252     32     12.70     BFF00000h     247     184    74.49
C0000000h     249     59     23.69     C0200000h     236     102    43.22
C0300000h     252     6      2.38      C0E00000h     229     14     6.11
C0F00000h     241     54     22.41     C1000000h     216     47     21.76
C1100000h     247     159    64.37     C1200000h     232     232    100.00
C1300000h     244     177    72.54     C1400000h     238     238    100.00
C1500000h     253     126    49.80     C1600000h     246     33     13.41
C2100000h     262     255    97.33     C2200000h     249     249    100.00
C2300000h     255     255    100.00    C2400000h     222     175    78.83
C2500000h     255     135    52.94     C2800000h     261     261    100.00
C2900000h     252     43     17.06     C2C00000h     234     234    100.00
C2D00000h     242     37     15.29     C3000000h     258     258    100.00
C3100000h     224     30     13.39     C3400000h     241     241    100.00
C3500000h     216     29     13.43     C3800000h     243     243    100.00
C3900000h     242     29     11.98     C3C00000h     223     223    100.00
C3D00000h     253     34     13.44     C4000000h     268     268    100.00
C4100000h     219     31     14.16     C4400000h     240     240    100.00
C4500000h     238     35     14.71   

WHAT THIS MEANS:

Program generated one million random pointers, and tried to write to
the pointer.  Of these, over 98% caused GP faults: This is exactly
what we want in a protected operating system.  However, over 15,000
of the random memory writes succeeded.  The program was nice and
wrote a value identical to that already found at the address, so no
harm was done, but severe damage could easily have been done by
a malicious or even errant program.

For example, the area writeable by each Win32 process includes
the page directory, the GDT, the TSS (including the I/O permission
bitmap), the VMM service table, and the four-megabyte "high linear"
area of each virtual machine (the System VM and each DOS box).

Besides Windows 95, RANDRW can also be run under other environments,
such as Win32s and Phar Lap's TNT DOS extender.  Here are some
comparative result from one machine:

Environment                 # hits          # seconds
-----------                 ------          ---------

Win95; Console; 1 DOS box    5,010              103
Win95; Console; 3 DOS box    7,802              107
Win95; Console; 6 DOS box   10,723              121
Win95; not Console; 0 box    3,844              108
Win95; not Console; 3 box    8,103              152
Win32s; no DOS box           6,198              263 
Win32s; 3 DOS box            7,285              263
OS/2, 1 DOS box              1,394             [488] *
OS/2, 6 DOS boxes            1,397             [514] *
TNT under WfW; 3 DOS box       349              188
TNT under DOS                  114               56
Windows NT 3.5                  56             [373] **

* OS/2 figures supplied by Greg White (CIS 73657,277), on different
  hardware, so #seconds aren't comparable.
      
** NT 3.5 figures supplied by Bill Potvin (CIS 70540,120), on
   different hardware, so #seconds aren't comparable.

In Windows 95, the hit rate increases as the number of DOS boxes
increases.  This is because Win95 maps the high-linear four megabyte
address space of each VM into each Win32 process.

The hit rate for Win32s seems roughly equivalent to that of Win95.

RANDRW takes considerably longer to run under Win32s than under
Win95, even though Win32s (unlike Win95) does not provide preemptive
multitasking for Win32 apps, so no other Windows program can run
while RANDRW is executing (unless if RANDRW is running under TNT
under Win3, in which case it is preemptively multitasked).  Thus,
the system is stalled until RANDRW completes.  This is a good example
of how Win95 provides better "tasking robustness" than Win32s, even
if its "memory robustness" is no better.

The TNT hit rate is substantially lower than that of Win95 or Win32s.

Note too that, when running in DOS outside Windows, TNT executes RANDRW
is considerably less time.  This even though with a lower hit rate, there
is a higher GP fault rate.
*/

#include <stdlib.h>
#include <stdio.h>
#include <limits.h>
#include <time.h>
#define WIN32_LEAN_AND_MEAN
#include "windows.h"
#include "excpt.h"

FILE *f;

#define NOT_CONSOLE

#ifdef NOT_CONSOLE
#define OPEN_LOG()       fopen("randrw.log", "w")
#define PUTS(s)          MessageBox(0, (s), "RANDRW", MB_OK)
void fail(const char *s) { PUTS(s); exit(1); }
#else
#define OPEN_LOG()       stdout
#define PUTS(s)          fprintf(stderr, "%s\n", (s))
void fail(const char *s) { PUTS(s); exit(1); }
#endif

/**********************************************************************/

/* The rand() function in 32-bit Microsoft C, as in most (?) 32-bit
   C run-time libraries, has RAND_MAX = 0x7FFF (STDLIB.H).  This is
   too small.  We need even coverage of the entire range 0-UINT_MAX.
   This simple random-number generator (from George Marsaglia,
   "The Mathematics of Random Number Generators," Proceedings of
   Symposia in Applied Mathematics v. 46 (1992)) has a period of
   4.3 x 10^9 -- just what we need. */
unsigned seed;
unsigned my_srand(unsigned x) { seed = x; }
unsigned my_rand(void) { return (seed = (69069.0 * (seed + 1.0))); }

/**********************************************************************/

unsigned attempts[4096] = {0};
unsigned meg[4096] = {0};

#define SET_MEG(p)      (((unsigned) (p)) >> 20)
#define GET_MEG(x)      ((x) << 20)

void print_meg()
{
    int i;
    fprintf(f, "Megabyte     Attempts  Hits    % Hits\n");
    fprintf(f, "---------    --------  ----    ------\n");
    for (i=0; i<4096; i++)
    if (meg[i] != 0)
    {
        fprintf(f, "%08lXh\t%u\t%u\t%0.2f\n", 
            GET_MEG(i), 
            attempts[i], meg[i],
            (double) meg[i] * 100.0 / (double) attempts[i]);
    }
}

/**********************************************************************/

#ifdef NOT_CONSOLE
#define argc __argc
#define argv __argv
extern int __argc;
extern char **__argv;

int PASCAL WinMain(HANDLE hInstance, HANDLE hPrevInstance,
    LPSTR lpszCmdLine, int nCmdShow)
#else
main(int argc, char *argv[])
#endif
{
    int iter = (argc < 2) ? 1000000 : atoi(argv[1]);
    int i, x, num_hits = 0, did_hit;
    unsigned *p;
    unsigned runtime;
    time_t t1, t2;

    f = OPEN_LOG();
    time(&t1);
    my_srand(t1);
    for (i=0; i<iter; i++)
    {
        p = (unsigned *) my_rand();
        attempts[SET_MEG(p)]++;
    
        // use structured exception handling to catch illegal writes
        // and figure out which ones the OS lets us get away with
        _try { did_hit = 1; x = *p; *p = x; }
        _except  (EXCEPTION_EXECUTE_HANDLER) { did_hit = 0; }

        if (did_hit) 
        { 
            num_hits++; 
            meg[SET_MEG(p)]++; 
        }
    }
    time(&t2);
    runtime = t2 - t1;
    
    fprintf(f, "\n");
    print_meg();
    fprintf(f, "\n");
    
    fprintf(f, "%u attempts\n", i);
    fprintf(f, "%u hits\n", num_hits);
    fprintf(f, "%0.2f%% hit rate\n", (100.0 * num_hits) / i);
    fprintf(f, "%u seconds\n", runtime);
    
    if (runtime != 0)
    {
        fprintf(f, "%u attempts/second\n", i/runtime);
        fprintf(f, "%u hits/second\n", num_hits/runtime);
    }
    
    fflush(f);
    fclose(f);
    PUTS("Done!");
    return 0;
}

