/*
 *  $Id: libnet_build_arp.c,v 1.1.1.1 2000/05/25 00:28:49 route Exp $
 *
 *  libnet
 *  libnet_build_arp.c - ARP packet assembler
 *
 *  Copyright (c) 1998 - 2001 Mike D. Schiffman <mike@infonexus.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#if (HAVE_CONFIG_H)
#include "../include/config.h"
#endif
#include "../include/libnet.h"


#if (WIN32 || __linux__)
 static u_char enet_dst[6] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff}; /* default arp value is broadcast */
#endif

int
libnet_build_arp(u_short hrd, u_short pro, u_char hln, u_char pln, u_short op,
            u_char *sha, u_char *spa, u_char *tha, u_char *tpa,
            const u_char *payload, int payload_s, u_char *buf)
{
    struct libnet_arp_hdr arp_hdr;

    if (!buf)
    {
        return (-1);
    }

    arp_hdr.ar_hrd = htons(hrd);       /* hardware address type */
    arp_hdr.ar_pro = htons(pro);       /* protocol address type */
    arp_hdr.ar_hln = hln;              /* hardware address length */
    arp_hdr.ar_pln = pln;              /* protocol address length */
    arp_hdr.ar_op  = htons(op);        /* opcode command */
    memcpy(arp_hdr.ar_sha, sha, hln);  /* sender hardware address */
    memcpy(arp_hdr.ar_spa, spa, pln);  /* sender protocol (IP) address */
    memcpy(arp_hdr.ar_tha, tha, hln);  /* target hardware address */
    memcpy(arp_hdr.ar_tpa, tpa, pln);  /* target protocol (IP) address */

    if (payload && payload_s) 
    {
        /*
         *  Unchecked runtime error for buf + LIBNET_ARP_H payload to be
         *  greater than the allocated heap memory.
         */
        memcpy(buf + LIBNET_ARP_H, payload, payload_s);
    }
    memcpy(buf, &arp_hdr, sizeof(arp_hdr));
    return (1);
}

#if (WIN32 || __linux__)
/* Below are the Arp_Table data manipulation functions */

static void Arp_Table_free(Arp_Table pTable)
{
 Arp_Table p,q;
 for (p=pTable;p!=NULL;p=q)
  {
   q=(p->next);
   free(p);
  }
}

static Arp_Table Arp_Table_alloc(void)
{
 return (Arp_Table)calloc(1,sizeof(struct arp_table));
}

static int cmp_ArpTable_ipadress (Arp_Table *a,Arp_Table *b)
{
 return ( ((*a)->ipadress==(*b)->ipadress )?0:
          (((*a)->ipadress>(*b)->ipadress)?1:-1)
        );
}

static int cmp_ipadress_ArpTable (u_long *keyval,Arp_Table *datanum)
{
 return ( ((*keyval)==(*datanum)->ipadress )?0:
          (((*keyval)>(*datanum)->ipadress)?1:-1)
        );
}

static void Arp_Table_add(u_long ip, u_char *mac, struct libnet_link_int *l)
{
 int i;
 Arp_Table q;

 /* move the data pointer to last entry (which is NULL) */
 for (q=l->g_Info->Arp_Database;q!=NULL;q=q->next);
 /* increase Arp_Index and dataspace for the new entry */
 l->g_Info->nArp_Index++;
 l->g_Info->Arp_Index=(Arp_Table *)realloc(l->g_Info->Arp_Index,l->g_Info->nArp_Index*sizeof(Arp_Table));
 l->g_Info->Arp_Index[l->g_Info->nArp_Index-1]=q=Arp_Table_alloc();

 /* fill in the data */
 q->ipadress=ip;
 for (i=0;i<6;i++) q->mac[i]=*(mac+i);
 q->next=NULL; 

 /* Sort the Arp_Index */
 qsort( (void *)l->g_Info->Arp_Index,
        l->g_Info->nArp_Index,
	  sizeof(Arp_Table),
	  (int (*)(const void *,const void *))cmp_ArpTable_ipadress
      );
}

/* Below are ARP requesters */

#if (WIN32)
 static void gettimeofday(struct timeval *tp, void *foo)
  {
   u_long elapsed_ms;
   
   elapsed_ms=GetTickCount();
   
   tp->tv_sec=elapsed_ms/1000;
   tp->tv_usec=(elapsed_ms%1000)*1000;
  }
#endif

#define Libnet_Timeval_Msec_Substract(a,b) ((((a).tv_sec - (b).tv_sec) * 1000) + ((a).tv_usec - (b).tv_usec) / 1000) /* Timeval subtract in milliseconds */


/* this function should NOT be called outside the library */
static void libnet_do_arp(u_long ip, struct libnet_link_int *l)
{
 #define IP_UCHAR_COMP(x, y) \
    		      (x[0] == y[0] && x[1] == y[1] && x[2] == y[2] && x[3] == y[3])
 int n;
 u_char *buf, errbuf[256], *packet, *ip_p;
 struct libnet_ethernet_hdr *p;
 struct libnet_arp_hdr *a;
 pcap_t *pd;
 struct pcap_pkthdr pc_hdr;
 struct timeval time_1, time_2;

    /*
     *  Initialize a packet.
     */
    if (libnet_init_packet(LIBNET_ARP_H + LIBNET_ETH_H, &buf) == -1)
    {
        perror("libnet_init_packet memory:");
        return;
    }

    /*
     *  Open the pcap device.
     */
    pd = pcap_open_live(l->device, LIBNET_ARP_H + LIBNET_ETH_H, 1, 50, errbuf);
    if (pd == NULL)
    {
        fprintf(stderr, "pcap_open_live: %s\n", errbuf);
        return;
    }

    /*
     *  Ethernet header
     */
    libnet_build_ethernet(
            enet_dst,               /* broadcast ethernet address */
            l->g_Info->MAC,    	/* source ethernet address */
            ETHERTYPE_ARP,          /* this frame holds an ARP packet */
            NULL,                   /* payload */
            0,                      /* payload size */
            buf);                   /* packet header memory */

    /*
     *  ARP header
     */
    libnet_build_arp(
            ARPHRD_ETHER,           /* hardware address type */
            ETHERTYPE_IP,           /* protocol address type */
            ETHER_ADDR_LEN,         /* hardware address length */
            4,                      /* protocol address length */
            ARPOP_REQUEST,          /* packet type - ARP request */
            l->g_Info->MAC,    	/* source (local) ethernet address */
            (u_char *)&(l->g_Info->LocalIp),    /* source (local) IP address */
            enet_dst,               /* target's ethernet address (broadcast) */
            (u_char *)&ip,          /* target's IP address */
            NULL,                   /* payload */
            0,                      /* payload size */
            buf + LIBNET_ETH_H);    /* packet header memory */

    n = libnet_write_link_layer(	l, 
						l->device, 
						buf, 
						LIBNET_ARP_H + LIBNET_ETH_H
					  );
    if (n == -1)
    {
        fprintf(stderr, "libnet_write_link_layer choked\n");
        return;
    }

    gettimeofday(&time_1,NULL);

    for (;(packet = ((u_char *)pcap_next(pd, &pc_hdr)));)
    {
        gettimeofday(&time_2,NULL);
        if ( (Libnet_Timeval_Msec_Substract(time_2,time_1)) > 500 ) break;
	
        p = (struct libnet_ethernet_hdr *)(packet);
        if (ntohs(p->ether_type) == ETHERTYPE_ARP)
        {
           a = (struct libnet_arp_hdr *)(packet + LIBNET_ETH_H);
           if (ntohs(a->ar_op) == ARPOP_REPLY)
            {
               ip_p = (u_char *)&(ip); 
               if (IP_UCHAR_COMP(a->ar_spa, ip_p))
                {
		     Arp_Table_add(ip,&(a->ar_sha[0]),l);
		     return;
                }
            }
        }
    }


    libnet_destroy_packet(&buf);

    return;
}

u_char *
libnet_get_remote_mac(u_long ip, struct libnet_link_int *l)
{
 Arp_Table *pArpTable;
 int i;

 /* Has a network socket been opened ? */
 if (!l)
  {
   fprintf(stderr, "libnet_get_remote_mac : link interface is not opened\n");
   return(NULL);
  }

 /* is this the first time use of the Arp requester ? */
 if (!l->g_Info->Arp_Database)
  {
   /* init the Arp Table with our own IP/MAC */
   l->g_Info->nArp_Index=1; /* one entry */
   l->g_Info->Arp_Index=(Arp_Table *)calloc(1,l->g_Info->nArp_Index*sizeof(Arp_Table)); /* allocate Arp_Index size */
   l->g_Info->Arp_Index[0]=l->g_Info->Arp_Database=Arp_Table_alloc(); /* allocate data size, Arp_Index points first data entry */
   l->g_Info->Arp_Database->ipadress=l->g_Info->LocalIp; /* fill ip for arp resolv */
   for (i=0;i<6;i++) l->g_Info->Arp_Database->mac[i]=l->g_Info->MAC[i]; /* fill mac for arp resolv */
   l->g_Info->Arp_Database->next=NULL; /* this first entry is also the last one for now */   
  }

 /* check cache for IP */
 if ((pArpTable=bsearch((void *)(&(ip)),
                       (void *)l->g_Info->Arp_Index,
                       l->g_Info->nArp_Index,
                       sizeof(Arp_Table),
                       (int (*)(const void *,const void *))cmp_ipadress_ArpTable)))
  return ((*pArpTable)->mac);

 /* search IP on LAN */
 libnet_do_arp(ip,l);
 if ((pArpTable=bsearch((void *)(&(ip)),
                       (void *)l->g_Info->Arp_Index,
                       l->g_Info->nArp_Index,
                       sizeof(Arp_Table),
                       (int (*)(const void *,const void *))cmp_ipadress_ArpTable)))
  return ((*pArpTable)->mac);

 /* check cache for router */
 if ((pArpTable=bsearch((void *)(&(l->g_Info->DefaultGateway)),
                       (void *)l->g_Info->Arp_Index,
                       l->g_Info->nArp_Index,
                       sizeof(Arp_Table),
                       (int (*)(const void *,const void *))cmp_ipadress_ArpTable)))
  {
   /* fill in the matching IP */
   Arp_Table_add(ip,(*pArpTable)->mac,l);
   return ((*pArpTable)->mac);
  }

 /* search Routeur on LAN */
 libnet_do_arp(l->g_Info->DefaultGateway,l);
 if ((pArpTable=bsearch((void *)(&(l->g_Info->DefaultGateway)),
                       (void *)l->g_Info->Arp_Index,
                       l->g_Info->nArp_Index,
                       sizeof(Arp_Table),
                       (int (*)(const void *,const void *))cmp_ipadress_ArpTable)))
  {
   /* fill in the matching IP */
   Arp_Table_add(ip,(*pArpTable)->mac,l);
   return ((*pArpTable)->mac);
  }

 /* Default */
 /* fill in the matching IP */
 Arp_Table_add(ip,&(enet_dst[0]),l);
 return (&(enet_dst[0]));
}

void
libnet_reset_arp_database(struct libnet_link_int *l)
{
 if (l->g_Info->Arp_Database) 
  {
   Arp_Table_free(l->g_Info->Arp_Database);
   free(l->g_Info->Arp_Index);
   l->g_Info->Arp_Database=NULL;
   l->g_Info->Arp_Index=NULL;
  }
}
#endif

/* EOF */
