{$M $400,0,0 } { wen don't need heap! }

{
  A sample TSR interface driver for JVFAX 7.0. You can modify it for
  use with your own interface. This is free software.
  You should use BP 7.0 to compile it.

  JVFAX recognizes the presence of a TSR interface driver by scanning
  the interrupt vector table. A TSR driver is identified by the presence
  of the character sequence 'DK8JV-JVFAX' at @<interrupt vector^>+2.
  By this, you can use any free entry in the interrupt vector
  table for the communication with JVFAX.

  If JVFAX has detected the presence of a TSR interface driver it will
  ignore all its internal interface configuration settings and will
  only use the TSR driver for the communication with the interface.

  Since I wrote this TSR during my summer holiday where I only had a
  small laptop computer with no assembler installed, I simply wrote it
  in (Borland) Pascal. Of course, you can save precious amounts of memory
  space by writing the entire TSR in assembler code. On the other hand,
  this TSR can be loaded high, so that a lack of memory should be
  no problem.

  The terms 'synchroneous', or 'asynchroneous' which are used in this
  program need some explanation:

   Asynchroneous means that all the timing is done by JVFAX itself.
   JVFAX reads from or writes to the interface in fixed time intervals
   that are equal to the pixel rate. JVFAX calculates the pixel rate
   as follows (for FAX): <pixel rate> :=  <index of cooperation>* Pi *
   <drum speed in turns/s>. For example, a weather chart with an IOC of 572
   and a drum speed of 120 LPM is received with a data rate of 3620 pixels/s.
   See English.doc for further explanations of <data rate>. JVFAX calls
   the get_pixel or the put_pixel routine with a frequency equal to this
   data rate. At each call to these routines, the interface has nothing else
   to do but to put the representation of the actual pixel intensity in the
   AL register (get_pixel) or to use the information in the AL register to
   generate an equivalent audio frequncy.

   In synchroneous mode however, the interface has to do all the timing
   instead. Communication with JVFAX will be done using some kind of
   handshake via the AH register. The whole communication works in
   a highly asynchroneously manner:
   18 times per second, JVFAX will ask for data available from the inter-
   face (get_pixel) or whether new data can be output to the interface. JVFAX
   tries to load or output as many data as possible in one burst. (it will
   continue in-/output as long as the AH register returns a 1)
   As you can see from the above, either your interface or your TSR must
   provide some memory to store several hundreds of pixels since data
   transfer takes only place 18 times per second.
   The reason for this complicated procdedure is that is should enable JVFAX
   to run in a multitasking environment such as MS-Windows or OS/2.
   Another benefit of the synchroneous mode is that for polar orbiting
   satellites the data rate can be derived from the 2400 Hz audio subcarrier.
   For JVFAX, the data rate must be 4800 pixels/s for the automatic phasing
   routine to work, so it would be a good idea to derive the 4800 Hz by a
   PLL circuit from the 2400 Hz subcarrier.
   (Only applicable to NOOA reception) If, for these satellites the data
   rate is locked to the audio carrier, the so called 'Banana' effect can
   be eliminated.

   In synchroneous mode, JVFAX hooks to the interrupt vector 1Ch. This
   interrupt is normally called by the timer interrupt service routine.
   If your TSR will modify this routine you must take care that the
   interrupt 1Ch is called by your modified ISR at the default rate.
   (approx. 18 times/s)

   All in all, constructing an interface for asynchromeous mode is much
   easier than doing the same for the synchroneous mode. I didn't have the
   opportunity yet to test all the routines for synchroneous operation.
   Please contact me if you should find any errors in the routines or if
   you need further informaton.

   July 1994, DK8JV

}

program jvtsr;


uses dos;

{------------------------------------------------------------------------}

{ general variables for the TSR, need not be changed normally }

type
  str15 = string[15];

const
  tsrname: str15='EASYFAX/SPKR';{ can be used to identify the TSR by name }
                                { will be displayed the in JVFAX main menu }
                                { should not exceed 15 characters }

  oldintno: byte = $FA;         { default interrupt to hook in the TSR }


var
  tsralreadyinstalled: boolean; { true if a TSR driver is already installed }
  thistsrinstalled:    boolean; { true if this driver is already installed }

  oldintvec:           pointer; { saves original vector for deinstallation }
  installedtsrname:    str15;   { if a TSR is already installed, these vars }
  installedtsrno:      byte;    { will reflect its name and vector number }

  i,j:                 integer; { soms scratch variables }


{************************************************************************}


{ add variables here that you need in your interfacing routines }
{ for our sample TSR, we have only a few }

const
  portaddr:   word = $3f8;    { default base addr COM1:, can be changed }
                              { by a command line parameter }
  mdmconmask: byte =    0;    { saves state of the 'switch' setting, will }
                              { control state of the DTR line here }

VAR
  tontab:     array[0..127] of word; { index table for audio generation by }
                                     { the speaker }
  mdmstat:    word;
  mdmcon:     word;                  { used to store some addresses }
                                     { relative to the port base addresses }

{------------------------------------------------------------------------}

{ Write here whatever auxillary routines you need. For our sample TSR,
  we need three routines: One to set up an index table to derive divider
  factors for the speaker timer from the intensity values passed to the TSR's
  put_pixel routine. The second procedure is to set up the serial port at
  <portaddr> to 57600 baud for communication with the Easyfax.
  The last routine is a delay routine that we need when we send control
  codes to the Easyfax to give it a bit of response time between subsequent
  codes.
}

procedure inittontab; far; { initializes the index table with divider }
                           { factors for the speaker timer }
var
  i: byte;
const
  os = 1193181;

begin
  for i := 0 to 63 do
    tontab[i] := round(os/(1900-400+(2*400/63)*i));
  tontab[125] := round(os/1100);
  tontab[126] := round(os/1200);
  tontab[127] := round(os/1300);
end;


procedure initserialport;
var
  i: word;

begin
  PORT[portaddr+3] := $80;        { set port to 57600 Baud (for command }
  I := ROUND(115200.0/57600);     { transmission to the Easyfax }
  PORT[portaddr] := LO(I);
  PORT[portaddr+1] := HI(I);
  PORT[portaddr+3] := 7;
  PORT[portaddr+1] := 0;
  PORT[portaddr+4] := 8;

  mdmstat := portaddr+6;          { save address of modem status register }
  mdmcon := portaddr+4;           { and of modem control register }

end;


procedure delay(d:LONGINT);       { a simple delay routine }
VAR
  H,M,S,T: WORD;
  l1 :longint;

BEGIN
  GETTIME(H,M,S,T);
  l1 :=longint(360000)*(H mod 24)+longint(6000)*M+longint(100)*S+T;
  repeat gettime(h,m,s,t) until
    abs((longint(360000)*(H mod 24)+longint(6000)*M+longint(100)*S+T)-l1)>d/10;
END;


{************************************************************************}

{ the following routines must be modified according to the needs of your
  interface }

procedure get_pixel; far; assembler;

{ must be compiled far

  Input parameters:   none

  Output parameters:  AL: pixel value

                      for synchroneous operation:

                      AH: 1 if a valid pixel date is available,
                          0 if no new data available

  must restore all registers other than AX on exit


  For FAX-Operation, the return value may have up to 8 significant bits.
  If the number of significant bits is less, these bits must be left-
  aligned, and the remaining low-bits should be set to zero.
  For SSTV operation, there are always 6 bits evaluated,
  left aligned. The two least significant bits reflect the state of the
  SSTV sync pulse, according to the following table:


          Bit 1 0     identifies:
          ========================
              H L     1050-1100 Hz
              L L     1150-1250 Hz
              L H     1250-1350 Hz

  If your interface cannot differentiate between the three sync audio
  frequency intervals it should always set both bits to zero if a sync
  pulse is present.


}

asm
  PUSH     BX
  PUSH     DX                {data transfer from the Easyfax is done via}
  PUSH     CX                {the four modem status lines. to transfer}
  push     ds                {8 data bits, data is read using a multiplex}
  mov      bx, seg @data     {technique}
  mov      ds,bx

  MOV      DX,mdmstat        {read high-Nibble}
  IN       AL,DX
  MOV      bl,al             {save in BL}
  XOR      AL,AL             {switch multiplexer to lower nibble by setting}
  or       al,mdmconmask     {RTS to low; care for proper 'switch setting'}
  MOV      DX,mdmcon         {(=DTR line)}
  OUT      DX,AL
  MOV      DX,mdmstat
  IN       AL,DX             {the multiplexer needs some time for switchig}
  IN       AL,DX             {so we waste some time here by reading the}
  IN       AL,DX             {port several times}
  IN       AL,DX
  IN       AL,DX
  IN       AL,DX
  IN       AL,DX
  IN       AL,DX
  IN       AL,DX
  IN       AL,DX             {this should be enough}

  MOV      CL,4
  SHR      AL,CL             {shift into lower nibble}
  and      BL,$F0
  OR       BL,AL             {and combine with high nibble (saved in BL)}

  MOV      AL,2              {reset multiplexer (set RTS to high level)}
  or       al,mdmconmask     {care for proper 'switch' setting}
  MOV      DX,mdmcon
  OUT      DX,AL

  mov      AL,BL             {return value in AL}
  mov      AH,1              {not necessary for asynchroneous mode}

  pop      DS
  pop      CX
  pop      DX
  pop      BX
end;


{------------------------------------------------------------------------}

procedure put_pixel; far; assembler;

{ must be compiled far

  Input parameters:   AL: pixel value; 6 bit.  0 = black (1500 Hz)
                                              63 = white (2300 Hz)

                                           1..62 = all intermediate freqs.

                          special AL values:

                            AL = 125:  transmit 1100 Hz;
                            AL = 126:  transmit 1200 Hz;
                            AL = 127:  transmit 1300 Hz;

  Output parameters:

                      for synchroneous operation:

                      AH: 1 if date could be sent,
                          0 if wasn't ready to accept a new pixel

  must restore all registers other than AX on exit

}

asm
  push ds
  push bx
  push ax
  mov bx, seg @data
  mov ds,bx

  xor ah,ah                 {this routine is very simple: use the AL value}
  shl ax,1                  {multiplied by 2 as an index to the divisor}
  mov bx, offset tontab     {table. Then, load the divisor and output it}
  add bx,ax                 {to the speaker timer port}

  mov  al,[bx]
  out  $42,al
  inc  bx
  mov  al,[bx]
  out  $42,al
  pop ax
  pop bx
  pop ds
  mov ah,1                  {not necessary for asynchroneous operation}
end;

{------------------------------------------------------------------------}

procedure initfunc; assembler; {function 0}

{ must perform basic initialization routines for the interface.
  Will be called once on JVFAX startup

  Entry parameters:  none

  Exit parameters:   AL:    Bit 0 set if RX-interface is synchroneous
                     AL:    Bit 1 set if TX-interface is synchroneous
                     ES:DI: Pointer to a Pascal string containing
                            the interface's / TSR's name

                     AH:    number of significant bits that the inter-
                            face can deliver in FAX-RX mode

}

asm

 { do any general driver/ interface initialization here ... }
 { for this example, we need not do any initialization here }

 mov al, 0           { RX and TX are both asynchroneous }
                     { change if necessary }
 mov bx, seg @data   { get pointer to 'tsrname' }
 mov es, bx
 mov di, offset tsrname
 mov ah,8            { the Easyfax can deliver up to 8 bit RX data }
end;

{------------------------------------------------------------------------}

procedure exitfunc; assembler; {function 9}

{ to perform basic deinitialization routines.
  Will be called once on JVFAX program end

  Entry parameters:  none

  Exit parameters:   none

}

asm

 { do any general driver/ interface initialization here ... }
 { for this example, we need not do anything }

end;

{------------------------------------------------------------------------}

procedure removetsr; assembler; {function $FF}

{ routine to free all memory used by the TSR. Is needed to remove the
  TSR by calling it with the command line parameter '-u'.
  You can change this routine if necessary to free other ressources that
  your driver eventually might use }

asm
  mov AH,$49             {free environment}
  mov bx,prefixseg
  mov es,bx
  mov di,$2c
  mov bx,ES:[di]
  mov es,bx
  int $21

  mov AH,$49             {free program memory}
  mov bx,prefixseg
  mov es,bx
  int $21

  mov al,oldintno        {restore original interrupt vector}
  push ds
  lds dx,oldintvec
  mov ah,$25
  int $21
  pop ds
end;

{------------------------------------------------------------------------}

procedure initforrx; assembler; {function 1}

{ should initialize the interface for receive operation
  this function is called everytime that JVFAX will switch to FAX or
  SSTV recption, or if a mode change in reception takes place

  Entry parameters:  BX:    Integer part of desired data rate in
                            pixels/s, can be ignored for asynchroneous
                            operation.

                     DX:    fractional part of desired data rate times
                            65,536. For the FAX modes, there are
                            normally only data rates that can be
                            expressed in integer numbers. For SSTV how-
                            ever, there may be non-integer values that
                            will make it necessary to evaluate the DX
                            register, too.

                            The register pair BXDS forms a long integer
                            variable that contains
                            frac(<actual data_rate> * 65,536)


                     CX:    Number of pixels/scan_line. Because for the
                            SSTV-Modes, there are quite often data rates
                            which cannot be expressed in ordinal numbers,
                            this can serve as an additional information.

                     AL:    JVFAX mode number. Numbers >= 20 are SSTV-
                            modes.

  Exit parameters:   BX:    actual data rate, or 0 to override
                            synchroneous operation by asynchroneous
                            mode.

                     ES:DI: Pointer the get_pixel routine which must
                            perform data input

Caution: If you want your interface to communicate with JVFAX in
         synchroneous mode, the interface's data rate >MUST< be greater or
         equal to the desired data passed by JVFAX in the BX register.
         If you return a value in BX that is less than the original BX
         value, JVFAX will use asynchroneous mode, instead. Please keep
         in mind, that if using synchroneous mode for automatic
         NOAA/METEOR reception, the data rate must be 4800 pixels/s.
         For the digital header evaluation of Meteosat transmissions, the
         data rate must be set to 3360 pixels/s.

}

asm
  call initserialport

  mov bx, seg @code
  mov es,bx
  mov di, offset cs:get_pixel  {return address of get_pixel routine}
end;                           {since we're in asynch mode, value of BX}
                               {doesn't matter}

{------------------------------------------------------------------------}

procedure initfortx; assembler; {function 2}

{ should initialize the interface for transmit operation
  will be called by JVFAX everytime that a transmission is started

  Entry parameters:  BX:    Integer part of desired data rate in
                            pixels/s, can be ignored for asynchroneous
                            operation.

                     DX:    fractional part of desired data rate times
                            65,536. For the FAX modes, there are
                            normally only data rates that can be
                            expressed in integer numbers. For SSTV how-
                            ever, there may be non-integer values that
                            will make it necessary to evaluate the DX
                            register, too.

                            The register pair BXDS forms a long integer
                            variable that contains
                            frac(<actual data_rate> * 65,536)


                     CX:    Number of pixels/scan_line. Because for the
                            SSTV-Modes, there are quite often data rates
                            which cannot be expressed in ordinal numbers,
                            this can serve as an additional information.

                     AL:    JVFAX mode number. Numbers >= 20 are SSTV-
                            modes.


  Exit parameters:   BX:    actual data rate, or 0 for asynchroneous
                            operation

                     ES:DI: Pointer the put_pixel routine which must
                            perform data output

                     AL:    set to 1 if your TSR uses the PC's internal
                            speaker driver


Caution: If you want your interface to communicate with JVFAX in
         synchroneous mode, the interface's data rate >MUST< be greater or
         equal to the desired data passed by JVFAX in the BX register.
         If you return a value in BX that is less than the original BX
         value, JVFAX will use asynchroneous mode, instead.

}

asm

  { we use the PC speaker as an example, so do some initialization }


  call inittontab


  mov  al,$B6                   { set correct timer mode }
  out  $43,al
  in   al,$61                   { enable speaker output }
  or   al,3
  out  $61,al

  mov bx, seg @code
  mov es,bx
  mov di, offset cs:put_pixel
  mov bx,0
  mov al,1                      { we want to output date via the speaker }
end;                            { Set AL to 1; else JVFAX will use the }
                                { speaker timer itself to retrieve lost
                                { pixels caused by frequent disk access }

{------------------------------------------------------------------------}

procedure initdev; {function 3}

{ should set the interface to the correct deviation. Will be called only
  in RX mode. There's no deviation switching in transmit mode, since
  standard deviation for amateur radio use is 400 Hz.


  Entry parameters:  CX:    desired deviation in Hz. Special cases:
                              CX =  0: AM-1 mode
                              CX = -1: AM-2 mode
                              CX = -2: SSTV-Mode

                            currently, JVFAX supports switching to the
                            following deviations:

                              150 Hz
                              200 Hz
                              300 Hz
                              350 Hz
                              400 Hz
                              500 Hz
                              600 Hz
                              700 Hz
                              800 Hz

                            if your interface cannot be set to one of
                            these deviations it should be set to the
                            nearest possible value.

                     optionally evaluate the BX-Register:

                     BH:      1 to set a TX-hardware filter on, 0 for off
                     BL:      1 to set a RX-hardware filter on, o for off


  Exit parameters:   none

  There can be selected two AM modes from within JVFAX. Since the geo-
  stationary satellites use another modulation index than the polar
  orbiting satellites, you can use the two different calls to switch
  between two contrast settings in your interface.

}

var
  dev:  integer;
  ctrl: byte;
  rxf:  byte;
  txf:  byte;

begin
  asm
    mov dev,cx          {save deviation in <dev>}
    mov al,80           {80 is Easyfax's RX hardware filter off command}
    cmp bl,1
    jne @w1
    inc al              {81 is RX filter on command}
@w1:mov rxf,al          {save rx filter state in <rxf>}
    mov al,82           {82 is Easyfax's TX hardware filter off command}
    cmp bh,1
    jne @w2
    inc al              {81 is TX filter on command}
@w2:mov txf,al          {save TX filter state in <txf>}

  end;

  case dev of
     0:  ctrl := $49;   {AM 1} {get Easyfax deviation control code}
    -1:  ctrl := $4a;   {AM 2} {dependant on the content of <dev>}
    -2:  ctrl := $4b;   {SSTV}

   150:  ctrl := $41;
   200:  ctrl := $42;
   300:  ctrl := $43;
   350:  ctrl := $44;
   400:  ctrl := $45;
   500:  ctrl := $46;
   600:  ctrl := $47;
   700:  ctrl := $47;
   800:  ctrl := $48;
   ELSE  ctrl := $45;
  end;

  PORT[PORTADDR] := rxf;  {output rx filter command}
  delay(200);             {wait a bit}
  PORT[PORTADDR] := txf;  {output tx filter command}
  delay(200);             {wait a bit}
  PORT[PORTADDR] := ctrl; {output deviation command}
  delay(100);
end;

{------------------------------------------------------------------------}

procedure settoidle; assembler; {function 4}


{ should set the interface idle mode
  is called everytime that JVFAX ends receive or transmit mode.

  Entry parameters:  none

  Exit parameters:   none
}

asm
  in al,$61       { shut off speaker by disabling speaker output }
  and al,$FE      { nothing to do concerning the Easyfax }
  out $61,al
end;

{------------------------------------------------------------------------}

procedure setswitch; assembler; {function 5}

{ this procedure is called after the initforrx routine (function 1) and at
  each change of the 'switch' setting in JVFAX.
  Can be used to establish a simple 2-channel receiver switching, etc.

  Entry parameters:  AL:    switch state, 1: on, 0: off

  Exit parameters:   none
}

asm
  mov mdmconmask, AL       { here, we save the switch state for later }
end;                       { use in the get_pixel routine. Will be used }
                           { there to control the DTR line }

{************************************************************************}

{ what follows here need not be modified normally...}

procedure mainproc; assembler;

{ Main entry point for calls to the TSR. The address of this routine will
  be hooked in the interrupt vector table by the TSR's installation routine.
  It serves as a kind of function dispatcher.
  The different functions are selected by the state of the AH register when
  this interrupt routine is called.
  The get_pixel and put_pixel routines however, are called directly
  by far calls from JVFAX to minimize code overhead.


  The functions are:

     AH =    0:   initialize interface (called once at JVFAX start)
     AH =    1:   set interface to RX mode
     AH =    2:   set interface to TX mode
     AH =    3:   set actual deviation (for receive mode only)
     AH =    4:   set interface to idle mode
     AH =    5:   set switch on/off  (optionally)
     AH =    9:   deinitialize interface (called once at JVFAX end)

     AH =  $FF:   function to uninstall the TSR (not called by JVFAX)

}

asm
  jmp @@start           { this structure must not be changed because it }
  db 'DK8JV-JVFAX'      { serves JVFAX to identify the presence of the }
                        { TSR interface driver. Normally there's no need }
@@start:                { to change this routine. DK8JV-JVFAX is the }
  push ds               { 'magic' keyword that JVFAX will search for }
  mov bx, seg @data
  mov ds, bx

@@func0:                { function dispatcher. Not quite elegant, but it }
  cmp ah,0              { works.... }
  jne @@func1
  call initfunc
  jmp @@ende;

@@func1:
  cmp ah,1
  jne @@func2
  call initforrx
  jmp @@ende;

@@func2:
  cmp ah,2
  jne @@func3
  call initfortx
  jmp @@ende;

@@func3:
  cmp ah,3
  jne @@func4
  call initdev
  jmp @@ende;

@@func4:
  cmp ah,4
  jne @@func5
  call settoidle
  jmp @@ende;

@@func5:
  cmp ah,5
  jne @@func9
  call setswitch
  jmp @@ende;

@@func9:
  cmp ah,9
  jne @@funcff
  call exitfunc
  jmp @@ende;

@@funcff:
  cmp ah,$FF
  jne @@ende
  call removetsr
  jmp @@ende;

@@ende:
  pop ds
  iret
end;


{------------------------------------------------------------------------}

function Hexstr(W:Word): string;

{ auxillary routine to convert numbers into hex format}

const
  Digits : array[0..$F] of Char = '0123456789ABCDEF';

begin
  if w>255 then
  begin
    Hexstr[0] := #4;
    Hexstr[1] := Digits[hi(W) shr 4];
    Hexstr[2] := Digits[hi(W) and $F];
    Hexstr[3] := Digits[lo(W) shr 4];
    Hexstr[4] := Digits[lo(W) and $F];
  end else
  begin
    Hexstr[0] := #2;
    Hexstr[1] := Digits[lo(W) shr 4];
    Hexstr[2] := Digits[lo(W) and $F];
  end;
end;

{------------------------------------------------------------------------}

function scanparameters(s:string): string;

{ auxillary function to scan the command line for parameters}

var
  i,j: byte;
  t: string;
  ok : boolean;

begin
  for j := 1 to length(s) do s[j] := upcase(s[j]);
  if paramcount=0 then
  begin
    scanparameters := '';
    exit;
  end;
  i := 0;
  ok := false;
  while (i<=paramcount) and not ok do
  begin
    t := paramstr(i);
    for j := 1 to length(t) do t[j] := upcase(t[j]);
    if copy(t,1,length(s))=s then ok := true else inc(i);
  end;
  if ok then scanparameters := t else scanparameters :='';
end;

{------------------------------------------------------------------------}

procedure checkiftsrisinstalled;

{ checks if a JVFAX TSR is already istalled }

type ident = record             { structure to scan for the 'magic' word }
                jump: word;
                id: packed array[1..11] of char;
             end;
var
  i: integer;
  ok: boolean;
  sptr: pointer;
  regs: registers;

begin
  tsralreadyinstalled := false;
  thistsrinstalled := false;
  i := 0; ok := false;

  while (i<=255) and not ok do  { search each entry in the Interrupt }
  begin                         { vector table if it is a call to a JVFAX }
    getintvec(i,sptr);          { TSR }
    if ident(sptr^).id='DK8JV-JVFAX' then ok := true else inc(i);
  end;
  if ok then
  begin
    tsralreadyinstalled := true;
    with regs do
    begin             { if a TSR driver is installed, see if it is this }
      ah := 0;        { TSR by calling the initialize interface routine }
      intr(i,regs);   { and look what is it's name }
      installedtsrname := string(ptr(es,di)^);
      installedtsrno := i;
      thistsrinstalled := (tsrname = installedtsrname);
    end;
  end;
end;

{------------------------------------------------------------------------}

procedure uninstall;

{ uninstalls TSR using TSR function $FF, need not be changed normally }

var
  regs: registers;
begin
  writeln;
  writeln;
  write('+++ '+tsrname+': ');
  if not tsralreadyinstalled then
  begin
    writeln('TSR is not loaded so it cannot be uninstalled!');
  end else
    if not thistsrinstalled then
    begin
      writeln('cannot uninstall, another TSR '''+installedtsrname+'''');
      writeln('is present on IRQ ',hexstr(installedtsrno),
                                 'h  which is different from this one');
    end else
    begin
      with Regs do
      begin
        ah := $FF;
        intr(installedtsrno,regs);
      end;
      writeln('successfully removed from memory!');
    end;
  halt;
end;


{------------------------------------------------------------------------}

procedure install;

{ installs the TSR by setting the interrupt vector and making the
  program stay resident, need not be changed normally }

begin
  writeln;
  writeln;
  write('+++ '+tsrname+': ');
  if tsralreadyinstalled then
  begin
    if thistsrinstalled then writeln('is already installed on IRQ '
           ,hexstr(installedtsrno),'h!')
      else writeln('failure, another JVFAX-TSR is already installed on IRQ '
           ,hexstr(installedtsrno),'h!');
    halt;
  end;
  getintvec(oldintno,oldintvec);  {save old vector for deinstallation}
  setintvec(oldintno,@mainproc);  {and set pointer to our own routine}

  writeln('successfully installed on IRQ ',hexstr(oldintno),'h');

  swapvectors;                    {restore all vectors changed by the}
                                  {Borland Pascal run time library}
  keep(0);
end;

{************************************************************************}

{ the following routine should be changed according to your needs }


procedure checkforparameters;

{ you can change this routine to interpret the command line parameters
  that you need
  You can also use this routine to do some status display
}

begin
  if scanparameters('-P')<>'' then  { see if user wants another Port }
  begin
    val('$'+copy(scanparameters('-P'),3
                              ,length(scanparameters('-P'))-2),i,j);
    if j=0 then portaddr := i;
  end;

  writeln;
  writeln;
  writeln(paramstr(0)+': '+tsrname+' TSR interface driver for JVFAX 7.0');
  writeln('interfaces the Easyfax for RX and the PC speaker for TX');
  writeln;
  writeln('Possible command line parameters:');
  writeln;
  writeln('    -u       uninstall driver');
  writeln('    -pXXX    use port addr XXXh for the Easyfax (currently: '
                                                  ,hexstr(portaddr),'h.)');
  writeln('    -iXX     use IRQ XXh for communication with JVFAX '
                                   +'(currently: ',hexstr(oldintno),'h.)');
end;


{************************************************************************}


{ start of main program }

begin
  checkiftsrisinstalled;            { see if this or another JVFAX-TSR
                                      is already loaded }

{ the following command line parameters should always be available }

  if scanparameters('-U')='-U' then { try to uninstall if user wants to }
    uninstall;

  if scanparameters('-I')<>'' then  { see if user wants another IRQ }
  begin
    val('$'+copy(scanparameters('-I'),3,length(scanparameters('-I'))-2)
      ,i,j);
    if j=0 then oldintno := i;
  end;


  checkforparameters;               { this routine should interpret all
                                      other command line parameters
                                      that you want to establish for
                                      your interface driver }

  install;                          { now install driver! }


  writeln;                          { we should never reach here !!!! }
  writeln;
  writeln('+++Severe error! ,tsrname could not be installed!');
  setintvec(oldintno, oldintvec);

end.                                { that's all }